(****************************************************************************)
(*                 The Calculus of Inductive Constructions                  *)
(*                                                                          *)
(*                                Projet Coq                                *)
(*                                                                          *)
(*                     INRIA        LRI-CNRS        ENS-CNRS                *)
(*              Rocquencourt         Orsay          Lyon                    *)
(*                                                                          *)
(*                                 Coq V6.3                                 *)
(*                               July 1st 1999                              *)
(*                                                                          *)
(****************************************************************************)
(*                               discharge.ml                               *)
(****************************************************************************)

open Std;;
open Pp;;
open More_util;;
open Vectops;;
open Libobject;;
open Library;;
open Names;;
open Impuniv;;
open Generic;;
open Term;;
open Printer;;
open Constrtypes;;
open Constraintab;;

open Environ;;
open Termenv;;
open Dischcore;;
open Variables;;
open Constants;;
open Indtypes;;
open Declare;;
open Class;;
open Classops;;
open Pretty;;
open Recordops;;
open Recordobj;;

(* C3| Section closing *)

let make_expands sp cmap =
    List.map (fun (k,_) -> (Const(coerce_path k sp),DO_REPLACE)) cmap
;;

(* Why CCI and not k ? *)
let recalc_sp sp =
  let (_,spid,k) = repr_path sp in
  Lib.make_path CCI spid
;;

let process_object sec_sp (ids_to_discard,work_alist) (sp,lobj) =
let tag = object_tag lobj
in match tag with

    "VARIABLE" ->
    let (id,(a,inf),stre,impl,sticky,u) = outVariable lobj in
    if stre = (DischargeAt sec_sp) or ids_to_discard <> [] then
      (add_constraints_object (sp,u);
       (id::ids_to_discard,work_alist))
    else
      let (u',(expmod_a,expmod_inf)) =
        with_universes 
          (fun (a',inf') ->
	     (expmod_type work_alist a',
	      option_app (fun(x,y)->(expmod_type work_alist x,y)) inf'))
	  (Lib.make_path OBJ id,u,(a,inf)) in
      let obj' = inVariable(id,(expmod_a,expmod_inf),stre,impl,sticky,u')
         in add_named_object (id,OBJ) obj';
            (ids_to_discard,work_alist)

  | "CONSTANT" ->
    let (cmap,stre,u) = outConstant lobj in
    if stre = (DischargeAt sec_sp) then
      (add_constraints_object (sp,u);
       (ids_to_discard,(make_expands sp cmap)@work_alist))
    else let ((_,spid,spk)) = repr_path sp in
         let newsp = Lib.make_path CCI spid in
         let (u',(mods,cmap')) =
            with_universes (process_recipe_cmap (sp,newsp)
                              (ids_to_discard,work_alist)) (newsp,u,cmap) in
         let obj' = inConstant(cmap',stre,u')
         in add_named_object (spid,OBJ) obj';
            List.iter (fun (k,cb) ->
                         if not cb.cONSTOPAQUE then
                           set_transparent_sp (coerce_path k newsp)) cmap;
            (ids_to_discard,mods@work_alist)

  | "MUTUALINDUCTIVE" ->
    let (mimap,u) = outMutualInductive lobj in
    let ((_,spid,spk)) = repr_path sp in
    let newsp = Lib.make_path CCI spid in
    let (u',(mods,mimap')) =
      with_universes
	(process_mimap (sp,newsp) (ids_to_discard,work_alist))
 	(newsp,u,mimap) in
    let obj' = inMutualInductive (mimap',u')
    in add_named_object (spid,OBJ) obj';
        (ids_to_discard,mods@work_alist)

  | "UNIVERSES" ->
      (add_anonymous_object lobj;
       (ids_to_discard,work_alist))

  | "CLASS" -> let ((cl,clinfo) as x) = outClass lobj in
               if clinfo.cL_STRE = (DischargeAt sec_sp) 
               then (ids_to_discard,work_alist)
               else (let y = process_class sec_sp x in
                     (add_anonymous_object (inClass y);
                      (ids_to_discard,work_alist)))
  | "COERCION" -> let (((_,coeinfo),_,_)as x) = outCoercion lobj in
               if coeinfo.cOE_STRE = (DischargeAt sec_sp) 
               then (ids_to_discard,work_alist)
               else (let ((_,_,clt) as y),idf,ps = process_coercion sec_sp x in
                     (add_anonymous_object (inCoercion y); 
                      coercion_syntax idf ps clt;
                      (ids_to_discard,work_alist)))
                    
  | "STRUCTURE" ->
      let (sp,info) = outStruc lobj in
      let newsp = recalc_sp sp in
      let (_,mib) = mind_of_path (ccisp_of newsp) in
      let strobj =
	{ s_CONST= info.s_CONST;
	  s_PARAM= mib.mINDNPARAMS;
	  s_PROJ= List.map (option_app recalc_sp) info.s_PROJ} in
      (add_anonymous_object (inStruc (newsp,strobj)));
      (ids_to_discard,work_alist)
  | "OBJDEF1" -> let sp = outObjDef1 lobj in
                 let ((_,spid,_)) = repr_path sp in
                 (try objdef_declare spid with
                    _ -> ());
                 (ids_to_discard,work_alist)

  | _ -> (ids_to_discard,work_alist)
;;

let process_item sec_sp acc = function
    (sp,Lib.LEAF lobj) -> process_object sec_sp acc (sp,lobj)
  | (_,Lib.Import(s,isp,locally)) ->
      (Library.import_export_module s locally;acc)
  | (_,_) -> acc
;;

let update sec_sp decls =
let (ids_to_discard,work_alist) =
    List.fold_left (process_item sec_sp) ([],[]) decls in
let modified = map_succeed
    (function (_,(DO_ABSTRACT(_,[]))) -> failwith "caught"
       | (Const sp,(DO_ABSTRACT(_,l))) -> Const sp
       | (MutInd (x_0,x_1),(DO_ABSTRACT(_,l))) -> MutInd (x_0,x_1)
       | _ -> failwith "caught") work_alist
in (ids_to_discard,modified)
;;

let pop_path s sp =
  let (sl,id,k) = repr_path sp in
  if sl <> [] & s = (List.hd sl) then (make_path (List.tl sl) id k)
    else anomaly "pop_path"
;;

(* we now combine everything into a close_section function *)
(* eliminates all variables and constants local to the section *)

(* Only displays modified terms in CCI *)

let display_modified secid (discarded,modified) =
  let disp_sp sp =
    if (kind_of_path sp) = CCI then
      let (_,cb) = const_of_path sp in
        (hOV 0 
           [< 'sTR"Constant "; 'sTR(print_basename sp); 'bRK(1,2);
              'sTR": " ; prtype cb.cONSTTYPE; 'fNL >])
    else [<>] in
  let disp_induc sp tyi =
    if (kind_of_path sp) = CCI then
      let (_,mib) = mind_of_path sp in 
      let nameind = mib.mINDPACKETS.(tyi).mINDTYPENAME in
        [< 'sTR "Type "; 
           'sTR(print_basename_mind sp nameind) ; 'bRK(1,2);
           'sTR": " ; prterm mib.mINDPACKETS.(tyi).mINDARITY.body; 'fNL >]
    else [<>]
  in
    mSG (prlist (function 
		     (Const sp) -> disp_sp (pop_path secid sp)
                   | (MutInd (sp,tyi)) -> disp_induc (pop_path secid sp) tyi
		   | _ -> assert false)
           (List.rev modified))
;;

let close_section verbosely str sp sec_ctxt =
  Lib.close_dir sp {Lib.exports=[]; Lib.export_fn = None};
  backtrack_caches();
  let kept = update sp (List.rev sec_ctxt) in
  if verbosely then display_modified str kept
;;

(* Unsafe: does not check if a proof is still in progress... *)
let close_section_or_module verbosely s =
  let sp =
    try Library.find_dir (Some s)
    with Library.Unknown_dir -> error ("No open section or module "^s)
  in
  match Lib.map sp with
      Lib.OpenDir(_,{Lib.module_p = is_module}) ->
        let last_open = Library.find_dir None in
        (if last_open <> sp then
          let name = string_of_id (basename last_open) in
          if is_module then error (name^" must be closed in module "^s)
             else error (name^" must be closed first"));
        let contents = Lib.contents_after (Some sp) in
        if is_module
          then close_module sp contents
          else close_section verbosely s sp contents
    | _ -> anomaly "close_section_or_module: not a module"
;;

(* $Id: discharge.ml,v 1.22 1999/10/29 23:19:05 barras Exp $ *)
