/* readclass.c -- parse and check structure of *.class files
 * Written by Charles Briscoe-Smith; refer to the file LEGAL for details.
 */

/* Interface definitions */

/* Load in a class file, check the structure and return true if all is
   okay. */
extern int readclass(const char filename[]);


#ifndef SEEN_readclass_h

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <string.h>
#include "types.h"

static u1
readu1(FILE *fp)
{
  return getc(fp);
}

static u2
readu2(FILE *fp)
{
  int c1, c2;

  c1=getc(fp);
  c2=getc(fp);
  return c1<<8 | c2;
}

static u4
readu4(FILE *fp)
{
  int c1, c2, c3, c4;

  c1=getc(fp);
  c2=getc(fp);
  c3=getc(fp);
  c4=getc(fp);
  return c1<<24 | c2<<16 | c3<<8 | c4;
}

#define error(str, a1) \
{ \
  fprintf(stderr, "%s: Error: " str "\n", filename, a1); \
  return 1; \
}

u2
readconst(FILE *fp, cp_info *buf)
{
  int len;
  int i;
  u2 tag;
  static int cont_flag=0;
  static u4 cont_val;

  if (cont_flag) {
    cont_flag=0;
    buf->u4=cont_val;
  }
  tag=readu1(fp);
  switch (tag) {
  case CONSTANT_Class:
  case CONSTANT_String:
    buf->u2=readu2(fp);
    break;
  case CONSTANT_Fieldref:
  case CONSTANT_Methodref:
  case CONSTANT_InterfaceMethodref:
  case CONSTANT_NameAndType:
    buf->u2x2.i1=readu2(fp);
    buf->u2x2.i2=readu2(fp);
    break;
  case CONSTANT_Integer:
  case CONSTANT_Float:
    buf->u4=readu4(fp);
    break;
  case CONSTANT_Long:
  case CONSTANT_Double:
    buf->u4=readu4(fp);
    cont_val=readu4(fp);
    cont_flag=1;
    break;
  case CONSTANT_Utf8:
    len=readu2(fp);
    if (len>0) {
      buf->utf8=(u1*) malloc(len+1);
      buf->utf8[len]=0;
    }
    for (i=0; i<len; i++) {
      buf->utf8[i]=readu1(fp);
    }
    break;
  default:
    fprintf(stderr, "invalid constant tag %d\n", tag);
  }
  return tag;
}

int
readclass(const char filename[])
{
  FILE *fp;
  u4 in4;
  u2 minor, major;
  u2 const_count, iface_count, field_count, meth_count, attr_count;
  int i1;
  cp_info *cp;
  u2 access, this, super;
  u2 *ifaces;
  u2 *tag;

  printf("\nClass file %s\n\n", filename);
  fp=fopen(filename, "rb");
  if (fp == 0) error("can't open: %s", strerror(errno));
  in4=readu4(fp);
  if (in4 != 0xCAFEBABE) error("invalid magic number 0x%08lX", in4);
  printf("Magic: 0x%08lX\n", in4);
  minor=readu2(fp);
  major=readu2(fp);
  if (major != 45) error("major version %d not supported", major);
  printf("Major version: %d\n", major);
  if (minor > 3) error("minor version %d not supported", minor);
  printf("Minor version: %d\n", minor);
  const_count=readu2(fp);
  if (const_count == 0) error("constant count is %d", const_count);
  printf("Number of constants: %d\n", const_count);
  cp=(cp_info*) calloc(const_count, sizeof(cp_info));
  tag=(u2*) calloc(const_count, sizeof(u2));
  for (i1=1; i1<const_count; i1++) {
    tag[i1]=readconst(fp, &cp[i1]);
    if (tag[i1]==CONSTANT_Long
       || tag[i1]==CONSTANT_Double) {
      i1++;
    }
  }
  for (i1=0; i1<const_count; i1++) {
    u4f u4f;
    u8d u8d;
    printf("Constant %d: ", i1);
    if (i1==0) printf("reserved\n");
    else if (tag[i1-1]==CONSTANT_Long
	     || tag[i1-1]==CONSTANT_Double) printf("continuation word\n");
    else switch (tag[i1]) {
    case CONSTANT_Class:
      printf("Class: %d (%s)\n", cp[i1].u2,
	     cp[cp[i1].u2].utf8);
      break;
    case CONSTANT_Fieldref:
      printf("Fieldref: %d, %d (%s, %s, %s)\n",
             cp[i1].u2x2.i1,
             cp[i1].u2x2.i2,
	     cp[cp[cp[i1].u2x2.i1].u2].utf8,
	     cp[cp[cp[i1].u2x2.i2].u2x2.i1].utf8,
	     cp[cp[cp[i1].u2x2.i2].u2x2.i2].utf8);
      break;
    case CONSTANT_Methodref:
      printf("Methodref: %d, %d (%s, %s, %s)\n",
             cp[i1].u2x2.i1,
	     cp[i1].u2x2.i2,
	     cp[cp[cp[i1].u2x2.i1].u2].utf8,
	     cp[cp[cp[i1].u2x2.i2].u2x2.i1].utf8,
	     cp[cp[cp[i1].u2x2.i2].u2x2.i2].utf8);
      break;
    case CONSTANT_InterfaceMethodref:
      printf("InterfaceMethodref: %d, %d (%s, %s, %s)\n",
             cp[i1].u2x2.i1,
             cp[i1].u2x2.i2,
	     cp[cp[cp[i1].u2x2.i1].u2].utf8,
	     cp[cp[cp[i1].u2x2.i2].u2x2.i1].utf8,
	     cp[cp[cp[i1].u2x2.i2].u2x2.i2].utf8);
      break;
    case CONSTANT_String:
      printf("String: %d (\"%s\")\n", cp[i1].u2,
	     cp[cp[i1].u2].utf8);
      break;
    case CONSTANT_Integer:
      printf("Integer: %ld\n", cp[i1].u4);
      break;
    case CONSTANT_Float:
      u4f.u=cp[i1].u4;
      printf("Float: %g\n", u4f.f);
      break;
    case CONSTANT_Long:
      printf("Long: %qd\n", ((u8) cp[i1].u4 << 32) + cp[i1+1].u4);
      break;
    case CONSTANT_Double:
      u8d.u=((u8) cp[i1].u4 << 32) + cp[i1+1].u4;
      printf("Double: %g\n", u8d.d);
      break;
    case CONSTANT_NameAndType:
      printf("NameAndType: %d, %d\n", cp[i1].u2x2.i1, cp[i1].u2x2.i2);
      break;
    case CONSTANT_Utf8:
      printf("Utf8: \"%s\"\n", cp[i1].utf8);
      break;
    }
  }

  access=readu2(fp);
  printf("Access flags:%s%s%s%s%s\n",
         access & 0x0001 ? " public" : "",
	 access & 0x0010 ? " final" : "",
	 access & 0x0400 ? " abstract" : "",
	 access & 0x0200 ? " interface" : " class",
	 access & 0x0020 ? "" : " (without super flag; generated by old Sun compiler)");
  this=readu2(fp);
  printf("This class: %d\n", this);
  super=readu2(fp);
  printf("Superclass: %d\n", super);
  iface_count=readu2(fp);
  printf("Number of interfaces: %d:", iface_count);
  ifaces=(u2*) calloc(iface_count, sizeof(u2));
  for (i1=0; i1<iface_count; i1++) {
    ifaces[i1]=readu2(fp);
    printf(" %d", ifaces[i1]);
  }
  field_count=readu2(fp);
  printf("\nNumber of fields: %d\n", field_count);
  meth_count=readu2(fp);
  printf("Number of methods: %d\n", meth_count);
  attr_count=readu2(fp);
  printf("Number of attributes: %d\n", attr_count);
  if (feof(fp)) {
    error("%s", "premature end of file");
  }
  return 0;
}

#endif /* SEEN_readclass_h */

