/* install-mbr for installing a master boot record
   Copyright (C) 1999 Neil Turton

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

See file COPYING for details */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <getopt.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <errno.h>
#include <assert.h>
typedef unsigned char uchar;
#include "mbr.h"

#define BUFFER_SIZE 512
#define CODE_SIZE (512-66)
#define PTN_TABLE_OFFSET CODE_SIZE
#define PTN_TABLE_SIZE 66
#define SIG_PTR_OFFSET (CODE_SIZE-2)

#define DEFAULT_TIMEOUT 18

const char *prog_name="install-mbr";

struct change_params {
  uchar flags;
#define CPFLAG_DRIVE         1
#define CPFLAG_DEFAULT       2
#define CPFLAG_TIMEOUT_VALID 4
  /* The following bit is used to solve conflicts between "-t <timeout>"
   * and "-i a".  It is set by "-t <timeout>" and cleared by "-i a".
   * If it is set, "-i a" is ignored. */
#define CPFLAG_TIMEOUT       8
#define CPFLAG_ENABLED      16
#define CPFLAG_INTERRUPT    32
  uchar drive;
  uchar deflt;
  uchar timeout;
  uchar enabled, enabled_mask;
#define CPE_1      ((uchar)1)
#define CPE_2      ((uchar)2)
#define CPE_3      ((uchar)4)
#define CPE_4      ((uchar)8)
#define CPE_F      ((uchar)16)
#define CPE_A      ((uchar)128)
  uchar interrupt, interrupt_mask;
#define CPI_SHIFT  1
#define CPI_KEY   2
#define CPI_ALWAYS 4
#define CPI_MASK   (CPI_SHIFT|CPI_KEY|CPI_ALWAYS)
};

struct desc {
  uchar *data;
  char *path;
  int flags;
#define DESC_FLAG_FREE_DATA 1 /* Should we free the data? */
#define DESC_FLAG_SPECIFIED 2
};

struct data {
  struct desc install, params, target, table;
  unsigned int offset;
  int flags;
#define DFLAG_KEEP     1
#define DFLAG_RESET    2
#define DFLAG_LIST     4
#define DFLAG_NO_ACT   8
#define DFLAG_VERBOSE 16
#define DFLAG_FORCE   32
  struct change_params changes;
};

/* Version specific routines *********************************************/

static struct mbr_params *
find_params(const struct desc *dsc, int quiet)
{
  uchar *mbr=dsc->data;
  uint ptr;
  struct mbr_params *params;
  uint ver_compat;

#if 0
  fprintf(stderr, "Looking for params at %p\n", mbr);
#endif

  ptr=mbr[SIG_PTR_OFFSET]+256*mbr[SIG_PTR_OFFSET+1];
  if(ptr>SIG_PTR_OFFSET-sizeof(struct mbr_params) ||
     memcmp(mbr+ptr, MP_SIG, sizeof(params->sig))!=0)
  {
    /* Due to a silly little mistake, version 1.0.0 got released with
       the signature pointer uninitialised.  This means that everyone
       has a different pointer, but the signature in this version is
       at 0x13b */
    ptr=0x13b;
    /* Parameters not found. */
    if(memcmp(mbr+ptr, MP_SIG, sizeof(MP_SIG)))
    {
      if(quiet)
	return NULL;

      fprintf(stderr, "%s:%s: Failed to find MBR signature.\n",
	      prog_name, dsc->path);
      exit(1);
    }
  }
  params=(struct mbr_params *)(mbr+ptr);
  ver_compat=params->ver_compat[0]+256*params->ver_compat[1];
  if(ver_compat > MP_VERSION)
  {
    uint version=params->version[0]+256*params->version[1];

    if(quiet)
      return NULL;

    fprintf(stderr, "%s:%s: Cannot handle MBR version %x "
	    "(backwards compatible to %x)\n",
	    prog_name, dsc->path, version, ver_compat);
    exit(1);
  }
  return params;
}

static void
modify_params(struct desc *dsc, struct change_params *cpp)
{
  struct mbr_params *params=find_params(dsc, 0);
  assert(MP_VERSION==0);

  if(cpp->flags & CPFLAG_DRIVE)
    params->drive=cpp->drive;

  if(cpp->flags & CPFLAG_DEFAULT)
  {
    params->deflt&=~MP_DEFLT_BITS;
    switch(cpp->deflt)
    {
    case '1': params->deflt|=0; break;
    case '2': params->deflt|=1; break;
    case '3': params->deflt|=2; break;
    case '4': params->deflt|=3; break;
    case 'F': case 'f': params->deflt|=4; break;
    case 'D': case 'd': params->deflt|=7; break;

    default:
      fprintf(stderr, "%s: Invalid default partition: %c\n",
	      prog_name, cpp->deflt);
      exit(1);
    };
  }
  /* The interaction of the timeout with the interrupt always bit is a
     little subtle.
     CPIA_MASK CPIA TIMEOUT TIMEOUT_VAL comment
     X         1    0       X           Set always
     X         X    1       (1)         Set timeout
     X         0    X       1           Set timeout
     0         0    0       0           Set default timeout
     1         0    0       0           Nothing set - do nothing
  */

  if((cpp->interrupt & CPI_ALWAYS) != 0 /* If setting CPI_ALWAYS */
     && (cpp->flags & CPFLAG_TIMEOUT) == 0) /* and not overridden */
    params->delay=0xff; /* Set to always interrupt */
  else if(cpp->flags & CPFLAG_TIMEOUT_VALID)
    params->delay=cpp->timeout;
  /* Timeout is not valid and not setting CPI_ALWAYS, so the only
     action we can be taking is clearing CPI_ALWAYS */
  else if((cpp->interrupt_mask & CPI_ALWAYS)==0)
    params->delay=DEFAULT_TIMEOUT;

  if(cpp->interrupt & CPI_SHIFT)
    params->deflt|=MP_DEFLT_ISHIFT;
  else if((cpp->interrupt_mask & CPI_SHIFT)==0)
    params->deflt&=~MP_DEFLT_ISHIFT;

  if(cpp->interrupt & CPI_KEY)
    params->deflt|=MP_DEFLT_IKEY;
  else if((cpp->interrupt_mask & CPI_KEY)==0)
    params->deflt&=~MP_DEFLT_IKEY;

  assert(MP_FLAG_EN1 == CPE_1);
  assert(MP_FLAG_EN2 == CPE_2);
  assert(MP_FLAG_EN3 == CPE_3);
  assert(MP_FLAG_EN4 == CPE_4);
  assert(MP_FLAG_ENF == CPE_F);
  assert(MP_FLAG_ENA == CPE_A);
  params->flags=(params->flags & cpp->enabled_mask)|cpp->enabled;
}

/* Fetch the parameters from src to cp.  If quiet is set, return true
   on failure, instead of calling exit(1). */
static int
fetch_params(struct change_params *cpp, const struct desc *src, int quiet)
{
  const struct mbr_params *params=find_params(src, quiet);
  assert(MP_VERSION==0);
  if(params==NULL)
    return 1;
  
  cpp->flags=CPFLAG_DRIVE|CPFLAG_DEFAULT|CPFLAG_ENABLED|CPFLAG_INTERRUPT;

  cpp->drive=params->drive;
  switch(params->deflt & MP_DEFLT_BITS)
  {
  case 0: cpp->deflt='1'; break;
  case 1: cpp->deflt='2'; break;
  case 2: cpp->deflt='3'; break;
  case 3: cpp->deflt='4'; break;
  case 4: cpp->deflt='F'; break;
  case 7: cpp->deflt='D'; break;
  default:
      fprintf(stderr, "%s: Invalid default partition found: %d\n",
	      prog_name, params->deflt);
      exit(1);
  };

  if(params->delay == 0xff)
  {
    cpp->interrupt=CPI_ALWAYS;
  }
  else
  {
    cpp->timeout=params->delay;
    cpp->flags|=CPFLAG_TIMEOUT_VALID|CPFLAG_TIMEOUT;
    cpp->interrupt=0;
  }
  cpp->interrupt_mask=~(CPI_ALWAYS|CPI_SHIFT|CPI_KEY);
  if(params->deflt & MP_DEFLT_ISHIFT)
    cpp->interrupt|=CPI_SHIFT;
  if(params->deflt & MP_DEFLT_IKEY)
    cpp->interrupt|=CPI_KEY;

  assert(MP_FLAG_EN1 == CPE_1);
  assert(MP_FLAG_EN2 == CPE_2);
  assert(MP_FLAG_EN3 == CPE_3);
  assert(MP_FLAG_EN4 == CPE_4);
  assert(MP_FLAG_ENF == CPE_F);
  assert(MP_FLAG_ENA == CPE_A);
  cpp->enabled=params->flags;
  cpp->enabled_mask=(uchar)~(uchar)(CPE_1|CPE_2|CPE_3|CPE_4|CPE_F|CPE_A);
  return 0;
}

/* Data handling *********************************************************/

static void
show_mbr(struct desc *dsc)
{
  struct change_params cp;
  fetch_params(&cp, dsc, 0);
  if(cp.flags & CPFLAG_DRIVE)
    printf("Drive: 0x%x\n", cp.drive);
  if(cp.flags & CPFLAG_DEFAULT)
    printf("Default: %c\n", cp.deflt);
  if(cp.flags & CPFLAG_TIMEOUT)
    printf("Timeout: %d/18 seconds\n", cp.timeout);
  if(cp.flags & CPFLAG_ENABLED)
  {
    printf("Enabled:");
    if(cp.enabled & CPE_1)
      printf(" 1");
    if(cp.enabled & CPE_2)
      printf(" 2");
    if(cp.enabled & CPE_3)
      printf(" 3");
    if(cp.enabled & CPE_4)
      printf(" 4");
    if(cp.enabled & CPE_F)
      printf(" F");
    if(cp.enabled & CPE_A)
      printf(" A");
      printf("\n");
  }
  if(cp.flags & CPFLAG_INTERRUPT)
  {
    printf("Interrupt:");
    if(cp.interrupt & CPI_SHIFT)
      printf(" Shift");
    if(cp.interrupt & CPI_KEY)
      printf(" Key");
    if(cp.interrupt & CPI_ALWAYS)
      printf(" Always");
    printf("\n");
  }
}

/* Copy parameters from src to dest.  Don't complain if quiet is
   set.  Return true if copy failed. */
static int
copy_params(struct desc *dest, struct desc *src, int quiet)
{
  struct change_params cp;

#if 0
  fprintf(stderr, "Fetching parameters from %s\n", src->path);
#endif

  /* Do the copying via the version-neutral "change_params" structure */
  if(fetch_params(&cp, src, quiet))
    return 1;

#if 0
  fprintf(stderr, "Writing parameters to %s\n", dest->path);
#endif
  modify_params(dest, &cp);
  return 0;
}

static void
copy_code(struct desc *dest, const struct desc *src)
{
  memcpy(dest->data, src->data, CODE_SIZE);
  /* Any errors should be reported against the source. */
  dest->path = src->path;
}

/* Copy the partition table */
static void
copy_table(struct desc *dest, const struct desc *src)
{
  memcpy(dest->data+PTN_TABLE_OFFSET, src->data+PTN_TABLE_OFFSET,
	 PTN_TABLE_SIZE);
}

/* IO routines ***********************************************************/

typedef ssize_t io_func(int, void *, size_t);

static size_t
do_io(io_func *fn, const struct desc *dsc, int fd, size_t min_size)
{
  int res;
  int done=0;
  while(done<BUFFER_SIZE)
  {
    res=fn(fd, dsc->data, BUFFER_SIZE-done);
    if(res<0 && errno!=EINTR)
    {
      fprintf(stderr, "%s: Error reading %s: %s\n", prog_name,
	      dsc->path, strerror(errno));
      exit(1);
    }
    if(res==0)
    {
      if(done>=min_size)
	return done;
      fprintf(stderr, "%s: Unexpected EOF while reading %s\n",
	      prog_name, dsc->path);
      exit(1);
    }
    done+=res;
  }
  return done;
}

static size_t
read_from_fd(const struct desc *dsc, int fd, size_t min_size)
{
  return do_io(read, dsc, fd, min_size);
}

/* Bah.  The prototype for write is different from that for read by
   a const.  Surely it is compatible, but ANSI C say no. */
static ssize_t
my_write(int fd, void *buf, size_t size)
{
  return write(fd, buf, size);
}

static size_t
write_to_fd(const struct desc *dsc, int fd, size_t min_size)
{
  return do_io(my_write, dsc, fd, min_size);
}

static void
read_file(struct desc *dsc)
{
  int fd=open(dsc->path, O_RDONLY|O_NOCTTY);
  if(fd==-1)
  {
    fprintf(stderr, "%s: Failed to open %s: %s\n",
	    prog_name, dsc->path, strerror(errno));
    exit(1);
  }
  if(dsc->flags & DESC_FLAG_FREE_DATA)
    free(dsc->data);
  dsc->data=malloc(512);
  if(dsc->data==NULL)
  {
    fprintf(stderr, "%s: Malloc failed\n", prog_name);
    exit(1);
  }
  dsc->flags|=DESC_FLAG_FREE_DATA;
  read_from_fd(dsc, fd, BUFFER_SIZE);
  if(close(fd)==-1)
  {
    fprintf(stderr, "%s: Error during close of %s: %s\n",
	    prog_name, dsc->path, strerror(errno));
    exit(1);
  }
}

/* The install code ******************************************************/

static void
install_mbr(struct data *dp)
{
  uchar *code, *params;
  int mbr_fd = -1;
  size_t mbr_size;
  uchar mbr_in[BUFFER_SIZE];
  uchar mbr_out[BUFFER_SIZE];
  struct desc target_orig={mbr_in, dp->target.path, 0};

  dp->target.data=mbr_out;

  /* Read the code file if specified.  Otherwise, use the
     built-in version. */
  if(dp->install.flags & DESC_FLAG_SPECIFIED)
    read_file(&(dp->install));
  else
  {
    dp->install.path="<internal>";
    dp->install.data=MBR_START;
    dp->install.flags=0;
  }

  /* Read the parameter file if specified. */
  if(dp->params.flags & DESC_FLAG_SPECIFIED)
    read_file(&(dp->params));
  else
  {
    dp->params.path=dp->install.path;
    dp->params.data=dp->install.data;
    dp->params.flags=0;
  }

  /* Start with an empty record. */
  memset(mbr_out, 0, sizeof(mbr_out));

  /* Open the target file/device. */
  mbr_fd=open(dp->target.path,
	      ((dp->flags & DFLAG_NO_ACT)
	       ? (O_RDONLY|O_NOCTTY)
	       : (O_RDWR|O_NOCTTY))
	      | ((dp->flags & DFLAG_FORCE) ? O_CREAT : 0),
	      S_IRUSR|S_IWUSR|S_IRGRP|S_IWGRP|S_IROTH|S_IWOTH);

  if(mbr_fd==-1)
  {
    fprintf(stderr, "%s: Failed to open %s: %s\n",
	    prog_name, dp->target.path, strerror(errno));
    exit(1);
  }

  /* Read the old MBR in.  This contains at least the partition
     table. */
  if(lseek(mbr_fd, dp->offset, SEEK_SET)==-1)
  {
    fprintf(stderr, "%s:%s: Failed to seek: %s\n",
	    prog_name, dp->target.path, strerror(errno));
    exit(1);
  }
  read_from_fd(&target_orig, mbr_fd,
	       dp->flags & DFLAG_FORCE ? 0 : BUFFER_SIZE);
  /* Check the signature */
  if ((dp->flags & DFLAG_FORCE) == 0 &&
      (mbr_in[BUFFER_SIZE-2] != 0x55 ||
       mbr_in[BUFFER_SIZE-1] != 0xAA))
  {
    fprintf(stderr, "%s:%s: No boot signature found.  "
	    "Use --force to override.\n", prog_name,
	    dp->target.path);
    exit(1);
  }

  if(dp->table.flags & DESC_FLAG_SPECIFIED)
    read_file(&(dp->table));
  else
  {
    dp->table.data = target_orig.data;
    dp->table.path = target_orig.path;
    dp->table.flags = 0;
  }

  /* Copy the code unless the user asks not to. */
  if((dp->flags & DFLAG_KEEP)==0)
  {
    if(dp->flags & DFLAG_VERBOSE)
      fprintf(stderr, "Copying code from %s\n", dp->install.path);
    copy_code(&(dp->target), &(dp->install));
  } else {
    if(dp->flags & DFLAG_VERBOSE)
      fprintf(stderr, "Copying code from %s\n", &target_orig);
    copy_code(&(dp->target), &target_orig);
  }

  /* Install parameters from params if the user wants it, or 
     we fail to copy parameters from the input. */
  if((dp->params.flags & DESC_FLAG_SPECIFIED)!=0
     || ((dp->flags & DFLAG_RESET)==0 &&
	 copy_params(&(dp->target), &target_orig, 1)))
  {
    if(dp->flags & DFLAG_VERBOSE)
      fprintf(stderr,"Resetting parameters from %s\n",
	      dp->params.path);
    copy_params(&(dp->target), &(dp->params), 0);
  }
  
  if(dp->flags & DFLAG_VERBOSE)
    fprintf(stderr,"Modifying parameters.\n");

  /* Modify the parameters according to the change_params structure. */
  if(dp->changes.flags)
    modify_params(&(dp->target), &(dp->changes));

  /* Copy the partition table. */
  copy_table(&(dp->target), &(dp->table));

  if(dp->flags & DFLAG_LIST)
  {
    show_mbr(&(dp->target));
  }
  mbr_out[BUFFER_SIZE-2]=0x55;
  mbr_out[BUFFER_SIZE-1]=0xAA;

  if((dp->flags & DFLAG_NO_ACT) == 0)
  {
    /* Write the result out. */
    if(lseek(mbr_fd, dp->offset, SEEK_SET)==-1)
    {
      fprintf(stderr, "%s:%s: Failed to seek: %s\n",
	      prog_name, dp->target.path, strerror(errno));
      exit(1);
    }
    write_to_fd(&(dp->target), mbr_fd, BUFFER_SIZE);
  };

  /* Close the target */
  if(close(mbr_fd))
  {
    fprintf(stderr, "%s: Error while closing %s: %s\n",
	    prog_name, dp->target.path, strerror(errno));
    exit(1);
  };
}

/* Utility function ******************************************************/

static unsigned int
get_uint(const char *str, const char *name)
{
  char *end_ptr;
  unsigned int result;
  if(str!=NULL && str[0]!=0)
  {
    result=(int)strtoul(str, &end_ptr, 0);
    if(*end_ptr==0)
      return result;
  }
  fprintf(stderr, "%s: %s must be a number: %s\n",
	  prog_name, name, str);
  exit(1);
}

/* Option processing *****************************************************/

static void
show_usage(FILE *f)
{
  fprintf(f, "Usage: %s [options] <target>\n", prog_name);
}

static void
suggest_help(void)
{
  fprintf(stderr, "Try '%s --help' for more information.\n",
	  prog_name);
}

static void
show_help(void)
{
  show_usage(stdout);
  printf("Options:\n"
	 "  -f, --force                       "
	 "Override some sanity checks.\n"
	 "  -I <path>, --install <path>       "
	 "Install code from the specified file.\n"
	 "  -k, --keep                        "
	 "Keep the current code in the MBR.\n"
	 "  -l, --list                        "
	 "Just list the parameters.\n"
	 "  -n, --no-act                      "
	 "Don't install anything.\n"
	 "  -o <offset>, --offset <offset>    "
	 "Install the MBR at byte offset <offset>.\n"
	 "  -P <path>, --parameters <path>    "
	 "Get parameters from <path>.\n"
	 "  -r, --reset                       "
	 "Reset the parameters to the default state.\n"
	 "  -T <path>, --table <path>         "
	 "Get partition table from <path>.\n"
	 "  -v, --verbose                     "
	 "Operate verbosely.\n"
	 "  -V, --version                     "
	 "Show version.\n"
	 "  -h, --help                        "
	 "Display this message.\n"
	 "Parameters:\n"
	 "  -d <drive>, --drive <drive>       "
	 "Set BIOS drive number.\n"
	 "  -e <option>, --enable <option>    "
	 "Select enabled boot option.\n"
	 "  -i <mode>, --interrupt <keys>     "
	 "Set interrupt mode. (a/c/s/cs/n)\n"
	 "  -p <partn>, --partition <partn>   "
	 "Set boot partition (0=whole disk).\n"
	 "  -t <timeout>, --timeout <timeout> "
	 "Set the timeout in 1/18 second.\n"
	 "Interrupt modes:\n"
	 "  's'=Interrupt if shift or ctrl is pressed.\n"
	 "  'k'=Interrupt if other key pressed.\n"
	 "  'a'=Interrupt always.\n"
	 "  'n'=Interrupt never.\n"
	 "Boot options:\n"
	 "  '1','2','3' or '4' - Partition 1,2,3 or 4.\n"
	 "  'F' - 1st floppy drive.\n"
	 "  'A' - Advanced mode.\n"
	 "Report bugs to neilt@chiark.greenend.org.uk\n"
	 );
}

static void
parse_enabled(struct change_params *params, const char *arg)
{
  char mode='+';
  if((params->flags & CPFLAG_ENABLED)==0)
  {
    /* The first time round, default to override */
    mode='=';
    params->flags|=CPFLAG_ENABLED;
  }

  while(*arg)
  {
    char c=*arg++;
    uchar bit=0;
    switch(c)
    {
    case '=':
      params->enabled_mask=0;
      params->enabled=0;
      /* Fall through */
    case '+': case '-':
      mode=c;
      /* Skip to the next iteration round the loop */
      continue;

    case '1': bit=CPE_1; break;
    case '2': bit=CPE_2; break;
    case '3': bit=CPE_3; break;
    case '4': bit=CPE_4; break;
    case 'F': case 'f': bit=CPE_F; break;
    case 'A': case 'a': bit=CPE_A; break;

    default:
      fprintf(stderr, "%s: Invalid boot option: %c\n", prog_name, c);
      exit(1);
    }
    if(mode=='=')
      params->enabled_mask=0;
    else
      params->enabled_mask&=~bit;

    if(mode=='-')
      params->enabled&=~bit;
    else
      params->enabled|=bit;
  }
}

static void
parse_interrupt(struct change_params *params, const char *arg)
{
  char mode='+';
  if((params->flags & CPFLAG_INTERRUPT)==0)
  {
    /* The first time round, default to override */
    mode='=';
    params->flags|=CPFLAG_INTERRUPT;
  }
  while(*arg)
  {
    char c=*arg++;
    uchar bit=0;
    switch(c)
    {
    case '=':
      params->interrupt_mask=0;
      params->interrupt=0;
      /* Fall through */
    case '+': case '-':
      mode=c;
      /* Skip to the next iteration round the loop */
      continue;

    case 's': case 'S': bit=CPI_SHIFT; break;
    case 'k': case 'K': bit=CPI_KEY; break;
    case 'a': case 'A': bit=CPI_ALWAYS;
      if(mode!='-') /* Don't enforce the timeout */
	params->flags&=~CPFLAG_TIMEOUT;
      break;
    case 'n': case 'N':
      params->interrupt_mask&=~CPI_MASK;
      if(mode=='-') {
	params->interrupt|=CPI_MASK;
      } else {
	params->interrupt&=~CPI_MASK;
      };
      continue;

    default:
      fprintf(stderr, "%s: Invalid interrupt option: %c\n", prog_name, c);
      exit(1);
    }
    if(mode=='=')
      params->interrupt_mask=0;
    else
      params->interrupt_mask&=~bit;

    if(mode=='-')
      params->interrupt&=~bit;
    else
      params->interrupt|=bit;
  }
}

static void
parse_options(struct data *my_data, int argc, char *argv[])
{
  int c;
  char to_do=0;
  static struct option long_options[] =
  {
    /* {name, has_arg, flag, val} */
    {"force",      0, 0, 'f'},
    {"install",    1, 0, 'I'},
    {"keep",       0, 0, 'k'},
    {"list",       0, 0, 'l'},
    {"no-act",     0, 0, 'n'},
    {"offset",     1, 0, 'o'},
    {"parameters", 1, 0, 'P'},
    {"reset",      0, 0, 'r'},
    {"table",      1, 0, 'T'},
    {"verbose",    0, 0, 'v'},
    {"version",    0, 0, 'V'},
    {"help",       0, 0, 'h'},

    {"drive",      1, 0, 'd'},
    {"enabled",    1, 0, 'e'},
    {"interrupt",  1, 0, 'i'},
    {"partition",  1, 0, 'p'},
    {"timeout",    1, 0, 't'},
    {0, 0, 0, 0},
  };

  prog_name=strrchr(argv[0], '/');
  if(prog_name==NULL)
    prog_name=argv[0];
  else
    prog_name++;

  while(1)
  {
    int long_index=0;
    c=getopt_long(argc, argv, "fI:klno:P:rT:vVhd:e:i:p:t:",
		  long_options, &long_index);
    if(c==EOF)
      break;
    switch(c)
    {
    case 'f':
      my_data->flags |= DFLAG_FORCE;
      break;

    case 'I': /* code to install */
      my_data->install.path=optarg;
      my_data->install.flags=DESC_FLAG_SPECIFIED;
      to_do=1;
      break;

    case 'k': /* Keep the current code */
      my_data->flags |= DFLAG_KEEP;
      to_do=1;
      break;

    case 'l': /* List mode */
      my_data->flags |= DFLAG_LIST;
      break;

    case 'n': /* Don't install */
      my_data->flags |= DFLAG_NO_ACT;
      break;

    case 'o': /* Offset */
      my_data->offset=get_uint(optarg, "Offset");
      break;

    case 'P': /* Parameters from file */
      my_data->params.path=optarg;
      my_data->params.flags=DESC_FLAG_SPECIFIED;
      to_do=1;
      break;

    case 'r': /* Reset parameters */
      my_data->flags |= DFLAG_RESET;
      to_do=1;
      break;

    case 'T': /* Partition table from file */
      my_data->table.path=optarg;
      my_data->table.flags=DESC_FLAG_SPECIFIED;
      to_do=1;
      break;

    case 'v': /* verbose */
      my_data->flags |= DFLAG_VERBOSE;
      break;
      
    case 'V': /* version */
      printf("install-mbr %s\n", VERSION);
      printf("Copyright (C) 1999 Neil Turton.\n");
      exit(0);
      break;

    case 'h': /* help */
      show_help();
      exit(0);

    case 'd': /* drive */
      {
	unsigned int drive=get_uint(optarg, "Drive");
	if(drive>0xfe)
	{
	  fprintf(stderr, "%s: Invalid drive number: 0x%x\n",
		  prog_name, drive);
	  exit(1);
	};
	my_data->changes.flags|=CPFLAG_DRIVE;
	my_data->changes.drive=drive;
	to_do=1;
      }
      break;

    case 'e':
      parse_enabled(&(my_data->changes), optarg);
      to_do=1;
      break;

    case 'i':
      parse_interrupt(&(my_data->changes), optarg);
      to_do=1;
      break;

    case 'p':
      if(optarg[0]==0 || optarg[1]!=0)
      {
	fprintf(stderr, "%s: Invalid partition number: %d\n",
		prog_name, optarg);
	exit(1);
      }
      my_data->changes.flags|=CPFLAG_DEFAULT;
      my_data->changes.deflt=*optarg;
      to_do=1;
      break;

    case 't':
      {
	unsigned int timeout=get_uint(optarg, "Timeout");
	if(timeout>0xfe)
	{
	  fprintf(stderr, "%s: Invalid timeout: %d\n",
		  prog_name, timeout);
	  exit(1);
	}
	my_data->changes.flags|=CPFLAG_TIMEOUT_VALID|CPFLAG_TIMEOUT;
	my_data->changes.timeout=timeout;
	to_do=1;
      }
      break;

    default:
      fprintf(stderr, "%s: Bad return from getopt_long: %d(%c)\n",
	      prog_name, c);
      /* Fall-through */
    case '?':
      suggest_help();
      exit(1);
    }
  }
  if(argc-optind!=1)
  {
    show_usage(stderr);
    suggest_help();
    exit(1);
  }
  my_data->target.path=argv[optind];
  my_data->target.flags=DESC_FLAG_SPECIFIED;

  /* Don't do anything if all we're asked to do is list. */
  if((my_data->flags & DFLAG_LIST) && !to_do)
    my_data->flags |= DFLAG_NO_ACT;
}

int
main(int argc, char *argv[])
{
  struct data my_data={
    /* install */
    {NULL, NULL, 0},
    /* params */
    {NULL, NULL, 0},
    /* target */
    {NULL, NULL, 0},
    /* table */
    {NULL, NULL, 0},
    /* offset */
    0,
    /* flags */
    0,
    /* change_params structure */
    {0, 0, 0, 0, 0, 0xff, 0, 0xff }};

  assert(MBR_SIZE==BUFFER_SIZE);

  parse_options(&my_data, argc, argv);
  install_mbr(&my_data);
  return 0;
}

/*  */
/* Local Variables: */
/* mode:c */
/* c-file-offsets: ((substatement-open . 0)) */
/* End: */
