/* -*- mode: C++; tab-width: 4 -*- */
// ===========================================================================
//	PPStream.h				   1993-1998 Metrowerks Inc. All rights reserved.
// ===========================================================================
//
//	Abstract class for reading/writing an ordered sequence of bytes

#ifndef _PPSTREAM_H_
#define _PPSTREAM_H_

#include "Byteswapping.h"		// Canonical
#include "ErrorHandling.h"		// ErrCode


// ---------------------------------------------------------------------------

enum StreamFromType {
	kStreamFromStart = 1,
	kStreamFromEnd,
	kStreamFromMarker
};

// ---------------------------------------------------------------------------

class	PPStream {
public:
					PPStream();
	virtual			~PPStream();

	virtual void	SetMarker(
							uae_s32			inOffset,
							StreamFromType		inFromWhere);
	virtual uae_s32	GetMarker() const;

	virtual void	SetLength(
							uae_s32			inLength);
	virtual uae_s32	GetLength() const;

	Bool			AtEnd() const
					{
						return GetMarker() >= GetLength();
					}

						// Write Operations

	virtual ErrCode	PutBytes(
								const void		*inBuffer,
								uae_s32			ioByteCount);

	PPStream&		operator << (const char* inString)
					{
						WriteCString(inString);
						return (*this);
					}

	PPStream&		operator << (const string& inString)
					{
						WriteString(inString);
						return (*this);
					}

	PPStream&		operator << (uae_s8 inNum)
					{
						Canonical (inNum);
						PutBytes(&inNum, sizeof(inNum));
						return (*this);
					}

	PPStream&		operator << (uae_u8 inNum)
					{
						Canonical (inNum);
						PutBytes(&inNum, sizeof(inNum));
						return (*this);
					}

	PPStream&		operator << (char inChar)
					{
						Canonical (inChar);
						PutBytes(&inChar, sizeof(inChar));
						return (*this);
					}

	PPStream&		operator << (uae_s16 inNum)
					{
						Canonical (inNum);
						PutBytes(&inNum, sizeof(inNum));
						return (*this);
					}

	PPStream&		operator << (uae_u16 inNum)
					{
						Canonical (inNum);
						PutBytes(&inNum, sizeof(inNum));
						return (*this);
					}

	PPStream&		operator << (uae_s32 inNum)
					{
						Canonical (inNum);
						PutBytes(&inNum, sizeof(inNum));
						return (*this);
					}

	PPStream&		operator << (uae_u32 inNum)
					{
						Canonical (inNum);
						PutBytes(&inNum, sizeof(inNum));
						return (*this);
					}

	PPStream&		operator << (bool inBool)
					{
						Canonical (inBool);
						PutBytes(&inBool, sizeof(inBool));
						return (*this);
					}

						// Read Operations

	virtual ErrCode	GetBytes(
								void			*outBuffer,
								uae_s32			ioByteCount);
	uae_s32			PeekData(
							void			*outButter,
							uae_s32			inByteCount);

	PPStream&		operator >> (char* outString)
					{
						ReadCString(outString);
						return (*this);
					}

	PPStream&		operator >> (string& outString)
					{
						ReadString(outString);
						return (*this);
					}

	PPStream&		operator >> (uae_s8 &outNum)
					{
						GetBytes(&outNum, sizeof(outNum));
						Canonical (outNum);
						return (*this);
					}

	PPStream&		operator >> (uae_u8 &outNum)
					{
						GetBytes(&outNum, sizeof(outNum));
						Canonical (outNum);
						return (*this);
					}

	PPStream&		operator >> (char &outChar)
					{
						GetBytes(&outChar, sizeof(outChar));
						Canonical (outChar);
						return (*this);
					}

	PPStream&		operator >> (uae_s16 &outNum)
					{
						GetBytes(&outNum, sizeof(outNum));
						Canonical (outNum);
						return (*this);
					}

	PPStream&		operator >> (uae_u16 &outNum)
					{
						GetBytes(&outNum, sizeof(outNum));
						Canonical (outNum);
						return (*this);
					}

	PPStream&		operator >> (uae_s32 &outNum)
					{
						GetBytes(&outNum, sizeof(outNum));
						Canonical (outNum);
						return (*this);
					}

	PPStream&		operator >> (uae_u32 &outNum)
					{
						GetBytes(&outNum, sizeof(outNum));
						Canonical (outNum);
						return (*this);
					}

	PPStream&		operator >> (bool &outBool)
					{
						GetBytes(&outBool, sizeof(outBool));
						Canonical (outBool);
						return (*this);
					}

		// Data-specific read/write functions
		//   There is an equivalent Shift operator for each one
		//	 except WritePtr/ReadPtr (since Ptr is really a char*,
		//	 which is the same as a C string).

	uae_s32			WriteCString(
							const char		*inString);
	uae_s32			ReadCString(
							char*			outString);

	uae_s32			WriteString(
							const string&	inString);
	uae_s32			ReadString(
							string&			outString);

protected:
	uae_s32			mMarker;
	uae_s32			mLength;
};

#endif
