/* -*- mode: C++; tab-width: 4 -*- */
/* ================================================================================== */
/* Copyright (c) 1998-1999 3Com Corporation or its subsidiaries. All rights reserved. */
/* ================================================================================== */

#ifndef _METAMEMORY_H_
#define _METAMEMORY_H_

#include "PalmHeap.h"			// PalmHeap
#include "ErrorHandling.h"		// Errors::EAccessType


class MetaMemory
{
	public:
		static void				Initialize				(void);
		static void				Reset					(void);
		static void				Save					(SessionFile&);
		static void				Load					(SessionFile&);
		static void				Dispose					(void);

		// Called to mark and unmark some areas of memory.

		static void				MarkTotalAccess			(uaecptr begin, uaecptr end);

		static void				SetAccess				(uaecptr begin, uaecptr end, uae_u8 bits);

		static void				MarkLowMemory			(uaecptr begin, uaecptr end);
		static void				MarkSystemGlobals		(uaecptr begin, uaecptr end);
		static void				MarkHeapHeader			(uaecptr begin, uaecptr end);
		static void				MarkMPT					(uaecptr begin, uaecptr end);
		static void				MarkChunkHeader			(uaecptr begin, uaecptr end);
		static void				MarkChunkTrailer		(uaecptr begin, uaecptr end);
		static void				MarkLowStack			(uaecptr begin, uaecptr end);
		static void				MarkUnlockedChunk		(uaecptr begin, uaecptr end);
		static void				MarkFreeChunk			(uaecptr begin, uaecptr end);

		static void				MarkScreen				(uaecptr begin, uaecptr end);
		static void				UnmarkScreen			(uaecptr begin, uaecptr end);

		static void				MarkCPUBreak			(uaecptr opcodeLocation);
		static void				UnmarkCPUBreak			(uaecptr opcodeLocation);

		// Called when memory needs to be marked as initialized or not.

#if FOR_LATER
		static void				MarkUninitialized		(uaecptr begin, uaecptr end);
		static void				MoveUninitialized		(uaecptr source, uaecptr dest, uae_u32 size);
		static void				MarkInitialized			(uaecptr begin, uaecptr end);
		static void				MarkLongInitialized		(uaecptr);	// Inlined, defined below
		static void				MarkWordInitialized		(uaecptr);	// Inlined, defined below
		static void				MarkByteInitialized		(uaecptr);	// Inlined, defined below
#endif

		// Called on Chunk-based operations.

		static void				MemChunkNew				(VoidPtr, UInt attributes);
		static void				MemChunkFree			(VoidPtr);

		// Called on Pointer-based operations.

		static void				MemPtrNew				(VoidPtr);
		static void				MemPtrResize			(VoidPtr, UInt oldSize);

		// Called on Handle-based operations.

		static void				MemHandleNew			(VoidHand);
		static void				MemHandleResize			(VoidHand, UInt oldSize);
		static void				MemHandleFree			(VoidHand);

		// Called when a chunk is locked or unlocked.

		static void				MemLocalIDToLockedPtr	(VoidPtr);
		static void				MemHandleLock			(VoidHand);
		static void				MemHandleUnlock			(VoidHand);
		static void				MemHandleResetLock		(VoidHand);
		static void				MemPtrResetLock			(VoidPtr);
		static void				MemPtrUnlock			(VoidPtr);

		// Called to brute-force sync with the heap state.  Called
		// after things like MemHeapCompact, MemHeapInit, MemHeapScramble,
		// etc.  Implicitly called after above functions like MemFooNew.

		static void				SyncHeap				(const PalmHeap::HeapInfo& heapInfo);
		static void				SyncDynamicHeap			(void);
		static void				SyncAllHeaps			(void);
		static void				AssertSynced			(void);

		// Can a normal application access this memory location?
		// Forbidden locations are low-memory, system globals, screen
		// memory, memory manager structures, unlocked handles, and
		// uninitialized memory.

		static Bool				CanAppGetLong			(uae_u8*);	// Inlined, defined below
		static Bool				CanAppGetWord			(uae_u8*);	// Inlined, defined below
		static Bool				CanAppGetByte			(uae_u8*);	// Inlined, defined below

		static Bool				CanAppSetLong			(uae_u8*);	// Inlined, defined below
		static Bool				CanAppSetWord			(uae_u8*);	// Inlined, defined below
		static Bool				CanAppSetByte			(uae_u8*);	// Inlined, defined below

		// Can the non-Memory Manager parts of the system access this memory location?
		// Forbidden locations are low-memory, memory manager structures,
		// unlocked handles, and uninitialized memory.

		static Bool				CanSystemGetLong		(uae_u8*);	// Inlined, defined below
		static Bool				CanSystemGetWord		(uae_u8*);	// Inlined, defined below
		static Bool				CanSystemGetByte		(uae_u8*);	// Inlined, defined below

		static Bool				CanSystemSetLong		(uae_u8*);	// Inlined, defined below
		static Bool				CanSystemSetWord		(uae_u8*);	// Inlined, defined below
		static Bool				CanSystemSetByte		(uae_u8*);	// Inlined, defined below

		// Can the Memory Managers parts of the system access this memory location?
		// Forbidden locations are low-memory and uninitialized memory.

		static Bool				CanMemMgrGetLong		(uae_u8*);	// Inlined, defined below
		static Bool				CanMemMgrGetWord		(uae_u8*);	// Inlined, defined below
		static Bool				CanMemMgrGetByte		(uae_u8*);	// Inlined, defined below

		static Bool				CanMemMgrSetLong		(uae_u8*);	// Inlined, defined below
		static Bool				CanMemMgrSetWord		(uae_u8*);	// Inlined, defined below
		static Bool				CanMemMgrSetByte		(uae_u8*);	// Inlined, defined below

		static Errors::EAccessType
								GetWhatHappened			(uaecptr iAddress, long size, Bool forRead);
		static Errors::EAccessType
								AllowForBugs			(uaecptr iAddress, long size, Bool forRead, Errors::EAccessType);

		// Accessors for getting ranges of memory.

		static uaecptr			GetLowMemoryBegin		(void);
		static uaecptr			GetLowMemoryEnd			(void);
		static uaecptr			GetSysGlobalsBegin		(void);
		static uaecptr			GetSysGlobalsEnd		(void);
		static uaecptr			GetScreenBegin			(void);
		static uaecptr			GetScreenEnd			(void);
		static uaecptr			GetHeapHdrBegin			(UInt heapID);
		static uaecptr			GetHeapHdrEnd			(UInt heapID);

		// Predicates to help determine what kind of access error
		// just occurred.

		static Bool				IsLowMemory				(uaecptr testAddress, uae_u32 size);
		static Bool				IsSystemGlobal			(uaecptr testAddress, uae_u32 size);
		static Bool				IsScreenBuffer			(uaecptr testAddress, uae_u32 size);
		static Bool				IsMemMgrData			(uaecptr testAddress, uae_u32 size);
		static Bool				IsUnlockedChunk			(uaecptr testAddress, uae_u32 size);
		static Bool				IsFreeChunk				(uaecptr testAddress, uae_u32 size);
		static Bool				IsUninitialized			(uaecptr testAddress, uae_u32 size);
		static Bool				IsStack					(uaecptr testAddress, uae_u32 size);
		static Bool				IsLowStack				(uaecptr testAddress, uae_u32 size);
		static Bool				IsAllocatedChunk		(uaecptr testAddress, uae_u32 size);

		static Bool				IsScreenBuffer8			(uae_u8* metaAddress);	// Inlined, defined below
		static Bool				IsScreenBuffer16		(uae_u8* metaAddress);	// Inlined, defined below
		static Bool				IsScreenBuffer32		(uae_u8* metaAddress);	// Inlined, defined below
		static Bool				IsScreenBuffer			(uae_u8* metaAddress, uae_u32 size);

		static Bool				IsCPUBreak				(uaecptr opcodeLocation);
		static Bool				IsCPUBreak				(uae_u8* metaLocation);

	private:
		struct ChunkCheck
		{
			uaecptr	testAddress;
			uae_u32	size;
			Bool	result;
		};

		struct WhatHappenedData
		{
			Errors::EAccessType	result;
			uaecptr				address;
			uae_u32				size;
			Bool				forRead;
		};

		static void				MarkRange				(uaecptr start, uaecptr end, uae_u8 v);
		static void				UnmarkRange				(uaecptr start, uaecptr end, uae_u8 v);
		static void				MarkUnmarkRange			(uaecptr start, uaecptr end,
														 uae_u8 andValue, uae_u8 orValue);

		static DWord			GetChunkSize			(VoidPtr	p);
		static DWord			GetChunkLockCount		(VoidPtr	p);

		static PalmHeap::EWalkerProcResult
								SyncOneHeap				(const PalmHeap::HeapInfo& heapInfo,
														 void* userData);
		static PalmHeap::EWalkerProcResult
								SyncOneChunk			(const PalmHeap::HeapInfo& heapInfo,
														 const PalmHeap::ChunkInfo& chunkInfo,
														 void* userData);
		static PalmHeap::EWalkerProcResult
								AssertChunkSynced		(const PalmHeap::HeapInfo& heapInfo,
														 const PalmHeap::ChunkInfo& chunkInfo,
														 void* userData);
		static PalmHeap::EWalkerProcResult
								WhatHappenedHeapCallback(const PalmHeap::HeapInfo& heapInfo,
														 void* userData);
		static PalmHeap::EWalkerProcResult
								WhatHappenedChunkCallback(const PalmHeap::HeapInfo& heapInfo,
														 const PalmHeap::ChunkInfo& chunkInfo,
														 void* userData);
		static PalmHeap::EWalkerProcResult
								CheckForChunkHeader		(const PalmHeap::HeapInfo& heapInfo,
														 const PalmHeap::ChunkInfo& chunkInfo,
														 void* userData);
		static PalmHeap::EWalkerProcResult
								CheckForUnlockedChunk	(const PalmHeap::HeapInfo& heapInfo,
														 const PalmHeap::ChunkInfo& chunkInfo,
														 void* userData);
		static PalmHeap::EWalkerProcResult
								CheckForFreeChunk		(const PalmHeap::HeapInfo& heapInfo,
														 const PalmHeap::ChunkInfo& chunkInfo,
														 void* userData);
		static PalmHeap::EWalkerProcResult
								CheckForAllocatedChunk	(const PalmHeap::HeapInfo& heapInfo,
														 const PalmHeap::ChunkInfo& chunkInfo,
														 void* userData);

		static Bool				ValidChunk				(VoidPtr);
		static Bool				ValidHandle				(VoidHand);


		static Bool				memfilled				(const uae_u8* ptr, uae_u8 value, long len);
		/*
			Memory is laid out as follows:

				+-------------------+
				| Low Memory		|	Off Limits to everyone except IRQ handlers, SysSleep, and SysDoze
				+-------------------+
				|					|
				| System Globals	|	Off Limits to applications
				|					|
				+-------------------+
				| Dynamic Heap		|
				|+-----------------+|
				|| Heap Header	   ||	Off Limits to applications and non-MemMgr system
				|+-----------------+|
				|| Chunks		   ||	Off Limits to apps and non-MemMgr system if unlocked
				|+-----------------+|
				|| Free chunk	   ||	Off Limits to applications and non-MemMgr system
				|+-----------------+|
				|| Chunks		   ||
				|+-----------------+|
				|| Chunks		   ||
				|+-----------------+|
				|| Chunks		   ||
				|+-----------------+|
				|| Stack Chunk	   ||	Off Limits to everyone if below A7
				|+-----------------+|
				|| Screen Chunk	   ||	Off Limits to applications
				|+-----------------+|
				+-------------------+

			Any memory location can be marked as "uninitialized", which means that
			it can be written to but not read from.  An exception to this would be
			the parts of the memory manager that moves around blocks (which may
			contain uninitialized sections).
		*/

		enum
		{
			kNoAppAccess		= 0x01,
			kNoSystemAccess		= 0x02,
			kNoMemMgrAccess		= 0x04,

#if FOR_LATER
			kUninitialized		= 0x10,	// Bytes have not been written to.
#endif
			kScreenBuffer		= 0x20,	// Screen buffer; update host screen if these bytes are changed.
			kStopCPU			= 0x40,	// Halt CPU emulation and check to see why (stored on even bytes).
			kPatchResource		= 0x80,	// Bytes belong to a system update resource (stored on even bytes).
			kCodeCoverage		= 0x80,	// Bytes have been executed during code coverage test (stored on odd bytes).

			kLowMemoryBits		= kNoAppAccess | kNoSystemAccess | kNoMemMgrAccess,
			kGlobalsBits		= kNoAppAccess,
			kMPTBits			= kNoAppAccess,
			kMemStructBits		= kNoAppAccess | kNoSystemAccess,
			kLowStackBits		= kNoAppAccess | kNoSystemAccess | kNoMemMgrAccess,
			kFreeChunkBits		= kNoAppAccess | kNoSystemAccess,	//kInaccessibleChunk,
			kUnlockedChunkBits	= kNoAppAccess | kNoSystemAccess,	//kInaccessibleChunk,
			kScreenBits			= kNoAppAccess | kScreenBuffer,
			kAccessBitMask		= kNoAppAccess | kNoSystemAccess | kNoMemMgrAccess,
			kFreeAccessBits		= 0
		};
};


// Macros to take a byte of bits and produce an
// 8-bit, 16-bit, or 32-bit version of it.

#define META_BITS_32(bits)			\
	(((uae_u32) (bits)) << 24) |	\
	(((uae_u32) (bits)) << 16) |	\
	(((uae_u32) (bits)) <<  8) |	\
	(((uae_u32) (bits)))

#define META_BITS_16(bits)			\
	(((uae_u16) (bits)) <<  8) |	\
	(((uae_u16) (bits)))

#define META_BITS_8(bits)			\
	(((uae_u8) (bits)))

// Macros to fetch the appropriate 8-bit, 16-bit, or
// 32-bit value from meta-memory.
//
// Fetch the values with byte operations in order to
// support CPUs that don't allow, say, 16-bit fetches
// on odd boundaries or 32-bit fetches on odd 16-bit
// boundaries.

#define META_VALUE_32(p)					\
	((((uae_u32)((uae_u8*) p)[0]) << 24) |	\
	 (((uae_u32)((uae_u8*) p)[1]) << 16) |	\
	 (((uae_u32)((uae_u8*) p)[2]) <<  8) |	\
	 (((uae_u32)((uae_u8*) p)[3])))

#define META_VALUE_16(p)					\
	((((uae_u16)((uae_u8*) p)[0]) << 8) |	\
	 (((uae_u16)((uae_u8*) p)[1])))

#define META_VALUE_8(p)		\
	(*(uae_u8*) p)

// Macros to define CanAppGetLong, CanAppSetLong,
// CanSystemGetLong, CanSystemSetLong, etc.

#define DEFINE_FUNCTIONS(kind, bits)							\
																\
inline Bool MetaMemory::Can##kind##Long (uae_u8* metaAddress)	\
{																\
	const uae_u32	kMask = META_BITS_32 (bits);				\
																\
	return (META_VALUE_32 (metaAddress) & kMask) == 0;			\
}																\
																\
inline Bool MetaMemory::Can##kind##Word (uae_u8* metaAddress)	\
{																\
	const uae_u16	kMask = META_BITS_16 (bits);				\
																\
	return (META_VALUE_16 (metaAddress) & kMask) == 0;			\
}																\
																\
inline Bool MetaMemory::Can##kind##Byte (uae_u8* metaAddress)	\
{																\
	const uae_u8	kMask = META_BITS_8 (bits);					\
																\
	return (META_VALUE_8 (metaAddress) & kMask) == 0;			\
}


#if FOR_LATER
DEFINE_FUNCTIONS(AppGet, kNoAppAccess | kUninitialized)
DEFINE_FUNCTIONS(AppSet, kNoAppAccess)

DEFINE_FUNCTIONS(SystemGet, kNoSystemAccess | kUninitialized)
DEFINE_FUNCTIONS(SystemSet, kNoSystemAccess)

DEFINE_FUNCTIONS(MemMgrGet, kNoMemMgrAccess | kUninitialized)
DEFINE_FUNCTIONS(MemMgrSet, kNoMemMgrAccess)
#else
DEFINE_FUNCTIONS(AppGet, kNoAppAccess)
DEFINE_FUNCTIONS(AppSet, kNoAppAccess)

DEFINE_FUNCTIONS(SystemGet, kNoSystemAccess)
DEFINE_FUNCTIONS(SystemSet, kNoSystemAccess)

DEFINE_FUNCTIONS(MemMgrGet, kNoMemMgrAccess)
DEFINE_FUNCTIONS(MemMgrSet, kNoMemMgrAccess)
#endif


#if FOR_LATER
inline void MetaMemory::MarkLongInitialized (uaecptr p)
{
	const uae_u32	kMask	=	META_BITS_32 (kUninitialized);

	META_VALUE_32 (p) &= ~kMask;
}
#endif

#if FOR_LATER
inline void MetaMemory::MarkWordInitialized (uaecptr p)
{
	const uae_u16	kMask	=	META_BITS_16 (kUninitialized);

	META_VALUE_16 (p) &= ~kMask;
}
#endif

#if FOR_LATER
inline void MetaMemory::MarkByteInitialized (uaecptr p)
{
	const uae_u8	kMask	=	META_BITS_8 (kUninitialized);

	META_VALUE_8 (p) &= ~kMask;
}
#endif


inline Bool MetaMemory::IsScreenBuffer (uae_u8* metaAddress, uae_u32 size)
{
	if (size == 1)
	{
		const uae_u8 kMask = META_BITS_8 (kScreenBuffer);

		return (META_VALUE_8 (metaAddress) & kMask) != 0;
	}
	else if (size == 2)
	{
		const uae_u16 kMask = META_BITS_16 (kScreenBuffer);

		return (META_VALUE_16 (metaAddress) & kMask) != 0;
	}
	else if (size == 4)
	{
		const uae_u32 kMask = META_BITS_32 (kScreenBuffer);

		return (META_VALUE_32 (metaAddress) & kMask) != 0;
	}

	const uae_u8 kMask = META_BITS_8 (kScreenBuffer);

	for (uae_u32 ii = 0; ii < size; ++ii)
	{
		if (((*(uae_u8*) (metaAddress + ii)) & kMask) != 0)
		{
			return true;
		}
	}

	return false;
}

inline Bool MetaMemory::IsScreenBuffer8 (uae_u8* metaAddress)
{
	const uae_u8 kMask = META_BITS_8 (kScreenBuffer);

	return (META_VALUE_8 (metaAddress) & kMask) != 0;
}

inline Bool MetaMemory::IsScreenBuffer16 (uae_u8* metaAddress)
{
	const uae_u16 kMask = META_BITS_16 (kScreenBuffer);

	return (META_VALUE_16 (metaAddress) & kMask) != 0;
}

inline Bool MetaMemory::IsScreenBuffer32 (uae_u8* metaAddress)
{
	const uae_u32 kMask = META_BITS_32 (kScreenBuffer);

	return (META_VALUE_32 (metaAddress) & kMask) != 0;
}


inline Bool MetaMemory::IsCPUBreak (uaecptr opcodeLocation)
{
	assert ((opcodeLocation & 1) == 0);
	return IsCPUBreak (get_meta_address (opcodeLocation));
}


inline Bool MetaMemory::IsCPUBreak (uae_u8* metaLocation)
{
	return ((*metaLocation) & kStopCPU) != 0;
}


#define META_CHECK(metaAddress, address, op, size, forRead)		\
	if (ROMBank::IsPCInRAM ())									\
	{															\
		if (!MetaMemory::CanApp##op (metaAddress))				\
		{														\
			ProbableCause (address, sizeof (size), forRead);	\
		}														\
	}															\
	else if (Patches::IsPCInMemMgr ())							\
	{															\
		if (!MetaMemory::CanMemMgr##op (metaAddress))			\
		{														\
			ProbableCause (address, sizeof (size), forRead);	\
		}														\
	}															\
	else														\
	{															\
		if (!MetaMemory::CanSystem##op (metaAddress))			\
		{														\
			ProbableCause (address, sizeof (size), forRead);	\
		}														\
	}

// ---------------------------------------------------------------------------
//		 MetaMemory::MarkTotalAccess
// ---------------------------------------------------------------------------

inline void MetaMemory::MarkTotalAccess (uaecptr begin, uaecptr end)
{
	UnmarkRange (begin, end, kAccessBitMask);
}


// ---------------------------------------------------------------------------
//		 MetaMemory::SetAccess
// ---------------------------------------------------------------------------

inline void MetaMemory::SetAccess (uaecptr begin, uaecptr end, uae_u8 bits)
{
	MarkUnmarkRange (begin, end, ~kAccessBitMask, bits);
}


// ---------------------------------------------------------------------------
//		 MetaMemory::MarkLowMemory
// ---------------------------------------------------------------------------
//	Mark the "low memory" range of memory (the first 256 bytes of
//	memory that hold exception vectors).

inline void MetaMemory::MarkLowMemory (uaecptr begin, uaecptr end)
{
	SetAccess (begin, end, kLowMemoryBits);
}


// ---------------------------------------------------------------------------
//		 MetaMemory::MarkSystemGlobals
// ---------------------------------------------------------------------------
//	Mark the "system globals" range of memory.  This range holds the
//	global variables used by the Palm OS, as well as the jump table.

inline void MetaMemory::MarkSystemGlobals (uaecptr begin, uaecptr end)
{
	SetAccess (begin, end, kGlobalsBits);
}


// ---------------------------------------------------------------------------
//		 MetaMemory::MarkHeapHeader
// ---------------------------------------------------------------------------

inline void MetaMemory::MarkHeapHeader (uaecptr begin, uaecptr end)
{
	SetAccess (begin, end, kMemStructBits);
}


// ---------------------------------------------------------------------------
//		 MetaMemory::MarkMPT
// ---------------------------------------------------------------------------

inline void MetaMemory::MarkMPT (uaecptr begin, uaecptr end)
{
	SetAccess (begin, end, kMPTBits);
}


// ---------------------------------------------------------------------------
//		 MetaMemory::MarkChunkHeader
// ---------------------------------------------------------------------------

inline void MetaMemory::MarkChunkHeader (uaecptr begin, uaecptr end)
{
	SetAccess (begin, end, kMemStructBits);
}


// ---------------------------------------------------------------------------
//		 MetaMemory::MarkChunkTrailer
// ---------------------------------------------------------------------------

inline void MetaMemory::MarkChunkTrailer (uaecptr begin, uaecptr end)
{
	SetAccess (begin, end, kMemStructBits);
}


// ---------------------------------------------------------------------------
//		 MetaMemory::MarkLowStack
// ---------------------------------------------------------------------------

inline void MetaMemory::MarkLowStack (uaecptr begin, uaecptr end)
{
	SetAccess (begin, end, kLowStackBits);
}


// ---------------------------------------------------------------------------
//		 MetaMemory::MarkUnlockedChunk
// ---------------------------------------------------------------------------

inline void MetaMemory::MarkUnlockedChunk (uaecptr begin, uaecptr end)
{
	SetAccess (begin, end, kUnlockedChunkBits);
}


// ---------------------------------------------------------------------------
//		 MetaMemory::MarkFreeChunk
// ---------------------------------------------------------------------------

inline void MetaMemory::MarkFreeChunk (uaecptr begin, uaecptr end)
{
	SetAccess (begin, end, kFreeChunkBits);
}


// ---------------------------------------------------------------------------
//		 MetaMemory::MarkScreen
// ---------------------------------------------------------------------------

inline void MetaMemory::MarkScreen (uaecptr begin, uaecptr end)
{
	SetAccess (begin, end, kScreenBits);
}


// ---------------------------------------------------------------------------
//		 MetaMemory::UnmarkScreen
// ---------------------------------------------------------------------------

inline void MetaMemory::UnmarkScreen (uaecptr begin, uaecptr end)
{
	MarkTotalAccess (begin, end);
	UnmarkRange (begin, end, kScreenBuffer);
}


// ---------------------------------------------------------------------------
//		 MetaMemory::MarkCPUBreak
// ---------------------------------------------------------------------------

inline void MetaMemory::MarkCPUBreak (uaecptr opcodeLocation)
{
	assert ((opcodeLocation & 1) == 0);

	uae_u8*	ptr = get_meta_address (opcodeLocation);

	*ptr |= kStopCPU;
}


// ---------------------------------------------------------------------------
//		 MetaMemory::UnmarkCPUBreak
// ---------------------------------------------------------------------------

inline void MetaMemory::UnmarkCPUBreak (uaecptr opcodeLocation)
{
	assert ((opcodeLocation & 1) == 0);

	uae_u8*	ptr = get_meta_address (opcodeLocation);

	*ptr &= ~kStopCPU;
}



#endif /* _METAMEMORY_H_ */
  
  
