/* -*- mode: C++; tab-width: 4 -*- */
/* ================================================================================== */
/* Copyright (c) 1998-1999 3Com Corporation or its subsidiaries. All rights reserved. */
/* ================================================================================== */

#include "EmulatorCommon.h"
#include "Bank_MC68328.h"

#include "Bank_ROM.h"			// ROMBank::IsPCInRAM
#include "Byteswapping.h"		// ByteswapWords
#include "CPU_REG.h"			// Software::BusError
#include "DebugMgr.h"			// Debug::CheckStepSpy
#include "Miscellaneous.h"		// GetHostTime
#include "RAM_ROM.h"			// gMemoryAccess
#include "SessionFile.h"		// WriteHwrDBallType, ReadHwrDBallType


typedef uae_u32	(*ReadFunction)(uaecptr address, int size);
typedef void	(*WriteFunction)(uaecptr address, int size, uae_u32 value);

// Macro to return the Dragonball address of the specified register

#define addressof(x)	(kMemoryStart + offsetof(HwrDBallType, x))


// Macros for reading/writing Dragonball registers.

#define READ_REGISTER(reg)	\
	((sizeof(g68328Regs.reg) == 1) ? do_get_mem_byte(&g68328Regs.reg) :	\
	 (sizeof(g68328Regs.reg) == 2) ? do_get_mem_word(&g68328Regs.reg) :	\
									 do_get_mem_long(&g68328Regs.reg))

#define WRITE_REGISTER(reg, value)	\
	(sizeof(g68328Regs.reg) == 1) ? do_put_mem_byte(&g68328Regs.reg, value) :	\
	(sizeof(g68328Regs.reg) == 2) ? do_put_mem_word(&g68328Regs.reg, value) :	\
									do_put_mem_long(&g68328Regs.reg, value)

class MC68328Reg
{
	public:
		static void		SetHandler				(ReadFunction, WriteFunction, uae_u32 start, int count);

		static uae_u32	UnsupportedRead			(uaecptr address, int size);
		static void		UnsupportedWrite		(uaecptr address, int size, uae_u32 value);

		static uae_u32	StdRead					(uaecptr address, int size);
		static void		StdWrite				(uaecptr address, int size, uae_u32 value);
		static void		NullWrite				(uaecptr address, int size, uae_u32 value);	// For read-only registers

		static uae_u32	pllFreqSelRead			(uaecptr address, int size);
		static uae_u32	portCDataRead			(uaecptr address, int size);
		static uae_u32	portDDataRead			(uaecptr address, int size);
		static uae_u32	portMDataRead			(uaecptr address, int size);
		static uae_u32	tmr1StatusRead			(uaecptr address, int size);
		static uae_u32	tmr2StatusRead			(uaecptr address, int size);
		static uae_u32	UartRead				(uaecptr address, int size);
		static uae_u32	rtcHourMinSecRead		(uaecptr address, int size);

		static void		csASelect1Write			(uaecptr address, int size, uae_u32 value);
		static void		csCSelect0Write			(uaecptr address, int size, uae_u32 value);
		static void		csCSelect1Write			(uaecptr address, int size, uae_u32 value);
		static void		intMaskHiWrite			(uaecptr address, int size, uae_u32 value);
		static void		intMaskLoWrite			(uaecptr address, int size, uae_u32 value);
		static void		intStatusHiWrite		(uaecptr address, int size, uae_u32 value);
		static void		portDDataWrite			(uaecptr address, int size, uae_u32 value);
		static void		portDIntReqEnWrite		(uaecptr address, int size, uae_u32 value);
		static void		portGDataWrite			(uaecptr address, int size, uae_u32 value);
		static void		tmr1StatusWrite			(uaecptr address, int size, uae_u32 value);
		static void		tmr2StatusWrite			(uaecptr address, int size, uae_u32 value);
		static void		wdCounterWrite			(uaecptr address, int size, uae_u32 value);
		static void		spiMasterControlWrite	(uaecptr address, int size, uae_u32 value);
		static void		UartWrite				(uaecptr address, int size, uae_u32 value);
		static void		lcdRegisterWrite		(uaecptr address, int size, uae_u32 value);
		static void		rtcControlWrite			(uaecptr address, int size, uae_u32 value);
		static void		rtcIntStatusWrite		(uaecptr address, int size, uae_u32 value);
		static void		rtcIntEnableWrite		(uaecptr address, int size, uae_u32 value);
};

// Values used to initialize the DragonBall registers.

const HwrDBallType	kInitial68328RegisterValues =
{
	0x0C,		//	Byte		scr;						// $000: System Control Register
	{ 0 },		//	Byte										___filler0[0x004-0x001];

	// The following ID stuff is not present on earlier chips (before ??)
	0x00,		//	Byte		chipID;						// $004: Chip ID Register
	0x00,		//	Byte		maskID;						// $005: Mask ID Register
	0x00,		//	Word		swID;						// $006: Software ID Register
	{ 0 },		//	Byte										___filler1[0x100-0x008];				 

	0x0000,		//	Word		csAGroupBase;				// $100: Chip Select Group A Base Register
	0x0000,		//	Word		csBGroupBase;				// $102: Chip Select Group B Base Register
	0x0000,		//	Word		csCGroupBase;				// $104: Chip Select Group C Base Register
	0x0000,		//	Word		csDGroupBase;				// $106: Chip Select Group D Base Register

	0x0000,		//	Word		csAGroupMask;				// $108: Chip Select Group A Mask Register
	0x0000,		//	Word		csBGroupMask;				// $10A: Chip Select Group B Mask Register
	0x0000,		//	Word		csCGroupMask;				// $10C: Chip Select Group C Mask Register
	0x0000,		//	Word		csDGroupMask;				// $10E: Chip Select Group D Mask Register

	0x00010006,	//	DWord		csASelect0;					// $110: Group A Chip Select 0 Register
	0x00010006,	//	DWord		csASelect1;					// $114: Group A Chip Select 1 Register
	0x00010006,	//	DWord		csASelect2;					// $118: Group A Chip Select 2 Register
	0x00010006,	//	DWord		csASelect3;					// $11C: Group A Chip Select 3 Register

	0x00000000,	//	DWord		csBSelect0;					// $120: Group B Chip Select 0 Register
	0x00000000,	//	DWord		csBSelect1;					// $124: Group B Chip Select 1 Register
	0x00000000,	//	DWord		csBSelect2;					// $128: Group B Chip Select 2 Register
	0x00000000,	//	DWord		csBSelect3;					// $12C: Group B Chip Select 3 Register

	0x00010006,	//	DWord		csCSelect0;					// $130: Group C Chip Select 0 Register
	0x00010006,	//	DWord		csCSelect1;					// $134: Group C Chip Select 1 Register
	0x00010006,	//	DWord		csCSelect2;					// $138: Group C Chip Select 2 Register
	0x00010006,	//	DWord		csCSelect3;					// $13C: Group C Chip Select 3 Register

	0x00000000,	//	DWord		csDSelect0;					// $140: Group D Chip Select 0 Register
	0x00000000,	//	DWord		csDSelect1;					// $144: Group D Chip Select 1 Register
	0x00000000,	//	DWord		csDSelect2;					// $148: Group D Chip Select 2 Register
	0x00000000,	//	DWord		csDSelect3;					// $14C: Group D Chip Select 3 Register

	0x0000,		//	Word		csDebug;					// $150: Chip Select debug register
	{ 0 },		//	Byte										___filler2[0x200-0x152];

	0x2400,		//	Word		pllControl;					// $200: PLL Control Register
	0x0123,		//	Word		pllFreqSel;					// $202: PLL Frequency Select Register
	0x0000,		//	Word		pllTest;					// $204: PLL Test Register
	{ 0 },		//	Byte										__filler44;
	0x1F,		//	Byte		pwrControl;					// $207: Power Control Register

	{ 0 },		//	Byte										___filler3[0x300-0x208];

	0x00,		//	Byte		intVector;					// $300: Interrupt Vector Register
	{ 0 },		//	Byte										___filler4;
	0x0000,		//	Word		intControl;					// $302: Interrupt Control Register
	0x00FF,		//	Word		intMaskHi;					// $304: Interrupt Mask Register/HIGH word
	0xFFFF,		//	Word		intMaskLo;					// $306: Interrupt Mask Register/LOW word
	0x00FF,		//	Word		intWakeupEnHi;				// $308: Interrupt Wakeup Enable Register
	0xFFFF,		//	Word		intWakeupEnLo;				// $30A: Interrupt Wakeup Enable Register
	0x0000,		//	Word		intStatusHi;				// $30C: Interrupt Status Register/HIGH word
	0x0000,		//	Word		intStatusLo;				// $30E: Interrupt Status Register/LOW word
	0x0000,		//	Word		intPendingHi;				// $310: Interrupt Pending Register
	0x0000,		//	Word		intPendingLo;				// $312: Interrupt Pending Register

	{ 0 },		//	Byte 										___filler4a[0x400-0x314];

	0x00,		//	Byte		portADir;					// $400: Port A Direction Register
	0x00,		//	Byte		portAData;					// $401: Port A Data Register
	{ 0 },		//	Byte										___filler5;
	0x00,		//	Byte		portASelect;				// $403: Port A Select Register

	{ 0 },		//	Byte										___filler6[4];

	0x00,		//	Byte		portBDir;					// $408: Port B Direction Register
	0x00,		//	Byte		portBData;					// $409: Port B Data Register
	{ 0 },		//	Byte										___filler7;
	0x00,		//	Byte		portBSelect;				// $40B: Port B Select Register

	{ 0 },		//	Byte										___filler8[4];

	0x00,		//	Byte		portCDir;					// $410: Port C Direction Register
	0x00,		//	Byte		portCData;					// $411: Port C Data Register
	{ 0 },		//	Byte										___filler9;
	0x00,		//	Byte		portCSelect;				// $413: Port C Select Register

	{ 0 },		//	Byte										___filler10[4];

	0x00,		//	Byte		portDDir;					// $418: Port D Direction Register
	0x00,		//	Byte		portDData;					// $419: Port D Data Register
	0xFF,		//	Byte		portDPullupEn;				// $41A: Port D Pull-up Enable
	{ 0 },		//	Byte										___filler11;
	0x00,		//	Byte		portDPolarity;				// $41C: Port D Polarity Register
	0x00,		//	Byte		portDIntReqEn;				// $41D: Port D Interrupt Request Enable
	{ 0 },		//	Byte										___filler12;
	0x00,		//	Byte		portDIntEdge;				// $41F: Port D IRQ Edge Register

	0x00,		//	Byte		portEDir;					// $420: Port E Direction Register
	0x00,		//	Byte		portEData;					// $421: Port E Data Register
	0x80,		//	Byte		portEPullupEn;				// $422: Port E Pull-up Enable
	0x80,		//	Byte		portESelect;				// $423: Port E Select Register

	{ 0 },		//	Byte										___filler14[4];

	0x00,		//	Byte		portFDir;					// $428: Port F Direction Register
	0x00,		//	Byte		portFData;					// $429: Port F Data Register
	0xFF,		//	Byte		portFPullupEn;				// $42A: Port F Pull-up Enable
	0xFF,		//	Byte		portFSelect;				// $42B: Port F Select Register

	{ 0 },		//	Byte										___filler16[4];

	0x00,		//	Byte		portGDir;					// $430: Port G Direction Register
	0x00,		//	Byte		portGData;					// $431: Port G Data Register
	0xFF,		//	Byte		portGPullupEn;				// $432: Port G Pull-up Enable
	0xFF,		//	Byte		portGSelect;				// $433: Port G Select Register

	{ 0 },		//	Byte										___filler18[4];

	0x00,		//	Byte		portJDir;					// $438: Port J Direction Register
	0x00,		//	Byte		portJData;					// $439: Port J Data Register
	{ 0 },		//	Byte										___filler19;
	0x00,		//	Byte		portJSelect;				// $43B: Port J Select Register

	{ 0 },		//	Byte										___filler19a[4];

	0x00,		//	Byte		portKDir;					// $440: Port K Direction Register
	0x00,		//	Byte		portKData;					// $441: Port K Data Register
	0xFF,		//	Byte		portKPullupEn;				// $442: Port K Pull-up Enable
	0xFF,		//	Byte		portKSelect;				// $443: Port K Select Register

	{ 0 },		//	Byte										___filler21[4];

	0x00,		//	Byte		portMDir;					// $448: Port M Direction Register
	0x00,		//	Byte		portMData;					// $449: Port M Data Register
	0xFF,		//	Byte		portMPullupEn;				// $44A: Port M Pull-up Enable Register
	0xFF,		//	Byte		portMSelect;				// $44B: Port M Select Register

	{ 0 },		//	Byte										___filler22[4];

	{ 0 },		//	Byte										___filler23[0x500-0x450];

	0x0000,		//	Word		pwmControl;					// $500: PWM Control Register
	0x0000,		//	Word		pwmPeriod;					// $502: PWM Period Register
	0x0000,		//	Word		pwmWidth;					// $504: PWM Width Register
	0x0000,		//	Word		pwmCounter;					// $506: PWM Counter

	{ 0 },		//	Byte										___filler24[0x600-0x508];

	0x0000,		//	Word		tmr1Control;				// $600: Timer 1 Control Register
	0x0000,		//	Word		tmr1Prescaler;				// $602: Timer 1 Prescaler Register
	0xFFFF,		//	Word		tmr1Compare;				// $604: Timer 1 Compare Register
	0x0000,		//	Word		tmr1Capture;				// $606: Timer 1 Capture Register
	0x0000,		//	Word		tmr1Counter;				// $608: Timer 1 Counter Register
	0x0000,		//	Word		tmr1Status;					// $60A: Timer 1 Status Register

	0x0000,		//	Word		tmr2Control;				// $60C: Timer 2 Control Register
	0x0000,		//	Word		tmr2Prescaler;				// $60E: Timer 2 Prescaler Register
	0xFFFF,		//	Word		tmr2Compare;				// $610: Timer 2 Compare Register
	0x0000,		//	Word		tmr2Capture;				// $612: Timer 2 Capture Register
	0x0000,		//	Word		tmr2Counter;				// $614: Timer 2 Counter Register
	0x0000,		//	Word		tmr2Status;					// $616: Timer 2 Status Register

	0x0000,		//	Word		wdControl;					// $618: Watchdog Control Register
	0x0000,		//	Word		wdReference;				// $61A: Watchdog Reference Register
	0x0000,		//	Word		wdCounter;					// $61C: Watchdog Counter

	{ 0 },		//	Byte										___filler25[0x700-0x61E];

	0x0000,		//	Word		spiSlave;					// $700: SPI Slave Register

	{ 0 },		//	Byte										___filler26[0x800-0x702];

	0x0000,		//	Word		spiMasterData;				// $800: SPI Master Data Register
	0x0000,		//	Word		spiMasterControl;			// $802: SPI Master Control Register

	{ 0 },		//	Byte										___filler27[0x900-0x804];

	0x0000,		//	Word		uControl;					// $900: Uart Control Register
	0x003F,		//	Word		uBaud;						// $902: Uart Baud Control Register
	0x0000,		//	Word		uReceive;					// $904: Uart Receive Register
	0x0000,		//	Word		uTransmit;					// $906: Uart Transmit Register
	0x0000,		//	Word		uMisc;						// $908: Uart Miscellaneous Register

	{ 0 },		//	Byte										___filler28[0xA00-0x90A];

	0x00000000,	//	DWord		lcdStartAddr;				// $A00: Screen Starting Address Register
	{ 0 },		//	Byte										___filler29;
	0xFF,		//	Byte		lcdPageWidth;				// $A05: Virtual Page Width Register
	{ 0 },		//	Byte										___filler30[2];
	0x03FF,		//	Word		lcdScreenWidth;				// $A08: Screen Width Register
	0x01FF,		//	Word		lcdScreenHeight;			// $A0A: Screen Height Register
	{ 0 },		//	Byte										___filler31[0xA18-0xA0C];
	0x0000,		//	Word		lcdCursorXPos;				// $A18: Cursor X Position
	0x0000,		//	Word		lcdCursorYPos;				// $A1A:	Cursor Y Position
	0x0101,		//	Word		lcdCursorWidthHeight;		// $A1C: Cursor Width and Height
	{ 0 },		//	Byte										___filler32;
	0x7F,		//	Byte		lcdBlinkControl;			// $A1F: Blink Control Register
	0x00,		//	Byte		lcdPanelControl;			// $A20: Panel Interface Control Register
	0x00,		//	Byte		lcdPolarity;				// $A21: Polarity Config Register
	0x00,		//	Byte										___filler33;
	0x00,		//	Byte		lcdACDRate;					// $A23: ACD (M) Rate Control Register
	0x00,		//	Byte										___filler34;
	0x00,		//	Byte		lcdPixelClock;				// $A25: Pixel Clock Divider Register
	0x00,		//	Byte										___filler35;
	0x40,		//	Byte		lcdClockControl;			// $A27: Clocking Control Register
	0x00,		//	Byte										___filler36;
	0x3E,		//	Byte		lcdLastBufferAddr;			// $A29: Last Buffer Address Register
	0x00,		//	Byte										___filler37;
	0x3F,		//	Byte		lcdOctetTermCount;			// $A2B: Octet Terminal Count Register
	0x00,		//	Byte										___filler38;
	0x00,		//	Byte		lcdPanningOffset;			// $A2D: Panning Offset Register
	{ 0 },		//	Byte										___filler39[3];
	0xB9,		//	Byte		lcdFrameRate;				// $A31: Frame Rate Control Modulation Register
	0x1073,		//	Word		lcdGrayPalette;				// $A32: Gray Palette Mapping Register
	0x00,		//	Byte		lcdReserved;				// $A34: Reserved

	{ 0 },		//	Byte										___filler40[0xB00-0xA35];

	0x00000000,	//	DWord		rtcHourMinSec;				// $B00: RTC Hours, Minutes, Seconds Register
	0x00000000,	//	DWord		rtcAlarm;					// $B04: RTC Alarm Register
	0x00000000,	//	DWord		rtcReserved;				// $B08: RTC Reserved
	0x0000,		//	Word		rtcControl;					// $B0C: RTC Control Register
	0x0000,		//	Word		rtcIntStatus;				// $B0E: RTC Interrupt Status Register
	0x0000,		//	Word		rtcIntEnable;				// $B10: RTC Interrupt Enable Register
	0x0000		//	Word		stopWatch;					// $B12: Stopwatch Minutes
};


#pragma mark -


// ===========================================================================
//		 Register Bank Accessors
// ===========================================================================
// These functions provide fetch and store access to the emulator's register
// memory.

static AddressBank	gAddressBank =
{
	MC68328Bank::GetLong,
	MC68328Bank::GetWord,
	MC68328Bank::GetByte,
	MC68328Bank::SetLong,
	MC68328Bank::SetWord,
	MC68328Bank::SetByte,
	MC68328Bank::GetRealAddress,
	MC68328Bank::ValidAddress,
	NULL,
	NULL
};

static const uae_u32	kMemoryStart = hwr68328Base;

static HwrDBallType		g68328Regs;

static bool				gHotSyncButtonDown;		// Use "bool" instead of "Bool" so that
												// we save a consistant type to the
												// session file.
static uae_u32			gTmr2CurrentMilliseconds;
static uae_u32			gTmr2StartMilliseconds;	// !!! Need to zap this when TEN bit
												// of Timer Control register is cleared.
static uae_u16			gLastTmr1Status;
static uae_u16			gLastTmr2Status;
static uae_u8			gPortDEdge;

static const uae_u32	ADDRESS_MASK = 0x0000FFF0;

// These functions provide fetch and store access to the
// emulator's register memory.

static ReadFunction		gReadFunctions [sizeof (HwrDBallType)];
static WriteFunction	gWriteFunctions [sizeof (HwrDBallType)];


/***********************************************************************
 *
 * FUNCTION:	MC68328Bank::Initialize
 *
 * DESCRIPTION:	Standard initialization function.  Responsible for
 *				initializing this sub-system when a new session is
 *				created.  May also be called from the Load function
 *				to share common functionality.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void MC68328Bank::Initialize (void)
{
	// Memory bank 0xFFFF is managed by the functions in MC68328Bank.

	Memory::InitializeBanks (gAddressBank, bankindex (kMemoryStart), 1);

	// Install the handlers for each register.

	MC68328Bank::InstallHandlers ();

	// Also reset here and now so that the hardware registers will be
	// set up.  We need them set up so that we know how to configure
	// our RAM from the chip selects.

	MC68328Bank::Reset ();
}


/***********************************************************************
 *
 * FUNCTION:	MC68328Bank::Reset
 *
 * DESCRIPTION:	Standard reset function.  Sets the sub-system to a
 *				default state.  This occurs not only on a Reset (as
 *				from the menu item), but also when the sub-system
 *				is first initialized (Reset is called after Initialize)
 *				as well as when the system is re-loaded from an
 *				insufficient session file.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void MC68328Bank::Reset (void)
{
	g68328Regs = kInitial68328RegisterValues;

	// Byteswap all the words in the Dragonball registers (if necessary).

	Canonical (g68328Regs);
	ByteswapWords (&g68328Regs, sizeof(g68328Regs));

	gHotSyncButtonDown = false;

	gLastTmr1Status = 0;
	gLastTmr2Status = 0;
	gPortDEdge = 0;

	Platform::CloseSerialPort ();
}


/***********************************************************************
 *
 * FUNCTION:	MC68328Bank::Save
 *
 * DESCRIPTION:	Standard save function.  Saves any sub-system state to
 *				the given session file.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void MC68328Bank::Save (SessionFile& f)
{
	StWordSwapper				swapper1 (&g68328Regs, sizeof(g68328Regs));
//	StCanonical<HwrDBallType>	swapper2 (g68328Regs);
	f.WriteHwrDBallType (g68328Regs);
	f.FixBug (SessionFile::kBugByteswappedStructs);

	const long	kCurrentVersion = 1;

	Chunk		chunk;
	ChunkStream	s (chunk);

	s << kCurrentVersion;
	s << gHotSyncButtonDown;
	s << gLastTmr1Status;
	s << gLastTmr2Status;
	s << gPortDEdge;

	f.WriteDBallState (chunk);
}


/***********************************************************************
 *
 * FUNCTION:	MC68328Bank::Load
 *
 * DESCRIPTION:	Standard load function.  Loads any sub-system state
 *				from the given session file.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void MC68328Bank::Load (SessionFile& f)
{
	if (f.ReadHwrDBallType (g68328Regs))
	{
		// The Windows version of Poser 2.1d29 and earlier did not write
		// out structs in the correct format.  The fields of the struct
		// were written out in Little-Endian format, not Big-Endian.  To
		// address this problem, the bug has been fixed, and a new field
		// is added to the file format indicating that the bug has been
		// fixed.  With the new field (the "bug bit"), Poser can identify
		// old files from new files and read them in accordingly.
		// 
		// With the bug fixed, the .psf files should now be interchangeable
		// across platforms (modulo other bugs...).

		if (!f.IncludesBugFix (SessionFile::kBugByteswappedStructs))
		{
			Canonical (g68328Regs);
		}
		ByteswapWords (&g68328Regs, sizeof(g68328Regs));

		// Re-open the serial port if needed.

		Bool	isOn = (READ_REGISTER (portGData) & hwrTD1PortGSerialOn) != 0;

		if (isOn)
		{
			Platform::OpenSerialPort ();
		}

		// Resync the UART sub-system.

		Bool	sendTxData = false;
		MC68328Bank::UARTStateChanged (sendTxData);

		// Reset gMemAccessFlags.fProtect_SRAMSet

		gMemAccessFlags.fProtect_SRAMSet = (READ_REGISTER (csASelect1) & 0x0008) != 0;
	}
	else
	{
		f.SetCanReload (false);
	}

	Chunk		chunk;
	if (f.ReadDBallState (chunk))
	{
		long		version;
		ChunkStream	s (chunk);

		s >> version;

		if (version >= 1)
		{
			s >> gHotSyncButtonDown;
			s >> gTmr2CurrentMilliseconds;
			s >> gTmr2StartMilliseconds;
			s >> gLastTmr1Status;
			s >> gLastTmr2Status;
			s >> gPortDEdge;
		}
	}
	else
	{
		f.SetCanReload (false);
	}
}


/***********************************************************************
 *
 * FUNCTION:	MC68328Bank::Dispose
 *
 * DESCRIPTION:	Standard dispose function.  Completely release any
 *				resources acquired or allocated in Initialize and/or
 *				Load.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void MC68328Bank::Dispose (void)
{
}


// ---------------------------------------------------------------------------
//		 MC68328Bank::InstallHandlers
// ---------------------------------------------------------------------------

void MC68328Bank::InstallHandlers (void)
{
	// The Dragonball register bank is further sub-divided into separate handlers
	// for each memory address.  First, install the default handlers for reading
	// and writing (which basically report the access as invalid).

	MC68328Reg::SetHandler (MC68328Reg::UnsupportedRead, MC68328Reg::UnsupportedWrite,
						kMemoryStart, sizeof (HwrDBallType));

	// Now add standard/specialized handers for the defined registers.

#define INSTALL_HANDLER(read, write, reg) \
	MC68328Reg::SetHandler (MC68328Reg::read, MC68328Reg::write, \
							addressof (reg), sizeof (g68328Regs.reg))

	INSTALL_HANDLER (StdRead,			StdWrite,				scr);
	INSTALL_HANDLER (StdRead,			StdWrite,				csAGroupBase);
	INSTALL_HANDLER (StdRead,			StdWrite,				csBGroupBase);
	INSTALL_HANDLER (StdRead,			StdWrite,				csCGroupBase);
	INSTALL_HANDLER (StdRead,			StdWrite,				csDGroupBase);
	INSTALL_HANDLER (StdRead,			StdWrite,				csAGroupMask);
	INSTALL_HANDLER (StdRead,			StdWrite,				csBGroupMask);
	INSTALL_HANDLER (StdRead,			StdWrite,				csCGroupMask);
	INSTALL_HANDLER (StdRead,			StdWrite,				csDGroupMask);
	INSTALL_HANDLER (StdRead,			StdWrite,				csASelect0);
	INSTALL_HANDLER (StdRead,			csASelect1Write,		csASelect1);
	INSTALL_HANDLER (StdRead,			StdWrite,				csASelect2);
	INSTALL_HANDLER (StdRead,			StdWrite,				csASelect3);
	INSTALL_HANDLER (StdRead,			StdWrite,				csBSelect0);
	INSTALL_HANDLER (StdRead,			StdWrite,				csBSelect1);
	INSTALL_HANDLER (StdRead,			StdWrite,				csBSelect2);
	INSTALL_HANDLER (StdRead,			StdWrite,				csBSelect3);
	INSTALL_HANDLER (StdRead,			csCSelect0Write,		csCSelect0);
	INSTALL_HANDLER (StdRead,			csCSelect1Write,		csCSelect1);
	INSTALL_HANDLER (StdRead,			StdWrite,				csCSelect2);
	INSTALL_HANDLER (StdRead,			StdWrite,				csCSelect3);
	INSTALL_HANDLER (StdRead,			StdWrite,				csDSelect0);
	INSTALL_HANDLER (StdRead,			StdWrite,				csDSelect1);
	INSTALL_HANDLER (StdRead,			StdWrite,				csDSelect2);
	INSTALL_HANDLER (StdRead,			StdWrite,				csDSelect3);
	INSTALL_HANDLER (StdRead,			StdWrite,				csDebug);
	INSTALL_HANDLER (StdRead,			StdWrite,				pllControl);
	INSTALL_HANDLER (pllFreqSelRead,	StdWrite,				pllFreqSel);
	INSTALL_HANDLER (StdRead,			StdWrite,				pllTest);
	INSTALL_HANDLER (StdRead,			StdWrite,				pwrControl);
	INSTALL_HANDLER (StdRead,			StdWrite,				intVector);
	INSTALL_HANDLER (StdRead,			StdWrite,				intControl);
	INSTALL_HANDLER (StdRead,			intMaskHiWrite,			intMaskHi);
	INSTALL_HANDLER (StdRead,			intMaskLoWrite,			intMaskLo);
	INSTALL_HANDLER (StdRead,			StdWrite,				intWakeupEnHi);
	INSTALL_HANDLER (StdRead,			StdWrite,				intWakeupEnLo);
	INSTALL_HANDLER (StdRead,			intStatusHiWrite,		intStatusHi);
	INSTALL_HANDLER (StdRead,			NullWrite,				intStatusLo);
	INSTALL_HANDLER (StdRead,			NullWrite,				intPendingHi);
	INSTALL_HANDLER (StdRead,			NullWrite,				intPendingLo);
	INSTALL_HANDLER (StdRead,			StdWrite,				portADir);
	INSTALL_HANDLER (StdRead,			StdWrite,				portAData);
	INSTALL_HANDLER (StdRead,			StdWrite,				portASelect);
	INSTALL_HANDLER (StdRead,			StdWrite,				portBDir);
	INSTALL_HANDLER (StdRead,			StdWrite,				portBData);
	INSTALL_HANDLER (StdRead,			StdWrite,				portBSelect);
	INSTALL_HANDLER (StdRead,			StdWrite,				portCDir);
	INSTALL_HANDLER (portCDataRead,		StdWrite,				portCData);
	INSTALL_HANDLER (StdRead,			StdWrite,				portCSelect);
	INSTALL_HANDLER (StdRead,			StdWrite,				portDDir);
	INSTALL_HANDLER (portDDataRead,		portDDataWrite,			portDData);
	INSTALL_HANDLER (StdRead,			StdWrite,				portDPullupEn);
	INSTALL_HANDLER (StdRead,			StdWrite,				portDPolarity);
	INSTALL_HANDLER (StdRead,			portDIntReqEnWrite,		portDIntReqEn);
	INSTALL_HANDLER (StdRead,			StdWrite,				portDIntEdge);
	INSTALL_HANDLER (StdRead,			StdWrite,				portEDir);
	INSTALL_HANDLER (StdRead,			StdWrite,				portEData);
	INSTALL_HANDLER (StdRead,			StdWrite,				portEPullupEn);
	INSTALL_HANDLER (StdRead,			StdWrite,				portESelect);
	INSTALL_HANDLER (StdRead,			StdWrite,				portFDir);
	INSTALL_HANDLER (StdRead,			lcdRegisterWrite,		portFData);
	INSTALL_HANDLER (StdRead,			StdWrite,				portFPullupEn);
	INSTALL_HANDLER (StdRead,			StdWrite,				portFSelect);
	INSTALL_HANDLER (StdRead,			StdWrite,				portGDir);
	INSTALL_HANDLER (StdRead,			portGDataWrite,			portGData);
	INSTALL_HANDLER (StdRead,			StdWrite,				portGPullupEn);
	INSTALL_HANDLER (StdRead,			StdWrite,				portGSelect);
	INSTALL_HANDLER (StdRead,			StdWrite,				portJDir);
	INSTALL_HANDLER (StdRead,			StdWrite,				portJData);
	INSTALL_HANDLER (StdRead,			StdWrite,				portJSelect);
	INSTALL_HANDLER (StdRead,			StdWrite,				portKDir);
	INSTALL_HANDLER (StdRead,			StdWrite,				portKData);
	INSTALL_HANDLER (StdRead,			StdWrite,				portKPullupEn);
	INSTALL_HANDLER (StdRead,			StdWrite,				portKSelect);
	INSTALL_HANDLER (StdRead,			StdWrite,				portMDir);
	INSTALL_HANDLER (portMDataRead,		StdWrite,				portMData);
	INSTALL_HANDLER (StdRead,			StdWrite,				portMPullupEn);
	INSTALL_HANDLER (StdRead,			StdWrite,				portMSelect);
	INSTALL_HANDLER (StdRead,			StdWrite,				pwmControl);
	INSTALL_HANDLER (StdRead,			StdWrite,				pwmPeriod);
	INSTALL_HANDLER (StdRead,			StdWrite,				pwmWidth);
	INSTALL_HANDLER (StdRead,			NullWrite,				pwmCounter);
	INSTALL_HANDLER (StdRead,			StdWrite,				tmr1Control);
	INSTALL_HANDLER (StdRead,			StdWrite,				tmr1Prescaler);
	INSTALL_HANDLER (StdRead,			StdWrite,				tmr1Compare);
	INSTALL_HANDLER (StdRead,			StdWrite,				tmr1Capture);
	INSTALL_HANDLER (StdRead,			NullWrite,				tmr1Counter);
	INSTALL_HANDLER (tmr1StatusRead,	tmr1StatusWrite,		tmr1Status);
	INSTALL_HANDLER (StdRead,			StdWrite,				tmr2Control);
	INSTALL_HANDLER (StdRead,			StdWrite,				tmr2Prescaler);
	INSTALL_HANDLER (StdRead,			StdWrite,				tmr2Compare);
	INSTALL_HANDLER (StdRead,			StdWrite,				tmr2Capture);
	INSTALL_HANDLER (StdRead,			NullWrite,				tmr2Counter);
	INSTALL_HANDLER (tmr2StatusRead,	tmr2StatusWrite,		tmr2Status);
	INSTALL_HANDLER (StdRead,			StdWrite,				wdControl);
	INSTALL_HANDLER (StdRead,			StdWrite,				wdReference);
	INSTALL_HANDLER (StdRead,			wdCounterWrite,			wdCounter);
	INSTALL_HANDLER (StdRead,			StdWrite,				spiSlave);
	INSTALL_HANDLER (StdRead,			StdWrite,				spiMasterData);
	INSTALL_HANDLER (StdRead,			spiMasterControlWrite,	spiMasterControl);
	INSTALL_HANDLER (UartRead,			UartWrite,				uControl);
	INSTALL_HANDLER (UartRead,			UartWrite,				uBaud);
	INSTALL_HANDLER (UartRead,			UartWrite,				uReceive);
	INSTALL_HANDLER (UartRead,			UartWrite,				uTransmit);
	INSTALL_HANDLER (UartRead,			UartWrite,				uMisc);
	INSTALL_HANDLER (StdRead,			lcdRegisterWrite,		lcdStartAddr);
	INSTALL_HANDLER (StdRead,			lcdRegisterWrite,		lcdPageWidth);
	INSTALL_HANDLER (StdRead,			lcdRegisterWrite,		lcdScreenWidth);
	INSTALL_HANDLER (StdRead,			lcdRegisterWrite,		lcdScreenHeight);
	INSTALL_HANDLER (StdRead,			StdWrite,				lcdCursorXPos);
	INSTALL_HANDLER (StdRead,			StdWrite,				lcdCursorYPos);
	INSTALL_HANDLER (StdRead,			StdWrite,				lcdCursorWidthHeight);
	INSTALL_HANDLER (StdRead,			StdWrite,				lcdBlinkControl);
	INSTALL_HANDLER (StdRead,			lcdRegisterWrite,		lcdPanelControl);
	INSTALL_HANDLER (StdRead,			StdWrite,				lcdPolarity);
	INSTALL_HANDLER (StdRead,			StdWrite,				lcdACDRate);
	INSTALL_HANDLER (StdRead,			StdWrite,				lcdPixelClock);
	INSTALL_HANDLER (StdRead,			StdWrite,				lcdClockControl);
	INSTALL_HANDLER (StdRead,			StdWrite,				lcdLastBufferAddr);
	INSTALL_HANDLER (StdRead,			StdWrite,				lcdOctetTermCount);
	INSTALL_HANDLER (StdRead,			StdWrite,				lcdPanningOffset);
	INSTALL_HANDLER (StdRead,			StdWrite,				lcdFrameRate);
	INSTALL_HANDLER (StdRead,			StdWrite,				lcdGrayPalette);
	INSTALL_HANDLER (StdRead,			StdWrite,				lcdReserved);
	INSTALL_HANDLER (rtcHourMinSecRead,	StdWrite,				rtcHourMinSec);
	INSTALL_HANDLER (StdRead,			StdWrite,				rtcAlarm);
	INSTALL_HANDLER (StdRead,			StdWrite,				rtcReserved);
	INSTALL_HANDLER (StdRead,			rtcControlWrite,		rtcControl);
	INSTALL_HANDLER (StdRead,			rtcIntStatusWrite,		rtcIntStatus);
	INSTALL_HANDLER (StdRead,			rtcIntEnableWrite,		rtcIntEnable);
	INSTALL_HANDLER (StdRead,			StdWrite,				stopWatch);
}


// ---------------------------------------------------------------------------
//		 MC68328Bank::GetLong
// ---------------------------------------------------------------------------

uae_u32 MC68328Bank::GetLong (uaecptr iAddress)
{
#if PROFILE_MEMORY
	gMemoryAccess[kREGLongRead]++;
	if (iAddress & 2)
		gMemoryAccess[kREGLongRead2]++;
#endif

#if (CHECK_FOR_ADDRESS_ERROR)
	if ((iAddress & 1) != 0)
	{
		Software::AddressError ();
	}
#endif

#if (PREVENT_USER_REGISTER_GET)
	if (gMemAccessFlags.fProtect_RegisterGet && ROMBank::IsPCInRAM ())
	{
		PreventedAccess (iAddress, 4, true);
	}
#endif

#if (VALIDATE_REGISTER_GET)
	if (gMemAccessFlags.fValidate_RegisterGet && !ValidAddress (iAddress, 4))
	{
		InvalidAccess (iAddress, 4, true);
	}
#endif

#if HAS_PROFILING
	CYCLE_GETLONG (WAITSTATES_REGISTERS);
#endif

	uae_u32			offset = iAddress - kMemoryStart;
	ReadFunction	fn = MC68328Reg::UnsupportedRead;

	if (offset < sizeof (HwrDBallType))
	{
		fn = gReadFunctions[offset];
	}

	assert (fn);
	return fn (iAddress, 4);
}


// ---------------------------------------------------------------------------
//		 MC68328Bank::GetWord
// ---------------------------------------------------------------------------

uae_u32 MC68328Bank::GetWord (uaecptr iAddress)
{
#if PROFILE_MEMORY
	gMemoryAccess[kREGWordRead]++;
#endif

#if (CHECK_FOR_ADDRESS_ERROR)
	if ((iAddress & 1) != 0)
	{
		Software::AddressError ();
	}
#endif

#if (PREVENT_USER_REGISTER_GET)
	if (gMemAccessFlags.fProtect_RegisterGet && ROMBank::IsPCInRAM ())
	{
		PreventedAccess (iAddress, 2, true);
	}
#endif

#if (VALIDATE_REGISTER_GET)
	if (gMemAccessFlags.fValidate_RegisterGet && !ValidAddress (iAddress, 2))
	{
		InvalidAccess (iAddress, 2, true);
	}
#endif

#if HAS_PROFILING
	CYCLE_GETWORD (WAITSTATES_REGISTERS);
#endif

	uae_u32			offset = iAddress - kMemoryStart;
	ReadFunction	fn = MC68328Reg::UnsupportedRead;

	if (offset < sizeof (HwrDBallType))
	{
		fn = gReadFunctions[offset];
	}

	assert (fn);
	return fn (iAddress, 2);
}


// ---------------------------------------------------------------------------
//		 MC68328Bank::GetByte
// ---------------------------------------------------------------------------

uae_u32 MC68328Bank::GetByte (uaecptr iAddress)
{
#if PROFILE_MEMORY
	gMemoryAccess[kREGByteRead]++;
#endif

#if (PREVENT_USER_REGISTER_GET)
	if (gMemAccessFlags.fProtect_RegisterGet && ROMBank::IsPCInRAM ())
	{
		PreventedAccess (iAddress, 1, true);
	}
#endif

#if (VALIDATE_REGISTER_GET)
	if (gMemAccessFlags.fValidate_RegisterGet && !ValidAddress (iAddress, 1))
	{
		InvalidAccess (iAddress, 1, true);
	}
#endif

#if HAS_PROFILING
	CYCLE_GETBYTE (WAITSTATES_REGISTERS);
#endif

	uae_u32			offset = iAddress - kMemoryStart;
	ReadFunction	fn = MC68328Reg::UnsupportedRead;

	if (offset < sizeof (HwrDBallType))
	{
		fn = gReadFunctions[offset];
	}

	assert (fn);
	return fn (iAddress, 1);
}


// ---------------------------------------------------------------------------
//		 MC68328Bank::SetLong
// ---------------------------------------------------------------------------

void MC68328Bank::SetLong (uaecptr iAddress, uae_u32 iLongValue)
{
#if PROFILE_MEMORY
	gMemoryAccess[kREGLongWrite]++;
	if (iAddress & 2)
		gMemoryAccess[kREGLongWrite2]++;
#endif

#if (CHECK_FOR_ADDRESS_ERROR)
	if ((iAddress & 1) != 0)
	{
		Software::AddressError ();
	}
#endif

#if (PREVENT_USER_REGISTER_SET)
	if (gMemAccessFlags.fProtect_RegisterSet && ROMBank::IsPCInRAM ())
	{
		PreventedAccess (iAddress, 4, false);
	}
#endif

#if (VALIDATE_REGISTER_SET)
	if (gMemAccessFlags.fValidate_RegisterSet && !ValidAddress (iAddress, 4))
	{
		InvalidAccess (iAddress, 4, false);
	}
#endif

#if HAS_PROFILING
	CYCLE_PUTLONG (WAITSTATES_REGISTERS);
#endif

	uae_u32			offset = iAddress - kMemoryStart;
	WriteFunction	fn = MC68328Reg::UnsupportedWrite;

	if (offset < sizeof (HwrDBallType))
	{
		fn = gWriteFunctions[offset];
	}

	assert (fn);
	fn (iAddress, 4, iLongValue);

	// See if any interesting memory locations have changed.  If so,
	// CheckStepSpy will change gBreakReason, which we'll check later.

	Debug::CheckStepSpy (iAddress, 4);
}


// ---------------------------------------------------------------------------
//		 MC68328Bank::SetWord
// ---------------------------------------------------------------------------

void MC68328Bank::SetWord (uaecptr iAddress, uae_u32 iWordValue)
{
#if PROFILE_MEMORY
	gMemoryAccess[kREGWordWrite]++;
#endif

#if (CHECK_FOR_ADDRESS_ERROR)
	if ((iAddress & 1) != 0)
	{
		Software::AddressError ();
	}
#endif

#if (PREVENT_USER_REGISTER_SET)
	if (gMemAccessFlags.fProtect_RegisterSet && ROMBank::IsPCInRAM ())
	{
		PreventedAccess (iAddress, 2, false);
	}
#endif

#if (VALIDATE_REGISTER_SET)
	if (gMemAccessFlags.fValidate_RegisterSet && !ValidAddress (iAddress, 2))
	{
		InvalidAccess (iAddress, 2, false);
	}
#endif

#if HAS_PROFILING
	CYCLE_PUTWORD (WAITSTATES_REGISTERS);
#endif

	uae_u32			offset = iAddress - kMemoryStart;
	WriteFunction	fn = MC68328Reg::UnsupportedWrite;

	if (offset < sizeof (HwrDBallType))
	{
		fn = gWriteFunctions[offset];
	}

	assert (fn);
	fn (iAddress, 2, iWordValue);

	// See if any interesting memory locations have changed.  If so,
	// CheckStepSpy will change gBreakReason, which we'll check later.

	Debug::CheckStepSpy (iAddress, 2);
}


// ---------------------------------------------------------------------------
//		 MC68328Bank::SetByte
// ---------------------------------------------------------------------------

void MC68328Bank::SetByte (uaecptr iAddress, uae_u32 iByteValue)
{
#if PROFILE_MEMORY
	gMemoryAccess[kREGByteWrite]++;
#endif

#if (PREVENT_USER_REGISTER_SET)
	if (gMemAccessFlags.fProtect_RegisterSet && ROMBank::IsPCInRAM ())
	{
		PreventedAccess (iAddress, 1, false);
	}
#endif

#if (VALIDATE_REGISTER_SET)
	if (gMemAccessFlags.fValidate_RegisterSet && !ValidAddress (iAddress, 1))
	{
		InvalidAccess (iAddress, 1, false);
	}
#endif

#if HAS_PROFILING
	CYCLE_PUTBYTE (WAITSTATES_REGISTERS);
#endif

	uae_u32			offset = iAddress - kMemoryStart;
	WriteFunction	fn = MC68328Reg::UnsupportedWrite;

	if (offset < sizeof (HwrDBallType))
	{
		fn = gWriteFunctions[offset];
	}

	assert (fn);
	fn (iAddress, 1, iByteValue);

	// See if any interesting memory locations have changed.  If so,
	// CheckStepSpy will change gBreakReason, which we'll check later.

	Debug::CheckStepSpy (iAddress, 1);
}


// ---------------------------------------------------------------------------
//		 MC68328Bank::ValidAddress
// ---------------------------------------------------------------------------

int MC68328Bank::ValidAddress (uaecptr iAddress, uae_u32 iSize)
{
	UNUSED_PARAM(iSize)

	uae_u32			offset = iAddress - kMemoryStart;
	WriteFunction	fn = MC68328Reg::UnsupportedWrite;

	if (offset < sizeof (HwrDBallType))
	{
		fn = gWriteFunctions[offset];
	}	

	int result = (fn != MC68328Reg::UnsupportedWrite);

	assert (result);

	return result;
}


// ---------------------------------------------------------------------------
//		 MC68328Bank::GetRealAddress
// ---------------------------------------------------------------------------

uae_u8* MC68328Bank::GetRealAddress (uaecptr iAddress)
{
	uae_u8*	loc = ((uae_u8*) &g68328Regs) + (iAddress - kMemoryStart);

	return loc;
}


// ---------------------------------------------------------------------------
//		 MC68328Bank::InvalidAccess
// ---------------------------------------------------------------------------

void MC68328Bank::InvalidAccess (uaecptr iAddress, long size, Bool forRead)
{
	UNUSED_PARAM(iAddress)
	UNUSED_PARAM(size)
	UNUSED_PARAM(forRead)

	Software::BusError ();
}


// ---------------------------------------------------------------------------
//		 MC68328Bank::PreventedAccess
// ---------------------------------------------------------------------------

void MC68328Bank::PreventedAccess (uaecptr iAddress, long size, Bool forRead)
{
	Memory::PreventedAccess (iAddress, size, forRead, Errors::kRegisterAccess);
}


// ---------------------------------------------------------------------------
//		 MC68328Bank::Cycle
// ---------------------------------------------------------------------------
// Handles periodic events that need to occur when the processor cycles (like
// updating timer registers).  This function is called in two places from
// Emulator::Execute.  Interestingly, the loop runs 3% FASTER if this function
// is in its own separate function instead of being inline.

void MC68328Bank::Cycle (Bool sleeping)
{
#define TIE_TICKS_TO_HOST	0

	// See if the host tick counter has changed.

#if (TIE_TICKS_TO_HOST)
	Bool	ticked = HWRegisters::GetCurrentMilliseconds () != gTmr2CurrentMilliseconds;
#else
	Bool	ticked = true;
#endif

	// Determine whether timer 2 is enabled.

	Bool	enabled = (READ_REGISTER (tmr2Control) & hwr328TmrControlEnable) != 0;

	if (enabled && ticked)
	{
		gTmr2CurrentMilliseconds = HWRegisters::GetCurrentMilliseconds ();

		// If so, increment the timer.

#if (TIE_TICKS_TO_HOST)
		const uae_u32	kCyclesPerMilliseconds	= hwrTD1Frequency / 1000;	// !!! Should get from GSysClockFreq, but let's be aware of performance issues.
		uae_u32			elapsedMilliseconds		= HWRegisters::GetCurrentMilliseconds () - gTmr2StartMilliseconds;
		uae_u32			elapsedCycles			= elapsedMilliseconds * kCyclesPerMilliseconds;
		uae_u32			prescaler				= READ_REGISTER (tmr2Prescaler);
		uae_u32			tmr2Counter				= elapsedCycles / (prescaler + 1);

		WRITE_REGISTER (tmr2Counter, tmr2Counter);
#else
		WRITE_REGISTER (tmr2Counter, READ_REGISTER (tmr2Counter) + (sleeping ? 1 : 12));
#endif

		// Determine whether the timer has reached the specified count.

		if (sleeping || READ_REGISTER (tmr2Counter) > READ_REGISTER (tmr2Compare))
		{
			// Flag the occurrence of the successful comparison.

			WRITE_REGISTER (tmr2Status, READ_REGISTER (tmr2Status) | hwr328TmrStatusCompare);

			// If the Free Run/Restart flag is not set, clear the counter.

			if ((READ_REGISTER (tmr2Control) & hwr328TmrControlFreeRun) == 0)
			{
				WRITE_REGISTER (tmr2Counter, 0);
				gTmr2StartMilliseconds = gTmr2CurrentMilliseconds;
			}

			// If the timer interrupt is enabled, post an interrupt.

			if ((READ_REGISTER (tmr2Control) & hwr328TmrControlEnInterrupt) != 0)
			{
				WRITE_REGISTER (intPendingLo, READ_REGISTER (intPendingLo) | hwr328IntLoTimer2);
				MC68328Bank::UpdateInterrupts ();
			}
		}
	}
}


// ---------------------------------------------------------------------------
//		 MC68328Bank::CycleSlowly
// ---------------------------------------------------------------------------
// Handles periodic events that need to occur when the processor cycles (like
// updating timer registers).  This function is called in two places from
// Emulator::Execute.  Interestingly, the loop runs 3% FASTER if this function
// is in its own separate function instead of being inline.

void MC68328Bank::CycleSlowly (Bool sleeping)
{
	UNUSED_PARAM(sleeping)

	// Check to see if the RTC alarm is ready to go off.  First see
	// if the RTC is enabled, and that the alarm event isn't already
	// registered (the latter check is just an optimization).

	if ((READ_REGISTER (rtcIntEnable) & hwr328RTCIntEnableAlarm) != 0 &&
		(READ_REGISTER (rtcIntStatus) & hwr328RTCIntStatusAlarm) == 0)
	{
		uae_u32	rtcAlarm = READ_REGISTER (rtcAlarm);

		long	almHour	 = (rtcAlarm & hwr328RTCAlarmHoursMask) >> hwr328RTCAlarmHoursOffset;
		long	almMin	 = (rtcAlarm & hwr328RTCAlarmMinutesMask) >> hwr328RTCAlarmMinutesOffset;
		long	almSec	 = (rtcAlarm & hwr328RTCAlarmSecondsMask) >> hwr328RTCAlarmSecondsOffset;
		long	almInSeconds = (almHour * 60 * 60) + (almMin * 60) + almSec;

		long	nowHour;
		long	nowMin;
		long	nowSec;
		::GetHostTime (&nowHour, &nowMin, &nowSec);
		long	nowInSeconds = (nowHour * 60 * 60) + (nowMin * 60) + nowSec;

		if (almInSeconds <= nowInSeconds)
		{
			WRITE_REGISTER (rtcIntStatus, READ_REGISTER (rtcIntStatus) | hwr328RTCIntStatusAlarm);
			MC68328Bank::UpdateRTCInterrupts ();
		}
	}
}


// ---------------------------------------------------------------------------
//		 MC68328Bank::GetInterruptLevel
// ---------------------------------------------------------------------------

uae_s32 MC68328Bank::GetInterruptLevel (void)
{
	uae_u16	intStatusHi = READ_REGISTER (intStatusHi);
	uae_u16	intStatusLo = READ_REGISTER (intStatusLo);

	// Level 7 = IRQ7.

	if ((intStatusHi & hwr328IntHiNMI) != 0)
		return 7;

	// Level 6 = SPIS, TMR1, IRQ6.

	if ((intStatusHi & (hwr328IntHiTimer1 | hwr328IntHiSPIS | hwr328IntHiIRQ6)) != 0)
		return 6;

	// Level 5 = PEN.

	if ((intStatusHi & hwr328IntHiPen) != 0)
		return 5;

	// Level 4 = SPIM, TMR2, UART, WDT, RTC, KB, PWM, INT0 - INT7.

	if ((intStatusLo & (	hwr328IntLoAllKeys |
							hwr328IntLoPWM |
							hwr328IntLoKbd |
						//	hwr328IntLoLCDC |
							hwr328IntLoRTC |
							hwr328IntLoWDT |
							hwr328IntLoTimer2 |
							hwr328IntLoSPIM)) != 0)
		return 4;

	// Level 3 = IRQ3.

	if ((intStatusHi & hwr328IntHiIRQ3) != 0)
		return 3;

	// Level 2 = IRQ2.

	if ((intStatusHi & hwr328IntHiIRQ2) != 0)
		return 2;

	// Level 1 = IRQ1.

	if ((intStatusHi & hwr328IntHiIRQ1) != 0)
		return 1;

	// Level 0.

	return -1;
}


// ---------------------------------------------------------------------------
//		 MC68328Bank::GetInterruptBase
// ---------------------------------------------------------------------------

uae_s32 MC68328Bank::GetInterruptBase (void)
{
	return READ_REGISTER (intVector) & 0xF8;
}


// ---------------------------------------------------------------------------
//		 MC68328Bank::ButtonEvent
// ---------------------------------------------------------------------------
// Handles a Palm device button event by updating the appropriate registers.

/*
	There  appears to be two ways to read the keys.  One (the way that
	KeyBootKeys does it) is to configure Port D and then to read  Port D
	Data.  When doing this, the bit assignments are as follows:

		#define	keyBitPower			0x01			// Power key
		#define	keyBitPageUp		0x02			// Page-up
		#define	keyBitPageDown		0x04			// Page-down
		#define	keyBitHard1			0x08			// App #1
		#define	keyBitHard2			0x10			// App #2
		#define	keyBitHard3			0x20			// App #3
		#define	keyBitHard4			0x40			// App #4

	Additionally:

		#define	hwrTD1PortDNoExtPower		0x80			// (L) External DC input

	KeyCurrentState also returns the current state of the keys, but
	their state is determined differently.  KeyCurrentState merely
	returns the value of the system global GKeyGlobalsP->keyState.
	This field was filled in during a keyboard interrupt (IRQ4).  In
	that interrupt handler, the state of the keys is determined
	by the INT0-INT6 bits in the interrupt status register, as well
	as the IRQ1 bit (which corresponds to whether or not the cradle
	button is being pressed.  If that button is down, the following
	bit is defined and returned by KeyCurrentState:

		#define	keyBitCradle		0x80			// Button on cradle
*/

void MC68328Bank::ButtonEvent (	Bool	iButton_IsDown,
								uae_s32	iButton)
{
	uae_u8	previousPortDData = READ_REGISTER (portDData);
	uae_u8	portDData = previousPortDData;

	if (iButton_IsDown)
	{
		portDData |= iButton;
	}
	else
	{
		portDData &= ~iButton;
	}

	WRITE_REGISTER (portDData, portDData);

	// Set the interrupt bits for the bits that went from off to on.

	gPortDEdge |= portDData & ~previousPortDData;

	// Set the new interrupt state.

	MC68328Bank::UpdatePortDInterrupts ();
}


// ---------------------------------------------------------------------------
//		 MC68328Bank::HotSyncEvent
// ---------------------------------------------------------------------------
// Handles a HotSync button event by updating the appropriate registers.

void MC68328Bank::HotSyncEvent (Bool iButton_IsDown)
{
	// If the button changes state, set or clear the HotSync interrupt.

	uae_u16	intPendingHi = READ_REGISTER (intPendingHi);

	if (iButton_IsDown)
	{
		intPendingHi |= hwr328IntHiIRQ1;
		gHotSyncButtonDown = true;
	}
	else
	{
		intPendingHi &= ~hwr328IntHiIRQ1;
		gHotSyncButtonDown = false;
	}

	WRITE_REGISTER (intPendingHi, intPendingHi);

	MC68328Bank::UpdateInterrupts ();
}


// ---------------------------------------------------------------------------
//		 MC68328Bank::GetLCDDepth
// ---------------------------------------------------------------------------

int MC68328Bank::GetLCDDepth (void)
{
	Bool	grayscale = (READ_REGISTER (lcdPanelControl) & hwr328LcdPanelControlGrayScale) != 0;

	return grayscale ? 2 : 1;
}


// ---------------------------------------------------------------------------
//		 MC68328Bank::GetLCDRowBytes
// ---------------------------------------------------------------------------

int MC68328Bank::GetLCDRowBytes (void)
{
	int width = READ_REGISTER (lcdPageWidth) * 2;

	return width;
}


// ---------------------------------------------------------------------------
//		 MC68328Bank::GetLCDWidth
// ---------------------------------------------------------------------------

int MC68328Bank::GetLCDWidth (void)
{
	int width = READ_REGISTER (lcdScreenWidth) + 1;

	return width;
}


// ---------------------------------------------------------------------------
//		 MC68328Bank::GetLCDHeight
// ---------------------------------------------------------------------------

int MC68328Bank::GetLCDHeight (void)
{
	int height = READ_REGISTER (lcdScreenHeight) + 1;

	return height;
}


// ---------------------------------------------------------------------------
//		 MC68328Bank::GetLCDStartAddr
// ---------------------------------------------------------------------------

uaecptr MC68328Bank::GetLCDStartAddr (void)
{
	uaecptr	addr = READ_REGISTER (lcdStartAddr);

	return addr;
}


// ---------------------------------------------------------------------------
//		 MC68328Bank::GetLCDOffset
// ---------------------------------------------------------------------------

int MC68328Bank::GetLCDOffset (void)
{
	int	offset = READ_REGISTER (lcdPanningOffset) & 0x0F;

	return offset;
}


// ---------------------------------------------------------------------------
//		 MC68328Bank::LCDIsOn
// ---------------------------------------------------------------------------

Bool MC68328Bank::LCDIsOn (void)
{
	Bool	on = (READ_REGISTER (portFData) & hwrTD1PortFLCDEnableOn) != 0;

	return on;
}


// ---------------------------------------------------------------------------
//		 MC68328Bank::BacklightIsOn
// ---------------------------------------------------------------------------

Bool MC68328Bank::BacklightIsOn (void)
{
	Bool	on = (READ_REGISTER (portGData) & hwrTD1PortGBacklightOn) != 0;

	return on;
}


// ---------------------------------------------------------------------------
//		 MC68328Bank::TurnSoundOff
// ---------------------------------------------------------------------------

void MC68328Bank::TurnSoundOff (void)
{
	uae_u16	pwmControl = READ_REGISTER (pwmControl);
	WRITE_REGISTER (pwmControl, pwmControl & ~hwr328PWMControlEnable);
}


// ---------------------------------------------------------------------------
//		 MC68328Bank::GetDynamicHeapSize
// ---------------------------------------------------------------------------

uae_u32 MC68328Bank::GetDynamicHeapSize (void)
{
	uae_u32	result = 0;

	uae_u32	csCSelect0 = READ_REGISTER (csCSelect0) & ADDRESS_MASK;
	uae_u32	csCSelect1 = READ_REGISTER (csCSelect1) & ADDRESS_MASK;

	if (csCSelect0 == 0x0000 && csCSelect1 == 0x0000)
	{
		result = 0 * 1024L;
	}
	else if (csCSelect0 == 0x0070 && csCSelect1 == 0x0000)
	{
		result = 32 * 1024L;
	}
	else if (csCSelect0 == 0x00F0 && csCSelect1 == 0x0000)
	{
		result = 64 * 1024L;
	}
	else if (csCSelect0 == 0x00F0 && csCSelect1 == 0x0070)
	{
		result = 96 * 1024L;
	}
	else if (csCSelect0 == 0x01F0 && csCSelect1 == 0x0000)
	{
		result = 128 * 1024L;
	}
	else if (csCSelect0 == 0x03F0 && csCSelect1 == 0x0000)
	{
		result = 256 * 1024L;
	}
	else
	{
		assert (false);
	}

	return result;
}


// ---------------------------------------------------------------------------
//		 MC68328Bank::GetROMSize
// ---------------------------------------------------------------------------

uae_u32 MC68328Bank::GetROMSize (void)
{
	uae_u32	result = 2 * 1024L * 1024L;

	return result;
}


// ---------------------------------------------------------------------------
//		 MC68328Bank::CanStop
// ---------------------------------------------------------------------------

Bool MC68328Bank::CanStop (void)
{
	// Make sure Timer 2 is enabled or the RTC interrupt is enabled.

	if ((READ_REGISTER (tmr2Control) & hwr328TmrControlEnable) != 0)
		return true;

	if ((READ_REGISTER (rtcIntEnable) & hwr328RTCIntEnableAlarm) != 0)
		return true;

	return false;
}


// ---------------------------------------------------------------------------
//		 MC68328Bank::UpdateInterrupts
// ---------------------------------------------------------------------------
// Determines whether an interrupt has occurred by copying the Interrupt
// Pending Register to the Interrupt Status Register.

void MC68328Bank::UpdateInterrupts (void)
{
	// Copy the Interrupt Pending Register to the Interrupt Status
	// Register, but ignore interrupts that are being masked.

	// Note: this function is not sensitive to the byte ordering of the registers,
	// so their contents don't need to be accessed via READ_REGISTER or WRITE_REGISTER.

	g68328Regs.intStatusHi = g68328Regs.intPendingHi & ~g68328Regs.intMaskHi;
	g68328Regs.intStatusLo = g68328Regs.intPendingLo & ~g68328Regs.intMaskLo;

	// If the Interrupt Status Register isn't clear, flag an interrupt.

	if (g68328Regs.intStatusHi || g68328Regs.intStatusLo)
	{
		regs.spcflags |= SPCFLAG_INT;
	}
}


// ---------------------------------------------------------------------------
//		 MC68328Bank::UpdatePortDInterrupts
// ---------------------------------------------------------------------------
// Determine what interrupts need to be generated based on the current
// settings in portDData and gPortDEdge.

void MC68328Bank::UpdatePortDInterrupts (void)
{
	// Update INT0-INT7 of the Interrupt-Pending register (bits 8-15 of the low word).

	// First, get those bits and clear them out.

	uae_u32	intPendingLo	= READ_REGISTER (intPendingLo) & 0x00FF;

	// Determine the new interrupt settings. An interrupt is signalled if:
	//
	//	- its gPortDEdge bit is set along with its corresponding portDIntEdge bit, or
	//	- its portDData bit is set but its corresponding portDIntEdge bit is not, and
	//	- that interrupt is enabled in portDIntReqen
	//
	// (Technically, it seems to me that we should also be looking at portDPolarity when
	//  examining edge interrupts to signal falling or rising edges.  However, those bits
	//  are always set by the current Palm OS ROMs (making them active low or falling edge),
	//  so we can work assuming that for now.  It also seems to me that we should be
	//  looking at portDDir so that we signal interrupts only on input bits, and not
	//  output bits (as happens with the up/down keys when we go to sleep).)

	uae_u8	portDIntEdge	= READ_REGISTER (portDIntEdge);
	uae_u8	portDData		= READ_REGISTER (portDData);
	uae_u8	portDIntReqEn	= READ_REGISTER (portDIntReqEn);
	uae_u8	newBits			= (gPortDEdge & portDIntEdge | portDData & ~portDIntEdge) & portDIntReqEn;

	// Merge in the new values and write out the result.

	intPendingLo |=	newBits << 8;
	WRITE_REGISTER (intPendingLo, intPendingLo);

	// Respond to the new interrupt state.

	MC68328Bank::UpdateInterrupts ();
}


// ---------------------------------------------------------------------------
//		 MC68328Bank::UpdateRTCInterrupts
// ---------------------------------------------------------------------------
// Determine whether to set or clear the RTC bit in the interrupt pending
// register based on the current RTC register values.

void MC68328Bank::UpdateRTCInterrupts (void)
{
	// See if the RTC is enabled.

	Bool	rtcEnabled = (READ_REGISTER (rtcControl) & hwr328RTCControlRTCEnable) != 0;

	// See if there are any RTC events that need to trigger an interrupt.

#define BITS_TO_CHECK				( \
	hwr328RTCIntEnableSec			| \
	hwr328RTCIntEnable24Hr			| \
	hwr328RTCIntEnableAlarm			| \
	hwr328RTCIntEnableMinute		| \
	hwr328RTCIntEnableStopWatch		)

	uae_u16	rtcIntStatus = READ_REGISTER (rtcIntStatus);
	uae_u16	rtcIntEnable = READ_REGISTER (rtcIntEnable);
	uae_u16	rtcIntPending = rtcIntStatus & rtcIntEnable & BITS_TO_CHECK;

	Bool	havePendingEvents = rtcIntPending != 0;

	// If the RTC is enabled and there are pending events, set the interrupt.
	// Otherwise, clear the interrupt.

	uae_u16	intPendingLo = READ_REGISTER (intPendingLo);

	if (rtcEnabled && havePendingEvents)
	{
		intPendingLo |= hwr328IntLoRTC;		// have events, so set interrupt
	}
	else
	{	intPendingLo &= ~hwr328IntLoRTC;	// no events, so clear interrupt
	}

	// Update the interrupt pending register.

	WRITE_REGISTER (intPendingLo, intPendingLo);

	// Respond to the new interrupt state.

	MC68328Bank::UpdateInterrupts ();
}


// ---------------------------------------------------------------------------
//		 MC68328Bank::UARTStateChanged
// ---------------------------------------------------------------------------

void MC68328Bank::UARTStateChanged (Bool sendTxData)
{
	UART::State	state (UART::kUART_Dragonball);
	MarshalUARTState (state);
	UART::StateChanged (state, sendTxData);
	UnmarshalUARTState (state);

	MC68328Bank::UpdateUARTInterrupts (state);
}


// ---------------------------------------------------------------------------
//		 MC68328Bank::UpdateUARTState
// ---------------------------------------------------------------------------

void MC68328Bank::UpdateUARTState (Bool refreshRxData)
{
//	if (UART::GetStateNeedsUpdating ())
	{
		UART::State	state (UART::kUART_Dragonball);
		MarshalUARTState (state);
		UART::UpdateState (state, refreshRxData);
		UnmarshalUARTState (state);

		MC68328Bank::UpdateUARTInterrupts (state);
	}
}


// ---------------------------------------------------------------------------
//		 MC68328Bank::UpdateUARTInterrupts
// ---------------------------------------------------------------------------

void MC68328Bank::UpdateUARTInterrupts (const UART::State& state)
{
	// Generate the appropriate interrupts.  Don't generate interrupts for
	// TX_FIFO_EMPTY and TX_FIFO_HALF; from the manual's overview: "The transmitter
	// will not generate another interrupt until the FIFO has completely emptied."
	//
	// Actually, this code is not quite right.  If TX_AVAIL_ENABLE is false but one
	// of the other TX_xxx_ENABLEs is true, then we want to generate an interrupt
	// for one of those.  With the test below, that will never happen.  For now,
	// this is OK, as the Palm OS doesn't enable any of the TX interrupts.

	if (state.RX_FULL_ENABLE	&& state.RX_FIFO_FULL	||
		state.RX_HALF_ENABLE	&& state.RX_FIFO_HALF	||
		state.RX_RDY_ENABLE		&& state.DATA_READY		||
	//	state.TX_EMPTY_ENABLE	&& state.TX_FIFO_EMPTY	||
	//	state.TX_HALF_ENABLE	&& state.TX_FIFO_HALF	||
		state.TX_AVAIL_ENABLE	&& state.TX_AVAIL)
	{
		// Set the UART interrupt.

		WRITE_REGISTER (intPendingLo, READ_REGISTER (intPendingLo) | hwr328IntLoUART);
	}
	else
	{
		// Clear the UART interrupt.

		WRITE_REGISTER (intPendingLo, READ_REGISTER (intPendingLo) & ~hwr328IntLoUART);
	}

	// Respond to the new interrupt state.

	MC68328Bank::UpdateInterrupts ();
}


// ---------------------------------------------------------------------------
//		 MC68328Bank::MarshalUARTState
// ---------------------------------------------------------------------------

void MC68328Bank::MarshalUARTState (UART::State& state)
{
	uae_u16	uControl	= READ_REGISTER (uControl);
	uae_u16	uBaud		= READ_REGISTER (uBaud);
	uae_u16	uReceive	= READ_REGISTER (uReceive);
	uae_u16	uTransmit	= READ_REGISTER (uTransmit);
	uae_u16	uMisc		= READ_REGISTER (uMisc);

	state.UART_ENABLE		= (uControl & hwr328UControlUARTEnable) != 0;
	state.RX_ENABLE			= (uControl & hwr328UControlRxEnable) != 0;
	state.TX_ENABLE			= (uControl & hwr328UControlTxEnable) != 0;
	state.RX_CLK_CONT		= (uControl & hwr328UControlRxClock1x) != 0;
	state.PARITY_EN			= (uControl & hwr328UControlParityEn) != 0;
	state.ODD_EVEN			= (uControl & hwr328UControlParityOdd) != 0;
	state.STOP_BITS			= (uControl & hwr328UControlStopBits2) != 0;
	state.CHAR8_7			= (uControl & hwr328UControlDataBits8) != 0;
	state.GPIO_DELTA_ENABLE	= (uControl & hwr328UControlGPIODeltaEn) != 0;	// 68328 only
//	state.OLD_ENABLE		= (uControl & hwrEZ328UControlOldDataEn) != 0;	// 68EZ328 only
	state.CTS_DELTA_ENABLE	= (uControl & hwr328UControlCTSDeltaEn) != 0;
	state.RX_FULL_ENABLE	= (uControl & hwr328UControlRxFullEn) != 0;
	state.RX_HALF_ENABLE	= (uControl & hwr328UControlRxHalfEn) != 0;
	state.RX_RDY_ENABLE		= (uControl & hwr328UControlRxRdyEn) != 0;
	state.TX_EMPTY_ENABLE	= (uControl & hwr328UControlTxEmptyEn) != 0;
	state.TX_HALF_ENABLE	= (uControl & hwr328UControlTxHalfEn) != 0;
	state.TX_AVAIL_ENABLE	= (uControl & hwr328UControlTxAvailEn) != 0;

	// Baud control register bits
	// These are all values the user sets; we just look at them.

	state.GPIO_DELTA		= (uBaud & hwr328UBaudGPIODelta) != 0;			// 68328 only
	state.GPIO				= (uBaud & hwr328UBaudGPIOData) != 0;			// 68328 only
	state.GPIO_DIR			= (uBaud & hwr328UBaudGPIODirOut) != 0;			// 68328 only
	state.GPIO_SRC			= (uBaud & hwr328UBaudGPIOSrcBaudGen) != 0;		// 68328 only
//	state.UCLK_DIR			= (uBaud & hwrEZ328UBaudUCLKDirOut) != 0;		// 68EZ328 only
	state.BAUD_SRC			= (uBaud & hwr328UBaudBaudSrcGPIO) != 0;
	state.DIVIDE			= (uBaud & hwr328UBaudDivider) >> hwr328UBaudDivideBitOffset;
	state.PRESCALER			= (uBaud & hwr328UBaudPrescaler);

	// Receive register bits
	// These are all input bits; we set them, not the user.

	state.RX_FIFO_FULL		= (uReceive & hwr328UReceiveFIFOFull) != 0;
	state.RX_FIFO_HALF		= (uReceive & hwr328UReceiveFIFOHalf) != 0;
	state.DATA_READY		= (uReceive & hwr328UReceiveDataRdy) != 0;
//	state.OLD_DATA			= (uReceive & hwrEZ328UReceiveOldData) != 0;	// 68EZ328 only
	state.OVRUN				= (uReceive & hwr328UReceiveOverrunErr) != 0;
	state.FRAME_ERROR		= (uReceive & hwr328UReceiveFrameErr) != 0;
	state.BREAK				= (uReceive & hwr328UReceiveBreakErr) != 0;
	state.PARITY_ERROR		= (uReceive & hwr328UReceiveParityErr) != 0;
	state.RX_DATA			= (uReceive & hwr328UReceiveData);

	// Transmitter register bits
	// We set everything except TX_DATA; the user sets that
	// value and ONLY that value.

	state.TX_FIFO_EMPTY		= (uTransmit & hwr328UTransmitFIFOEmpty) != 0;
	state.TX_FIFO_HALF		= (uTransmit & hwr328UTransmitFIFOHalf) != 0;
	state.TX_AVAIL			= (uTransmit & hwr328UTransmitTxAvail) != 0;
	state.SEND_BREAK		= (uTransmit & hwr328UTransmitSendBreak) != 0;
	state.IGNORE_CTS		= (uTransmit & hwr328UTransmitIgnoreCTS) != 0;
//	state.BUSY				= (uTransmit & hwrEZ328UTransmitBusy) != 0;		// 68EZ328 only
	state.CTS_STATUS		= (uTransmit & hwr328UTransmitCTSStatus) != 0;
	state.CTS_DELTA			= (uTransmit & hwr328UTransmitCTSDelta) != 0;
	state.TX_DATA			= (uTransmit & hwr328UTransmitData);

	// Misc register bits
	// These are all values the user sets; we just look at them.

//	state.BAUD_TEST			= (uMisc & hwrEZ328UMiscBaudTest) != 0;			// 68EZ328 only
	state.CLK_SRC			= (uMisc & hwr328UMiscClkSrcGPIO) != 0;
	state.FORCE_PERR		= (uMisc & hwr328UMiscForceParityErr) != 0;
	state.LOOP				= (uMisc & hwr328UMiscLoopback) != 0;
//	state.BAUD_RESET		= (uMisc & hwrEZ328UMiscBaudReset) != 0;		// 68EZ328 only
//	state.IR_TEST			= (uMisc & hwrEZ328UMiscIRTestEn) != 0;			// 68EZ328 only
	state.RTS_CONT			= (uMisc & hwr328UMiscRTSThruFIFO) != 0;
	state.RTS				= (uMisc & hwr328UMiscRTSOut) != 0;
	state.IRDA_ENABLE		= (uMisc & hwr328UMiscIRDAEn) != 0;
	state.IRDA_LOOP			= (uMisc & hwr328UMiscLoopIRDA) != 0;
//	state.RX_POL			= (uMisc & hwrEZ328UMiscRXPolarityInv) != 0;	// 68EZ328 only
//	state.TX_POL			= (uMisc & hwrEZ328UMiscTXPolarityInv) != 0;	// 68EZ328 only
}


// ---------------------------------------------------------------------------
//		 MC68328Bank::UnmarshalUARTState
// ---------------------------------------------------------------------------

void MC68328Bank::UnmarshalUARTState (const UART::State& state)
{
	uae_u16	uControl	= 0;
	uae_u16	uBaud		= 0;
	uae_u16	uReceive	= 0;
	uae_u16	uTransmit	= 0;
	uae_u16	uMisc		= 0;

	if (state.UART_ENABLE)		uControl |= hwr328UControlUARTEnable;
	if (state.RX_ENABLE)		uControl |= hwr328UControlRxEnable;
	if (state.TX_ENABLE)		uControl |= hwr328UControlTxEnable;
	if (state.RX_CLK_CONT)		uControl |= hwr328UControlRxClock1x;
	if (state.PARITY_EN)		uControl |= hwr328UControlParityEn;
	if (state.ODD_EVEN)			uControl |= hwr328UControlParityOdd;
	if (state.STOP_BITS)		uControl |= hwr328UControlStopBits2;
	if (state.CHAR8_7)			uControl |= hwr328UControlDataBits8;
	if (state.GPIO_DELTA_ENABLE)uControl |= hwr328UControlGPIODeltaEn;	// 68328 only
//	if (state.OLD_ENABLE)		uControl |= hwrEZ328UControlOldDataEn;	// 68EZ328 only
	if (state.CTS_DELTA_ENABLE)	uControl |= hwr328UControlCTSDeltaEn;
	if (state.RX_FULL_ENABLE)	uControl |= hwr328UControlRxFullEn;
	if (state.RX_HALF_ENABLE)	uControl |= hwr328UControlRxHalfEn;
	if (state.RX_RDY_ENABLE)	uControl |= hwr328UControlRxRdyEn;
	if (state.TX_EMPTY_ENABLE)	uControl |= hwr328UControlTxEmptyEn;
	if (state.TX_HALF_ENABLE)	uControl |= hwr328UControlTxHalfEn;
	if (state.TX_AVAIL_ENABLE)	uControl |= hwr328UControlTxAvailEn;

	// Baud control register bits
	// These are all values the user sets; we just look at them.

	if (state.GPIO_DELTA)		uBaud |= hwr328UBaudGPIODelta;		// 68328 only
	if (state.GPIO)				uBaud |= hwr328UBaudGPIOData;		// 68328 only
	if (state.GPIO_DIR)			uBaud |= hwr328UBaudGPIODirOut;		// 68328 only
	if (state.GPIO_SRC)			uBaud |= hwr328UBaudGPIOSrcBaudGen;	// 68328 only
//	if (state.UCLK_DIR)			uBaud |= hwrEZ328UBaudUCLKDirOut;	// 68EZ328 only
	if (state.BAUD_SRC)			uBaud |= hwr328UBaudBaudSrcGPIO;

	uBaud |= (state.DIVIDE << hwr328UBaudDivideBitOffset) & hwr328UBaudDivider;
	uBaud |= (state.PRESCALER) & hwr328UBaudPrescaler;

	// Receive register bits
	// These are all input bits; we set them, not the user.

	if (state.RX_FIFO_FULL)		uReceive |= hwr328UReceiveFIFOFull;
	if (state.RX_FIFO_HALF)		uReceive |= hwr328UReceiveFIFOHalf;
	if (state.DATA_READY)		uReceive |= hwr328UReceiveDataRdy;
//	if (state.OLD_DATA)			uReceive |= hwrEZ328UReceiveOldData;	// 68EZ328 only
	if (state.OVRUN)			uReceive |= hwr328UReceiveOverrunErr;
	if (state.FRAME_ERROR)		uReceive |= hwr328UReceiveFrameErr;
	if (state.BREAK)			uReceive |= hwr328UReceiveBreakErr;
	if (state.PARITY_ERROR)		uReceive |= hwr328UReceiveParityErr;

	uReceive |= (state.RX_DATA) & hwr328UReceiveData;

	// Transmitter register bits
	// We set everything except TX_DATA; the user sets that
	// value and ONLY that value.

	if (state.TX_FIFO_EMPTY)	uTransmit |= hwr328UTransmitFIFOEmpty;
	if (state.TX_FIFO_HALF)		uTransmit |= hwr328UTransmitFIFOHalf;
	if (state.TX_AVAIL)			uTransmit |= hwr328UTransmitTxAvail;
	if (state.SEND_BREAK)		uTransmit |= hwr328UTransmitSendBreak;
	if (state.IGNORE_CTS)		uTransmit |= hwr328UTransmitIgnoreCTS;
//	if (state.BUSY)				uTransmit |= hwrEZ328UTransmitBusy;		// 68EZ328 only
	if (state.CTS_STATUS)		uTransmit |= hwr328UTransmitCTSStatus;
	if (state.CTS_DELTA)		uTransmit |= hwr328UTransmitCTSDelta;

	uTransmit |= (state.TX_DATA) & hwr328UTransmitData;

	// Misc register bits
	// These are all values the user sets; we just look at them.

//	if (state.BAUD_TEST)		uMisc |= hwrEZ328UMiscBaudTest;			// 68EZ328 only
	if (state.CLK_SRC)			uMisc |= hwr328UMiscClkSrcGPIO;
	if (state.FORCE_PERR)		uMisc |= hwr328UMiscForceParityErr;
	if (state.LOOP)				uMisc |= hwr328UMiscLoopback;
//	if (state.BAUD_RESET)		uMisc |= hwrEZ328UMiscBaudReset;		// 68EZ328 only
//	if (state.IR_TEST)			uMisc |= hwrEZ328UMiscIRTestEn;			// 68EZ328 only
	if (state.RTS_CONT)			uMisc |= hwr328UMiscRTSThruFIFO;
	if (state.RTS)				uMisc |= hwr328UMiscRTSOut;
	if (state.IRDA_ENABLE)		uMisc |= hwr328UMiscIRDAEn;
	if (state.IRDA_LOOP)		uMisc |= hwr328UMiscLoopIRDA;
//	if (state.RX_POL)			uMisc |= hwrEZ328UMiscRXPolarityInv;	// 68EZ328 only
//	if (state.TX_POL)			uMisc |= hwrEZ328UMiscTXPolarityInv;	// 68EZ328 only

	WRITE_REGISTER (uControl, uControl);
	WRITE_REGISTER (uBaud, uBaud);
	WRITE_REGISTER (uReceive, uReceive);
	WRITE_REGISTER (uTransmit, uTransmit);
	WRITE_REGISTER (uMisc, uMisc);
}


#pragma mark -


// ===========================================================================
//		 Register Accessors
// ===========================================================================

// ---------------------------------------------------------------------------
//		 MC68328Reg::SetHandler
// ---------------------------------------------------------------------------

void MC68328Reg::SetHandler (ReadFunction read, WriteFunction write,
							uae_u32 start, int count)
{
	int	index = start - kMemoryStart;
	for (int ii = 0; ii < count; ++ii, ++index)
	{
		gReadFunctions[index] = read;
		gWriteFunctions[index] = write;
	}
}


// ---------------------------------------------------------------------------
//		 MC68328Reg::UnsupportedRead
// ---------------------------------------------------------------------------

uae_u32 MC68328Reg::UnsupportedRead (uaecptr address, int size)
{
	if (!CEnableFullAccess::AccessOK ())
	{
		MC68328Bank::InvalidAccess (address, size, true);
	}

	return ~0;
}


// ---------------------------------------------------------------------------
//		 MC68328Reg::UnsupportedWrite
// ---------------------------------------------------------------------------

void MC68328Reg::UnsupportedWrite (uaecptr address, int size, uae_u32 value)
{
	UNUSED_PARAM(value)

	if (!CEnableFullAccess::AccessOK ())
	{
		MC68328Bank::InvalidAccess (address, size, false);
	}
}


// ---------------------------------------------------------------------------
//		 MC68328Reg::StdRead
// ---------------------------------------------------------------------------

uae_u32 MC68328Reg::StdRead (uaecptr address, int size)
{
	// The order of these tests is based on the frequency at which
	// the registers are accessed.

	if (size == 1)
		return do_get_mem_byte (MC68328Bank::GetRealAddress (address));

	if (size == 2)
		return do_get_mem_word (MC68328Bank::GetRealAddress (address));

	return do_get_mem_long (MC68328Bank::GetRealAddress (address));
}


// ---------------------------------------------------------------------------
//		 MC68328Reg::StdWrite
// ---------------------------------------------------------------------------

void MC68328Reg::StdWrite (uaecptr address, int size, uae_u32 value)
{
	// The order of these tests is based on the frequency at which
	// the registers are accessed.

	if (size == 2)
		do_put_mem_word (MC68328Bank::GetRealAddress (address), value);

	else if (size == 4)
		do_put_mem_long (MC68328Bank::GetRealAddress (address), value);

	else
		do_put_mem_byte (MC68328Bank::GetRealAddress (address), value);
}


// ---------------------------------------------------------------------------
//		 MC68328Reg::NullWrite
// ---------------------------------------------------------------------------

void MC68328Reg::NullWrite (uaecptr address, int size, uae_u32 value)
{
	UNUSED_PARAM(address)
	UNUSED_PARAM(size)
	UNUSED_PARAM(value)

	// Do nothing (for read-only registers).
}


// ---------------------------------------------------------------------------
//		 MC68328Reg::pllFreqSelRead
// ---------------------------------------------------------------------------

uae_u32 MC68328Reg::pllFreqSelRead (uaecptr address, int size)
{
	// Simulate the rising and falling of the CLK32 signal so that functions like
	// HwrPreRAMInit, HwrShutDownPLL, PrvSetPLL, and PrvShutDownPLL won't hang.

	uae_u16	pllFreqSel = READ_REGISTER (pllFreqSel) ^ 0x8000;
	WRITE_REGISTER (pllFreqSel, pllFreqSel);

	// Finish up by doing a standard read.

	return MC68328Reg::StdRead (address, size);
}


// ---------------------------------------------------------------------------
//		 MC68328Reg::portCDataRead
// ---------------------------------------------------------------------------

uae_u32 MC68328Reg::portCDataRead (uaecptr address, int size)
{
	// This makes the power on key work. If the signal is not asserted, the
	// unit will not transition between asleep and awake (cf. HwrSleep, HwrWake).

	uae_u8	portCData = READ_REGISTER (portCData) | hwr328PortCNMI;
	WRITE_REGISTER (portCData, portCData);

	// Finish up by doing a standard read.

	return MC68328Reg::StdRead (address, size);
}


// ---------------------------------------------------------------------------
//		 MC68328Reg::portDDataRead
// ---------------------------------------------------------------------------
//	Used for:
//		Reading keys (KeySleep, KeyHandleInterrupt, prvKeyScan, KeyInit)
//		Hardware ID (HwrIdentifyFeatures)

uae_u32 MC68328Reg::portDDataRead (uaecptr address, int size)
{
	// See if they're looking for the hardware ID.

			uae_u8	portEDir		= READ_REGISTER (portEDir);
			uae_u8	portEData		= READ_REGISTER (portEData);
			uae_u8	portEPullupEn	= READ_REGISTER (portEPullupEn);
	const	uae_u8	kMask			= hwrTD1PortENoBacklight;

	if ((portEDir & kMask) == kMask &&
		(portEData & kMask) == 0 &&
		(portEPullupEn & kMask) == 0)
	{
		uae_u8	idBits = 0;

// -*- NEW DEVICE -*-

		// Determine the hardware ID.

		switch (Emulator::GetHardwareDevice ())
		{
			case kDevicePilot1000:
			case kDevicePilot5000:
			case kDevicePalmPilotPersonal:
			case kDevicePalmPilotProfessional:
			case kDevicePalmPilotWithUpgrade:
				idBits = keyBitHard1 | keyBitHard2 | keyBitHard3 | keyBitHard4;
				break;

			case kDevicePalmIII:
				idBits = keyBitHard2 | keyBitHard3 | keyBitHard4;
				break;

			case kDevicePalmVII:
				idBits = keyBitHard3 | keyBitHard4;
				break;

			case kDeviceQualComm:
				idBits = keyBitHard1 | keyBitHard4;
				break;

			case kDevicePalmIIIx:
			case kDevicePalmV:
			case kDeviceAustin:
			case kDevicePalmVIIEZ:
				Platform::Debugger ();
				break;

			default: assert (false);
		}

		// Return its complement.

		return ~idBits;
	}

	// Finish up by doing a standard read.

	return MC68328Reg::StdRead (address, size);
}


// ---------------------------------------------------------------------------
//		 MC68328Reg::portMDataRead
// ---------------------------------------------------------------------------

uae_u32 MC68328Reg::portMDataRead (uaecptr address, int size)
{
	// Ensure that bit hwrTD1PortMDockIn is set.  If it's clear, HotSync
	// will sync via the modem instead of the serial port.

	uae_u8	portMData = READ_REGISTER (portMData) | hwrTD1PortMDockIn;
	WRITE_REGISTER (portMData, portMData);

	// Finish up by doing a standard read.

	return MC68328Reg::StdRead (address, size);
}


// ---------------------------------------------------------------------------
//		 MC68328Reg::tmr1StatusRead
// ---------------------------------------------------------------------------

uae_u32 MC68328Reg::tmr1StatusRead (uaecptr address, int size)
{
	uae_u16	tmr1Counter = READ_REGISTER (tmr1Counter) + 16;
	uae_u16	tmr1Compare = READ_REGISTER (tmr1Compare);
	uae_u16	tmr1Control = READ_REGISTER (tmr1Control);

	// Increment the timer.

	WRITE_REGISTER (tmr1Counter, tmr1Counter);

	// If the timer has passed the specified value...

	if ( (tmr1Counter - tmr1Compare) < 16 )
	{
		// Set the flag saying the timer timed out.

		uae_u16	tmr1Status = READ_REGISTER (tmr1Status) | hwr328TmrStatusCompare;
		WRITE_REGISTER (tmr1Status, tmr1Status);

		// If it's not a free-running timer, reset it to zero.

		if ( (tmr1Control & hwr328TmrControlFreeRun) == 0 )
			WRITE_REGISTER (tmr1Counter, 0);
	}

	gLastTmr1Status |= READ_REGISTER (tmr1Status);	// remember this guy for later (see MC68328Reg::tmr1StatusWrite())

	// Finish up by doing a standard read.

	return MC68328Reg::StdRead (address, size);
}


// ---------------------------------------------------------------------------
//		 MC68328Reg::tmr2StatusRead
// ---------------------------------------------------------------------------

uae_u32 MC68328Reg::tmr2StatusRead (uaecptr address, int size)
{
#if 0	// (Greg doesn't do this for Timer 2...I wonder why)

	/*
		ram_rom.cpp: DBReg::tmr2StatusRead: Hmm, I don't update the TMR2 count
		value. As with everything else that's missing in my DragonBall
		emulation, it's probably because I never found anything that needed it.
		If you know otherwise, by all means implement it. Also, the magic value
		of 16 counts per status read seemed to be about right for the programmed
		timer 1 frequency. It isn't likely to be right for timer 2.

						-- Greg
	*/

	uae_u16	tmr2Counter = READ_REGISTER (tmr2Counter) + 16;
	uae_u16	tmr2Compare = READ_REGISTER (tmr2Compare);
	uae_u16	tmr2Control = READ_REGISTER (tmr2Control);

	// Increment the timer.

	WRITE_REGISTER (tmr2Counter, tmr2Counter);

	// If the timer has passed the specified value...

	if ( (tmr2Counter - tmr2Compare) < 16 )
	{
		// Set the flag saying the timer timed out.

		uae_u16	tmr2Status = READ_REGISTER (tmr1Status) | hwr328TmrStatusCompare;
		WRITE_REGISTER (tmr2Status, tmr2Status);

		// If it's not a free-running timer, reset it to zero.

		if ( (tmr2Control & hwr328TmrControlFreeRun) == 0 )
			WRITE_REGISTER (tmr2Counter, 0);
	}
#endif

	gLastTmr2Status |= READ_REGISTER (tmr2Status);	// remember this guy for later (see MC68328Reg::tmr2StatusWrite())

	// Finish up by doing a standard read.

	return MC68328Reg::StdRead (address, size);
}


// ---------------------------------------------------------------------------
//		 MC68328Reg::UartRead
// ---------------------------------------------------------------------------

uae_u32 MC68328Reg::UartRead (uaecptr address, int size)
{
	// If this is a full read, get the next byte from the FIFO.

	Bool	refreshRxData = (address == addressof (uReceive)) && (size == 2);

	// See if there's anything new ("Put the data on the bus")

	MC68328Bank::UpdateUARTState (refreshRxData);

	// Finish up by doing a standard read.

	return MC68328Reg::StdRead (address, size);
}


// ---------------------------------------------------------------------------
//		 MC68328Reg::rtcHourMinSecRead
// ---------------------------------------------------------------------------

uae_u32 MC68328Reg::rtcHourMinSecRead (uaecptr address, int size)
{
	// Get the desktop machine's time.

	long	hour, min, sec;
	::GetHostTime (&hour, &min, &sec);

	// Update the register.

	WRITE_REGISTER (rtcHourMinSec, (hour << hwr328RTCHourMinSecHoursOffset)
								| (min << hwr328RTCHourMinSecMinutesOffset)
								| (sec << hwr328RTCHourMinSecSecondsOffset));

	// Finish up by doing a standard read.

	return MC68328Reg::StdRead (address, size);
}


// ---------------------------------------------------------------------------
//		 MC68328Reg::csASelect1Write
// ---------------------------------------------------------------------------

void MC68328Reg::csASelect1Write (uaecptr address, int size, uae_u32 value)
{
	// Do a standard update of the register.

	MC68328Reg::StdWrite (address, size, value);

	// Check its new state and update our ram-protect flag.

	gMemAccessFlags.fProtect_SRAMSet = (READ_REGISTER (csASelect1) & 0x0008) != 0;
}


// ---------------------------------------------------------------------------
//		 MC68328Reg::csCSelect0Write
// ---------------------------------------------------------------------------

void MC68328Reg::csCSelect0Write (uaecptr address, int size, uae_u32 value)
{
	uae_u32	csCSelect0 = READ_REGISTER (csCSelect0);

	// Do a standard update of the register.

	MC68328Reg::StdWrite (address, size, value);

	// Check to see if the unprotected memory range changed.

	if ((csCSelect0 & ADDRESS_MASK) != (READ_REGISTER (csCSelect0) & ADDRESS_MASK))
	{
		regs.spcflags |= SPCFLAG_RESET_BANKS;
	}
}


// ---------------------------------------------------------------------------
//		 MC68328Reg::csCSelect1Write
// ---------------------------------------------------------------------------

void MC68328Reg::csCSelect1Write (uaecptr address, int size, uae_u32 value)
{
	uae_u32	csCSelect1 = READ_REGISTER (csCSelect1);

	// Do a standard update of the register.

	MC68328Reg::StdWrite (address, size, value);

	// Check to see if the unprotected memory range changed.

	if ((csCSelect1 & ADDRESS_MASK) != (READ_REGISTER (csCSelect1) & ADDRESS_MASK))
	{
		regs.spcflags |= SPCFLAG_RESET_BANKS;
	}
}


// ---------------------------------------------------------------------------
//		 MC68328Reg::intMaskHiWrite
// ---------------------------------------------------------------------------

void MC68328Reg::intMaskHiWrite (uaecptr address, int size, uae_u32 value)
{
	// Do a standard update of the register.

	MC68328Reg::StdWrite (address, size, value);

	// Respond to the new interrupt state.

	MC68328Bank::UpdateInterrupts ();
}


// ---------------------------------------------------------------------------
//		 MC68328Reg::intMaskLoWrite
// ---------------------------------------------------------------------------

void MC68328Reg::intMaskLoWrite (uaecptr address, int size, uae_u32 value)
{
	// Do a standard update of the register.

	MC68328Reg::StdWrite (address, size, value);

	// Respond to the new interrupt state.

	MC68328Bank::UpdateInterrupts ();
}


// ---------------------------------------------------------------------------
//		 MC68328Reg::intStatusHiWrite
// ---------------------------------------------------------------------------

void MC68328Reg::intStatusHiWrite (uaecptr address, int size, uae_u32 value)
{
	// IRQ1, IRQ2, IRQ3, IRQ6 and IRQ7 are cleared by writing to their
	// respective status bits.  We handle those there. Since there are
	// no interrupt status bits like this in intStatusLo, we don't need
	// a handler for that register; we only handle intStatusHi.

	// Even though this is a 16-bit register as defined by the Palm headers,
	// it's a 32-bit register according to Dragonball docs, and is in fact
	// accessed that way in the kernal files (cf. HwrIRQ4Handler). In those
	// cases, we're still only interested in access to the IRQ# bits, so we
	// can turn 4-byte accesses into 2-byte accesses.

	if (size == 4)
		value >>= 16;

	// Take into account the possibility of 1-byte accesses, too. If we're
	// accessing the upper byte, just return. If we're accessing the lower
	// byte, we can treat it as a 2-byte access.

	else if (size == 1 && address == addressof (intStatusHi))
		return;

	// Now we can treat the rest of this function as a word-write to intStatusHi.

	uae_u16	intPendingHi = READ_REGISTER (intPendingHi);

	//	For each interrupt:
	//		If we're writing to that interrupt's status bit and its edge bit is set:
	//			- clear the interrupt's pending bit
	//			- respond to the new interrupt state.

	#undef CLEAR_PENDING_INTERRUPT
	#define CLEAR_PENDING_INTERRUPT(edge, irq)							\
		if ( (READ_REGISTER (intControl) & edge) && (value & (irq)) )	\
		{																\
			intPendingHi &= ~(irq);										\
		}

	CLEAR_PENDING_INTERRUPT (hwr328IntCtlEdge1, hwr328IntHiIRQ1);
	CLEAR_PENDING_INTERRUPT (hwr328IntCtlEdge2, hwr328IntHiIRQ2);
	CLEAR_PENDING_INTERRUPT (hwr328IntCtlEdge3, hwr328IntHiIRQ3);
	CLEAR_PENDING_INTERRUPT (hwr328IntCtlEdge6, hwr328IntHiIRQ6);

	// IRQ7 is not edge-programmable, so clear it if we're merely writing to it.

	if (value & hwr328IntHiNMI)
	{
		intPendingHi &= ~(hwr328IntHiNMI);
	}

	// If we're emulating the user pressing the hotsync button, make sure the
	// interrupt stays asserted.  (!!! Should we use the same technique for
	// other buttons, too?  It doesn't seem to be needed right now, but doing
	// that may more closely mirror the hardware.)

	if (gHotSyncButtonDown)
	{
		intPendingHi |= hwr328IntHiIRQ1;
	}
	else
	{
		intPendingHi &= ~hwr328IntHiIRQ1;
	}

	WRITE_REGISTER (intPendingHi, intPendingHi);
	MC68328Bank::UpdateInterrupts ();
}


// ---------------------------------------------------------------------------
//		 MC68328Reg::portDDataWrite
// ---------------------------------------------------------------------------

void MC68328Reg::portDDataWrite (uaecptr address, int size, uae_u32 value)
{
	UNUSED_PARAM(address)
	UNUSED_PARAM(size)

	// Clear the interrupt bits that are having a 1 written to them.

	gPortDEdge &= ~value;

	// Set the new interrupt state.

	MC68328Bank::UpdatePortDInterrupts ();
}


// ---------------------------------------------------------------------------
//		 MC68328Reg::portDIntReqEnWrite
// ---------------------------------------------------------------------------

void MC68328Reg::portDIntReqEnWrite (uaecptr address, int size, uae_u32 value)
{
	// Do a standard update of the register.

	MC68328Reg::StdWrite (address, size, value);

	// Set the new interrupt state.

	MC68328Bank::UpdatePortDInterrupts ();
}


// ---------------------------------------------------------------------------
//		 MC68328Reg::portGDataWrite
// ---------------------------------------------------------------------------

void MC68328Reg::portGDataWrite (uaecptr address, int size, uae_u32 value)
{
	// Get the current value.

	uae_u16	portGData = READ_REGISTER (portGData);

	// Chain to lcdRegisterWrite first.  It will update the register, and do
	// anything else it needs to.  We let _it_ update the register because
	// it needs to also first fetch the original value of the register for
	// comparison purposes.

	lcdRegisterWrite (address, size, value);

	// If the serial driver enable bit changed, react to it.

	Bool	wasOn = (portGData & hwrTD1PortGSerialOn) != 0;
	Bool	isOn = (READ_REGISTER (portGData) & hwrTD1PortGSerialOn) != 0;

	if (isOn && !wasOn)
	{
		Platform::OpenSerialPort ();
		
		Bool	sendTxData = false;
		MC68328Bank::UARTStateChanged (sendTxData);
	}
	else if (!isOn && wasOn)
	{
		Platform::CloseSerialPort ();
	}
}


// ---------------------------------------------------------------------------
//		 MC68328Reg::tmr1StatusWrite
// ---------------------------------------------------------------------------

void MC68328Reg::tmr1StatusWrite (uaecptr address, int size, uae_u32 value)
{
	UNUSED_PARAM(address)
	UNUSED_PARAM(size)

	assert (size == 2);	// This function's a hell of a lot easier to write if
						// we assume only full-register access.

	// Get the current value.

	uae_u16	tmr1Status = READ_REGISTER (tmr1Status);

	// If the user had previously read the status bits while they
	// were set, then it's OK for them to be clear now.  Otherwise,
	// we have to merge any set status bits back in.

	tmr1Status &= value | ~gLastTmr1Status;	// gLastTmr1Status was set in MC68328Reg::tmr1StatusRead()

	WRITE_REGISTER (tmr1Status, tmr1Status);

	gLastTmr1Status = 0;
	if ( (tmr1Status & hwr328TmrStatusCompare) == 0 )
	{
		uae_u16	intPendingHi = READ_REGISTER (intPendingHi) & ~hwr328IntHiTimer1;
		WRITE_REGISTER (intPendingHi, intPendingHi);

		// Respond to the new interrupt state.

		MC68328Bank::UpdateInterrupts ();
	}
}


// ---------------------------------------------------------------------------
//		 MC68328Reg::tmr2StatusWrite
// ---------------------------------------------------------------------------

void MC68328Reg::tmr2StatusWrite (uaecptr address, int size, uae_u32 value)
{
	UNUSED_PARAM(address)
	UNUSED_PARAM(size)

	assert (size == 2);	// This function's a hell of a lot easier to write if
						// we assume only full-register access.

	// Get the current value.

	uae_u16	tmr2Status = READ_REGISTER (tmr2Status);

	// If the user had previously read the status bits while they
	// were set, then it's OK for them to be clear now.  Otherwise,
	// we have to merge any set status bits back in.

	tmr2Status &= value | ~gLastTmr2Status;	// gLastTmr2Status was set in MC68328Reg::tmr2StatusRead()

	WRITE_REGISTER (tmr2Status, tmr2Status);

	gLastTmr2Status = 0;
	if ( (tmr2Status & hwr328TmrStatusCompare) == 0 )
	{
		uae_u16	intPendingLo = READ_REGISTER (intPendingLo) & ~hwr328IntLoTimer2;
		WRITE_REGISTER (intPendingLo, intPendingLo);

		// Respond to the new interrupt state.

		MC68328Bank::UpdateInterrupts ();
	}
}


// ---------------------------------------------------------------------------
//		 MC68328Reg::wdCounterWrite
// ---------------------------------------------------------------------------

void MC68328Reg::wdCounterWrite (uaecptr address, int size, uae_u32 value)
{
	UNUSED_PARAM(address)
	UNUSED_PARAM(size)
	UNUSED_PARAM(value)

	// Always set it to zero (a write to this register always resets it).

	WRITE_REGISTER (wdCounter, 0);
}


// ---------------------------------------------------------------------------
//		 MC68328Reg::spiMasterControlWrite
// ---------------------------------------------------------------------------

void MC68328Reg::spiMasterControlWrite (uaecptr address, int size, uae_u32 value)
{
	// Do a standard update of the register.

	MC68328Reg::StdWrite (address, size, value);

	// Get the current value.

	uae_u16	spiMasterControl = READ_REGISTER (spiMasterControl);

	// Check to see if data exchange and interrupts are enabled.

	#define BIT_MASK (hwr328SPIMControlExchange | hwr328SPIMControlIntEnable)
	if ((spiMasterControl & BIT_MASK) != 0)
	{
		// If so, assert the interrupt and clear the exchange bit.

		spiMasterControl |= hwr328SPIMControlIntStatus;
		spiMasterControl &= ~hwr328SPIMControlExchange;

		WRITE_REGISTER (spiMasterControl, spiMasterControl);

/*
		// If we wanted digitizer data, load it into the SPIM data register.

		switch (READ_REGISTER (portFData) & hwrTD1PortFPanelMask)
		{
			case (hwrTD1PortFPanelCfgXMeas):
				WRITE_REGISTER (spiMasterData, (0xFF - Hardware::fgPen_HorzLocation) * 2);
				break;

			case (hwrTD1PortFPanelCfgYMeas):
				WRITE_REGISTER (spiMasterData, (0xFF - Hardware::fgPen_VertLocation) * 2);
				break;
		}
*/
	}
}


// ---------------------------------------------------------------------------
//		 MC68328Reg::UartWrite
// ---------------------------------------------------------------------------

void MC68328Reg::UartWrite(uaecptr address, int size, uae_u32 value)
{
	// Do a standard update of the register.

	MC68328Reg::StdWrite (address, size, value);

	// If this write included the TX_DATA field, signal that it needs to
	// be transmitted.

	Bool	sendTxData =
				((address == addressof (uTransmit)) && (size == 2)) ||
				((address == addressof (uTransmit) + 1) && (size == 1));

	// React to any changes.

	MC68328Bank::UARTStateChanged (sendTxData);
}


// ---------------------------------------------------------------------------
//		 MC68328Reg::lcdRegisterWrite
// ---------------------------------------------------------------------------

void MC68328Reg::lcdRegisterWrite(uaecptr address, int size, uae_u32 value)
{
	// First, get the old value in case we need to see what changed.

	uae_u32	oldValue = MC68328Reg::StdRead (address, size);

	// Do a standard update of the register.

	MC68328Reg::StdWrite (address, size, value);

	// Note what changed.

	if (address == addressof (lcdScreenWidth))
	{
		Screen::InvalidateAll ();
	}
	else if (address == addressof (lcdScreenHeight))
	{
		Screen::InvalidateAll ();
	}
	else if (address == addressof (lcdPanelControl))
	{
		if (((value ^ oldValue) & hwr328LcdPanelControlGrayScale) != 0)
		{
			Screen::InvalidateAll ();
		}
	}
	else if (address == addressof (lcdStartAddr))
	{
		// Make sure the low-bit is always zero.

		uae_u32	lcdStartAddr = READ_REGISTER (lcdStartAddr) & 0xFFFFFFFE;
		WRITE_REGISTER (lcdStartAddr, lcdStartAddr);

		Screen::InvalidateAll ();
	}
	else if (address == addressof (lcdPageWidth))
	{
		if (value != oldValue)
		{
			Screen::InvalidateAll ();
		}
	}
	else if (address == addressof (portFData))
	{
		if (((value ^ oldValue) & hwrTD1PortFLCDEnableOn) != 0)
		{
			Screen::InvalidateAll ();
		}
	}
	else if (address == addressof (portGData))
	{
		if (((value ^ oldValue) & hwrTD1PortGBacklightOn) != 0)
		{
			Screen::InvalidateAll ();
		}
	}
}


// ---------------------------------------------------------------------------
//		 MC68328Reg::rtcControlWrite
// ---------------------------------------------------------------------------

void MC68328Reg::rtcControlWrite(uaecptr address, int size, uae_u32 value)
{
	// Do a standard update of the register.

	MC68328Reg::StdWrite (address, size, value);

	// Respond to the new interrupt state.

	MC68328Bank::UpdateRTCInterrupts ();
}


// ---------------------------------------------------------------------------
//		 MC68328Reg::rtcIntStatusWrite
// ---------------------------------------------------------------------------

void MC68328Reg::rtcIntStatusWrite(uaecptr address, int size, uae_u32 value)
{
	// Status bits are cleared by writing ones to them.

	// If we're doing a byte-write to the upper byte, shift the byte
	// so that we can treat the operation as a word write.  If we're
	// doing a byte-write to the lower byte, this extension will happen
	// automatically.

	if (address == addressof (rtcIntStatus) && size == 1)
		value <<= 8;

	// Get the current value.

	uae_u16	rtcIntStatus = READ_REGISTER (rtcIntStatus);

	// Clear the requested bits.

	rtcIntStatus &= ~value;

	// Update the register.

	WRITE_REGISTER (rtcIntStatus, rtcIntStatus);

	// Respond to the new interrupt state.

	MC68328Bank::UpdateRTCInterrupts ();
}


// ---------------------------------------------------------------------------
//		 MC68328Reg::rtcIntEnableWrite
// ---------------------------------------------------------------------------

void MC68328Reg::rtcIntEnableWrite(uaecptr address, int size, uae_u32 value)
{
	// Do a standard update of the register.

	MC68328Reg::StdWrite (address, size, value);

	// Respond to the new interrupt state.

	MC68328Bank::UpdateRTCInterrupts ();
}


