/*
 * (C) Copyright Michel Casabianca 1999  All rights reserved.
 *
 * The program is provided "as is" without any warranty express or
 * implied, including the warranty of non-infringement and the implied
 * warranties of merchantibility and fitness for a particular purpose.
 * The Copyright owner will not be liable for any damages suffered by
 * you as a result of using the Program. In no event will the Copyright
 * owner be liable for any special, indirect or consequential damages or
 * lost profits even if the Copyright owner has been advised of the
 * possibility of their occurrence.
 */

package xslp.xml.parsers;

import org.xml.sax.*;
import java.util.*;
import java.io.*;

/**
 * Implementation of a simple catalog based on property files. <br>
 * Sample of such a catalog file:<br>
 * <pre>
 * # DTD pour les textes des documents du site
 * -//EOR//DTD texte//FR=file:///home/casa/doc/eor/site/projet/source/dtds/texte.dtd
 * -//EOR//DTD livre//FR=file:///home/casa/doc/eor/site/projet/source/dtds/livre.dtd
 * # DTD pour les articles
 * -//EOR//DTD article//FR=file:///home/casa/doc/eor/site/projet/source/dtds/article.dtd
 * # DTD pour les brves
 * -//EOR//DTD breve//FR=file:///home/casa/doc/eor/site/projet/source/dtds/breve.dtd
 * # DTD pour l'index du site
 * -//EOR//DTD index//FR=file:///home/casa/doc/eor/site/projet/source/dtds/index.dtd
 * </pre> 
 * @author <A HREF="mailto:casa@sdv.fr">Michel Casabianca</A>
 */
public class SimpleCatalog extends Hashtable 
    implements EntityResolver
{
    /**
     * Constructor without parameters (creates empty catalog)
     *
     */
    public SimpleCatalog() {}

    /**
     * Constructor loading a catalog
     *
     * @param file the file of the catalog to load
     */
    public SimpleCatalog(String file) {
	    loadCatalog(file);
    }

    /**
     * Loads the named catalog (appending it to the existing entries)
     *
     * @param file the file to load
     */
    public void loadCatalog(String file) {
        //-- check for null file (kvisco)
        if (file == null) return;
        try {
            FileReader fileReader = new FileReader(file);
            loadCatalog(fileReader);
            fileReader.close();
	    } catch(IOException e) {e.printStackTrace();}
    } //-- loadCatalog
    
    /**
     * Loads a catalog from the given Reader 
     * (appending it to the existing entries)
     *
     * @param reader the Reader to load the catalog from
     */
    public void loadCatalog(Reader reader) {
	    try {
	        // open catalog file
	        BufferedReader input = new BufferedReader(reader);
	        String line=null;
	        // reads lines of the catalog
	        while((line=input.readLine())!=null) {
		        line=line.trim();
		        if(!line.equals("") && !line.startsWith("#") 
		            && !(line.indexOf("=")==-1)) 
		        {
		            String id=line.substring(0,line.indexOf("="));
		            String uri=line.substring(line.indexOf("=")+1,
					            line.length());
		            // puts the pair id/uri in the catalog
		            put(id,uri);
		        }
	        }
	    } catch(IOException e) {e.printStackTrace();}
    } //-- loadCatalog

    /**
     * Resolves an entity reference (implements EntityResolver)
     *
     * @param publicId the public id
     * @param systemId the system id
     * @return input source corresponding to the entity
     */
    public InputSource resolveEntity(String publicId,String systemId) {
	    // try to find the public id
	    //-- check for null publicId (kvisco)
	    if ((publicId != null)&& (get(publicId)!=null)) {
	        return new InputSource((String) get(publicId));
	    }
	    // return InputSource for system id
    	return new InputSource(systemId);
    }
} //-- SimpleCatalog
