/*
 * (C) Copyright Keith Visco 1999  All rights reserved.
 *
 * The program is provided "as is" without any warranty express or
 * implied, including the warranty of non-infringement and the implied
 * warranties of merchantibility and fitness for a particular purpose.
 * The Copyright owner will not be liable for any damages suffered by
 * you as a result of using the Program. In no event will the Copyright
 * owner be liable for any special, indirect or consequential damages or
 * lost profits even if the Copyright owner has been advised of the
 * possibility of their occurrence.
 */

package com.kvisco.xsl;

/**
 * Represents a String result
 * @author <a href="mailto:kvisco@ziplink.net">Keith Visco</a>
**/
public class StringResult implements ExprResult {

    private String value = null;
    
      //----------------/
     //- Constructors -/
    //----------------/
    
    /**
     * Creates a new StringResult with the default value
    **/
    public StringResult() {
        super();        
    } //-- StringResult
    
    /**
     * Creates a new String Result with the given value
    **/
    public StringResult(String value) {
        this.value = value;
    } //-- StringResult
    
      //------------------/
     //- Public Methods -/
    //------------------/
    
    /**
     * Returns the value of this ExprResult as a boolean
     * @return the value of this ExprResult as a boolean
    **/
    public boolean booleanValue() {
        if (value == null) return false;
        return (value.length() > 0);
    } //-- booleanValue

    public boolean equals(ExprResult exprResult) {
        if (exprResult.getResultType() != ExprResult.STRING)
            return false;
        if (value == null) return false;
        return value.equals(exprResult.toString());
        
    } //-- equals
    
    /**
     * Returns the value of this StringResult
     * @return the value of this StringResult
    **/
    public String getValue() {
        return value;
    } //-- getValue
    
    /**
     * Returns the type of this ExprResult
     * @return the type of this ExprResult
     * @see ExprResult
    **/
    public short getResultType() {
        return ExprResult.STRING;
    } //-- getResultType
    
    /**
     * Returns the value of this ExprResult as a double
     * @return the value of this ExprResult as a double
    **/
    public double numberValue() {
        if (value != null) {
            try {
                return Double.valueOf(value.trim()).doubleValue();
            }
            catch (NumberFormatException nfe) {};
        }
        return Double.NaN;
    } //-- numberValue
    
    /**
     * Sets the value of this StringResult
     * @param value the String value to use as the result value
    **/
    public void setValue(String value) {
        this.value = value;
    } //-- setValue
    
    /**
     * Returns the Java Object this ExprResult represents
     * @return the Java Object this ExprResult represents
    **/
    public Object toJavaObject() {
        return value;
    } //-- toJavaObject
    
    /**
     * Returns the value of this ExprResult as a String
     * @return the value of this ExprResult as a String
    **/
    public String toString() {
        return value;
    } //--toString

} //-- StringResult