/*
 * (C) Copyright Keith Visco 1998  All rights reserved.
 *
 * The software is provided "as is" without any warranty express or
 * implied, including the warranty of non-infringement and the implied
 * warranties of merchantibility and fitness for a particular purpose.
 * The Copyright owner will not be liable for any damages suffered by 
 * you as a result of using the software. In no event will the Copyright
 * owner be liable for any special, indirect or consequential damages or 
 * lost profits even if the Copyright owner has been advised of the 
 * possibility of their occurrence.  
 */
package com.kvisco.xsl;

import java.io.PrintWriter;
import java.util.Hashtable;
import com.kvisco.util.List;
import org.w3c.dom.*;

/**
 * Represents a Basic XSL Selection Object. 
 * xsl:if, xsl:apply-templates, xsl:for-each
 * @author <a href="mailto:kvisco@ziplink.net">Keith Visco</a>
**/
public class Selection extends XSLObject {
        
    private static SelectExpr DEFAULT_EXPR = null;
        
    /** 
     * The selectExpr for this Selection
    **/ 
    private SelectExpr selectExpr = null;
    
    private Element element = null;
    
    public Selection(XSLStylesheet parentStylesheet, short type) {
        super(parentStylesheet,type);
        
        if (DEFAULT_EXPR == null) {
            try {
                DEFAULT_EXPR =
                    ExpressionParser.createSelectExpr("* | text()");
            }
            catch(InvalidExprException iee) {};
        }
    } //-- Selection
    
    /**
     * @return the SelectExpr of this Selection
    **/
    public SelectExpr getSelectExpr() {
        return this.selectExpr;
    } //--getSelectExpr
    
    /**
     * Returns an Array of any XSLSort elements for this Selection.<BR>
     * Only xsl:apply-templates or xsl:for-each can have xsl:sort elements
    **/
    public XSLSort[] getSortElements() {
        List list = new List();
        
        List actions = getActions();
        XSLObject xslObject;
        for (int i = 0; i < actions.size(); i++) {
            xslObject = (XSLObject)actions.get(i);
            if (xslObject.getType() == XSLObject.SORT)
                list.add(xslObject);
            else break;
        }
        return (XSLSort[]) list.toArray(new XSLSort[list.size()]);
    } //-- getSortElements
    
    /**
     * Selects all nodes that match this Selection's SelectExpr 
     * using the given context node and ids.
     * @param context the Node context for evaluate the 
     * SelectExpr with
     * @param idRefs the ID reference table to resolve IDExpr(s)
    **/
    public NodeSet selectNodes(Node context, ProcessorState ps) 
        throws InvalidExprException
    {
        if (selectExpr == null) {
            String selectAttr = getAttribute(Names.SELECT_ATTR);
            if ((selectAttr != null) && (selectAttr.length()>0))
                selectExpr = ExpressionParser.createSelectExpr(selectAttr);
            else selectExpr = DEFAULT_EXPR;
        }
        return selectExpr.selectNodes(context, ps);
    } //-- selectNodes
    
      //---------------------/
     //- Protected Methods -/
    //---------------------/
    
    /**
     * Allows subclasses to handle the append operation of the 
     * given XSLObject. Called from #appendAction
     * this method should return true, if there is nothing left to do,
     * otherwise false, if the super class should handle the append
    **/
    protected boolean handleAction(XSLObject xslObject) {
        if (getType() == XSLObject.CHOOSE) {
            switch (xslObject.getType()) {
                case XSLObject.IF: //-- for my hack..sorry
                case XSLObject.WHEN:
                case XSLObject.OTHERWISE:
                    break;
                default:
                    return true; //-- filter out unwanted elements
            }
        }
        return false;
    } //--  handleAction
        
} //-- Selection