/*
 * (C) Copyright Keith Visco 1999  All rights reserved.
 *
 * The program is provided "as is" without any warranty express or
 * implied, including the warranty of non-infringement and the implied
 * warranties of merchantibility and fitness for a particular purpose.
 * The Copyright owner will not be liable for any damages suffered by
 * you as a result of using the Program. In no event will the Copyright
 * owner be liable for any special, indirect or consequential damages or
 * lost profits even if the Copyright owner has been advised of the
 * possibility of their occurrence.
 */

package com.kvisco.xsl;
import org.w3c.dom.*;

/**
 * Represents a PrimaryExpr<BR>
 * <PRE>
 * from XSLT WD 19990421
 * [14] PrimaryExpr ::= VariableReference
 *                      | '(' Expr ')'
 *                      | Literal
 *                      | Number
 *                      | FunctionCall
 * </PRE>
 * @author Keith Visco (kvisco\@ziplink.net)
**/
public class PrimaryExpr implements Expr {


    public static final short VARIABLE_REFERENCE = 0;
    public static final short EXPR               = 1;
    public static final short LITERAL            = 2;
    public static final short NUMBER             = 3;
    public static final short FUNCTION_CALL      = 4;
    
    private Expr    expr    = null;
    private String  literal = null;
    private double  number  = Double.NaN;
    private short   type    = LITERAL;

      //----------------/
     //- Constructors -/
    //----------------/

    /**
     * Creates a new Literal Primary Expression
    **/
    protected PrimaryExpr(String literal) {
        this.literal = literal;
    } //-- PrimaryExpr
    
    /**
     * Creates a new PrimaryExpr with the given type
    **/
    protected PrimaryExpr(short type) {
        this.type = type;
    } //-- PrimaryExpr

      //------------------/
     //- Public Methods -/
    //------------------/
    
    /**
     * Returns the type of Expr this Expr represents
     * @return the type of Expr this Expr represents
    **/
    public short getExprType() {
        return Expr.PRIMARY;
    } //-- getExprType
    
    /**
     * Retrieves the type of this PrimaryExpr
     * @return the type of this PrimaryExpr
    **/
    public short getType() {
        return this.type;
    } //--getType
    
    /**
     * Returns the literal of this PrimaryExpr, or null if no literal
     * this PrimaryExpr does not represent a literal.
     * @return the literal of this PrimaryExpr
    **/
    public String getLiteral() {
        return literal;
    } //-- getLiteral
    
    public double getNumber() {
        return number;
    } //-- getNumber
    
    /**
     * Evalutes this PrimaryExpr using the given Node as
     * the context and ProcessorState
    **/
    public ExprResult evaluate(Node context, ProcessorState ps) 
        throws InvalidExprException
    {
        
        
        switch(type) {
            // VariableReference
            case VARIABLE_REFERENCE:
                ExprResult exprResult = ps.getVariable(literal);
                if (exprResult != null) return exprResult;
                break;
            // Literal
            case LITERAL:
                return new StringResult(literal);
            // '(' Expr ')'
            case EXPR:
                if (expr != null) return expr.evaluate(context, ps);
                break;
            // Number
            case NUMBER:
                return new NumberResult(number);
            // FunctionCall
            case FUNCTION_CALL:
                if (expr != null) {
                    return expr.evaluate(context, ps);
                }
            default:
                break;
        }
        
        //-- what should happen here?
        return new StringResult("undefined");
        
    } //-- evaluate

    
    /**
     * Returns the String representation of this PrimaryExpr
     * @return the String representation of this PrimaryExpr
    **/
    public String toString() {
        StringBuffer sb = new StringBuffer();
        
        switch(type) {
            // VariableReference
            case VARIABLE_REFERENCE:
                sb.append("$");
                sb.append(literal);
                break;
            // Literal
            case LITERAL:
                sb.append("'");
                sb.append(literal);
                sb.append("'");
                break;
            // '(' Expr ')'
            case EXPR:
                sb.append("(");
                if (expr != null) sb.append(expr.toString());
                sb.append(")");
                break;
            // Number
            case NUMBER:
                sb.append (String.valueOf(number));
                break;
            // FunctionCall
            case FUNCTION_CALL:
                if (expr != null)
                    sb.append(expr.toString());
                else 
                    sb.append("void()");
                break;
            default:
                break;
        }
        return sb.toString();
    } //-- toString
    
      //---------------------/
     //- Protected Methods -/
    //---------------------/
    
    /**
     * Sets the Expr to use with this PrimaryExpr.<BR>
     * @param expr the Expr to use with this PrimaryExpr
    **/
    protected void setExpr(Expr expr) {
        this.expr = expr;
    } //-- setExpr

    /**
     * Sets the Literal to use with this PrimaryExpr.<BR>
     * @param literal the literal string value
     * to use with this PrimaryExpr
    **/
    protected void setLiteral(String literal) {
        this.literal = literal;
    } //-- setLiteral
    
    protected void setNumber(String number) {
        try {
            this.number = Double.valueOf(number).doubleValue();
        }
        catch (NumberFormatException nfe) {
            this.number = Double.NaN;
        }
    } //-- setNumber

    protected void setNumber(double number) {
        this.number = number;
    } //-- setNumber
    
} //-- PrimaryExpr
