/*
 * (C) Copyright Keith Visco 1999  All rights reserved.
 *
 * The program is provided "as is" without any warranty express or
 * implied, including the warranty of non-infringement and the implied
 * warranties of merchantibility and fitness for a particular purpose.
 * The Copyright owner will not be liable for any damages suffered by
 * you as a result of using the Program. In no event will the Copyright
 * owner be liable for any special, indirect or consequential damages or
 * lost profits even if the Copyright owner has been advised of the
 * possibility of their occurrence.
 */

package com.kvisco.xsl;

import org.w3c.dom.Node;

public class NodeSet implements ExprResult {
    
    private int DEFAULT_SIZE = 25;
    
    private Node[] elements;
    
    private int initialSize = DEFAULT_SIZE;
    
    /**
     * The next available location in the elements array
    **/
    private int elementCount = 0;
    
    /**
     * Creates a new NodeSet with the default Size
    **/
    public NodeSet() {
        elements = new Node[DEFAULT_SIZE];
    } //-- List
    
    public NodeSet(int size) {
        initialSize = size;
        elements = new Node[size];
    } //-- List
    
    /**
     * Adds the specified Node to this NodeSet if it is not already
     * contained within in this NodeSet.
     * @param node the Node to add to the NodeSet
     * @return true if the Node is added to the NodeSet
    **/
    public boolean add(Node node) {
        if (!contains(node)) {
            if (elementCount == elements.length) increaseSize();
            elements[elementCount++] = node;
            return true;
        }
        return false;
    } //-- add

    /**
     * Copies the Set of nodes from nodeSet into this NodeSet;
    **/
    public void add(NodeSet nodeSet) {
        int nsz = nodeSet.size();
        ensureSize(nsz);
        //-- use the add function to prevent duplicates
        for (int i = 0; i < nsz; i++) add(nodeSet.get(i));
    } //-- add

    /**
     * Adds the specified Node to the NodeSet at the specified index,
     * as long as the Node is not already contained within the set
     * @param node the Node to add to the NodeSet
     * @return true if the Node is added to the NodeSet
     * @exception IndexOutOfBoundsException
    **/
    public boolean add(int index, Node node) 
        throws IndexOutOfBoundsException 
    {
        if ((index < 0) || (index > elementCount)) 
            throw new IndexOutOfBoundsException();
            
        if (contains(node)) return false;
        
        // make sure we have room to add the object
        if (elementCount == elements.length) increaseSize();
        
        if (index == elementCount) {
            elements[elementCount++] = node;
        }
        else {
            shiftUp(index);
            elements[index] = node;
            elementCount++;
        }
        return true;
    } //-- add
    
    /**
     * Returns the value of this ExprResult as a boolean
     * @return the value of this ExprResult as a boolean
    **/
    public boolean booleanValue() {
        return (size() > 0);
    } //-- booleanValue
    
    /**
     * Removes all elements from the list
    **/
    public void clear() {
        for (int i = 0; i < elementCount; i++) {
            elements[i] = null;
        }
        elementCount = 0;
    } //-- clear
    
    /**
     * Copies the nodes from this NodeSet into a new NodeSet
    **/
    public NodeSet copy() {
        NodeSet nodes = new NodeSet(elementCount);
        nodes.arrayCopyFrom(elements, 0, elementCount);
        return nodes;
    } //-- copy
        
    /**
     * Returns true if the specified Node is contained in the set.
     * if the specfied Node is null, then if the NodeSet contains a null
     * value, true will be returned.
     * @param node the element to search the NodeSet for
     * @return true if specified Node is contained in the NodeSet
    **/
    public boolean contains(Node node) {
        return (indexOf(node) >= 0);
    } //-- contains

    /**
     * Compares the specified object with this NodeSet for equality.
     * Returns true if and only if the specified Object is a NodeSet 
     * that is the same size as this NodeSet and all of its associated 
     * Nodes are contained within this NodeSet.
     * @return true if and only if the specified Object is a NodeSet 
     * that is the same size as this NodeSet and all of its associated 
     * Nodes are contained within this NodeSet.
    **/
    public boolean equals(Object obj) {
        if (obj == null) return false;
        if (!(obj instanceof NodeSet)) return false;
        NodeSet nSet = (NodeSet)obj;
        if (nSet.size() != size()) return false;
        for (int i = 0; i < size(); i++) {
            Node n1 = get(i);
            if (!nSet.contains(get(i))) return false;
               return false;
        }
        return true;
    } //-- equals
    
    public boolean equals(ExprResult exprResult) {
        if (exprResult.getResultType() == ExprResult.NODE_SET) {
            return this.equals((Object)exprResult);
        }
        return false;
    }
    /**
     * Returns the Node at the specified position in this NodeSet.
     * @param index the position of the Node to return
     * @exception IndexOutOfBoundsException 
    **/
    public Node get(int index) throws IndexOutOfBoundsException {
        if ((index < 0) || index >= elementCount) {
            throw new IndexOutOfBoundsException();
        }
        return elements[index];
    } //-- get
    
    /**
     * Returns the type of ExprResult that a NodeSet represents
     * @return the type of ExprResult that a NodeSet represents
     * @see ExprResult
    **/
    public short getResultType() {
        return ExprResult.NODE_SET;
    } //-- getResultType
    
    /**
     * Returns the hash code value for this NodeSet.
     * The hash code of a NodeSet is defined to be the sum of
     * the hashCode values for each Node in the set
     * @return the hash code value for this list
    **/
    public int hashCode() {
       int hashCode = 0;
       for (int i = 0; i < elementCount; i++) {
           Node node = elements[i];
           hashCode += (node==null) ? 0 : node.hashCode();
       }
       return hashCode;
    } //-- hashCode
    
    /**
     * Returns the index of the specified Node, 
     * or -1 if the Node is not contained in the NodeSet
     * @param node the Node to get the index for
    **/
    public int indexOf(Node node) {
        for (int i = 0; i < elementCount; i++) {
            if (node == elements[i]) return i;
        }
        return -1;
    } //-- indexOf

    /**
     * Returns true if there are no Nodes in the NodeSet.
     * @return true if there are no Nodes in the NodeSet.
    **/
    public boolean isEmpty() {
        return (elementCount == 0);
    } //-- isEmpty

    /**
     * Returns the value of this ExprResult as a double
     * @return the value of this ExprResult as a double
    **/
    public double numberValue() {
        if (size() > 0) {
            StringResult sr =
                new StringResult(XSLObject.getNodeValue(get(0)));
            return sr.numberValue();
        }
        return Double.NaN;
    } //-- numberValue
    /**
     * Removes the Node at the specified index from the NodeSet
     * @param index the position in the NodeSet to remove the Node from
     * @return the Node that was removed from the list
    **/
    public Node remove(int index) {
        
        if ((index < 0) || (index > elementCount)) return null;
        Node node = elements[index];
        shiftDown(index+1);
        --elementCount;
        return node;
    } //-- remove

    /**
     * Removes the the specified Node from the NodeSet
     * @param node the Node to remove from the NodeSet
     * @return true if the Node was removed from the list
    **/
    public boolean remove(Node node) {
        int index = indexOf(node);
        
        if (index > -1) {
            remove(index);
        } 
        else return false;
        
        return true;
    } //-- remove
    
     
    /**
     * Returns the number of elements in the NodeSet
     * @return the number of elements in the NodeSet
    **/
    public int size() {
        return elementCount;
    } //-- size
    
    /**
     * Returns an array containing all of the nodes in this NodeSet 
     * in proper sequence.
     * @return the array of Nodes of this NodeSet
    **/
    public Node[] toArray() {
        Node[] nodeArray = new Node[elementCount];
        System.arraycopy(elements,0,nodeArray,0,elementCount);
        return nodeArray;
    } //-- toArray

    /**
     * Returns an array containing all of the nodes in this NodeSet 
     * in proper sequence.
     * @return the array of Nodes of this NodeSet
    **/
    public Node[] toArray(Node[] dst) {
        
        Node[] nodeArray = null;
        
        if (dst.length >= elementCount) nodeArray = dst;
        else {
            nodeArray =  new Node[elementCount];
        }
        System.arraycopy(elements,0,nodeArray,0,elementCount);
        return nodeArray;
    } //-- toArray

    /**
     * Returns the Java Object this ExprResult represents
     * @return the Java Object this ExprResult represents
    **/
    public Object toJavaObject() {
        return this;
    } //-- toJavaObject

      //---------------------/
     //- Protected Methods -/
    //---------------------/
    
    protected void arrayCopyTo(Node[] dst, int dstStartIdx) {
        // copy from elements[0..length] tp nodes[start]
        System.arraycopy(elements, 0, dst, dstStartIdx, elementCount);
    } //-- arrayCopyTo
    
    protected void arrayCopyFrom(Node[] src, int startIdx, int endIdx) {
        int numberToCopy = endIdx-startIdx;
        if (numberToCopy < 1) return;        
        ensureSize(numberToCopy);
        System.arraycopy(src, startIdx, elements, elementCount, numberToCopy);
        elementCount+=numberToCopy;
    } //-- arrayCopyFrom
    
      //-------------------/
     //- Private Methods -/
    //-------------------/
    
    /**
     * increase the NodeSet capacity by a factor of its initial size
    **/
    private void increaseSize() {
        Object[] pointer = elements;
        int length = (pointer.length > 0) ? pointer.length : 1;
        elements = new Node[length*2];
        System.arraycopy(pointer, 0, elements, 0, pointer.length);
    } //-- increaseSize
    
    /**
     * makes sure we have enough spaces to hold count number of elements
     * @param count the number of elements we need to hold
    **/
    private void ensureSize(int count) {
        int freeSpace = (elements.length - elementCount);
        if (freeSpace < count) {
            int adjustment = count-freeSpace;
            Object[] pointer = elements;
            int length = elements.length + adjustment;
            elements = new Node[length];
            System.arraycopy(pointer, 0, elements, 0, pointer.length);
        }
    } //-- ensureSize
    
    /**
     * Shifts all elements at the specified index to down by 1
    **/
    private void shiftDown(int index) {
        if ((index <= 0) || (index >= elementCount)) return;
        System.arraycopy(elements, index, elements, index - 1, elementCount - index);        
        // clean up for gc
        elements[elementCount-1] = null;
    } //-- shiftDown
    
    /**
     * Shifts all elements at the specified index up by 1
    **/
    private void shiftUp(int index) {
        if (index == elementCount) return;
        if (elementCount == elements.length) increaseSize();
        System.arraycopy(elements, index, elements, index + 1, elementCount - index);
    } //-- shiftUp
    
    
    
} //-- NodeSet