/*************************************************************************/
/*                                                                       */
/*                Centre for Speech Technology Research                  */
/*                     University of Edinburgh, UK                       */
/*                    Copyright (c) 1994,1995,1996                       */
/*                        All Rights Reserved.                           */
/*                                                                       */
/*  Permission to use, copy, modify, distribute this software and its    */
/*  documentation for research, educational and individual use only, is  */
/*  hereby granted without fee, subject to the following conditions:     */
/*   1. The code must retain the above copyright notice, this list of    */
/*      conditions and the following disclaimer.                         */
/*   2. Any modifications must be clearly marked as such.                */
/*   3. Original authors' names are not deleted.                         */
/*  This software may not be used for commercial purposes without        */
/*  specific prior written permission from the authors.                  */
/*                                                                       */
/*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK        */
/*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING      */
/*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT   */
/*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE     */
/*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES    */
/*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN   */
/*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,          */
/*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF       */
/*  THIS SOFTWARE.                                                       */
/*                                                                       */
/*************************************************************************/
/*                       Author :  Paul Taylor                           */
/*                       Date   :  October 1994                          */
/*-----------------------------------------------------------------------*/
/*                      Command Line Utilities                           */
/*                                                                       */
/*  awb merged the help usage and argment definitions April 97           */
/*                                                                       */
/*=======================================================================*/
#include <stdlib.h>
#include "EST_unix.h"
#include "EST_String.h"
#include "EST_io_aux.h"
#include "EST_Token.h"
#include "EST_cutils.h"
#include "EST_TList.h"
#include "EST_string_aux.h"
#include "EST_cmd_line.h"

// This is reset by the command line options functions to argv[0]
EST_String est_progname = "ESTtools";

static int valid_option(const EST_Option &option,const char *arg,
			EST_String &sarg);
static void standard_options(int argc, char **argv, const EST_String &usage);
static void arg_error(const EST_String &progname, const EST_String &message);
static void parse_usage(const EST_String &progname, const EST_String &usage, 
			EST_Option &options, EST_Option &al);
static void output_man_options(const EST_String &usage);


static int legal_arg(EST_String arg, EST_String slist[], int num);
static int type_arg(EST_String option, EST_String arg, EST_String slist[], int num);

static EST_TBI* legal_arg(EST_String arg, EST_StrList slist);
static int type_arg(EST_String option, EST_String arg, EST_StrList slist);

int init_lib_ops(EST_Option &al, EST_Option &op)
{
    char *envname;

    // read environment variable operations file if specified
    if ((al.val("-N", 0) != "true") && 
	((envname = getenv("IA_OP_FILE")) != 0))
	if (op.load(getenv("IA_OP_FILE")) != read_ok)
	    exit (1);
    
    // read command line operations file if specified
    if (al.val("-c", 0) != "") 
	if (op.load(al.val("-c")) != read_ok)
	    exit (1);
    
    // override operations with command line options
    override_lib_ops(op, al); 

    if (al.val("-ops", 0) == "true") // print options if required
	cout << op;
    
    return 0;
}

//  An attempt to integrate help, usage and argument definitions in 
//  one string (seems to work)
//  Still to do: 
// *   adding arbitrary "-" at end of files (do we need that ?)
//     dealing with environment var specification of extra options
//     override options function (maybe no longer needed)
//     list of named values for argument
//     Way to identify mandatory argments
int parse_command_line2(int argc, 
		   char *argv[],
		   const EST_String &usage,
		   EST_StrList &files,
		   EST_Option &al, int make_stdio)
{
    // Parse the command line arguments returning them in a normalised
    // form in al, and files in files.
    int i;
    EST_Option options;
    EST_String arg;
    (void)make_stdio;

    // help, version, man_options are always supported
    standard_options(argc,argv,usage); 

    // Find valid options, arguments and defaults
    // sets defaults in al
    est_progname = argv[0];
    parse_usage(argv[0],usage,options,al);

    for (i=1; i < argc; i++)
    {
	if (!EST_String(argv[i]).contains("-",0))  // its a filename
	    files.append(argv[i]);
	else if (streq(argv[i],"-"))   // single "-" denotes stdin/out
	    files.append("-");
	else if (!valid_option(options,argv[i],arg))   
	{
	    arg_error(argv[0],
		      EST_String(": Unknown option \"")+argv[i]+"\"\n");
	}
	else                                       // valid option, check args
	{
	    if (options.val(arg) == "true")  // no argument required
		al.add_item(arg, "true");
	    else if (options.val(arg) == "<int>")
	    {
		if (i+1 == argc)
		    arg_error(argv[0],
			      EST_String(": missing int argument for \"")+
			      arg+"\"\n");
		i++;
		if (!EST_String(argv[i]).matches(RXint))
		    arg_error(argv[0],
			      EST_String(": argument for \"")+
			      arg+"\" not an int\n");
		al.add_item(arg,argv[i]);
	    }
	    else if ((options.val(arg) == "<float>") ||
		     (options.val(arg) == "<double>"))
	    {
		if (i+1 == argc)
		    arg_error(argv[0],
			      EST_String(": missing float argument for \"")+
			      arg+"\"\n");
		i++;
		if (!EST_String(argv[i]).matches(RXdouble))
		    arg_error(argv[0],
			      EST_String(": argument for \"")+
			      arg+"\" not a float\n");
		al.add_item(arg,argv[i]);
	    }
	    else if (options.val(arg) == "<string>")
	    {
		if (i+1 == argc)
		    arg_error(argv[0],
			      EST_String(": missing string argument for \"")+
			      arg+"\"\n");
		i++;
		al.add_item(arg,argv[i]);
	    }
	    else if (options.val(arg) == "<ofile>")
	    {
		if (i+1 == argc)
		    arg_error(argv[0],
			      EST_String(": missing ifile argument for \"")+
			      arg+"\"\n");
		i++;
		if (writable_file(argv[i]) == TRUE)
		    al.add_item(arg,argv[i]);
		else 
		    arg_error(argv[0],
			      EST_String(": output file not accessible \"")+
			      argv[i]+"\"\n");
	    }
	    else if (options.val(arg) == "<ifile>")
	    {
		if (i+1 == argc)
		    arg_error(argv[0],
			      EST_String(": missing ifile argument for \"")+
			      arg+"\"\n");
		i++;
		if (readable_file(argv[i]) == TRUE)
		    al.add_item(arg,argv[i]);
		else
		    arg_error(argv[0],
			      EST_String(": input file not accessible \"")+
			      argv[i]+"\"\n");

	    }
	    else if (options.val(arg) == "<star>")
	    {
		al.add_item(arg,EST_String(argv[i]).after(arg));
	    }
	    // else string list 
	    else
	    {
		arg_error(argv[0],
			  EST_String(": unknown argument type \"")+
			  options.val(argv[i])+"\" (misparsed usage string)\n");
	    }
	}
    }

    if (files.length() == 0)
	files.append("-");

    return 0;
}

static int valid_option(const EST_Option &options,const char *arg,
			EST_String &sarg)
{
    // Checks to see arg is declared as an option.
    // This would be trivial were it not for options containing *
    // The actual arg name is put in sarg
    EST_TBI *p;

    for (p=options.list.head(); p != 0; p=next(p))
    {
	if (options.key(p) == arg)
	{
	    sarg = arg;
	    return TRUE;
	}
	else if ((options.valp(p) == "<star>") &&
		 (EST_String(arg).contains(options.key(p),0)))
	{
	    sarg = options.key(p);
	    return TRUE;
	}
    }

    return FALSE;
}

static void parse_usage(const EST_String &progname, const EST_String &usage, 
			EST_Option &options, EST_Option &al)
{
    // Extract option definitions from usage and put them in options
    // If defaults are specified add them al
    EST_TokenStream ts;
    EST_Token t;

    ts.open_string(usage);
    ts.set_SingleCharSymbols("{}[]|");
    ts.set_PunctuationSymbols("");
    ts.set_PrePunctuationSymbols("");

    while (!ts.eof())
    {
	t = ts.get();
	if ((t.string().contains("-",0)) &&
	    (t.whitespace().contains("\n")))
	{   // An argument
	    if ((ts.peek().string() == "<string>") ||
		(ts.peek().string() == "<float>") ||
		(ts.peek().string() == "<double>") ||
		(ts.peek().string() == "<ifile>") ||
		(ts.peek().string() == "<ofile>") ||
		(ts.peek().string() == "<int>"))
	    {
		options.add_item(t.string(),ts.get().string());	
		if (ts.peek().string() == "{") // a default is given
		{
		    ts.get();
		    al.add_item(t.string(),ts.get().string());
		    if (ts.get() != "}")
			arg_error(progname,
			  EST_String(": malformed default value for \"")+
			  t.string()+"\" (missing closing brace)\n");
		}
	    }
	    else if (t.string().contains("*"))
		options.add_item(t.string().before("*"),"<star>");
	    // else check for explicit list of names 
	    else
		options.add_item(t.string(),"true");  // simple argument
	}
    }
}

static void arg_error(const EST_String &progname, const EST_String &message)
{
    // Output message  and pointer to more help then exit
    cerr << progname << message;
    cerr << "Type -h for help on options.\n";
    exit(-1);
}

static void standard_options(int argc, char **argv, const EST_String &usage)
{
    // A number of options are always supported 
    int i;

    for (i=1; i < argc; i++)
    {
	if (streq(argv[i],"-man_options"))
	{
	    output_man_options(usage);
	    exit(0);
	}
	if ((streq(argv[i],"-h")) ||
	    (streq(argv[i],"-help")) ||
	    (streq(argv[i],"-?")) ||
	    (streq(argv[i],"--help")))
	{
	    cout << usage << endl;
	    exit(0);
	}
	if (((streq(argv[i],"-version")) ||
	     (streq(argv[i],"--version")))&&
	    (!usage.contains("\n-v")))
	{
	    cout << argv[0] << ": " << est_tools_version << endl;
	    exit(0);
	}
    }

    return;
}

static void output_man_options(const EST_String &usage)
{
    EST_TokenStream ts;
    EST_Token t;
    int in_options = FALSE;

    ts.open_string(usage);
    ts.set_SingleCharSymbols("{}[]|");
    ts.set_PunctuationSymbols("");
    ts.set_PrePunctuationSymbols("");

    while (!ts.eof())
    {
	t = ts.get();
	if ((t.string().contains("-",0)) &&
	    (t.whitespace().contains("\n")))
	{   // An argument
	    fprintf(stdout,"\n.TP 8\n.BI \"%s \" ",(const char *)t.string());
	    if ((ts.peek().string() == "<string>") ||
		(ts.peek().string() == "<float>") ||
		(ts.peek().string() == "<double>") ||
		(ts.peek().string() == "<int>"))
		fprintf(stdout,"%s",(const char *)ts.get().string());
	    if ((ts.peek().string() == "{"))
	    {  // a default value
		ts.get();
		fprintf(stdout," \" {%s}\"",(const char *)ts.get().string());
		ts.get();
	    }
	    if (!ts.peek().whitespace().contains("\n"))
		fprintf(stdout,"\n");
	    in_options = TRUE;
	}
	else if (in_options)
	{
	    if (t.whitespace().contains("\n"))
		fprintf(stdout,"\n");
	    fprintf(stdout,"%s ",(const char *)t.string());
	}
    }
    if (in_options)
	fprintf(stdout,"\n");
	

}

int parse_command_line(int argc, 
		   char *argv[],
		   EST_String deflist,
		   EST_StrList &files,
		   EST_Option &al, int make_stdio)
{
    EST_String *sarg = new EST_String[argc + 1];
    int i, j, an;
    int num_args; // number of arguments (not files)
    int max_ops; // number of possible options.

    for (i = 0; i < argc - 1; ++i) // put cl args into strings
	sarg[i] = argv[i + 1];
    
    EST_String space(" ");
    deflist.gsub(RXwhite, space); // find out how may tokens are in list.
    max_ops =  deflist.freq(space) + 1;
    
    EST_String *slist = new EST_String[max_ops];// each string in one element of array
    split(deflist, slist, max_ops, RXwhite);
    
    num_args = 0;
    for (i = j = 0; i < argc - 1; ++i)
    {
	if (sarg[i].contains("-", 0)) // this is an argument, not a file
	{
	    if ((an = legal_arg(sarg[i], slist, max_ops)) == -1)
	    {
		cerr << "Error: illegal option " << sarg[i] << endl;
		exit (1);
	    }
	    if (slist[an].contains("*")) // special case of wildcard arglist.
	    {
		int n = slist[an].before("*").length();
		if (n == (signed int)sarg[i].length())
		{
		    if (legal_arg(sarg[i + 1], slist, max_ops) != -1)
		    {
			cerr<<"Error: EST_Option \"" <<sarg[i] << 
			    "\" takes an argument\n";
			exit (1);
		    }
		    al.add_item(sarg[i], sarg[1+i]);
		    i++;  /* note can't be on previous line due to */
		          /* order of evaluation problem           */
		}
		else
		    al.add_item(sarg[i].before(n), sarg[i].after(n - 1));
		//		++i;
	    }
	    else if (slist[an].contains(":")) // normal case of flag + arg
	    {
		if (legal_arg(sarg[i + 1], slist, max_ops) != -1)
		{
		    cerr<<"Error: EST_Option \""<< sarg[i] 
			<<"\" takes an argument\n";
		    exit (1);
		}
		if (type_arg(sarg[i], sarg[i + 1], slist, max_ops) == -1)
		    exit (1);
		al.add_item(sarg[i], sarg[1+i]);
		i++;  /* note can't be on previous line due to */
		      /* order of evaluation problem           */
	    }
	    else		// normal case of flag only.
		al.add_item(sarg[i], "true");
	}   
	else			//filename
	    files.append(sarg[i]);
    }
    
    // if stdin/out requested, make up dummy filenames of "-"
    
    if ((make_stdio) && (files.length() < 1))
	files.append("-");
    
    delete [] sarg;
    delete [] slist;
    
    return 0;
} 

int parse_command_line(int argc, 
		       char *argv[],
		       EST_String deflist,
		       EST_String *&files,
		       int &num_files,
		       EST_Option &al, int make_stdio)
{
    EST_String *sarg = new EST_String[argc + 1];
    int i, j, an;
    int num_args;		// number of arguments (not files)
    int max_ops;		// number of possible options.
    
    for (i = 0; i < argc - 1; ++i) // put cl args into strings
	sarg[i] = argv[i + 1];
    
    EST_String space(" ");
    deflist.gsub(RXwhite, space); // find out how may tokens are in list.
    max_ops =  deflist.freq(space) + 1;
    
    files = new EST_String[(argc < 2) ? 2 : argc]; // must have at least 2 args
    EST_String *slist = new EST_String[max_ops];   // each string in one element of array
    split(deflist, slist, max_ops, RXwhite);
    
    num_args = 0;
    
    for (i = j = 0; i < argc; ++i)
    {
	if (sarg[i].contains("-", 0)) // this is an argument, not a file
	{
	    if ((an = legal_arg(sarg[i], slist, max_ops)) == -1)
	    {
		cerr << "Error: illegal option " << sarg[i] << endl;
		exit (0);
	    }
	    if (slist[an].contains("*")) // special case of wildcard arglist.
	    {
		int n = slist[an].before("*").length();
		if (n == (signed int)sarg[i].length())
		{
		    if (legal_arg(sarg[i + 1], slist, max_ops) != -1)
		    {
			cerr<<"Error: EST_Option \"" <<sarg[i] << 
			    "\" takes an argument\n";
			exit (0);
		    }
		    al.add_item(sarg[i], sarg[++i]);
		}
		else
		    al.add_item(sarg[i].before(n), sarg[i].after(n - 1));
		//		++i;
	    }
	    else if (slist[an].contains(":")) // normal case of flag + arg
	    {
		if (legal_arg(sarg[i + 1], slist, max_ops) != -1)
		{
		    cerr<<"Error: EST_Option \""<< sarg[i] 
			<<"\" takes an argument\n";
		    exit (0);
		}
		if (type_arg(sarg[i], sarg[i + 1], slist, max_ops) == -1)
		    exit (0);
		al.add_item(sarg[i], sarg[++i]);
	    }
	    else		// normal case of flag only.
		al.add_item(sarg[i], "true");
	}   
	else			//filename
	    files[j++] = sarg[i];
    }
    num_files = j - 1;
    
    // if stdin/out requested, make up dummy filenames of "-"
    
    if (make_stdio)
    {
	if (num_files < 1)
	    files[0] = "-";
	
	if (num_files < 2)
	{
	    files[1] = "-";
	    num_files = 2;
	}
    }
    
    delete [] sarg;
    delete [] slist;
    
    return 0;
} 

static int legal_arg(EST_String arg, EST_String slist[], int num)
{				// returns true if arg is in slist.
    int j;
    EST_String t,s ;
    for (j = 0; j < num; ++j)
	if (slist[j].contains("*"))
	{
	    t = slist[j].before("*");
	    int l = t.length();
	    s = arg.before(l);
	    if (t == s)
		return j;
	}
	else 
	{
	    t = (slist[j].contains(":")) ? slist[j].before(":") : slist[j];
	    if (t == arg)
		return j;
	}    
    return -1;
}

EST_TBI * legal_arg(EST_String arg, EST_StrList slist)
{				// returns true if arg is in slist.
    EST_String t,s ;
    EST_TBI *p;
    for (p = slist.head(); p; p = next(p))
	if (slist(p).contains("*"))
	{
	    t = slist(p).before("*");
	    int l = t.length();
	    s = arg.before(l);
	    if (t == s)
		return p;
	}
	else if (slist(p).contains(arg, 0))
	    return p;
    
    return 0;
}

static int type_arg(EST_String option, EST_String arg, EST_String slist[], int num)
{				// returns true if arg is of correct type. Can't really check for strings
    // as they could look quite like integers etc.
    int j;
    for (j = 0; j < num; ++j)
	if (slist[j].contains(option))
	{
	    if (slist[j].after(":") == "i") // check for integer in slist
	    {
		if (!arg.matches(RXint)) // check if arg is an int
		{
		    cout << "Expecting an integer argument to option "
			<< option << endl;
		    return -1;
		}
	    }
	    else if ((slist[j].after(":") == "f")) // check for float or double
	    {
		if (!arg.matches(RXdouble)) // check if arg is a float
		{
		    cout << "Expecting a floating point argument to option "
			<< option << endl;
		    return -1;
		}
	    }
	    return j;
	}
    
    return -1;
}

static int type_arg(EST_String option, EST_String arg, EST_StrList slist)
{				// returns true if arg is of correct type. Can't really check for strings
    // as they could look quite like integers etc.
    EST_TBI *p;
    for (p = slist.head(); p; p = next(p))
	if (slist(p).contains(option))
	{
	    if (slist(p).after(":") == "i") // check for integer in slist
	    {
		if (!arg.matches(RXint)) // check if arg is an int
		{
		    cout << "Expecting an integer argument to option "
			<< option << endl;
		    return -1;
		}
	    }
	    else if ((slist(p).after(":") == "f")) // check for float or double
	    {
		if (!arg.matches(RXdouble)) // check if arg is a float
		{
		    cout << "Expecting a floating point argument to option "
			<< option << endl;
		    return -1;
		}
	    }
	    return 1;
	}
    return -1;
}

EST_String options_general(void)
{
    // The standard waveform input options 
    return
	EST_String("")+
	"-o <ofile>      output file" +
	"-otype <string> output file type\n";
}
