/*************************************************************************/
/*                                                                       */
/*                Centre for Speech Technology Research                  */
/*                     University of Edinburgh, UK                       */
/*                         Copyright (c) 1996                            */
/*                        All Rights Reserved.                           */
/*                                                                       */
/*  Permission to use, copy, modify, distribute this software and its    */
/*  documentation for research, educational and individual use only, is  */
/*  hereby granted without fee, subject to the following conditions:     */
/*   1. The code must retain the above copyright notice, this list of    */
/*      conditions and the following disclaimer.                         */
/*   2. Any modifications must be clearly marked as such.                */
/*   3. Original authors' names are not deleted.                         */
/*  This software may not be used for commercial purposes without        */
/*  specific prior written permission from the authors.                  */
/*                                                                       */
/*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK        */
/*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING      */
/*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT   */
/*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE     */
/*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES    */
/*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN   */
/*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,          */
/*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF       */
/*  THIS SOFTWARE.                                                       */
/*                                                                       */
/*************************************************************************/
/*                 Speech Tools Author: Paul Taylor                      */
/*                 Original Author: Tony Robinson                        */
/*                 Date   :  July 1994                                   */
/*-----------------------------------------------------------------------*/
/*                   LPC/cepstra calculation                             */
/*                                                                       */
/*=======================================================================*/

/* Note - this was based on code by Tony Robinson, which I had to
adapt to make it into C++. All bugs are my fault..  pault.*/

/*
 * Further modifications by RJC
 */

#include <sys/types.h>
#include <stdlib.h>
#include <math.h>
#include <iostream.h>
#include <fstream.h>
#include "EST_unix.h"
#include "EST_sigpr.h"
#include "EST_math.h"
#include "EST_ContourType.h"
#include "EST_track_aux.h"
#include "EST_TBuffer.h"

# define MAX_LPORDER	    128
# define MAX_ORDER	    128

static float lpredict(float *adc, int wsize, float *acf, float *ref, 
		      float *lpc, int order);
float *Float_hamming(int size);
static void residual_shorts(const short *orig, short *e, int wsize, const float *a, int order);
static void resynth_shorts(short *synth, const short *e, int wsize, float *a, int order);
static void resynth_noex_shorts(short *synth, int wsize, float *a, int order);

static inline float get_rms_2(const float window[], int length)
{
  float sum=0.0;
  for(int i=0; i<length;i++)
    sum += window[i]*window[i];

  return sum/length;
}

static inline float get_rms_2(const short window[], int length)
{
  float sum=0.0;
  for(int i=0; i<length;i++)
    {
      sum += (float)window[i]*(float)window[i];
    }

  return sum/length;
}

static inline float get_rms(const short window[], int length)
{
  return sqrt(get_rms_2(window, length));
}

static inline float get_rms(const float window[], int length)
{
  return sqrt(get_rms_2(window, length));
}

static inline short irint(float f) { return (short)(f+0.5); }
static inline short irint(double f) { return (short)(f+0.5); }

static float get_voiced(const float window[], int length, 
			const float acf[], int order)
{
  (void) window;
  (void) length;

  float sum=0.0;

  for(int i=0; i<=order; i++)
    sum += acf[i]*acf[i];

  return sqrt(sum/order) > 2.5e+7;
}

void lpc_cepstra(EST_Track &lpct, EST_Wave &sig, float length, float shift, int
		 lporder, int genlpc)

{
    float sum, m;
    static float acf[MAX_LPORDER], ref[MAX_LPORDER], lpc[MAX_LPORDER];
    static float ceps[MAX_LPORDER + 1];
    short *tmp_sig, *window_start;
    int i, k, pos;

    int frame_length = (int) (length * (float) sig.sample_rate() );
    int frame_shift = (int) (shift * (float) sig.sample_rate() );
    float *hamming = Float_hamming(frame_length);
    float *window = new float[frame_length];

//    cout << "length = " << length;
//    cout << "frame_length = " << frame_length << endl;
    
    int num_frames = (int)ceil((float)sig.num_samples() / frame_shift);
    int num_samples = ((num_frames - 1) * frame_shift) + frame_length;

    tmp_sig = new short[num_samples];
    for (i = 0; i < sig.num_samples(); ++i)
	tmp_sig[i] = sig.a(i);
    
    // desparate nightmare hack
    for (; i < num_samples; ++i)
	tmp_sig[i] = i%2; // to ensure that the cepstra is working on something
	
    lpct.resize(num_frames, lporder+1);
    
    for (k = 0 ; k < num_frames; ++k)
    {
	pos = frame_shift * k;
	window_start = tmp_sig + pos;
	sum = 0.0;
	for(i = 0; i < frame_length; i++)
	    sum += window_start[i];
	m = sum / frame_length;
		
	for(i = 0; i < frame_length; i++)
	    window[i] = hamming[i] * (window_start[i] - m);
		
	lpredict(window, frame_length, acf, ref, lpc, lporder);

	if (genlpc)
	{
	    for (i = 0; i <= lporder; ++i)
		lpct.a(k, i) = lpc[i];
	}
	else
	{
	    lpc2ceps(lpc, lporder, ceps, lporder);
		
	    for (i = 0; i < lporder; ++i)
		lpct.a(k, i) = ceps[i + 1];
	}
    }

    delete [] tmp_sig;
    delete [] window;
}

// default description of tracks created by lpc_analyse.

static EST_ChannelMapping lpc_mapping =
{
  { channel_length, 0 },
  { channel_coef0,  1 },
  { channel_coefN, 17 },
  { channel_unknown},	
};
static EST_TrackMap LPCTrackMap(lpc_mapping);

void lpc_analyse(EST_Track &lpct, EST_Wave &sig, 
		 EST_WindowType window_type,
		 float length, float shift, 
		 EST_TrackMap *description,
		 EST_Track *pms)

{
    float sum, m;
    static float acf[MAX_LPORDER], ref[MAX_LPORDER], lpc[MAX_LPORDER];
    short *tmp_sig, *window_start;
    int i, k, pos=0;

    // Use the description requested to control how we analyse

    if (!description)
      description = &LPCTrackMap;

    int lporder = description->get(channel_coefN) - description->get(channel_coef0);

    bool do_power = (bool)(description->get(channel_power) != NO_SUCH_CHANNEL);
    bool do_length=(bool)(description->get(channel_length) != NO_SUCH_CHANNEL);
    bool do_voiced=(bool)(description->get(channel_voiced) != NO_SUCH_CHANNEL);
    bool do_time = (bool)(description->get(channel_time) != NO_SUCH_CHANNEL);

    // set up windowing

    EST_WindowFunction *make_window = EST_WindowTypeMap.info(window_type).fn;

    int frame_length = (int) (length * (float) sig.sample_rate() );
    int frame_shift = (int) (shift * (float) sig.sample_rate() );
    int offset = (frame_length - frame_shift)/2;
    EST_FVector window_vals(make_window(frame_length));

    float *window = new float[frame_length*3];

    // see how long this is going to be
    int num_frames;

    if (pms)
      {
      num_frames = pms->num_frames();
      offset=0;
      }
    else
      num_frames = (int)ceil((float)sig.num_samples() / frame_shift);

    int num_samples = sig.num_samples();

    tmp_sig = new short[num_samples];
    for (i = 0; i < sig.num_samples(); ++i)
	tmp_sig[i] = sig.a(i);
    
    // desparate nightmare hack
    for (; i < num_samples; ++i)
	tmp_sig[i] = i%2; // to ensure that the analysis is working on something
	

    lpct.resize(num_frames, description->last_channel()+1);

    lpct.assign_map(description);
    lpct.set_contour_type(EST_ContourType::LPC);
    if (pms)
      lpct.set_space_type("VARI");
    else
      lpct.set_space_type("FIXED");

    for (k = 0 ; k < num_frames; ++k)
    {
	window_start = tmp_sig + pos - offset;
	sum = 0.0;
	int this_frame_length;

	if (pms)
	  {
	    float this_pm = pms->t(k);
	    // float next_pm = k<num_frames-1?pms->t(k+1):sig.end();
	    int this_pm_pos = (int)(this_pm * sig.sample_rate() + 0.5);
	    // int next_pm_pos = (int)(next_pm * sig.sample_rate() + 0.5);

	    this_frame_length = (int)((this_pm_pos - pos) + offset*2 + 0.5);
	  }
	else
	  {
	    this_frame_length = frame_length;
	  }

	if (this_frame_length > frame_length*3)
	  {
	    cerr << "frame too long - " << this_frame_length << "\n";
	  }

	if (pos + this_frame_length > num_samples)
	  this_frame_length = num_samples - pos;
	
	if (window_type != wf_rectangle)
	  {
	    if (pms)
	      window_vals = make_window(this_frame_length);

	    for(i = 0; i < this_frame_length; i++)
	      if (window_start + i >= tmp_sig)
		sum += window_start[i];

	    m = sum / this_frame_length;
		
	    for(i = 0; i < this_frame_length; i++)	
	      if (window_start + i >= tmp_sig && window_start + i < tmp_sig + num_samples)
		window[i] = window_vals(i) * (window_start[i] - m);
	      else
		window[i] = 0;
	  }
	else
	  for(i = 0; i < this_frame_length; i++)	
	    window[i] = window_start[i];
	  
	lpredict(window, this_frame_length, acf, ref, lpc, lporder);
		
	for (i = 0; i < lporder+1; ++i)
	  lpct.a(k, channel_coef0, i) = lpc[i];

	lpct.t(k) = (pos+0.0)/sig.sample_rate();

	if (do_length)
	  lpct.a(k, channel_length) = pms?this_frame_length: frame_shift;
	if (do_power)
	  lpct.a(k, channel_power) = get_rms(window, this_frame_length);
	if (do_voiced)
	  lpct.a(k, channel_voiced) = get_voiced(window, this_frame_length, acf, lporder);
	if (do_time)
	  lpct.a(k, channel_time) = lpct.t(k);

	// cout  << "frame " << k << " time " << lpct.t(k) << "\n";

	pos += this_frame_length - offset*2;
    }

    delete [] tmp_sig;
    delete [] window;
}

void lpc_residual(EST_Wave &residual, EST_Track &lpcs, EST_Wave &signal)

{
  short *res, *res_at;
  short *sig, *sig_at;
  int num_samples;
  int order;

  num_samples = sum_lengths(lpcs);

  order = get_order(lpcs);

  res = new short[num_samples];

  if (signal.length() < num_samples)
    {
      cout << "Warning: residual too short by " << (num_samples - signal.length()) << " samples\n";
      sig = new short[num_samples];
      memcpy((short *)sig, signal.data(), signal.length()*2);
      memset((short *)sig+signal.length(), 0, (num_samples - signal.length())*2);
    }
  else
    sig = signal.data();

  int i, wsize;

  res_at = res;
  sig_at = sig;
  

  for(i=0; i< lpcs.num_frames() ; res_at += wsize, sig_at += wsize, i++)
    {

      wsize = (int)lpcs.a(i, channel_length);

      if (res_at - res + wsize >= num_samples)
	wsize = num_samples - (res_at - res);

      if (wsize <1)
	break;

      if (res_at == res)
	{
	  // the first N samples are copied from the
	  // original signal. This is wrong, but unless
	  // we are analysing a small chunk the error won't show.

	  for(int j=0; j<order;j++)
	    res_at[j] = sig_at[j];
	  wsize -= order;
	  res_at +=order;
	  sig_at += order;
	}
      
      float filter[MAX_LPORDER];

      for(int c=0; c <= order; c++)
	filter[c] = lpcs.a(i, channel_coef0, c);


      residual_shorts(sig_at, res_at, wsize, filter, order);
      // printf("frame %d pos %d lpc[1] %f sample %d res %d\n", i, res_at-res, filter[1], sig_at[0], res_at[0]);
    }

  residual.set_data(res, num_samples, signal.sample_rate(), 1);

  if (sig != signal.data())
    delete[] (short *)sig;

}

void lpc_resynth(EST_Wave &signal, EST_Track &lpcs, 
		 EST_Wave &residual,
		 EST_Track *modifications,
		 EST_PpMod::Func *pp_mod,
		 EST_PpMod::Func *pp_mod_u)

{
  short *sig, *sig_at;
  short *res;
  int num_samples, num_frames, order;

  if (!pp_mod_u)
    pp_mod_u = pp_mod;

  if ((order = get_order(lpcs)) == 0)
    return;

  if (modifications)
    {
    num_samples = sum_lengths(*modifications);
    num_frames = modifications->num_frames();
    }
  else
    {
    num_samples = sum_lengths(lpcs);
    num_frames = lpcs.num_frames();
    }

  if (residual.length() < num_samples)
    {
      cout << "Warning: residual too short by " << (num_samples - residual.length()) << " samples\n";
      res = new short[num_samples];
      memcpy((short *)res, residual.data(), residual.length()*2);
      memset((short *)res+residual.length(), 0, (num_samples - residual.length())*2);
    }
  else
    res = residual.data();

  sig = new short[num_samples+order];

  sig_at = sig;

  while (sig_at < sig+order)
    *(sig_at++) = 0;
  
  lpc_resynth_chunk(sig_at,
		    lpcs,
		    res,
		    num_samples, 
		    0,
		    num_frames, order,	       
		    modifications,
		    pp_mod,
		    pp_mod_u);

  signal.set_data_copy(sig_at, num_samples, residual.sample_rate(), 1);

  delete[] sig;
  if (res != residual.data())
    delete[] (short *)res;
}

void lpc_resynth_chunk(short *sig,
		       EST_Track &lpcs,
		       const short *res,
		       int num_samples, 
		       int start_frame,
		       int num_frames, 
		       int order,	       
		       EST_Track *modifications,
		       EST_PpMod::Func *pp_mod,
		       EST_PpMod::Func *pp_mod_u,
		       short *res_return)
		       
{
  short *sig_at, *res_return_at;
  const short *pp_res;
  EST_TBuffer<short> res_buffer(300,100);
  int *res_pos=NULL;
  int orig_wsize;

  bool have_power = lpcs.has_channel(channel_power);
  bool have_new_power = (modifications != NULL 
			 && modifications->has_channel(channel_power));
  bool have_voiced = lpcs.has_channel(channel_voiced);

  res_pos = get_start_positions(lpcs);

  int i, wsize;

  sig_at = sig;
  res_return_at = res_return;

  for(i=start_frame; i< start_frame+num_frames ; sig_at += wsize, i++)
    {

      int frame = (int)(modifications?(modifications->a(i,channel_frame)):i);

      orig_wsize = (int)lpcs.a(frame, channel_length);
      wsize = (int)(modifications?(modifications->a(i,channel_length)):orig_wsize);

      // cout << "synth frame " << frame << "\n";

      if (sig_at - sig + wsize >= num_samples)
	wsize = num_samples - (sig_at - sig);

      if (wsize <1)
	break;

      float filter[MAX_LPORDER];

      for(int c=0; c <= order; c++)
	filter[c] = lpcs.a(frame, channel_coef0, c);

      if (pp_mod_u && have_voiced && lpcs.a(frame, channel_voiced) < 0.5)
	{
	  res_buffer.ensure(wsize, (bool)FALSE);
	  pp_res = res_buffer;
	  (*pp_mod_u)(res+res_pos[frame], orig_wsize, res_buffer, wsize);
	}
      else if (pp_mod)
	{
	  res_buffer.ensure(wsize, (bool)FALSE);
	  pp_res = res_buffer;
	  (*pp_mod)(res+res_pos[frame], orig_wsize, res_buffer, wsize);
	}
      else
	pp_res = res + res_pos[frame];

      resynth_shorts(sig_at, pp_res, wsize, filter, order);

      if (have_power || have_new_power)
	{
	  float target_rms;

	  if (have_new_power)
	    target_rms = modifications->a(i, channel_power);
	  else
	    target_rms = lpcs.a(frame, channel_power);
	  float resynth_rms = get_rms(sig_at, wsize);

	  // printf("rms: got %f want %f\n", resynth_rms, target_rms);

	  for(int s=0; s<wsize; s++)
	    sig_at[s] = (int)( (float)sig_at[s]*target_rms/resynth_rms + 0.5);
	}

      if (res_return_at)
	{
	  for(int r=0; r<wsize; r++)
	    res_return_at[r] = pp_res[r];
	  res_return_at += wsize;
	}
    }

  // cout << "chunk dur " << (sig_at-sig)/16000.0 << "\n";

  delete[] res_pos;
}

void lpc_resynth_split(EST_Wave &signal, EST_Track &lpcs, 
		       EST_Wave &residual,
		       EST_Track *modifications,
		       EST_PpMod::Func *pp_mod,
		       EST_PpMod::Func *pp_mod_u)

{
    short *sig;
    short *res;
    int num_samples, num_frames, order;

    if (!pp_mod_u)
	pp_mod_u = pp_mod;

    if ((order = get_order(lpcs)) == 0)
	return;

    if (modifications)
    {
	num_samples = sum_lengths(*modifications);
	num_frames = modifications->num_frames();
    }
    else
    {
	num_samples = sum_lengths(lpcs);
	num_frames = lpcs.num_frames();
    }

    if (residual.length() < num_samples)
    {
	cout << "Warning: residual too short by " << (num_samples - residual.length()) << " samples\n";
	res = new short[num_samples];
	memcpy((short *)res, residual.data(), residual.length()*2);
	memset((short *)res+residual.length(), 0, (num_samples - residual.length())*2);
    }
    else
	res = residual.data();

    sig = new short[num_samples];
    memset(sig, 0, num_samples*sizeof(short));

    lpc_resynth_chunk_split(sig,
			    lpcs,
			    res,
			    num_samples, 
			    0,
			    num_frames, 
			    order,	       
			    modifications,
			    pp_mod,
			    pp_mod_u);

    signal.set_data_copy(sig, num_samples, residual.sample_rate(), 1);

    delete[] sig;
    if (res != residual.data())
	delete[] (short *)res;
}

void lpc_resynth_chunk_split(short *sig,
			     EST_Track &lpcs,
			     const short *res,
			     int num_samples, 
			     int start_frame,
			     int num_frames, 
			     int order,	       
			     EST_Track *modifications,
			     EST_PpMod::Func *pp_mod,
			     EST_PpMod::Func *pp_mod_u,
			     short *res_return)
		       
{
    short *sig_at, *res_return_at;
    const short *pp_res;
    EST_TBuffer<short> 
	res_buffer(300,100), 
    pulse_buffer(300,100);
    int *res_pos=NULL;
    int orig_wsize;

    bool have_power = lpcs.has_channel(channel_power);
    bool have_new_power = (modifications != NULL 
			   && modifications->has_channel(channel_power));
    bool have_voiced = lpcs.has_channel(channel_voiced);

    res_pos = get_start_positions(lpcs);

    int i, wsize;

    sig_at = sig;
    res_return_at = res_return;

    memset(pulse_buffer, 0, order * sizeof(short));

    for(i=start_frame; i< start_frame+num_frames ; sig_at += wsize, i++)
    {
	int frame = (int)(modifications?(modifications->a(i,channel_frame)):i);

	orig_wsize = (int)lpcs.a(frame, channel_length);
	wsize = (int)(modifications?(modifications->a(i,channel_length)) :orig_wsize);

	if (sig_at - sig + wsize >= num_samples)
	    wsize = num_samples - (sig_at - sig);

	if (wsize <1)
	    break;

	float filter[MAX_LPORDER];

	for(int cc=0; cc <= order; cc++)
	    filter[cc] = lpcs.a(frame, channel_coef0, cc);

	if (pp_mod_u && have_voiced && lpcs.a(frame, channel_voiced) < 0.5)
	{
	    res_buffer.ensure(wsize, (bool)FALSE);
	    pp_res = res_buffer;
	    (*pp_mod_u)(res+res_pos[frame], orig_wsize, res_buffer, wsize);
	}
	else if (pp_mod)
	{
	    res_buffer.ensure(wsize, (bool)FALSE);
	    pp_res = res_buffer;
	    (*pp_mod)(res+res_pos[frame], orig_wsize, res_buffer, wsize);
	}
	else
	    pp_res = res + res_pos[frame];

	pulse_buffer.ensure(wsize+order, (short)0, order);

	// cout << "pulse\n";
	resynth_shorts(&(pulse_buffer.b()[order]), pp_res, 
		       wsize, filter, order);
	
	// printf("frame %d pos %d rpos %d lpc1 %f sample %d res %d\n", frame, sig_at - sig, res_pos[frame], filter[1], sig_at[0], res[res_pos[frame]]);
	
	float pulse_scale = 1.0;
	if (have_power || have_new_power)
	{
	    float target_rms;
	    
	    if (have_new_power)
		target_rms = modifications->a(i, channel_power);
	    else
		target_rms = lpcs.a(frame, channel_power);
	    
	    float pulse_rms = get_rms(&(pulse_buffer.b()[order]), wsize);
	    float ringing_rms = get_rms(sig_at, wsize);
	    float p_r_sum = 0;
	    
	    for(int ss=0; ss<wsize; ss++)
		p_r_sum += pulse_buffer.b()[order+ss] * sig_at[ss];
	    
	    float a = pulse_rms*pulse_rms;
	    float b = 2 * p_r_sum / wsize;
	    float c = ringing_rms*ringing_rms - target_rms*target_rms;
	    float tmp = b*b - 4*a*c;
	    
	    if (pulse_rms > 1 && tmp > 0)
		pulse_scale = ( -b + sqrt(tmp))/(2*a);
	    
	    // cout << "pulse " << pulse_rms << " ringing " << ringing_rms << " ring[10] " << sig_at[10] << " tmp " << tmp << " target = " << target_rms << " scale = " << pulse_scale << "\n";
	}
	
	for(int s=0; s<wsize; s++)
	    sig_at[s] = (int)( (float)sig_at[s] + pulse_buffer.b()[s+order]*pulse_scale + 0.5);
	
	// cout << "final = " << get_rms(sig_at, wsize) << "\n";
	
	if (frame+1 < lpcs.num_frames() && (sig_at-sig)+wsize*3 < num_samples)
	{
	    
	    for(int c=0; c <= order; c++)
		filter[c] = lpcs.a(frame+1, channel_coef0, c);
	    
	    // cout << "ringing\n";
	    resynth_noex_shorts(sig_at+wsize, wsize*2, filter, order);
	}
	
	if (res_return_at)
	{
	    for(int r=0; r<wsize; r++)
		res_return_at[r] = pp_res[r];
	    res_return_at += wsize;
	}
    }
    
    delete[] res_pos;
}

void lpc_resynth_split_old(EST_Wave &signal, EST_Track &lpcs, 
			   EST_Wave &residual,
			   EST_Track *modifications,
			   EST_PpMod::Func *pp_mod,
			   EST_PpMod::Func *pp_mod_u)

{
    short *sig, *sig_at;
    short *res, *pp_res;
    EST_TBuffer<short> res_buffer(300,100), 
    pulse_buffer(300,100);
    int num_samples, num_frames, 
    *res_pos=NULL;
    int order, orig_wsize;
    
    if (!pp_mod_u)
	pp_mod_u = pp_mod;
    
    bool have_power = lpcs.has_channel(channel_power);
    bool have_voiced = lpcs.has_channel(channel_voiced);
    
    if (lpcs.has_channel(channel_coefN))
	order = lpcs.channel_position(channel_coefN) - lpcs.channel_position(channel_coef0);
    else
	order = lpcs.num_channels()-lpcs.channel_position(channel_coef0)-1;
    
    if (modifications)
    {
	num_samples = sum_lengths(*modifications);
	num_frames = modifications->num_frames();
    }
    else
    {
	num_samples = sum_lengths(lpcs);
	num_frames = lpcs.num_frames();
    }
    
    
    sig = new short[num_samples];
    res_pos = get_start_positions(lpcs);
    
    res = residual.data();
    
    int i, wsize;
    
    sig_at = sig;
    
    memset(pulse_buffer, 0, order * sizeof(short));
    memset(sig, 0, num_samples*sizeof(short));
    
    for(i=0; i< num_frames ; sig_at += wsize, i++)
    {
	int frame = (int)(modifications?(modifications->a(i,channel_frame)):i);
	
	orig_wsize = (int)lpcs.a(frame, channel_length);
	wsize = (int)(modifications?(modifications->a(i,channel_length)):orig_wsize);
	
	if (sig_at - sig + wsize >= num_samples)
	    wsize = num_samples - (sig_at - sig);
	
	if (wsize <1)
	    break;
	
	float filter[MAX_LPORDER];
	
	for(int cc=0; cc <= order; cc++)
	    filter[cc] = lpcs.a(frame, channel_coef0, cc);
	
	if (pp_mod_u && have_voiced && lpcs.a(frame, channel_voiced) > 0.5)
	{
	    res_buffer.ensure(wsize, (bool)FALSE);
	    pp_res = res_buffer;
	    (*pp_mod)(res+res_pos[frame], orig_wsize, res_buffer, wsize);
	}
	else if (pp_mod)
	{
	    res_buffer.ensure(wsize, (bool)FALSE);
	    pp_res = res_buffer;
	    (*pp_mod)(res+res_pos[frame], orig_wsize, res_buffer, wsize);
	}
	else
	    pp_res = res + res_pos[frame];
	
	pulse_buffer.ensure(wsize+order, (short)0, order);
	
	// cout << "pulse\n";
	resynth_shorts(&(pulse_buffer.b()[order]), pp_res, wsize, filter, order);
	
	// printf("frame %d pos %d rpos %d lpc1 %f sample %d res %d\n", frame, sig_at - sig, res_pos[frame], filter[1], sig_at[0], res[res_pos[frame]]);
	
	float pulse_scale = 1.0;
	if (have_power)
	{
	    float target_rms = lpcs.a(frame, channel_power);
	    float pulse_rms = get_rms(&(pulse_buffer.b()[order]), wsize);
	    float ringing_rms = get_rms(sig_at, wsize);
	    float p_r_sum = 0;
	    
	    for(int ss=0; ss<wsize; ss++)
		p_r_sum += pulse_buffer.b()[order+ss] * sig_at[ss];
	    
	    float a = pulse_rms*pulse_rms;
	    float b = 2 * p_r_sum / wsize;
	    float c = ringing_rms*ringing_rms - target_rms*target_rms;
	    float tmp = b*b - 4*a*c;
	    
	    if (pulse_rms > 1 && tmp > 0)
		pulse_scale = ( -b + sqrt(tmp))/(2*a);
	    
	    // cout << "pulse " << pulse_rms << " ringing " << ringing_rms << " ring[10] " << sig_at[10] << " tmp " << tmp << " target = " << target_rms << " scale = " << pulse_scale << "\n";
	}
	
	for(int s=0; s<wsize; s++)
	    sig_at[s] = (int)( (float)sig_at[s] + pulse_buffer.b()[s+order]*pulse_scale + 0.5);
	
	// cout << "final = " << get_rms(sig_at, wsize) << "\n";
	
	if (frame+1 < num_frames && (sig_at-sig)+wsize+wsize < num_samples)
	{
	    for(int c=0; c <= order; c++)
		filter[c] = lpcs.a(frame+1, channel_coef0, c);
	    
	    // cout << "ringing\n";
	    resynth_noex_shorts(sig_at+wsize, wsize, filter, order);
	}
    }
    
    signal.set_data(sig, num_samples, residual.sample_rate(), 1);
    
    delete[] sig;
    delete[] res_pos;
}

void lpc_to_reflection(EST_Track &lpcs)
{
    // Convert standard lpc coefficients into reflection coefficients
    // This is done inline with no identification in the Track that
    // this is done.
    int f,i;
    float *buf,*rbuf;
    int ncoefs = lpcs.num_channels()-2;
    rbuf = new float[ncoefs];
    
    for (f=0; f < lpcs.num_frames(); f++)
    {
	buf = &lpcs(f,2);
	lpc2ref(buf,rbuf,ncoefs);
	for (i=0; i < ncoefs; i++)
	    lpcs(f,i+2) = rbuf[i];
    }
    
    delete [] rbuf;
}

float *Float_hamming(int size)
{
    float *window, k;
    int    i;
    
    k = 2.0 * M_PI / size;
    window = walloc(float,size);
    for(i = 0; i < size; i++)
	window[i] = 0.54 - 0.46 * cos(k * (i + 0.5));
    return(window);
}

/* watch out, these are all 0 .. order inclusive arrays */

static float lpredict(float *adc, int wsize, 
		      float *acf, float *ref, float *lpc,
		      int order) 
{
    int   i, j;
    float e, ci, sum;
    float tmp[MAX_ORDER];
    int stableorder=-1;
    
    if(order >= MAX_ORDER) 
	cerr << "lpc: " << order << "exceeds MAX_ORDER\n";
    
    for(i = 0; i <= order; i++) {
	sum = 0.0;
	for(j = 0; j < wsize - i; j++) sum += adc[j] * adc[j + i];
	acf[i] = sum;
    }
    
    /* find lpc coefficients */
    e = acf[0];
    lpc[0] = 1.0;
    for(i = 1; i <= order; i++) {
	ci = 0.0;
	for(j = 1; j < i; j++) 
	    ci += lpc[j] * acf[i-j];
	ref[i] = ci = (acf[i] - ci) / e;
	//Check stability of the recursion
	if (fabs(ci) < 1.000000) {
	    lpc[i] = ci;
	    for(j = 1; j < i; j++) 
		tmp[j] = lpc[j] - ci * lpc[i-j];
	    for(j = 1; j < i; j++) 
		lpc[j] = tmp[j];
	    e = (1 - ci * ci) * e;
	    stableorder = i;
	}
	else break;
    }
    if (stableorder != order) {
	fprintf(stderr,
		"warning:levinson instability, order restricted to %d\n",
		stableorder);
	for (;i<=order;i++)
	    lpc[i]=0.0;
    }   
    return(e);
}

float residual(const float *orig, float *e, int wsize, const float *a, int order) 
{
    int i, j;
    float maxv=0;
    
    for (i = 0; i < wsize; ++i)
    {
	e[i] = orig[i];
	for (j = 1; j <= order; ++j)
	    e[i] -= (a[j] * orig[i - j]);
	if (fabs(e[i]) > maxv)
	    maxv = fabs(e[i]);
    }
    return maxv;
}

float residual(const short *orig, float *e, int wsize, const float *a, int order) 
{
    int i, j;
    float maxv=0;
    
    for (i = 0; i < wsize; ++i)
    {
	e[i] = orig[i];
	for (j = 1; j <= order; ++j)
	    e[i] -= (a[j] * orig[i - j]);
	if (fabs(e[i]) > maxv)
	    maxv = fabs(e[i]);
    }
    return maxv;
}

static void residual_shorts(const short *orig, short *e, int wsize, const float *a, int order) 
{
    int i, j;
    
    for (i = 0; i < wsize; ++i)
    {
	double r = orig[i];
	for (j = 1; j <= order; ++j)
	    r -= (a[j] * orig[i - j]);
	
	e[i] = (short)(r+0.5);
    }
}

void resynth(float *synth, float *e, int wsize, float *a, int order) 
{
    int i, j;
    
    for (i = 0; i < wsize; ++i)
    {
	synth[i] = e[i];
	for (j = 1; j <= order; ++j)
	    synth[i] += (a[j] * e[i - j]);
    }
}

static void resynth_shorts(short *synth, const short *e, int wsize, float *a, int order) 
{
    int i, j; 
    
    for (i = 0; i < wsize; ++i)
    {
	double sample = e[i];
	
	for (j = 1; j <= order; ++j)
	    sample += (a[j] * synth[i - j]);
	
	synth[i] = (short)(sample+0.5);
	// cout << " res " << e[i] << " samp " << synth[i] << "\n";
    }
}

static void resynth_noex_shorts(short *synth, int wsize, float *a, int order) 
{
    int i, j; 
    
    for (i = 0; i < wsize; ++i)
    {
	double sample = 0;
	
	for (j = 1; j <= order; ++j)
	    sample += (a[j] * synth[i - j]);
	
	synth[i] = (short)(sample+0.5);
    }
}

