 /************************************************************************/
 /*                                                                      */
 /*                Centre for Speech Technology Research                 */
 /*                     University of Edinburgh, UK                      */
 /*                       Copyright (c) 1996,1997                        */
 /*                        All Rights Reserved.                          */
 /*                                                                      */
 /*  Permission to use, copy, modify, distribute this software and its   */
 /*  documentation for research, educational and individual use only, is */
 /*  hereby granted without fee, subject to the following conditions:    */
 /*   1. The code must retain the above copyright notice, this list of   */
 /*      conditions and the following disclaimer.                        */
 /*   2. Any modifications must be clearly marked as such.               */
 /*   3. Original authors' names are not deleted.                        */
 /*  This software may not be used for commercial purposes without       */
 /*  specific prior written permission from the authors.                 */
 /*                                                                      */
 /*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK       */
 /*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING     */
 /*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT  */
 /*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE    */
 /*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES   */
 /*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN  */
 /*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,         */
 /*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF      */
 /*  THIS SOFTWARE.                                                      */
 /*                                                                      */
 /************************************************************************/
 /*                 Author: Richard Caley (rjc@cstr.ed.ac.uk)            */
 /*                   Date: Thu Jun 19 1997                              */
 /************************************************************************/
 /*                                                                      */
 /* Converting between coefficiant type.                                 */
 /*                                                                      */
 /************************************************************************/

#include "EST_Track.h"
#include "EST_ContourType.h"
#include "EST_sigpr.h"

void ref2area(const float *ref, float *area, int order) {
    int   i;
    
    for(i = 1; i <= order; i++)
	area[i] = (1.0 - ref[i]) / (1.0 + ref[i]);
}

#define ALMOST1 0.99999
void ref2logarea(const float *ref, float *logarea, int order) 
{
    int   i;
    
    for(i = 1; i <= order; i++) {
	if(ref[i] > ALMOST1) logarea[i] = log((1.0 - ALMOST1) / (1.0 + ALMOST1));
	else if(ref[i] < -ALMOST1) logarea[i] = log((1.0 + ALMOST1)/(1.0-ALMOST1));
	else logarea[i] = log((1.0 - ref[i]) / (1.0 + ref[i]));
    }
}

void ref2truearea(const float *ref, float *area, int order) {
    int   i;
    
    area[1] = (1.0 - ref[1]) / (1.0 + ref[1]);
    for(i = 2; i <= order; i++) 
	area[i] = area[i - 1] * (1.0 - ref[i]) / (1.0 + ref[i]);
}

#define MAX_ABS_CEPS 4.0
void lpc2ceps(const float *lpc, int nlpc, float *ceps, int nceps) {
    int n;
    
    for(n = 1; n <= nlpc && n <= nceps; n++) {
	float sum = 0.0;
	int   k;
	
	for(k = 1; k < n; k++) sum += k * ceps[k] * lpc[n - k];
	ceps[n] = lpc[n] + sum / n;
    }
    
    /* be wary of these interpolated values */
    for(n = nlpc + 1; n <= nceps; n++) {
	float sum = 0.0;
	int   k;
	
	for(k = n - (nlpc - 1); k < n; k++) sum += k * ceps[k] * lpc[n - k];
	ceps[n] = sum / n;
    }
    
    /* very occasionally the above can go unstable, fudge if this happens */
    
    for(n = 1; n <= nceps; n++) {
	
	if( isnan(ceps[n]) ) ceps[n] = 0.0; // check if NaN -- happens on some frames of silence
	
	if(ceps[n] >  MAX_ABS_CEPS){
	    cerr << "WARNING : cepstral coeff " << n << " was " << ceps[n] << endl;
	    cerr << "lpc coeff " << n << " = " << lpc[n] << endl;
	    
	    ceps[n] =  MAX_ABS_CEPS;
	}
	
	if(ceps[n] < -MAX_ABS_CEPS) {
	    cerr << "WARNING : cepstral coeff " << n << " was " << ceps[n] << endl;
	    ceps[n] = -MAX_ABS_CEPS;
	}
    }
}

void lpc2ref(const float *lpc, float *rfc, int order)
{
    // LPC to reflection coefficients 
    // from code from Borja Etxebarria
    int i,j;
    float f,ai;
    float *vo,*vx;
    float *vn = new float[order];

    i = order-1;
    rfc[i] = ai = lpc[i];
    f = 1-ai*ai;
    i--;

    for (j=0; j<=i; j++)
	rfc[j] = (lpc[j]+((ai*lpc[i-j])))/f;

    /* vn=vtmp in previous #define */
    vo=rfc;

    for ( ;i>0; ) 
    {
	ai=vo[i];
	f = 1-ai*ai;
	i--;
	for (j=0; j<=i; j++)
	    vn[j] = (vo[j]+((ai*vo[i-j])))/f;
	
	rfc[i]=vn[i];

	vx = vn;
	vn = vo;
	vo = vx;
    }

    delete [] vn;
}

void ref2lpc(const float *rfc, float *lpc, int order)
{
    // Here we use Christopher Longet Higgin's alrgorihm converted to 
    // an equivalent by awb.  Its doesn't have hte reverse order or
    // negation requirement.
    float a,b;
    int n,k;

    for (n=0; n < order; n++)
    {
	lpc[n] = rfc[n];
	for (k=0; 2*(k+1) <= n+1; k++)
	{
	    a = lpc[k];
	    b = lpc[n-(k+1)];
	    lpc[k] = a-b*lpc[n];
	    lpc[n-(k+1)] = b-a*lpc[n];
	}
    }
}

void convert(EST_Track &tr, EST_ContourType from, EST_ContourType to)
{
    (void)tr;
    (void)from;
    (void)to;
}

