 /************************************************************************/
 /*                                                                      */
 /*                Centre for Speech Technology Research                 */
 /*                     University of Edinburgh, UK                      */
 /*                       Copyright (c) 1996,1997                        */
 /*                        All Rights Reserved.                          */
 /*                                                                      */
 /*  Permission to use, copy, modify, distribute this software and its   */
 /*  documentation for research, educational and individual use only, is */
 /*  hereby granted without fee, subject to the following conditions:    */
 /*   1. The code must retain the above copyright notice, this list of   */
 /*      conditions and the following disclaimer.                        */
 /*   2. Any modifications must be clearly marked as such.               */
 /*   3. Original authors' names are not deleted.                        */
 /*  This software may not be used for commercial purposes without       */
 /*  specific prior written permission from the authors.                 */
 /*                                                                      */
 /*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK       */
 /*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING     */
 /*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT  */
 /*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE    */
 /*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES   */
 /*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN  */
 /*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,         */
 /*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF      */
 /*  THIS SOFTWARE.                                                      */
 /*                                                                      */
 /*************************************************************************/
 /*                                                                       */
 /*                 Author: Richard Caley (rjc@cstr.ed.ac.uk)             */
 /*                   Date: Fri Aug 29 1997                               */
 /* --------------------------------------------------------------------  */
 /* Functions which modify the length of a pulse.                         */
 /*                                                                       */
 /*************************************************************************/

#include <stdio.h>
#include "EST_PpMod.h"

static short abs(short n) { return n<0?-n:n; }

static int find_min(const short *s, int p)
{
  int o=p, min=1000;

  for(int i=p-5; i<p+5; i++)
    if (abs(s[i]) < min)
      {
	min = abs(s[i]);
	o=i;
      }
  return o;
}

void EST_PpMod::stretch(const short *orig, int orig_length,
		     short *mod, int mod_length)
{
int i;
float j, j_step;

j_step = (orig_length+0.0)/mod_length;

for(i=0, j=0; i<mod_length; i++, j+=j_step)
  mod[i] = orig[(int)(j+0.5)];

}

void EST_PpMod::chop_1_3(const short *orig, int orig_length,
		     short *mod, int mod_length)
{
int num_copy_start = orig_length/3;
int num_copy_end = orig_length/6;

num_copy_start = find_min(orig, num_copy_start);
num_copy_end = orig_length - find_min(orig, orig_length - num_copy_end);

int i=0, j;

for(i=0, j=0; i<num_copy_start; i++, j++)
  mod[i] = orig[j];

for(j=num_copy_start; i<mod_length-num_copy_end; i++)
  {
    mod[i] = orig[j];
    j++;
    if (j > orig_length - num_copy_end)
      j=num_copy_start;
  }

for(j=orig_length -num_copy_end ; i < mod_length; i++, j++)
  mod[i] = orig[j];
}

void EST_PpMod::stretch_mid(const short *orig, int orig_length,
			short *mod, int mod_length)
{
  int i;
  float j, j_step;
  float ja, ja_step;
  int peak = (int)(mod_length*4.0/5.0+0.5);

  j_step = (orig_length+0.0)/mod_length - 1;
  ja_step = 2.0/(peak);

  // cout << "from " << orig_length << " to " << mod_length << "\n";

  for(i=0, j=0, ja=0; i < peak; i++, j+=(1+j_step*ja), ja+=ja_step)
    {
      // cout << i << " = " << (int)(j+0.5) << "\n";
      mod[i] = orig[(int)(j+0.5)];
    }

  ja_step = 2.0/(mod_length-peak);

  for(ja=2; i<mod_length; i++, j+=(1+j_step*ja), ja -=ja_step)
    {
      // cout << i << " = " << (int)(j+0.5) << "\n";
      mod[i] = orig[(int)(j+0.5)];
    }

}

void EST_PpMod::stretch_1_3(const short *orig, int orig_length,
		     short *mod, int mod_length)
{
int num_copy_start, num_copy_end;

if ( mod_length > orig_length / 2)
  {
    num_copy_start = orig_length/3;
    num_copy_end = orig_length/6;
  }
else if ( mod_length > orig_length * 3 / 8)
  {
    num_copy_start = orig_length/4;
    num_copy_end = orig_length/8;
  }
else
  {
    EST_PpMod::stretch(orig, orig_length, mod, mod_length);
    return;
  }

int i=0, j;

for(i=0, j=0; i<num_copy_start; i++, j++)
  mod[i] = orig[j];

float k=j;
float k_step = (orig_length-num_copy_start-num_copy_end+0.0)/(mod_length-num_copy_start-num_copy_end+0.0);

for(; i<mod_length-num_copy_end; i++)
  {
    mod[i] = orig[(int)(k+0.5)];
    k += k_step;
  }

for(j=orig_length -num_copy_end ; i < mod_length; i++, j++)
  mod[i] = orig[j];
}


// pp resampling code


#define MAX_PERIOD	(500)	/* bigger than vocoder.h */
#define MIN_PERIOD       (30)
#define	IBUF_SIZE	(MAX_PERIOD + 10)
#define	OBUF_SIZE	(2 * IBUF_SIZE)

#define irint(F) ((int)((F)+0.5))
#define srint(F) ((short)((F)+0.5))

static void     
hi_lift (short    *data,
	 int      data_len)
/*
    Self contained little beastie to give a +6db lift at half the
    sampling rate to the input data.  Useful when counteracting spectral
    slope imposed by repeated LP filtering in pulse resampling or when
    interpolating data.  Filter gain is unity at DC.

    H(z) = 1.6 / ( 1.0 + 0.6*(1/z) )  ... gives a pole at z = -0.6
*/
{
    static float   state=0.0;
    int   i;

    for ( i=0; i<data_len; i++ ) {
      state = 1.6 * (float)*data - 0.6 * state;
      *data++ = (short)state;
    }
}

static void
second_order_section ( float *x, float (*states)[2], float (*coefs)[5])

/*
    This function updates the states of a 2nd order section
    Coefficients are stored as above (simillar to fdesign).
*/
{
float   u;

   /* Feedback part */
   u = *x - (*states)[0] * (*coefs)[3] - (*states)[1] * (*coefs)[4];

   /* Feed forward part and section gain */
   *x = (*coefs)[0] * ( u + (*states)[0] * (*coefs)[1] + (*states)[1] * (*coefs)[2] );

   /* Update states */
   (*states)[1] = (*states)[0];
   (*states)[0] = u;
}                                            /* End of 2nd order section */


static void
iir_2 ( const short *ibuf, short *obuf, float states[4][2], float coefs[4][5], int size, int order )

{
   int	    i, j, sections;
   float   (*s)[2], (*c)[5];
   float            x;

   /* Main Loop,  through samples in the buffer */
   for ( i=0; i<size; i++ ) {

     x = *ibuf++;                            /* Get next sample into work/op area */
     sections = (order - 1)/2 + 1;           /* Calc number of sections */
     s = states;                             /* Re-init ptrs to staes */
     c = coefs;

     /* Loop Through sections in filter */
     for ( j=0; j<sections; j++ ) {
       second_order_section( &x, s++, c++ );
     }                                       /* End of Loop through filter sections */
     *obuf++ = irint (x);
   }                                         /* End of main loop */
}                                            /* End of iir_2 */



void EST_PpMod::resample(const short *ibuf, 
	    int old_pp, 
	    short *obuf, 
	    int new_pp)


/*
  Resamples pules by upsampling to 2*sasmpling rate and LP filtering to rate/2.
  Linear interpolation of samples to resample followed by reverse LP filter
  under the new time index ( the pulse spectra has been stretched by under sampling ).
  Finally downsample back to sampling rate. The filter states are cleared each
  frame, the reverse filtering needs to do this, and the forward filter certainly
  must following uv/v or frame repeats, which is most of the time. Also the nature
  of what we're doing implies we treat each pulse as separate from all previous ones
  anyhow.

*/
{
       int    i, j;
       float  tscale, t_old, x;

       int ioffset=0;
       int ooffset=0;

       /* Coeficients for 2nd order sections from fdesign program:

	  fdesign -aL -s10000.0 8 2500.0 LP_butter 
       
       Giving an 8th order (4section) Butterworth LP filter with 3db point
       at 1/4 sampling rate.  For use in upsampling by factor 2            */

       static float    coefs[ 4 ] [ 5 ] = {
	 { 0.418378, 2.0, 1.0, 0.0, 0.673514 },
	 { 0.321426, 2.0, 1.0, 0.0, 0.285702 },
	 { 0.273005, 2.0, 1.0, 0.0, 0.092019 },
	 { 0.252425, 2.0, 1.0, 0.0, 0.009701 }
       };
       static float    states1[ 4 ] [ 2 ] = {
	 { 0.0, 0.0 }, { 0.0, 0.0 },
	 { 0.0, 0.0 }, { 0.0, 0.0 }
       };
       static float    states2[ 4 ] [ 2 ] = {
	 { 0.0, 0.0 }, { 0.0, 0.0 },
	 { 0.0, 0.0 }, { 0.0, 0.0 }
       };

       static float    last_val = 0.0;
       short           up_buf1[ (2 * IBUF_SIZE) ];
       short           up_buf2[ (2 * OBUF_SIZE) ];
       short           rev[ (2 * OBUF_SIZE) ];
       int             up_new_pp, up_old_pp;


       ibuf += ioffset;
       obuf += ooffset;

       for ( i=0; i < new_pp; obuf[ i++ ] = 0 );  /* Clear buffers */
       for ( i=0; i < (2*IBUF_SIZE); up_buf1[ i++ ] = 0 );
       for ( i=0; i < (2*OBUF_SIZE); up_buf2[ i++ ] = 0 );
       for ( i=0; i < (2*OBUF_SIZE); rev[ i++ ] = 0 );
       for ( i=0; i < 4; i++ ) {         /* Clear filter states too */
	 states1[i][0] = states1[i][1] = 0.0;
	 states2[i][0] = states2[i][1] = 0.0;
       }

       for ( i=0, j=0; i<old_pp; i++ ) {           /* Upsample into up_buf1 */ 
	 x = ibuf[ i ];
	 up_buf1[ j++ ] = srint(x);
	 up_buf1[ j++ ] = srint(x);
       }

       j = 0;                                      /* index to input pulse */
       t_old = -1.0;                               /* corresponds to last sample */
       tscale = (float) old_pp / (float) new_pp;   /* maps modified to original */
       up_old_pp = 2*old_pp;
       up_new_pp = 2*new_pp;

       /* LP filter upsampled pulse */
       iir_2 ( up_buf1, up_buf1, states1, coefs, up_old_pp, 8 );

       /* Copy and time-scale the samples; do not try to interpolate the last one */
       /* to avoid trying to read up_buf1[ up_old_pp ] which is off the end.       */
       for ( i=0; i<( up_new_pp - 1 ); i++ ) {
	 t_old += tscale;
	 if ( (int)t_old >= j ) {
           j = (int)t_old + 1;
	   if ( j >= ( up_old_pp ) ) {
	     fprintf( stderr, 
	     "foedit: resampling read beyond end of i/p buffer, %d\n", up_old_pp );
	     j = up_old_pp - 1;
	   }
	   last_val = up_buf1[ j-1 ];
	 }

	 /* Interpolate upsampled version */
         up_buf2[i] = srint(( t_old-(float)(j-1) )*( up_buf1[j] - last_val ) + last_val);
       }
       /* Copy last sample to pin down the time scaling to the pulse ends */
       last_val = up_buf1[ up_old_pp - 1 ];
       up_buf2[ up_new_pp - 1 ] = srint(last_val);

       /* Reverse LP filter upsampled and freq stretched pulse */
       j = up_new_pp - 1;
       for ( i=0; i<up_new_pp; i++ )   rev[i] = up_buf2[j--];
       iir_2 ( rev, rev, states2, coefs, up_new_pp, 8 );
       j = up_new_pp - 1;
       for ( i=0; i<up_new_pp; i++ )   up_buf2[j--] = rev[i];

       for ( i=0; i<new_pp; i++ ) {            /* downsample from up_buf2 */ 
	 obuf [i] = up_buf2[ 2*i ];
       }

       /* A little hi-lift as double filtering and interpolation give too much
	  of a high roll-off */
       hi_lift ( obuf, new_pp );               /* 6db lift at sr/2 */
}


