/*************************************************************************/
/*                                                                       */
/*                Centre for Speech Technology Research                  */
/*                     University of Edinburgh, UK                       */
/*                      Copyright (c) 1995,1996                          */
/*                        All Rights Reserved.                           */
/*                                                                       */
/*  Permission to use, copy, modify, distribute this software and its    */
/*  documentation for research, educational and individual use only, is  */
/*  hereby granted without fee, subject to the following conditions:     */
/*   1. The code must retain the above copyright notice, this list of    */
/*      conditions and the following disclaimer.                         */
/*   2. Any modifications must be clearly marked as such.                */
/*   3. Original authors' names are not deleted.                         */
/*  This software may not be used for commercial purposes without        */
/*  specific prior written permission from the authors.                  */
/*                                                                       */
/*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK        */
/*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING      */
/*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT   */
/*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE     */
/*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES    */
/*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN   */
/*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,          */
/*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF       */
/*  THIS SOFTWARE.                                                       */
/*                                                                       */
/*************************************************************************/
/*                         Author :  Simon King                          */
/*                         Date   :  April 1995                          */
/*-----------------------------------------------------------------------*/
/*                  EST_FMatrix Class auxilliary functions               */
/*                                                                       */
/*=======================================================================*/

#include "EST_FMatrix.h"
#include "math.h"
#include <limits.h>

bool polynomial_fit(EST_FVector &x, EST_FVector &y, EST_FVector
		    &co_effs, int order)
{
    EST_FVector weights;
    weights.resize(x.n());
    for(int i=0; i<x.n(); ++i)
	weights(i) = 1.0;
    
    return polynomial_fit(x,y,co_effs,weights,order);
}

bool polynomial_fit(EST_FVector &x, EST_FVector &y, EST_FVector &co_effs, 
	       EST_FVector &weights, int order)
{
    if(order <= 0){
	cerr << "polynomial_fit : order must be >= 1" << endl;
	return false;
    }

    if(x.n() != y.n()){
	cerr << "polynomial_fit : x and y must have same dimension" << endl;
	return false;
    }

    if(weights.n() != y.n()){
	cerr << "polynomial_fit : weights must have same dimension as x and y" << endl;
	return false;
    }

    if(x.n() <= order){
	cerr << "polynomial_fit : x and y must have at least order+1 elements"
	    << endl;
	return false;
    }
    

    // matrix of basis function values
    EST_FMatrix A;
    A.resize(x.n(),order+1);
    
    EST_FVector y1;
    y1.resize(y.n());
    
    for(int row=0;row<y.n();row++){
	y1(row) = y(row) * weights(row);
	for(int col=0;col<=order;col++){
	    A(row,col) = pow(x(row),(float)col) * weights(row);
	    
	}
    }
    
    // could call pseudo_inverse, but save a bit by doing
    // it here since we need transpose(A) anyway
    
    
    EST_FMatrix At = transpose(A);
    EST_FMatrix At_A = At * A;
    EST_FMatrix At_A_inv = inverse(At_A);
    
    // error check
    if(At_A_inv.num_rows() == 0){
	cerr << "polynomial_fit : inverse failed" << endl;
	return false;
    }
    
    EST_FVector At_y1 = At * y1;
    co_effs = At_A_inv * At_y1;
    return true;
    
}

int matrix_max(const IMatrix &a)
{
    int i, j;
    int v = 0;			// warning - this should be set to MININT, but I don't know
    // where this is to be found
    
    for (i = 0; i < a.num_rows(); ++i)
	for (j = 0; j < a.num_columns(); ++j)
	    if (a.el(i, j) > v)
		v = a.el(i, j);
    
    return v;
}

float matrix_max(const EST_FMatrix &a)
{
    int i, j;
    float v = INT_MIN;
    
    for (i = 0; i < a.num_rows(); ++i)
	for (j = 0; j < a.num_columns(); ++j)
	    if (a.el(i, j) > v)
		v = a.el(i, j);
    
    return v;
}

int square(const EST_FMatrix &a)
{
    return a.num_rows() == a.num_columns();
}
// add all elements in matrix.
float sum(const EST_FMatrix &a)
{
    int i, j;
    float t = 0.0;
    for (i = 0; i < a.num_rows(); ++i)
	for (j = 0; j < a.num_columns(); ++j)
	    t += a.el(i, j);
    return t;
}

// set all elements not on the diagonal to zero.
EST_FMatrix diagonalise(const EST_FMatrix &a)
{
    int i;
    EST_FMatrix b(a, 0);	// initialise and fill b with zeros
    for (i = 0; i < a.num_rows(); ++i)
	b(i, i) = a.el(i, i);
    
    return b;
}

EST_FMatrix sub(const EST_FMatrix &a, int row, int col)
{
    int i, j, I, J;
    int n = a.num_rows() - 1;
    EST_FMatrix s(n, n);
    
    for (i = I = 0; i < n; ++i, ++I)
    {
	if (I == row)
	    ++I;
	for (j = J = 0; j < n; ++j, ++J)
	{
	    if (J == col)
		++J;
	    s(i, j) = a.el(I, J);
	}
    }
    
    //    cout << "sub: row " << row  << " col " << col << "\n" << s;
    
    return s;
}

EST_FMatrix row(const EST_FMatrix &a, int row)
{
    EST_FMatrix s(1, a.num_columns());
    int i;
    
    for (i = 0; i < a.num_columns(); ++i)
	s(0, i) = a.el(row, i);
    
    return s;
}

EST_FMatrix column(const EST_FMatrix &a, int col)
{
    EST_FMatrix s(a.num_rows(), 1);
    int i;
    
    for (i = 0; i < a.num_rows(); ++i)
	s(i, 0) = a.el(i, col);
    
    return s;
}

EST_FMatrix triangulate(const EST_FMatrix &a)
{
    EST_FMatrix b(a, 0);
    int i, j;
    
    for (i = 0; i < a.num_rows(); ++i)
	for (j = i; j < a.num_rows(); ++j)
	    b(j, i) = a.el(j, i);
    
    return b;
}

EST_FMatrix transpose(const EST_FMatrix &a)
{
    int i, j;
    EST_FMatrix t(a.num_columns(), a.num_rows());
    //    cout << "before trans " << a << endl;
    //    cout << "a r " << a.num_rows() << " c " << a.num_columns() << endl;
    //    cout << "t r " << t.num_rows() << " c " << t.num_columns() << endl;
    
    for (i = 0; i < t.num_rows(); ++i)
	for (j = 0; j < t.num_columns(); ++j)
	    t(i, j) = a.el(j, i);
    return t;
}

EST_FMatrix backwards(EST_FMatrix &a)
{
    int i, j, n;
    n = a.num_columns();
    EST_FMatrix t(n, n);
    
    for (i = 0; i < n; ++i)
	for (j = 0; j < n; ++j)
	    t(n - i - 1, n - j - 1) = a.el(i, j);
    
    return t;
}


// changed name from abs as it causes on at least on POSIX machine
// where int abs(int) is a macro
EST_FMatrix fmatrix_abs(const EST_FMatrix &a)
{
    int i, j;
    EST_FMatrix b(a, 0);
    
    for (i = 0; i < a.num_rows(); ++i)
	for (j = 0; j < a.num_columns(); ++j)
	    b(i, j) = fabs(a.el(i, j));
    
    return b;
}
extern "C" int mat_inv(int n, double **mat_in, double **inv_out);

EST_FMatrix inverse(EST_FMatrix &a)
{
    EST_FMatrix b;
    int i, j;
    int n = a.num_rows();
    double **mat_in, **inv_out;
    
    mat_in = new double*[n];
    inv_out = new double*[n];
    
    for (i = 0; i < n; ++i)
    {
	mat_in[i] = new double[n];
	inv_out[i] = new double[n];
	for (j = 0; j < n; ++j)
	{
	    mat_in[i][j] = a.el(i, j);
	    inv_out[i][j] = 0.0;
	}
    }
    
    if (mat_inv(n,mat_in, inv_out) != 0)
    {
	cerr << "Inverse: Matrix has singularity" << endl;
	return b;			// dah what do you do here ?
    }
    
    // simonk : added this
    b.resize(n,n);
    
    for (i = 0; i < n; ++i)
	for (j = 0; j < n; ++j)
	    b(i,j) = inv_out[i][j];
    
    return b;
}

EST_FMatrix pseudo_inverse(EST_FMatrix &a)
{
    
    // for non-square matrices
    // useful for solving linear eqns
    // (e.g. polynomial fitting)
    
    // is it square ?
    if( a.num_rows() == a.num_columns() )
	return inverse(a);
    
    if( a.num_rows() < a.num_columns() ){
	cerr << "pseudo-inverse requires rows >=  cols" << endl;
	return a;			// !? just waiting for exceptions I suppose
    }
    
    // this looks like it might be slow ... but who cares
    EST_FMatrix a_trans = transpose(a);
    EST_FMatrix atrans_a = a_trans * a;
    EST_FMatrix atrans_a_inverse = inverse(atrans_a);
    EST_FMatrix pseudo_inv = atrans_a_inverse * a_trans;
    
    // pseudo inverse has same dimensions as transpose(A)
    return pseudo_inv;
}


float determinant(const EST_FMatrix &a)
{
    int i, j;
    int n = a.num_rows();
    float det;
    if (!square(a))
    {
	cerr << "Tried to take determinant of non-square matrix\n";
	return 0.0;
    }
    
    EST_FVector A(n);
    
    if (n == 2)			// special case of 2x2 determinant
	return (a.el(0,0) * a.el(1,1)) - (a.el(0,1) * a.el(1,0));
    
    float p;
    
    // create cofactor matrix
    j = 1;
    for (i = 0; i < n; ++i)
    {
	p = (float)(i + j + 2);	// because i & j should start at 1
	//	cout << "power " <<p << endl;
	A(i) = pow(-1.0, p) * determinant(sub(a, i, j));
    }
    //    cout << "cofactor " << A;
    
    // sum confactor and original matrix 
    det = 0.0;
    for (i = 0; i < n; ++i)
	det += a.el(i, j)* A(i);
    
    return det;
}
