/*************************************************************************/
/*                                                                       */
/*                Centre for Speech Technology Research                  */
/*                     University of Edinburgh, UK                       */
/*                      Copyright (c) 1995,1996                          */
/*                        All Rights Reserved.                           */
/*                                                                       */
/*  Permission to use, copy, modify, distribute this software and its    */
/*  documentation for research, educational and individual use only, is  */
/*  hereby granted without fee, subject to the following conditions:     */
/*   1. The code must retain the above copyright notice, this list of    */
/*      conditions and the following disclaimer.                         */
/*   2. Any modifications must be clearly marked as such.                */
/*   3. Original authors' names are not deleted.                         */
/*  This software may not be used for commercial purposes without        */
/*  specific prior written permission from the authors.                  */
/*                                                                       */
/*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK        */
/*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING      */
/*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT   */
/*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE     */
/*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES    */
/*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN   */
/*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,          */
/*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF       */
/*  THIS SOFTWARE.                                                       */
/*                                                                       */
/*************************************************************************/
/*                         Author :  Paul Taylor                         */
/*                         Date   :  April 1995                          */
/*-----------------------------------------------------------------------*/
/*                        Matrix Class for floats                        */
/*                                                                       */
/*=======================================================================*/

#include <stdlib.h>
#include <stdio.h>
#include <fstream.h>
#include <math.h>
#include <limits.h>
#include "EST_String.h"
#include "EST_types.h"
#include "EST_FMatrix.h"
#include "EST_cutils.h"  // for swap functions 
#include "EST_Token.h"

static unsigned int fmatrix_magic = 0x5354464D; /* STFM */

IMatrix::IMatrix(IMatrix &a, int b) 
:EST_TMatrix<int>(a.num_rows(), a.num_columns())
{
    if (b < 0)
	return;
    if (b == 0)
	fill(0);
    def_val = 0;
}

void IMatrix::fill(int v)
{
    int i, j;
    for (i = 0; i < num_rows(); ++i)
	for (j = 0; j < num_columns(); ++j)
	    x[i][j] = v;
}

EST_FMatrix::EST_FMatrix(const EST_FMatrix &a, int b)
:EST_TMatrix<float>(a.num_rows(), a.num_columns())
{
    if (b < 0)
	return;
    if (b == 0)
	fill(0.0);
    def_val = 0.0;
}

void EST_FMatrix::fill(float v)
{
    int i, j;
    for (i = 0; i < num_rows(); ++i)
	for (j = 0; j < num_columns(); ++j)
	    x[i][j] = v;
}

EST_FMatrix & EST_FMatrix::operator+=(const EST_FMatrix &a)
{
    int i, j;
    if (a.num_columns() != num_columns())
    {
	cerr <<"Matrix addition error: bad number of columns\n";
	return *this;
    }
    if (a.num_rows() != num_rows())
    {
	cerr <<"Matrix addition error: bad number of rows\n";
	return *this;
    }
    for (i = 0; i < a.num_rows(); ++i)
	for (j = 0; j < a.num_columns(); ++j)
	    x[i][j] += a.x[i][j];

    return *this;
}

EST_FMatrix operator+(const EST_FMatrix &a, const EST_FMatrix &b)
{
    EST_FMatrix ab;
    int i, j;
    if (a.num_columns() != b.num_columns())
    {
	cerr <<"Matrix addition error: bad number of columns\n";
	return ab;
    }
    if (a.num_rows() != b.num_rows())
    {
	cerr <<"Matrix addition error: bad number of rows\n";
	return ab;
    }
    ab.resize(a.num_rows(), a.num_columns());
    for (i = 0; i < a.num_rows(); ++i)
	for (j = 0; j < a.num_columns(); ++j)
	    ab(i, j) = a.el(i, j) + b.el(i, j);

    return ab;
}

EST_FMatrix operator-(const EST_FMatrix &a,const EST_FMatrix &b)
{
    EST_FMatrix ab;
    int i, j;

    if (a.num_columns() != b.num_columns())
    {
	cerr <<"Matrix subtraction error: bad number of columns:" <<
	    a.num_columns() << " and " << b.num_columns() << endl;
	return ab;
    }
    if (a.num_rows() != b.num_rows())
    {
	cerr <<"Matrix subtraction error: bad number of rows\n";
	return ab;
    }
    ab.resize(a.num_rows(), a.num_columns());
    for (i = 0; i < a.num_rows(); ++i)
	for (j = 0; j < a.num_columns(); ++j)
	    ab(i, j) = a.el(i, j) - b.el(i, j);

    return ab;
}

EST_FMatrix operator*(const EST_FMatrix &a, const float x)
{
    EST_FMatrix b(a, 0);
    int i, j;

    for (i = 0; i < a.M; ++i)
	for (j = 0; j < a.N; ++j)
	    b.x[i][j] = a.x[i][j] * x;

    return b;
}

EST_FVector operator*(const EST_FMatrix &a, const EST_FVector &v)
{    
    EST_FVector b;
    b.resize(a.num_rows());
    
    if(a.num_columns() != v.n()){
	cerr <<"Matrix-vector multiplication error: matrix rows != vector size"
	     << endl;
	return b;
    }

    int i, j;
    for (i = 0; i < a.num_rows(); ++i){
	b(i) = 0.0;
	for (j = 0; j < a.num_columns(); ++j)
	    b(i) += a.el(i,j) * v(j);
    }
    return b;
}


#if 0
EST_FMatrix operator/(const EST_FMatrix &a, float x)
{
    return (a * (1/x));
}
#endif

EST_FMatrix operator*(const EST_FMatrix &a, const EST_FMatrix &b)
{
    EST_FMatrix ab;
    if (a.num_columns() != b.num_rows())
    {
	cerr <<"Matrix multiplication error: bad number of rows and columns\n";
	return ab;
    }

    ab.resize(a.num_rows(), b.num_columns());
    int i, j, k, n;
    n = a.num_columns();	// could also be b.num_rows()
    
    for (i = 0; i < a.num_rows(); ++i)
	for (k = 0; k < b.num_columns(); ++k)
	{
	    ab(i, k) = 0.0;
	    for (j = 0; j < n; ++j)
		ab(i, k) += a.el(i, j) * b.el(j, k);
	}
    
    return ab;
}

#if 0
EST_FMatrix inverse_old(EST_FMatrix &a)
{
    EST_FMatrix b;
    float det;
    if (!square(a))
    {
	cerr << "Tried to take inverse of non-square matrix\n";
	return b;
    }
    
      b = transpose(a);
//    b = backwards(a);
    cout << "trans " <<b;
    det = determinant(a);
    cout << "determinant " <<det << endl;
    
    b = b * (1/det);
    cout << "\ninverse\n" << b << endl << endl;
    return b;
}
#endif

void EST_FMatrix::copyin(float **inx, int rows, int cols)
{
    int i, j;

    resize(rows, cols);

    for (i = 0; i < rows; ++i)
	for (j = 0; j < cols; ++j)
	    x[i][j] = inx[i][j];
    
}

EST_write_status EST_FMatrix::save(const EST_String &file)
{
    int i, j;
    ostream *outf;
    if (file == "-")
	outf = &cout;
    else
	outf = new ofstream(file);
    
    for (i = 0; i < num_rows(); ++i)
    {
	for (j = 0; j < num_columns(); ++j)
	    *outf << x[i][j] << " ";
	*outf << endl;
    }
    
    if (outf != &cout)
	delete outf;

    return write_ok;
}

EST_write_status EST_FMatrix::binsave(const EST_String &file)
{
    // Binary save with short header for byte swap and sizes
    int i;
    FILE *fd;
    if (file == "-")
	fd = stdout;
    else if ((fd = fopen(file,"wb")) == NULL)
    {
	cerr << "EST_FMatrix: binsave: failed to open \"" << file << 
	    "\" for writing" << endl;
	return misc_write_error;
    }
    
    fwrite(&fmatrix_magic,sizeof(int),1,fd);
    fwrite(&M,sizeof(int),1,fd);    // rows 
    fwrite(&N,sizeof(int),1,fd);    // columns

    for (i = 0; i < num_rows(); ++i)
    {
	if (fwrite(x[i],sizeof(float),num_columns(),fd) != 
	    (unsigned int)num_columns())
	{
	    cerr << "EST_FMatrix: binsave: failed to write row " << i << "to \"" 
		<< file << "\"" << endl;
	    return misc_write_error;
	}
    }
    
    if (fd != stdout)
	fclose(fd);

    return write_ok;
}

EST_read_status EST_FMatrix::binload(const EST_String &file)
{
    // Binary load with short header for byte swap and sizes
    int i;
    unsigned int magic;
    int rows, cols, swap;
    FILE *fd;
    
    if (file == "-")
	fd = stdin;
    else if ((fd = fopen(file,"rb")) == NULL)
    {
	cerr << "EST_FMatrix: binload: failed to open \"" << file << 
	    "\" for reading" << endl;
	return misc_read_error;
    }
    
    fread(&magic,sizeof(int),1,fd);
    if (magic == fmatrix_magic)
	swap = FALSE;
    else if (magic == SWAPINT(fmatrix_magic))
	swap = TRUE;
    else
    {
	cerr << "EST_FMatrix: binload: file \"" << file << 
	    "\" not a binary matrix file" << endl;
	return wrong_format;
    }
    fread(&rows,sizeof(int),1,fd);    // rows 
    fread(&cols,sizeof(int),1,fd);    // columns
    if (swap)
	resize(SWAPINT(rows),SWAPINT(cols));
    else
	resize(rows,cols);

    for (i = 0; i < num_rows(); ++i)
    {
	if (fread(x[i],sizeof(float),num_columns(),fd) != 
	    (unsigned int)num_columns())
	{
	    cerr << "EST_FMatrix: binload: failed to read row " << i << "to \"" 
		<< file << "\"" << endl;
	    return misc_read_error;
	}
	if (swap)
	    swap_bytes_float(x[i],num_columns());
    }
    
    if (fd != stdin)
	fclose(fd);

    return format_ok;
}

EST_read_status EST_FMatrix::load(const EST_String &filename)
{
    EST_TokenStream ts, tt;
    EST_StrList sl;
    int i, j, n_rows, n_cols;
    EST_String t;
    EST_TBI *p;
    
    if (((filename == "-") ? ts.open(cin) : ts.open(filename)) != 0)
    {
	cerr << "Can't open track file " << filename << endl;
	return misc_read_error;
    }
    // set up the character constant values for this stream
    ts.set_SingleCharSymbols(";");
    
    // first read in as list
    for (n_rows = 0; !ts.eof(); ++n_rows)
	sl.append(ts.get_upto_eoln().string());

    tt.open_string(sl.first());
    for (n_cols = 0; !tt.eof(); ++n_cols)
    	tt.get().string();

    // resize track and copy values in
    resize(n_rows, n_cols);

    for (p = sl.head(), i = 0; p != 0; ++i, p = next(p))
    {
	tt.open_string(sl(p));
	for (j = 0; !tt.eof(); ++j)
	    x[i][j] = atof(tt.get().string());
	if (j != n_cols)
	{
	    cerr << "Wrong number of points in row " << i << endl;
	    cerr << "Expected " << n_cols << " got " << j << endl;
	    return misc_read_error;
	}
    }
    return format_ok;
}

EST_FVector& EST_FVector::operator*=(const EST_FVector &s)
{
    if(n() != s.n()){
	cerr << "Cannot elementwise multiply vectors of differing lengths" << endl;
	
	return *this;
    }

    for (int i = 0; i < n(); ++i)
	(*this)(i) *= s(i);

    return *this;
}

/*EST_FVector& EST_FVector::operator=(EST_FList &s)
{
    EST_TBI *p;
    int i;

    this->resize(s.length());
    for (i = 0, p = s.head(); p; ++i, p = next(p))
	(*this)(i) *= s(p);

    return *this;
}

*/

EST_read_status EST_FVector::load(EST_String filename)
{    
    EST_TokenStream ts;
    EST_String s;
    int i;

    i = 0;
    
    if (((filename == "-") ? ts.open(cin) : ts.open(filename)) != 0)
    {
	cerr << "can't open vector input file " << filename << endl;
	return misc_read_error;
    }
    ts.set_SingleCharSymbols(";");

    while (!ts.eof())
    {
	ts.get();
	++i;
    }
    cout << "file has " << i << " points\n";
    resize(i);

    ts.close();
    if (((filename == "-") ? ts.open(cin) : ts.open(filename)) != 0)
    {
	cerr << "can't open vector input file " << filename << endl;
	return misc_read_error;
    }

    for (i = 0; !ts.eof(); ++i)
    {
	s = ts.get().string();
	(*this)(i) = atof(s);
    }

    cout << *this;

    ts.close();
    return format_ok;
}

EST_read_status EST_DVector::load(EST_String filename)
{    
    EST_TokenStream ts;
    EST_String s;
    int i;

    i = 0;
    
    if (((filename == "-") ? ts.open(cin) : ts.open(filename)) != 0)
    {
	cerr << "can't open vector input file " << filename << endl;
	return misc_read_error;
    }
    ts.set_SingleCharSymbols(";");

    while (!ts.eof())
    {
	ts.get();
	++i;
    }
    cout << "file has " << i << " points\n";
    resize(i);

    ts.close();
    if (((filename == "-") ? ts.open(cin) : ts.open(filename)) != 0)
    {
	cerr << "can't open vector input file " << filename << endl;
	return misc_read_error;
    }

    for (i = 0; !ts.eof(); ++i)
    {
	s = ts.get().string();
	(*this)(i) = atof(s); // actually returns double
    }

    cout << *this;

    ts.close();
    return format_ok;
}
    



