/*************************************************************************/
/*                                                                       */
/*                Centre for Speech Technology Research                  */
/*                     University of Edinburgh, UK                       */
/*                      Copyright (c) 1995,1996                          */
/*                        All Rights Reserved.                           */
/*                                                                       */
/*  Permission to use, copy, modify, distribute this software and its    */
/*  documentation for research, educational and individual use only, is  */
/*  hereby granted without fee, subject to the following conditions:     */
/*   1. The code must retain the above copyright notice, this list of    */
/*      conditions and the following disclaimer.                         */
/*   2. Any modifications must be clearly marked as such.                */
/*   3. Original authors' names are not deleted.                         */
/*  This software may not be used for commercial purposes without        */
/*  specific prior written permission from the authors.                  */
/*                                                                       */
/*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK        */
/*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING      */
/*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT   */
/*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE     */
/*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES    */
/*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN   */
/*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,          */
/*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF       */
/*  THIS SOFTWARE.                                                       */
/*                                                                       */
/*************************************************************************/
/*                    Author :  Simon King                               */
/*                    Date   :  January 1997                             */
/*-----------------------------------------------------------------------*/
/*                Decoder class for stack decoder                        */
/*                                                                       */
/*=======================================================================*/
#include <stdlib.h>
#include <string.h>
#include "WordDecoder.h"

WordDecoder::WordDecoder(Hypothesis *initial_hypothesis_ptr,
			 Track_Acoustic_Model &amodel,
			 Lattice_Language_Model &lmodel,
			 float lm_sf,
			 float am_sf,
			 int m)
{

    cerr << "WordDecoder ctor" << endl;
    
    make_heap(m);
    max_heap_size = m;
    acoustic_model = &amodel;
    language_model = &lmodel;

    lm_scale_factor = lm_sf;
    am_scale_factor = am_sf;
    
    if(initial_hypothesis_ptr == NULL){

	initial_hypothesis_ptr = new Hypothesis;
	
	initial_hypothesis_ptr->acoustic_model_state 
	    = (void*)((Track_Acoustic_Model*)acoustic_model)->start_state();

	initial_hypothesis_ptr->language_model_state 
	    = (void*)((Lattice_Language_Model*)language_model)->start_state();

	//initial_hypothesis_ptr->path = NULL;
    }

    Heap::insert_error_t e = heap->insert(initial_hypothesis_ptr);
    switch(e){
    case Heap::inserted_ok:
	break;
	
    case Heap::heap_full:
	delete_hypothesis(initial_hypothesis_ptr);
	cerr << "Failed to insert initial hypothesis" << endl;
	break;
	
    case Heap::prune_item:
	delete_hypothesis(initial_hypothesis_ptr);
	break;
	
    }

};


WordDecoder::~WordDecoder(){

    cerr << "WordDecoder dtor" << endl;
    delete heap;
    
};


void
WordDecoder::make_heap(int m)
{
    // a standard heap
    heap = new LUBHeap(m,1000); // heap of max size m, HACK 1000 !!!

}


EST_TList<Hypothesis*> 
WordDecoder::generate_new_hypotheses(Hypothesis *current_hypothesis)
{

    EST_TList<void*> new_lm_states;
    EST_TList<Hypothesis*> new_hypotheses;
    Hypothesis* new_hypothesis;

    new_lm_states 
	= ((Lattice_Language_Model*)language_model)->extend(current_hypothesis->language_model_state);

    // make new hypotheses
    EST_TBI *lptr;
    for(lptr=new_lm_states.head();lptr!=NULL;lptr=next(lptr)){

	new_hypothesis = new Hypothesis(); // hmmm
	new_hypothesis->language_model_state = new_lm_states(lptr);

	((Track_Acoustic_Model*)acoustic_model)->copy_state(current_hypothesis->acoustic_model_state,new_hypothesis->acoustic_model_state);

	new_hypothesis->h_length = current_hypothesis->h_length;
	set_log_prob(new_hypothesis);
	set_eval_fn(new_hypothesis);

	//cerr << "Made new hypothesis: ";
	//print(new_hypothesis);

	new_hypotheses.append(new_hypothesis);

    }

    new_lm_states.clear(); // doesn't delete actual states
    delete_hypothesis(current_hypothesis);
    return new_hypotheses;
}


bool
WordDecoder::extend(Hypothesis *h)
{
    // using observations

    int current_word;
    current_word = ((Lattice_Language_Model*)language_model)->get_current_word_index(h->language_model_state);
    
    if( !((Track_Acoustic_Model*)acoustic_model)->extend(h->acoustic_model_state,current_word) )
	return false;
    
    h->h_length = ((Track_Acoustic_Model*)acoustic_model)->get_frame(h->acoustic_model_state);

    return true;

}


void
WordDecoder::set_eval_fn(Hypothesis *h)
{
    h->h_eval_fn = (h->h_partial_log_prob + h->h_estimated_remaining_log_prob)
	/ h->h_length; // temporary - will be A* soon


}


void 
WordDecoder::set_partial_log_prob(Hypothesis *h)
{
    h->h_partial_log_prob = 
	((Lattice_Language_Model*)language_model)->get_log_prob(h->language_model_state)
	* lm_scale_factor
	
	+ ((Track_Acoustic_Model*)acoustic_model)->get_log_prob(h->acoustic_model_state)
	* am_scale_factor;
    
}


void 
WordDecoder::set_estimated_remaining_log_prob(Hypothesis *h)
{
    h->h_estimated_remaining_log_prob = 0.0;
}

bool
WordDecoder::is_finished(Hypothesis *h)
{
    return (bool)(lm_is_finished(h) && am_is_finished(h) );
}

bool
WordDecoder::lm_is_finished(Hypothesis *h)
{
    return ((Lattice_Language_Model*)language_model)->is_finished(h->language_model_state);
}


bool
WordDecoder::am_is_finished(Hypothesis *h)
{
    return ((Track_Acoustic_Model*)acoustic_model)->is_finished(h->acoustic_model_state);
}


void
WordDecoder::print(Hypothesis *h)
{
    
    cerr << "len= " << h->h_length
	 << " eval= " << h->h_eval_fn
	 << " logp=" << h->h_partial_log_prob
	 << "+" << h->h_estimated_remaining_log_prob
	 << " ";

    ((Lattice_Language_Model*)language_model)->print_state(h->language_model_state);
    
    cerr << " ";
    
    ((Track_Acoustic_Model*)acoustic_model)->print_state(h->acoustic_model_state);
    
    cerr << " word="
	 << ((Lattice_Language_Model*)language_model)->get_current_word(h->language_model_state)
	 << endl;
    


}

void 
WordDecoder::delete_hypothesis(Hypothesis *h)
{
    ((Track_Acoustic_Model*)acoustic_model)->delete_state(h->acoustic_model_state);
    ((Lattice_Language_Model*)language_model)->delete_state(h->language_model_state);
}


void 
WordDecoder::control_heap()
{

    // keep heap size somewhere near half the max size

#ifdef DEBUG
    ((LUBHeap*)heap)->print_stats();
#endif

    if(heap->size() > max_heap_size * 0.5){

	((LUBHeap*)heap)->set_envelope(1); // move elsewhere

	EST_TList<Heap_Item*> to_prune =
	    ((LUBHeap*)heap)->prune();
	
	for(EST_TBI *tp=to_prune.head();tp!=NULL;tp=next(tp))
	    delete_hypothesis( (Hypothesis*)to_prune(tp) );

	to_prune.clear();
    }



}
