/*

Name:
DISPLAY.C

Description:
Ncurses display section of mikmod - bare unix version.

Should deal with all the quiet options and refresh() after ncurses calls,
so just call these functions direct wherever needed. 

Steve McIntyre <stevem@chiark.greenend.org.uk>

HISTORY
=======

v1.00 (06/12/96) - first "versioned" version
v1.01 (06/12/96) - fixed update_name_display for Solaris - add "(null)" if necessary
v1.02 (16/12/96) - minor cosmetic change to display_error()

*/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include "mikmod.h"

#ifdef NCURSES
#include <ncurses/ncurses.h>
#else
#include <curses.h>
#endif

extern curmod cur_mod;
extern int quiet;
extern float speed_constant;   /* multiplier for tempo,initialised to 1 */

void init_display(void)
{
	if(quiet) return;
	initscr(); 
	cbreak(); 
	noecho(); 
	nonl(); 
	nodelay(stdscr, TRUE);
	keypad(stdscr, TRUE); 
	strcpy(cur_mod.version,mikversion);
}

void display_version(void)
{
	if(quiet) return;

	/* always use display_verion first, so clear call is OK here... */
	clear(); 

	printw("%s",cur_mod.version);
	refresh();
}	

void display_driver(void)
{
	if(quiet) return;
	printw("%s",cur_mod.driver);
	refresh();
}	

void display_file(void)
{
	if(quiet) return;
	printw("%s",cur_mod.file_output);
	refresh();
}	

void display_name(void)
{
	if(quiet) return;
	printw("%s",cur_mod.name_type);
	refresh();
}	

void display_status(void)
{
	if(quiet) return;
	printw("%s",cur_mod.status);
	refresh();
}	

void display_pausebanner(void)
{
	if(quiet) return;
	addstr(pausebanner);
	refresh();
}	

void display_extractbanner(void)
{
	if(quiet) return;
	addstr(extractbanner);
	refresh();
}	

void display_loadbanner(void)
{
	if(quiet) return;
	addstr(loadbanner);
	refresh();
}	

void display_error(char *myerr, char *myerr_file)
{
	if(quiet) return;
	printw("Non-fatal Error:\n %s: \"%s\".\n",myerr,myerr_file);
	refresh();
	sleep(3);
}	

void display_driver_error(char *myerr)
{
	if(quiet) return;
	printw("Driver error: %s.\n",myerr);
	refresh();
}	

void display_all(void)
{
	if(quiet) return;
	display_version();
	display_driver();
	display_file();
	display_name();
	display_status();
}	

void update_driver_display(void)
{
	if(quiet) return;
	sprintf(cur_mod.driver,"%s: %d bit %s %s, %u Hz\n",
		md_driver->Name,
		(md_mode&DMODE_16BITS) ? 16:8,
		(md_mode&DMODE_INTERP) ? "interpolated":"normal",
		(md_mode&DMODE_STEREO) ? "stereo":"mono",
		md_mixfreq);
}

void update_file_display(void)
{
	if(quiet) return;
	sprintf(cur_mod.file_output,"                                                                                                                                                     ");
	sprintf(cur_mod.file_output,"File: %s%s\n",cur_mod.filename,d_text[cur_mod.deleted]); 
}

void update_name_display()
{
	if(quiet) return;
	sprintf(cur_mod.name_type,"Name: %s\n"
		"Type: %s, "
		"Periods: %s,%s\n",
		cur_mod.songname==NULL?"(null)":cur_mod.songname,
		cur_mod.modtype,
		(cur_mod.flags&UF_XMPERIODS) ? "XM type" : "mod type",
		(cur_mod.flags&UF_LINEAR) ? "Linear" : "Log");
}

void update_status_display()
{
	if(quiet) return;
	sprintf(cur_mod.status,"\rpat:%2d/%d pos:%2d spd:%d bpm:%3.0f%% vol:%d%% ",cur_mod.sngpos,cur_mod.numpat,cur_mod.patpos,mp_sngspd,speed_constant*100,mp_volume);
}

void exit_display()
{
	if(quiet) return;
	endwin();
}
