 /************************************************************************/
 /*                                                                      */
 /*                Centre for Speech Technology Research                 */
 /*                     University of Edinburgh, UK                      */
 /*                       Copyright (c) 1996,1997                        */
 /*                        All Rights Reserved.                          */
 /*                                                                      */
 /*  Permission to use, copy, modify, distribute this software and its   */
 /*  documentation for research, educational and individual use only, is */
 /*  hereby granted without fee, subject to the following conditions:    */
 /*   1. The code must retain the above copyright notice, this list of   */
 /*      conditions and the following disclaimer.                        */
 /*   2. Any modifications must be clearly marked as such.               */
 /*   3. Original authors' names are not deleted.                        */
 /*  This software may not be used for commercial purposes without       */
 /*  specific prior written permission from the authors.                 */
 /*                                                                      */
 /*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK       */
 /*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING     */
 /*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT  */
 /*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE    */
 /*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES   */
 /*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN  */
 /*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,         */
 /*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF      */
 /*  THIS SOFTWARE.                                                      */
 /*                                                                      */
 /************************************************************************/
 /*                                                                      */
 /*                 Author: Richard Caley (rjc@cstr.ed.ac.uk)            */
 /*                   Date: Mon Jun 30 1997                              */
 /* -------------------------------------------------------------------- */
 /* Trivial join module for things already cut to size.                  */
 /*                                                                      */
 /************************************************************************/

#include "festival.h"
#include "Unit.h"
#include "FramesJoin.h"
#include "../WaveSynthesis/FramesUnit.h"
#include "cut_unit_join.h"
#include "module_support.h"

static ModuleDescription description =
{
  "cut_unit_join", 1.0,
  "CSTR",
  "Richard Caley <rjc@cstr.ed.ac.uk>",
  {
    "Trivially join pre-cut units by abutting them.",
    NULL
  },
  {
    { "Segment",	"Contains the segments." },
    { "Unit",		"The units which are to be joined." },
    { NULL,NULL}
  },
  { {NULL,NULL}
  },
  {
    { "Join",		"Final stream of joins." },
    {NULL,NULL}
  },
  {
    { "JoinVerbose",	mpt_bool,	"nil",	"Print trace of Joining process." },
    {NULL,NULL,NULL,NULL}
  }
};

LISP cut_unit_join(LISP args)
{
  EST_Utterance *utt;

  EST_String segment_stream_name("Segment");
  EST_String    unit_stream_name("Unit");
  EST_String    join_stream_name("Join");

  EST_Stream *segment_stream=NULL, *unit_stream=NULL, *join_stream=NULL;

  unpack_module_args(args, 
		     utt, 
		     segment_stream_name, segment_stream, sat_existing,
		     unit_stream_name, unit_stream, sat_existing,
		     join_stream_name, join_stream, sat_replace);

  bool verbose = bool_parameter_get("JoinVerbose");

  EST_Stream_Item *unitem, *last;
  int last_endi=0;

  for(last=NULL,unitem=unit_stream->head(); unitem; unitem=next(unitem))
    {
      EST_Relation *links = &(unitem->rlink(segment_stream_name));
      int starti = links->item(0);
      int endi = links->item(1);
      if (last && starti <= last_endi)
	{
	  if (verbose)
	    cout << "overlap " << last_endi << " " << starti << " " << endi << "\n";

	  Unit *left_unit = (Unit *)last->contents();
	  Unit *right_unit = (Unit *)unitem->contents();
	  

	  if (verbose)
	    cout << "    units " << left_unit->name() << " " << right_unit->name() << "\n";

	  if (!left_unit->property("frames_subtype") ||
	      !left_unit->property("pre_cut"))
	    {
	      cwarn << "bad unit type " << left_unit->subtype_name() <<"\n";
	      continue;
	    }
	  if (!right_unit->property("frames_subtype") ||
	      !right_unit->property("pre_cut"))
	    {
	      cwarn << "bad unit type " << right_unit->subtype_name() <<"\n";
	      continue;
	    }

	  EST_Track *left_pm = ((FramesUnit *)left_unit)->pitchmarks();
	  EST_Track *right_pm = ((FramesUnit *)right_unit)->pitchmarks();
	  (void)right_pm;

	  FramesJoin *join = new FramesJoin(left_unit, left_pm->num_frames()-1,
					    right_unit, 0);

	  EST_Stream_Item item;
	  item.init("Join");
	  item.set_name(segment_stream->item(starti)->name());
	  item.set_contents((Join *)join, Unit::decrease_reference_count); 
	  ((Join *)join)->gc_ref();
	  link(item, *last);
	  link(item, *unitem);
	  item.set_feature("type", join->subtype_name());
	  item.set_feature("end_index", join->end_index());
	  item.set_feature("start_index", join->start_index());
	  join_stream->append(item);
	}

      last = unitem;
      last_endi = endi;
    }

  return NIL;
}

void cut_unit_join_init(void)

{
  proclaim_module("cut_unit_join", &description);

  init_module_subr("cut_unit_join", cut_unit_join, &description);
}

