/*************************************************************************/
/*                                                                       */
/*                Centre for Speech Technology Research                  */
/*                     University of Edinburgh, UK                       */
/*                       Copyright (c) 1996,1997                         */
/*                        All Rights Reserved.                           */
/*                                                                       */
/*  Permission to use, copy, modify, distribute this software and its    */
/*  documentation for research, educational and individual use only, is  */
/*  hereby granted without fee, subject to the following conditions:     */
/*   1. The code must retain the above copyright notice, this list of    */
/*      conditions and the following disclaimer.                         */
/*   2. Any modifications must be clearly marked as such.                */
/*   3. Original authors' names are not deleted.                         */
/*  This software may not be used for commercial purposes without        */
/*  specific prior written permission from the authors.                  */
/*                                                                       */
/*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK        */
/*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING      */
/*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT   */
/*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE     */
/*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES    */
/*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN   */
/*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,          */
/*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF       */
/*  THIS SOFTWARE.                                                       */
/*                                                                       */
/*************************************************************************/
/*                     Author :  Alan W Black                            */
/*                     Date   :  April 1996                              */
/*-----------------------------------------------------------------------*/
/*                                                                       */
/*               EST_Utterance access functions (from Lisp)              */
/*                                                                       */
/*=======================================================================*/
#include <stdio.h>
#include "EST_unix.h"
#include "festival.h"
#include "festivalP.h"

EST_Utterance *GETUTTVAL(LISP x)
{
    if (TYPEP(x,tc_utt))
	return (EST_Utterance *)UTTVAL(x);
    else
    {
	cerr << "SIOD: wta to GETUTTVAL" << endl;
	festival_error();
    }
    return NULL;
}

EST_Stream_Item *GETSTREAMITEMVAL(LISP x)
{
    if (TYPEP(x,tc_streamitem))
	return (EST_Stream_Item *)STREAMITEMVAL(x);
    else
    {
	cerr << "SIOD: wta to GETSTREAMVAL" << endl;
	festival_error();
    }
    return NULL;
}

LISP utt_iform(EST_Utterance &utt)
{
    return read_from_string(strintern(utt_iform_string(utt)));
}

const EST_String utt_iform_string(EST_Utterance &utt)
{
    if (utt.stream_present("IForm"))
	return utt.stream("IForm").head()->feature("iform").string();
    else
	return "0";
}

const EST_String utt_type(EST_Utterance &utt)
{
    if (utt.stream_present("IForm"))
	return utt.stream("IForm").head()->feature("type").string();
    else
	return "0";
}

static LISP utt_save(LISP utt, LISP fname)
{
    EST_Utterance *u = GETUTTVAL(utt);
    EST_String filename = get_c_string(fname);
    if (fname == NIL)
	filename = "save.utt";

    u->save(filename,2);

    return utt;
}

static LISP utt_load(LISP utt, LISP fname)
{
    EST_Utterance *u;
    if (utt == NIL)
	u = new EST_Utterance;
    else
	u = GETUTTVAL(utt);
    EST_String filename = get_c_string(fname);

    if (u->load(filename) != 0)
    {
	cerr << "utt.load: loading from \"" << filename << "\" failed" <<
	    endl;
	festival_error();
    }

    if (utt == NIL)
	return siod_make_utt(*u);
    else
	return utt;
}

static LISP utt_stream(LISP utt, LISP sname)
{
    EST_Utterance *u = GETUTTVAL(utt);
    EST_String streamname = get_c_string(sname);
    LISP streams = NIL;
    EST_Stream_Item *s;

    if (u->stream_present(streamname))
	for (s=u->stream(streamname).tail(); s != 0; s=prev(s))
	    streams = cons(siod_make_streamitem((void *)s),streams);
	
    return streams;
}

static LISP utt_stream_head(LISP utt, LISP sname)
{
    EST_Utterance *u = GETUTTVAL(utt);
    EST_String streamname = get_c_string(sname);

    if (u->stream_present(streamname))
	return siod_make_streamitem((void *)u->stream(streamname).head());
    else
	return NIL;
}

static LISP utt_stream_tail(LISP utt, LISP sname)
{
    EST_Utterance *u = GETUTTVAL(utt);
    EST_String streamname = get_c_string(sname);

    if (u->stream_present(streamname))
	return siod_make_streamitem((void *)u->stream(streamname).tail());
    else
	return NIL;
}

static LISP utt_streamnames(LISP utt)
{
    // Return list of stream names
    EST_Utterance *u = GETUTTVAL(utt);
    LISP streamnames = NIL;
    EST_TBI *p;

    for (p = u->s.head(); p; p = next(p))
	streamnames = cons(rintern(u->s(p).stream_name()),streamnames);

    return reverse(streamnames);
}

static LISP utt_streamitem_rel(LISP utt, LISP sitem, LISP relname)
{
    // List of all streamitems related to sitem by name relname
    EST_Utterance *u = GETUTTVAL(utt);
    EST_Stream_Item *s = GETSTREAMITEMVAL(sitem);
    LISP l_rels = NIL;
    EST_Relation *s_rels = s->link(get_c_string(relname));
    EST_TBI *p;

    for (p=s_rels->head(); p != 0; p=next(p))
    {
	const EST_Stream_Item &r = u->ritem(get_c_string(relname),(*s_rels)(p));
	l_rels = cons(siod_make_streamitem((void *)&r),l_rels);
    }

    return reverse(l_rels);
}

static LISP streamitem_next(LISP sitem)
{
    // Returns next stream_item or NIL if end
    EST_Stream_Item *s = GETSTREAMITEMVAL(sitem);
    EST_Stream_Item *n = next(s);
    
    if (n == 0)
	return NIL;
    else
	return siod_make_streamitem(n);
}

static LISP streamitem_prev(LISP sitem)
{
    // Returns next stream_item or NIL if end
    EST_Stream_Item *s = GETSTREAMITEMVAL(sitem);
    EST_Stream_Item *p = prev(s);
    
    if (p == 0)
	return NIL;
    else
	return siod_make_streamitem(p);
}

static LISP set_streamitem_name(LISP sitem, LISP newname)
{
    // Set a stream's name to newname
    EST_Stream_Item *s = GETSTREAMITEMVAL(sitem);
    
    if (s != 0)
	s->set_name(get_c_string(newname));
    return sitem;
}

static LISP set_streamitem_end(LISP sitem, LISP newend)
{
    // Set a stream's name to newname
    EST_Stream_Item *s = GETSTREAMITEMVAL(sitem);
    
    if (s != 0)
	s->set_end(get_c_float(newend));
    return sitem;
}

static LISP utt_streamitem_delete(LISP utt, LISP sitem)
{
    // Delete named stream item
    EST_Utterance *u = GETUTTVAL(utt);
    EST_Stream_Item *s = GETSTREAMITEMVAL(sitem);

    u->remove_item(s->stream_name(),s->addr());
    return utt;
}

static LISP utt_stream_delete(LISP utt, LISP name)
{
    // Delete named stream ite
    EST_Utterance *u = GETUTTVAL(utt);

    if (u->stream_present(get_c_string(name)))
	u->remove_stream(get_c_string(name));
    else
    {
	cerr << "utt.stream.delete: no stream named " << get_c_string(name)
	    << endl;
	festival_error();
    }

    return utt;
}

static LISP utt_streamitem_insert(LISP utt, LISP streamname, LISP item)
{
    // Add a new stream item
    EST_Utterance *u = GETUTTVAL(utt);
    EST_Stream_Item sitem;
    
    sitem.init(get_c_string(streamname));

    if (!u->stream_present(get_c_string(streamname)))
	u->create_stream(get_c_string(streamname));
    
    if (item == NIL)
	u->stream(get_c_string(streamname)).append(sitem);
    else
    {
	EST_Stream_Item *is = GETSTREAMITEMVAL(item);
	u->stream(get_c_string(streamname)).insert_after(is,sitem);
    }

    return utt;
}

static LISP utt_relate_items(LISP utt, LISP item1, LISP item2)
{
    // Link two stream items
    EST_Stream_Item *s1 = GETSTREAMITEMVAL(item1);
    EST_Stream_Item *s2 = GETSTREAMITEMVAL(item2);

    link(*s1,*s2);

    return utt;
}

static LISP utt_stream_create(LISP utt, LISP name)
{
    // Create new stream
    EST_Utterance *u = GETUTTVAL(utt);

    if (u->stream_present(get_c_string(name)))
    {
	cerr << "utt.stream.create: already a stream named " 
	    << get_c_string(name) << endl;
	festival_error();
    }
    else
	u->create_stream(get_c_string(name));

    return utt;
}

LISP siod_make_utt(EST_Utterance &u)
{
    u.add_ref();
    return siod_make_utt_cell((void *)&u);
}

void utt_free(LISP utt)
{
    EST_Utterance *u = GETUTTVAL(utt);

    if ((u != 0) &&
	(u->del_ref() == 0))
	delete u;

    UTTVAL(utt) = NULL;
}

LISP utt_rel(LISP utt)
{
    // relocate
    cerr << "EST_Utterance relocate\n";
    return utt;
}

LISP utt_mark(LISP utt)
{
    // Should mark all the LISP cells in it 
    // but at present we use the gc_(un)protect mechanism 
    return utt;
}

void utt_scan(LISP utt)
{
    (void)utt;
    cerr << "EST_Utterance scan (sweep?)\n";
    return;
}

void utt_cleanup(EST_Utterance &u)
{
    // Remove all streams in u except IForm 
    // This is called in the Initialization to ensure we can 
    // continue with a nice clean utterance
    EST_TBI *p;

    for (p = u.s.head(); p; p = next(p))
	if (u.s(p).stream_name() != "IForm")
	{
	    u.s(p).clear();
	    u.s.remove(p);
	}

}

LISP make_utterance(LISP args,LISP env)
{
    /* Make an utterance structure from given input */
    (void)env;
    EST_Utterance *u = new EST_Utterance;
    static int a = 0;
    EST_Stream_Item item, *iitem;
    EST_String t;
    LISP utt,lform;

    u->create_stream("IForm");  // build initial stream 

    item.init("IForm");
    item.set_addr(a);
    a++;
    u->stream("IForm").append(item);
    iitem = u->stream("IForm").tail();

    lform = car(cdr(args));
    t = get_c_string(car(args));
    iitem->set_feature("type",t);
    iitem->set_feature("iform",siod_sprint(lform));

    utt = siod_make_utt(*u);

    return utt;
}

static LISP streamitem_name(LISP sitem)
{
    EST_Stream_Item *s = GETSTREAMITEMVAL(sitem);

    return rintern(s->name());
}

static LISP streamitem_features(LISP sitem)
{
    // Return assoc list of features on this stream
    EST_Stream_Item *s = GETSTREAMITEMVAL(sitem);
    LISP features = NIL;
    EST_TBI *p;

    for (p=s->features.list.head(); p != 0; p=next(p))
    {
	const EST_Val &v = s->features.list.item(p).v;
	LISP fpair;
	
	if (v.type() == val_int)
	    fpair = make_param_int(s->features.list.item(p).k, v.Int());
	else if (v.type() == val_float)
	    fpair = make_param_float(s->features.list.item(p).k, v.Float());
	else
	    fpair = make_param_lisp(s->features.list.item(p).k, 
				    strintern(v.string()));
	features = cons(fpair,features);
    }

    return reverse(features);
}

static LISP streamitem_end(LISP sitem)
{
    EST_Stream_Item *s = GETSTREAMITEMVAL(sitem);

    return flocons(s->end());
}

void add_item_features(EST_Stream_Item &s,LISP features)
{
    // Add LISP speficied features to s;
    LISP f;

    for (f=features; f != NIL; f=cdr(f))
	if (streq("name",get_c_string(car(car(f)))))
	    s.set_name(get_c_string(car(cdr(car(f)))));
	else
	    s.set_feature(get_c_string(car(car(f))),
			  get_c_string(car(cdr(car(f)))));
	
}

void utterance_lisp_declare(void)
{
    // declare utterance specific Lisp functions 
    long kind;

    set_gc_hooks(tc_utt, 0, utt_rel,utt_mark,utt_scan,utt_free,NULL,&kind);
    // Standard functions
    init_fsubr("Utterance",make_utterance,
 "(Utterance TYPE DATA)\n\
  Build an utterance of type TYPE from DATA.  Different TYPEs require\n\
  different types of data.  New types may be defined by defUttType.\n\
  [see Utterance types]");
    init_subr_2("utt.load",utt_load,
 "(utt.load UTT FILENAME)\n\
  Loads UTT with the streams and stream items described in FILENAME.\n\
  The format is Xlabel-like as saved by utt.save.  If UTT is nil a new\n\
  utterance is created, loaded and returned.  If FILENAME is \"-\"\n\
  the data is read from stdin.");
    init_subr_2("utt.save",utt_save,
 "(utt.save UTT FILENAME)\n\
  Save UTT in FILENAME in an Xlabel-like format.  If FILENAME is \"-\"\n\
  then print output to stdout.");
    init_subr_2("utt.stream",utt_stream,
 "(utt.stream UTT STREAMNAME)\n\
  Return a list of stream items in STREAMNAME in UTT. \n\
  [see Accessing an utterance]");
    init_subr_2("utt.stream.head",utt_stream_head,
 "(utt.stream.head UTT STREAMNAME)\n\
  Return first streamitem in STREAMNAME in UTT, or nil if empty. \n\
  [see Accessing an utterance]");
    init_subr_2("utt.stream.tail",utt_stream_tail,
 "(utt.stream.tail UTT STREAMNAME)\n\
  Return last streamitem in STREAMNAME in UTT, or nil if empty. \n\
  [see Accessing an utterance]");
    init_subr_3("utt.streamitem.rel",utt_streamitem_rel,
 "(utt.streamitem.rel UTT STREAMITEM RELNAME)\n\
  Return a list of streams related to STREAMITEM in UTT by RELNAME.\n\
  [see Accessing an utterance]");
    init_subr_1("streamitem.name",streamitem_name,
 "(streamitem.name STREAMITEM)\n\
  Returns the stream name of STREAMITEM. [see Accessing an utterance]");
    init_subr_1("streamitem.end",streamitem_end,
 "(streamitem.end STREAMITEM)\n\
  Returns STREAMITEM's end time. [see Accessing an utterance]");
    init_subr_1("streamitem.next",streamitem_next,
 "(streamitem.next STREAMITEM)\n\
  Returns the next stream item from STREAMITEM or nil STREAMITEM is last\n\
  in the stream. [see Accessing an utterance]");
    init_subr_1("streamitem.prev",streamitem_prev,
 "(streamitem.prev STREAMITEM)\n\
  Returns the previous stream item from STREAMITEM or nil STREAMITEM is \n\
  first in the stream. [see Accessing an utterance]");
    init_subr_2("streamitem.set_name",set_streamitem_name,
 "(streamitem.set_name STREAMITEM NAME)\n\
  Sets STREAMITEM's name to NAME. [see Accessing an utterance]");
    init_subr_2("streamitem.set_end",set_streamitem_end,
 "(streamitem.set_end STREAMITEM ENDTIME)\n\
  Sets STREAMITEM's end to ENDTIME.  Great care should be taken with this\n\
  function to avoid the end times in a stream from becoming unordered.\n\
  [see Accessing an utterance]");
    init_subr_1("streamitem.features",streamitem_features,
 "(streamitem.features STREAMITEM)\n\
  Returns all features in STREAMITEM as an assoc list.");
    init_subr_2("utt.streamitem.delete",utt_streamitem_delete,
 "(utt.streamitem.delete UTT STREAMITEM)\n\
  Delete STREAMITEM from UTT.  [see Accessing an utterance]");
    init_subr_3("utt.streamitem.insert",utt_streamitem_insert,
 "(utt.streamitem.insert UTT STREAMNAME STREAMITEM)\n\
  Insert a new stream item in stream STREAMNAME in UTT, if STREAMITEM is\n\
  non-nil insert it after STREAMITEM, if STREAMITEM is nil or unspecified,\n\
  simply append it to the stream.  This will create the stream if it \n\
  doesn't currently exist.  [see Accessing an utterance]");
    init_subr_3("utt.relate_items",utt_relate_items,
 "(utt.relate_items UTT STREAMITEM1 STREAMITEM2)\n\
  Add relation between STREAMITEM1 and STREAMITEM2 in UTT.\n\
  [see Accessing an utterance]");
    init_subr_2("utt.stream.delete",utt_stream_delete,
 "(utt.stream.delete UTT STREAMNAME)\n\
  Delete stream called STREAMNAME from UTT.  This causes an error if no\n\
  stream call STREAMNAME exists. [see Accessing an utterance]");
    init_subr_2("utt.stream.create",utt_stream_create,
 "(utt.stream.create UTT STREAMNAME)\n\
  Create stream call STREAMNAME, causes an error if a stream of that\n\
  name already exists.  [see Accessing an utterance]");
    init_subr_1("utt.streamnames",utt_streamnames,
 "(utt.streamnames UTT)\n\
  Returns a list of the names of all streams in UTT.\n\
  [see Accessing an utterance]");
    init_subr_1("audio_mode",l_audio_mode,
 "(audio_mode MODE)\n\
 Control audio specific modes.  Five subcommands are supported. If\n\
 MODE is async, start the audio spooler so that Festival need not wait\n\
 for a waveform to complete playing before continuing.  If MODE is\n\
 sync wait for the audio spooler to empty, if running, and they cause\n\
 future plays to wait for the playing to complete before continuing.\n\
 Other MODEs are, close which waits for the audio spooler to finish\n\
 any waveforms in the queue and then closes the spooler (it will restart\n\
 on the next play), shutup, stops the current waveform playing and empties\n\
 the queue, and query which lists the files in the queue.  The queue may\n\
 be up to five waveforms long. [see Audio output]");

}
