/*************************************************************************/
/*                                                                       */
/*                Centre for Speech Technology Research                  */
/*                     University of Edinburgh, UK                       */
/*                       Copyright (c) 1996,1997                         */
/*                        All Rights Reserved.                           */
/*                                                                       */
/*  Permission to use, copy, modify, distribute this software and its    */
/*  documentation for research, educational and individual use only, is  */
/*  hereby granted without fee, subject to the following conditions:     */
/*   1. The code must retain the above copyright notice, this list of    */
/*      conditions and the following disclaimer.                         */
/*   2. Any modifications must be clearly marked as such.                */
/*   3. Original authors' names are not deleted.                         */
/*  This software may not be used for commercial purposes without        */
/*  specific prior written permission from the authors.                  */
/*                                                                       */
/*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK        */
/*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING      */
/*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT   */
/*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE     */
/*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES    */
/*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN   */
/*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,          */
/*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF       */
/*  THIS SOFTWARE.                                                       */
/*                                                                       */
/*************************************************************************/
/*                     Author :  Alan W Black                            */
/*                     Date   :  April 1996                              */
/*-----------------------------------------------------------------------*/
/*                                                                       */
/* Functions for accessing parts of the utterance through "features"     */
/* features are a cononical adressing method for relatively accessing    */
/* information in an utterance from a given stream.                      */
/*                                                                       */
/* A feature name is is a dotted separated string of names ended by a    */
/* a final feature.  The names before the final feature are navigational */
/* consiting of a few builtin forms (i.e. next and previous) or names of */
/* others streams taht are to be followed to.                            */
/*                                                                       */
/* For example:                                                          */
/*   "name"          is the name of the current stream item              */
/*   "n.name"        is the name of the next stream item                 */
/*   "n.n.name"      is the name of the next next stream item            */
/*   "Word.duration" is the duration of the word related to this item    */
/*   "p.Syllable.syl_break"                                              */
/*                   is the syllable break level of the syllable related */
/*                   the previous item.                                  */
/*   "Syllable.p.syl_break"                                              */
/*                   is the syllable break level of the syllable related */
/*                   this item's previous Syllable item.                 */
/*                                                                       */
/* Modifiers for relation names are identified by a colon                */
/*    "Syllable:num"   number of syllables related to this item          */
/*    "Segment:first"  first segment related to this item (default)      */
/*    "Segment:last"   last segment related to this item (default)       */
/*                                                                       */
/* The following features are defined for all stream items               */
/*     name start end duration addr                                      */
/* Other feature are defined through the C++ function festival_def_ff    */
/*                                                                       */
/* To extre features are defined here ph_* (i.. any feature prefixed by  */
/* ph_) and lisp_*.  ph_ function check the phoneme set anduse the       */
/* remainder of the name as a phone feature return its value for the     */
/* current item (which will normally be a Segment stream item).  e.g.    */
/* ph_vc or ph_vheight                                                   */
/* The lisp_* feature will call a lisp function (the name following      */
/* the lisp_) with two arguments the utterance and the stream item.      */
/* this allows arbitrary new features without recompilation              */
/*                                                                       */
/*=======================================================================*/
#include <stdio.h>
#include <stdlib.h>
#include "EST_unix.h"
#include <string.h>
#include "festival.h"
#include "festivalP.h"

static LISP ff_pref_assoc(const char *name, LISP alist);

// Really wanted a KVL to do this but it doesn't work 
static LISP ff_docstrings = NULL;
static EST_StringTrie ff_list;
static LISP ff_pref_list = NULL;

LISP lisp_val(const EST_Val &pv)
{
    if (pv.type() == val_unset)
    {
	cerr << "EST_PVal unse, can't build lisp value" << endl;
	festival_error();
	return NIL;
    }
    else if (pv.type() == val_int)
	return flocons(pv.Int());
    else if (pv.type() == val_float)
	return flocons(pv.Float());
    else if (pv.type() == val_string)
	return strintern(pv.string_only());
    else
    {
	cerr << "EST_PVal has unknown type, can't build lisp value" << endl;
	festival_error();
	return NIL;
    }
}

static EST_Val ff_name(EST_Utterance &u,EST_Stream_Item &i)
{
    (void)u;
    return EST_Val(i.name());
}

static EST_Val ff_start(EST_Utterance &u,EST_Stream_Item &i)
{
    (void)u;
    return EST_Val(i.start());
}

static EST_Val ff_end(EST_Utterance &u,EST_Stream_Item &i)
{
    (void)u;
    return EST_Val(i.end());
}

static EST_Val ff_dur(EST_Utterance &u,EST_Stream_Item &i)
{
    (void)u;
    return EST_Val(i.dur());
}

static EST_Val ff_addr(EST_Utterance &u,EST_Stream_Item &i)
{
    (void)u;
    return EST_Val(i.addr());
}

static EST_Val ff_lisp_func(EST_Utterance &u,EST_Stream_Item &i,
			    const EST_String &name)
{
    // This function is called for features functions startsing lisp_
    // It calls the lisp function following that with u and i
    // as arguments, the return value (which must be atomic) is
    // then passed back as a Val.  I'm not sure if this will be 
    // particularly efficient, but it will make development of
    // new features quicker as they can be done in Lisp without
    // changing the C++ code.
    EST_String lfunc_name = name.after("lisp_");
    LISP r,l;

    l = cons(rintern(lfunc_name),
	     cons(siod_make_utt(u),
		  cons(siod_make_streamitem(&i),NIL)));
    r = leval(l,NIL);
    if ((consp(r)) || (r == NIL))
    {
	cerr << "FFeature Lisp function: " << lfunc_name << 
	    " retruned non-atomic value" << endl;
	festival_error();
    }
    else if (numberp(r))
	return EST_Val(FLONM(r));
	return EST_Val(get_c_string(r));
}

void festival_def_ff(const EST_String &name,const EST_String &sname, 
		     FT_ff_func func,char *doc)
{
    //  define the given feature function with documentation
    (void)sname;

    if (ff_list.lookup(name) == 0)
    {
	ff_list.add(name,(void *)func);
	if (ff_docstrings == NIL)
	    gc_protect(&ff_docstrings);
	EST_String id = sname + "." + name;
	ff_docstrings = cons(cons(rintern(id),cstrcons(doc)),ff_docstrings);
	siod_set_lval("ff_docstrings",ff_docstrings);
    }
    else
    {
	cerr << "ffeature " << name << " duplicate definition" << endl;
	festival_error();
	
    }

    return;
}

void festival_def_ff_pref(const EST_String &pref,const EST_String &sname, 
			  FT_ff_pref_func func, char *doc)
{
    // define the given class of feature functions
    // All feature functions names with this prefix will go to this func
    LISP lpair;
    (void)sname;

    lpair = siod_assoc_str(pref,ff_pref_list);
    
    if (lpair == NIL)
    {
	if (ff_pref_list == NIL)
	    gc_protect(&ff_pref_list);
	ff_pref_list = cons(cons(rintern(pref),cons(MKPTR(func),NIL)),
		       ff_pref_list);
	EST_String id = sname + "." + pref;
	ff_docstrings = cons(cons(rintern(id),cstrcons(doc)),ff_docstrings);
	siod_set_lval("ff_docstrings",ff_docstrings);
    }
    else
    {
	cerr << "ffeature (prefix)" << pref << " duplicate definition" << endl;
	festival_error();
    }

    return;
}

static LISP ff_pref_assoc(const char *name, LISP alist)
{
    // Search list of ff_pref_funcs to see if name has an appropriate
    // prefix 
    LISP l;
    char *prefix;
    
    for (l=alist; CONSP(l); l=CDR(l))
    {
	prefix = get_c_string(car(CAR(l)));
	if (strstr(name,prefix) == name)
	    return CAR(l);
    }

    // not found
    return NIL;
}

static EST_String Feature_Separator = ".";
static EST_String Feature_SingleCharSymbol = ":";
static EST_String Feature_PunctuationSymbols = "";
static EST_String Feature_PrePunctuationSymbols = "";

static int ffeat_one_name(const char* name)
{
    // returns TRUE is name doesn't contain . or :
    int i;

    for (i=0; name[i] != '\0'; i++)
	if ((name[i]=='.') || (name[i]==':'))
	    return FALSE;
    return TRUE;
}

EST_Val ffeature(EST_Utterance &u,EST_Stream_Item &item,const EST_String &fname)
{
    // Select and apply feature function name to s and return result 
    FT_ff_pref_func pfunc;
    FT_ff_func func;
    LISP lpair;
    EST_TokenStream ts;
    EST_Stream_Item *s = &item;
    const char *modifier;

    if (ffeat_one_name(fname))
    {   // if its a simple name do it quickly
	if ((func = (FT_ff_func)ff_list.lookup(fname)) != 0)
	    return (func)(u,item);
	else if ((lpair = ff_pref_assoc(fname,ff_pref_list)) != NIL)
	{
	    pfunc = (FT_ff_pref_func)PTRVAL(CAR(CDR(lpair)));
	    return (pfunc)(u,item,fname);
	}
	else // it must be a feature name for this stream item
	    return EST_Val(item.feature(fname));
    }
    ts.open_string(fname);
    ts.set_WhiteSpaceChars(Feature_Separator);
    ts.set_SingleCharSymbols(Feature_SingleCharSymbol);
    ts.set_PunctuationSymbols(Feature_PunctuationSymbols);
    ts.set_PrePunctuationSymbols(Feature_PrePunctuationSymbols);

    while (!ts.eof())
    {
	const EST_String &Sname = ts.get().string();
	const char *name = Sname;
	if (streq(name,"n"))
	    s=next(s);
	else if (streq(name,"p"))
	    s=prev(s);
	else if (streq(name,"nn"))
	    s=next(next(s));
	else if (streq(name,"pp"))
	    s=prev(prev(s));
	else if ((func = (FT_ff_func)ff_list.lookup(Sname)) != 0)
	    return (func)(u,*s);
	else if ((lpair = ff_pref_assoc(name,ff_pref_list)) != NIL)
	{
	    pfunc = (FT_ff_pref_func)PTRVAL(CAR(CDR(lpair)));
	    return (pfunc)(u,*s,Sname);
	}
	else if (u.stream_present(Sname))
	{
	    EST_Relation *rels = s->link(Sname);
	    if (rels->head() == 0) /* aren't any relations */
		return EST_Val(0);
	    EST_String CSname = Sname; // copy it
	    if (ts.peek() == ":") // relation modified
	    {
		ts.get(); 
		modifier = ts.get().string();
		if (streq(modifier,"last"))
		    s = &u.ritem(CSname,(*rels)(rels->tail()));
		else if (streq(modifier,"num"))
		    return EST_Val(rels->length());
		else if (streq(modifier,"first"))
		    s = &u.ritem(CSname,(*rels)(rels->head()));
		else
		{
		    cerr << "ffeature: invalid relation modifier " <<
			modifier << endl;
		    festival_error();
		}
	    }
	    else		// treat it like "first"
		s = &u.ritem(CSname,(*rels)(rels->head()));
	}
	else // it must be a feature name for this stream item
	    return EST_Val(s->feature(name));

	if (s==0)
	    return EST_Val(0);
    }

    cerr << "Invalid ffeature name: \"" << fname << "\"" << endl;
    festival_error();

    return EST_Val(0);
}

static LISP lisp_streamitem_feat(LISP utt,LISP sitem, LISP name)
{
    // return the ffeature name for this stream 
    EST_Stream_Item *s;
    EST_Utterance *u;
    EST_String fname;

    s = GETSTREAMITEMVAL(sitem);
    u = GETUTTVAL(utt);
    if ((s == 0) || (u == 0))
    {
	cerr << "lisp_stream_feat: stream and/or utt wta\n";
	festival_error();
    }
    fname = get_c_string(name);

    return lisp_val(ffeature(*u,*s,fname));

}

static LISP lisp_streamitem_set_feat(LISP sitem, LISP name, LISP value)
{
    // return the ffeature name for this stream 
    EST_Stream_Item *s;
    EST_String fname;

    s = GETSTREAMITEMVAL(sitem);
    fname = get_c_string(name);
    if ((fname.contains(".")) ||
	(fname.contains(":")))
    {
	cerr << "stream.set_feat: cannot set fet name containing " <<
	    "\".\" or \":\"" << endl;
	festival_error();
    }
    if (consp(value))
    {
	cerr << "stream.set_feat: cannot set non-atomic value" << endl;
	festival_error();
    }
    else if (TYPEP(value,tc_flonum))
	s->set_feature(fname,get_c_float(value));
    else
	s->set_feature(fname,get_c_string(value));

    return value;
}

void festival_feature_funcs(void)
{
    // declare feature specific Lisp functions 

    festival_def_ff("name","any",ff_name,
    "ANY.name\n\
  The name of the stream item.  This may be applied to stream items of\n\
  any stream type.");
    festival_def_ff("start","any",ff_start,
    "ANY.start\n\
  The start time in seconds of the stream item.  This is calculated from\n\
  the end time of the previous item, which is defined as 0.0 if there is\n\
  no previous item.  This may be applied to any stream type but some\n\
  streams may not have end times.");
    festival_def_ff("end","any",ff_end,
    "ANY.end\n\
  The end time in seconds of the stream item.  This may apply to stream\n\
  items of any stream type, but some streams may not have end times.");
    festival_def_ff("duration","any",ff_dur,
    "ANY.duration\n\
  The duration of the stream item in seconds.  This is calculated from\n\
  the end time minus the end time of the previous stream item,\n\
  which will be 0.0 if there is no previous item.  This may be called\n\
  for any stream item, though some stream items may not have end times.");
    festival_def_ff("addr","any",ff_addr,
    "ANY.addr\n\
  A unique number for that stream item in that stream.  This allows\n\
  equality checks between stream items in the same stream.  This\n\
  may be applied to stream items of any stream type.");
    festival_def_ff_pref("lisp_","any",ff_lisp_func,
    "ANY.lisp_*\n\
  Apply Lisp function named after lisp_.  The function is called with\n\
  an utterance and stream item.  It must return an atomic value.\n\
  This method may be inefficient and is primarily desgined to allow\n\
  quick prototyping of new feature functions.");

    init_subr_3("utt.streamitem.feat",lisp_streamitem_feat,
    "(utt.streamitem.feat UTT STREAMITEM FEATNAME)\n\
   Return value of FEATNAME of STREAMITEM in UTT.");
    init_subr_3("streamitem.set_feat",lisp_streamitem_set_feat,
    "(streamitem.set_feat STREAMITEM FEATNAME VALUE)\n\
   Set FEATNAME to VALUE in STREAMITEM.");


}
