//============================================================================
//
//    SSSS    tt          lll  lll
//   SS  SS   tt           ll   ll
//   SS     tttttt  eeee   ll   ll   aaaa    "An Atari 2600 VCS Emulator"
//    SSSS    tt   ee  ee  ll   ll      aa
//       SS   tt   eeeeee  ll   ll   aaaaa   Copyright (c) 1995,1996,1997
//   SS  SS   tt   ee      ll   ll  aa  aa         Bradford W. Mott
//    SSSS     ttt  eeeee llll llll  aaaaa
//
//============================================================================

/**
  A device that emulates the Television Interface Adapator found in
  the Atari 2600.  The Television Interface Adapator is an integrated
  circuit designed to interface between an eight bit microprocessor
  and a television video modulator. It converts eight bit parallel data
  into serial outputs for the color, luminosity, and composite sync
  required by a video modulator.

  This class emulates the TIA by outputing the serial outputs into
  a frame buffer which can then be displayed on screen.

  @author  Bradford W. Mott
  @version $Id: TIA.cxx,v 1.3 1997/05/31 14:01:38 bwmott Exp $
*/

#include <assert.h>
#include <iostream.h>
#include <string.h>

#include "Snd.hxx"
#include "System.hxx"
#include "Term.hxx"
#include "TIA.hxx"

#define HBLANKOFFSET 0
#define HBLANK (68 - HBLANKOFFSET)

//============================================================================
// Constructor
//============================================================================
TIA::TIA(System& system)
    : Device(system)
{
  // Map all of my addresses in the system
  for(uWord address = 0; address < 8192; ++address)
  {
    if((address & 0x1080) == 0x0000)
    {
      mySystem.mapPeek(address, *this);
      mySystem.mapPoke(address, *this);
    }
  }

  // Ask the sound system to create me a sound object
  mySound = Sound::create();

  // Allocate buffers for two frame buffers
  myCurrentFrameBuffer = new uByte[160 * 300];
  myPreviousFrameBuffer = new uByte[160 * 300];

  #if defined OS_2
  DiveMap((PBYTE) myCurrentFrameBuffer);
  #endif

  // Compute all of the mask tables
  computeBallMaskTable();
  computeCollisionTable();
  computeMissleMaskTable();
  computePlayerMaskTable();
  computePlayerReflectTable();
  computePlayfieldMaskTable();

  // Put myself in a sane state
  reset();
}

//============================================================================
// Destructor
//============================================================================
TIA::~TIA()
{
  #if defined OS_2
    DiveUnMap();
  #endif

  delete mySound;
  delete[] myCurrentFrameBuffer;
  delete[] myPreviousFrameBuffer;
}

//============================================================================
// Reset to power on state
//============================================================================
void TIA::reset()
{
  // Clear frame buffers
  for(uLong i = 0; i < 160 * 300; ++i)
  {
    myCurrentFrameBuffer[i] = myPreviousFrameBuffer[i] = 0;
  }

  // Reset pixel pointer and drawing flag
  myFramePointer = myCurrentFrameBuffer;
  myDrawFrame = false;

  // Calculate color clock offsets for starting and stoping frame drawing
  myStartDisplayOffset = mySystem.properties().integer("Display.YStart") * 228;
  myStopDisplayOffset = mySystem.properties().integer("Display.Height") * 228 
      + myStartDisplayOffset;

  // Reasonable values to start and stop the current frame drawing
  myClockWhenFrameStarted = mySystem.m6507().cycles() * 3;
  myClockStartDisplay = myClockWhenFrameStarted + myStartDisplayOffset;
  myClockStopDisplay = myClockWhenFrameStarted + myStopDisplayOffset;
  myClockAtLastUpdate = myClockWhenFrameStarted;
  myClockToEndOfScanLine = 228;
  myVSYNCFinishClock = 0xFFFFFFFF;

  // Currently no objects are enabled
  myEnabledObjects = 0;

  // Some default values for the registers
  myVSYNC = 0;
  myVBLANK = 0;
  myNUSIZ0 = 0;
  myNUSIZ1 = 0;
  myCOLUP0 = 0;
  myCOLUP1 = 0;
  myCOLUPF = 0;
  myCOLUBK = 0;
  myCTRLPF = 0;
  myREFP0 = false;
  myREFP1 = false;
  myPF = 0;
  myGRP0 = 0;
  myGRP1 = 0;
  myDGRP0 = 0;
  myDGRP1 = 0;
  myENAM0 = false;
  myENAM1 = false;
  myENABL = false;
  myDENABL = false;
  myHMP0 = 0;
  myHMP1 = 0;
  myHMM0 = 0;
  myHMM1 = 0;
  myHMBL = 0;
  myVDELP0 = false;
  myVDELP1 = false;
  myVDELBL = false;
  myRESMP0 = false;
  myRESMP1 = false;
  myCollision = 0;
  myPOSP0 = 0;
  myPOSP1 = 0;
  myPOSM0 = 0;
  myPOSM1 = 0;
  myPOSBL = 0;

  // Some default values for the "current" variables
  myCurrentPF = 0;
  myCurrentGRP0 = 0;
  myCurrentGRP1 = 0;
  myCurrentBLMask = ourBallMaskTable[0][0];
  myCurrentM0Mask = ourMissleMaskTable[0][0][0];
  myCurrentM1Mask = ourMissleMaskTable[0][0][0];
  myCurrentP0Mask = ourPlayerMaskTable[0][0];
  myCurrentP1Mask = ourPlayerMaskTable[0][0];
  myCurrentPFMask = ourPlayfieldTable[0];

  // Get the number of color clocks to delay a player's graphics write by
  myPlayerDelay = mySystem.properties().integer("TIA.PlayerDelay");

  myLastHMOVEClock = 0;
  myM0CosmicArkMotionEnabled = false;
}
 
//============================================================================
// Execute instructions until the next frame has been generated
//============================================================================
void TIA::frame(bool drawFrame)
{
  // Indicate that we should draw the frame
  myDrawFrame = drawFrame;

  #ifndef OS_2    // remove for OS/2 - conflicts with DIVE support
  // Swap frame buffers if we're actually drawing the frame
  if(myDrawFrame)
  {
    uByte* tmp = myCurrentFrameBuffer;
    myCurrentFrameBuffer = myPreviousFrameBuffer;
    myPreviousFrameBuffer = tmp;
  }
  #endif

  // Reset frame buffer pointer
  myFramePointer = myCurrentFrameBuffer;

  // Execute instructions until frame is finished or the given number 
  // of cycles has been processed
  mySystem.m6507().execute(150000);
}

//============================================================================
// Answer the height of my frame buffer
//============================================================================
uWord TIA::height() const 
{ 
  return mySystem.properties().integer("Display.Height"); 
}

//============================================================================
// Answer sound object used for playing audio
//============================================================================
Sound& TIA::sound() const
{
  return *mySound;
}

//============================================================================
// Compute the ball mask table
//============================================================================
void TIA::computeBallMaskTable()
{
  // First, calculate masks for alignment 0
  for(Long size = 0; size < 4; ++size)
  {
    Long x;

    // Set all of the masks to false to start with
    for(x = 0; x < 160; ++x)
    {
      ourBallMaskTable[0][size][x] = false;
    }

    // Set the necessary fields true
    for(x = 0; x < 160 + 8; ++x)
    {
      if((x >= 0) && (x < (1 << size)))
      {
        ourBallMaskTable[0][size][x % 160] = true;
      }
    }

    // Copy fields into the wrap-around area of the mask
    for(x = 0; x < 160; ++x)
    {
      ourBallMaskTable[0][size][x + 160] = ourBallMaskTable[0][size][x];
    }
  }

  // Now, copy data for alignments of 1, 2 and 3
  for(uLong align = 1; align < 4; ++align)
  {
    for(uLong size = 0; size < 4; ++size)
    {
      for(uLong x = 0; x < 320; ++x)
      {
        ourBallMaskTable[align][size][x] = 
            ourBallMaskTable[0][size][(x + 320 - align) % 320];
      }
    }
  }
}

//============================================================================
// Compute the collision decode table
//============================================================================
void TIA::computeCollisionTable()
{
  for(int i = 0; i < 64; ++i)
  { 
    ourCollisionTable[i] = 0;

    if((i & myM0Bit) && (i & myP1Bit))    // M0-P1
      ourCollisionTable[i] |= 0x0001;

    if((i & myM0Bit) && (i & myP0Bit))    // M0-P0
      ourCollisionTable[i] |= 0x0002;

    if((i & myM1Bit) && (i & myP0Bit))    // M1-P0
      ourCollisionTable[i] |= 0x0004;

    if((i & myM1Bit) && (i & myP1Bit))    // M1-P1
      ourCollisionTable[i] |= 0x0008;

    if((i & myP0Bit) && (i & myPFBit))    // P0-PF
      ourCollisionTable[i] |= 0x0010;

    if((i & myP0Bit) && (i & myBLBit))    // P0-BL
      ourCollisionTable[i] |= 0x0020;

    if((i & myP1Bit) && (i & myPFBit))    // P1-PF
      ourCollisionTable[i] |= 0x0040;

    if((i & myP1Bit) && (i & myBLBit))    // P1-BL
      ourCollisionTable[i] |= 0x0080;

    if((i & myM0Bit) && (i & myPFBit))    // M0-PF
      ourCollisionTable[i] |= 0x0100;

    if((i & myM0Bit) && (i & myBLBit))    // M0-BL
      ourCollisionTable[i] |= 0x0200;

    if((i & myM1Bit) && (i & myPFBit))    // M1-PF
      ourCollisionTable[i] |= 0x0400;

    if((i & myM1Bit) && (i & myBLBit))    // M1-BL
      ourCollisionTable[i] |= 0x0800;

    if((i & myBLBit) && (i & myPFBit))    // BL-PF
      ourCollisionTable[i] |= 0x1000;

    if((i & myP0Bit) && (i & myP1Bit))    // P0-P1
      ourCollisionTable[i] |= 0x2000;

    if((i & myM0Bit) && (i & myM1Bit))    // M0-M1
      ourCollisionTable[i] |= 0x4000;
  }
}

//============================================================================
// Compute the missle mask table
//============================================================================
void TIA::computeMissleMaskTable()
{
  // First, calculate masks for alignment 0
  Long x, size, number;

  // Clear the missle table to start with
  for(number = 0; number < 8; ++number)
    for(size = 0; size < 4; ++size)
      for(x = 0; x < 160; ++x)
        ourMissleMaskTable[0][number][size][x] = false;

  for(number = 0; number < 8; ++number)
  {
    for(size = 0; size < 4; ++size)
    {
      for(x = 0; x < 160 + 72; ++x)
      {
        // Only one copy of the missle
        if((number == 0x00) || (number == 0x05) || (number == 0x07))
        {
          if((x >= 0) && (x < (1 << size)))
            ourMissleMaskTable[0][number][size][x % 160] = true;
        }
        // Two copies - close
        else if(number == 0x01)
        {
          if((x >= 0) && (x < (1 << size)))
            ourMissleMaskTable[0][number][size][x % 160] = true;
          else if(((x - 16) >= 0) && ((x - 16) < (1 << size)))
            ourMissleMaskTable[0][number][size][x % 160] = true;
        }
        // Two copies - medium
        else if(number == 0x02)
        {
          if((x >= 0) && (x < (1 << size)))
            ourMissleMaskTable[0][number][size][x % 160] = true;
          else if(((x - 32) >= 0) && ((x - 32) < (1 << size)))
            ourMissleMaskTable[0][number][size][x % 160] = true;
        }
        // Three copies - close
        else if(number == 0x03)
        {
          if((x >= 0) && (x < (1 << size)))
            ourMissleMaskTable[0][number][size][x % 160] = true;
          else if(((x - 16) >= 0) && ((x - 16) < (1 << size)))
            ourMissleMaskTable[0][number][size][x % 160] = true;
          else if(((x - 32) >= 0) && ((x - 32) < (1 << size)))
            ourMissleMaskTable[0][number][size][x % 160] = true;
        }
        // Two copies - wide
        else if(number == 0x04)
        {
          if((x >= 0) && (x < (1 << size)))
            ourMissleMaskTable[0][number][size][x % 160] = true;
          else if(((x - 64) >= 0) && ((x - 64) < (1 << size)))
            ourMissleMaskTable[0][number][size][x % 160] = true;
        }
        // Three copies - medium
        else if(number == 0x06)
        {
          if((x >= 0) && (x < (1 << size)))
            ourMissleMaskTable[0][number][size][x % 160] = true;
          else if(((x - 32) >= 0) && ((x - 32) < (1 << size)))
            ourMissleMaskTable[0][number][size][x % 160] = true;
          else if(((x - 64) >= 0) && ((x - 64) < (1 << size)))
            ourMissleMaskTable[0][number][size][x % 160] = true;
        }
      }

      // Copy data into wrap-around area
      for(x = 0; x < 160; ++x)
        ourMissleMaskTable[0][number][size][x + 160] = 
          ourMissleMaskTable[0][number][size][x];
    }
  }

  // Now, copy data for alignments of 1, 2 and 3
  for(uLong align = 1; align < 4; ++align)
  {
    for(number = 0; number < 8; ++number)
    {
      for(size = 0; size < 4; ++size)
      {
        for(x = 0; x < 320; ++x)
        {
          ourMissleMaskTable[align][number][size][x] = 
            ourMissleMaskTable[0][number][size][(x + 320 - align) % 320];
        }
      }
    }
  }
}

//============================================================================
// Compute the player mask table
//============================================================================
void TIA::computePlayerMaskTable()
{
  // First, calculate masks for alignment 0
  Long x, mode;

  // Set the player mask table to all zeros
  for(mode = 0; mode < 8; ++mode)
    for(x = 0; x < 160; ++x)
      ourPlayerMaskTable[0][mode][x] = 0x00;

  // Now, compute the player mask table
  for(mode = 0; mode < 8; ++mode)
  {
    for(x = 0; x < 160 + 72; ++x)
    {
      if(mode == 0x00)
      {
        if((x >= 0) && (x < 8))
          ourPlayerMaskTable[0][mode][x % 160] = 0x80 >> x;
      }
      else if(mode == 0x01)
      {
        if((x >= 0) && (x < 8))
          ourPlayerMaskTable[0][mode][x % 160] = 0x80 >> x;
        else if(((x - 16) >= 0) && ((x - 16) < 8))
          ourPlayerMaskTable[0][mode][x % 160] = 0x80 >> (x - 16);
      }
      else if(mode == 0x02)
      {
        if((x >= 0) && (x < 8))
          ourPlayerMaskTable[0][mode][x % 160] = 0x80 >> x;
        else if(((x - 32) >= 0) && ((x - 32) < 8))
          ourPlayerMaskTable[0][mode][x % 160] = 0x80 >> (x - 32);
      }
      else if(mode == 0x03)
      {
        if((x >= 0) && (x < 8))
          ourPlayerMaskTable[0][mode][x % 160] = 0x80 >> x;
        else if(((x - 16) >= 0) && ((x - 16) < 8))
          ourPlayerMaskTable[0][mode][x % 160] = 0x80 >> (x - 16);
        else if(((x - 32) >= 0) && ((x - 32) < 8))
          ourPlayerMaskTable[0][mode][x % 160] = 0x80 >> (x - 32);
      }
      else if(mode == 0x04)
      {
        if((x >= 0) && (x < 8))
          ourPlayerMaskTable[0][mode][x % 160] = 0x80 >> x;
        else if(((x - 64) >= 0) && ((x - 64) < 8))
          ourPlayerMaskTable[0][mode][x % 160] = 0x80 >> (x - 64);
      }
      else if(mode == 0x05)
      {
//      if((x >= 0) && (x < 16))
//        ourPlayerMaskTable[0][mode][x % 160] = 0x80 >> (x / 2);
        if((x > 0) && (x <= 16))
          ourPlayerMaskTable[0][mode][x % 160] = 0x80 >> ((x - 1) / 2);
      }
      else if(mode == 0x06)
      {
        if((x >= 0) && (x < 8))
          ourPlayerMaskTable[0][mode][x % 160] = 0x80 >> x;
        else if(((x - 32) >= 0) && ((x - 32) < 8))
          ourPlayerMaskTable[0][mode][x % 160] = 0x80 >> (x - 32);
        else if(((x - 64) >= 0) && ((x - 64) < 8))
          ourPlayerMaskTable[0][mode][x % 160] = 0x80 >> (x - 64);
      }
      else if(mode == 0x07)
      {
//      if((x >= 0) && (x < 32))
//        ourPlayerMaskTable[0][mode][x % 160] = 0x80 >> (x / 4);
        if((x > 0) && (x <= 32))
          ourPlayerMaskTable[0][mode][x % 160] = 0x80 >> ((x - 1) / 4);
      }
    }

    // Copy data into wrap-around area
    for(x = 0; x < 160; ++x)
      ourPlayerMaskTable[0][mode][x + 160] = ourPlayerMaskTable[0][mode][x];
  }

  // Now, copy data for alignments of 1, 2 and 3
  for(uLong align = 1; align < 4; ++align)
  {
    for(mode = 0; mode < 8; ++mode)
    {
      for(x = 0; x < 320; ++x)
      {
        ourPlayerMaskTable[align][mode][x] =
            ourPlayerMaskTable[0][mode][(x + 320 - align) % 320];
      }
    }
  }
}

//============================================================================
// Compute the player reflect table
//============================================================================
void TIA::computePlayerReflectTable()
{
  for(uWord i = 0; i < 256; ++i)
  {
    uByte r = 0;

    for(uWord t = 1; t <= 128; t *= 2)
    {
      r = (r << 1) | ((i & t) ? 0x01 : 0x00);
    }

    ourPlayerReflectTable[i] = r;
  } 
}

//============================================================================
// Compute playfield mask table
//============================================================================
void TIA::computePlayfieldMaskTable()
{
  Long x;

  // Compute playfield mask table for non-reflected mode
  for(x = 0; x < 160; ++x)
  {
    if(x < 16)
      ourPlayfieldTable[0][x] = 0x00001 << (x / 4);
    else if(x < 48)
      ourPlayfieldTable[0][x] = 0x00800 >> ((x - 16) / 4);
    else if(x < 80) 
      ourPlayfieldTable[0][x] = 0x01000 << ((x - 48) / 4);
    else if(x < 96) 
      ourPlayfieldTable[0][x] = 0x00001 << ((x - 80) / 4);
    else if(x < 128)
      ourPlayfieldTable[0][x] = 0x00800 >> ((x - 96) / 4);
    else if(x < 160) 
      ourPlayfieldTable[0][x] = 0x01000 << ((x - 128) / 4);
  }

  // Compute playfield mask table for reflected mode
  for(x = 0; x < 160; ++x)
  {
    if(x < 16)
      ourPlayfieldTable[1][x] = 0x00001 << (x / 4);
    else if(x < 48)
      ourPlayfieldTable[1][x] = 0x00800 >> ((x - 16) / 4);
    else if(x < 80) 
      ourPlayfieldTable[1][x] = 0x01000 << ((x - 48) / 4);
    else if(x < 112) 
      ourPlayfieldTable[1][x] = 0x80000 >> ((x - 80) / 4);
    else if(x < 144) 
      ourPlayfieldTable[1][x] = 0x00010 << ((x - 112) / 4);
    else if(x < 160) 
      ourPlayfieldTable[1][x] = 0x00008 >> ((x - 144) / 4);
  }
}

//============================================================================
// Helper function for the updateFrame method
//============================================================================
inline void TIA::updateFrameHelper(uLong clock)
{
  // Update frame one scanline at a time
  do
  {
    // Compute the number of clocks we're going to update
    uLong clocksToUpdate = 0;

    Long clocksFromStartOfScanLine = 228 - myClockToEndOfScanLine;

    // See if we're updating more than the current scanline
    if((myClockToEndOfScanLine + myClockAtLastUpdate) <= clock)
    {
      // Yes, we have more than one scanline to update so finish current one
      clocksToUpdate = myClockToEndOfScanLine;
      myClockToEndOfScanLine = 228;
      myClockAtLastUpdate += clocksToUpdate;
    }
    else
    {
      // No, so do as much of the current scanline as possible
      clocksToUpdate = clock - myClockAtLastUpdate;
      myClockToEndOfScanLine -= clocksToUpdate;
      myClockAtLastUpdate = clock;
    }

    // Reset color clock position if it has reached the right side
    if(clocksFromStartOfScanLine >= (228 - HBLANKOFFSET))
    {
      clocksFromStartOfScanLine = 0;
    }

    // Skip over as many horizontal blank clocks as we can
    if(clocksFromStartOfScanLine < HBLANK)
    {
      uLong tmp = (HBLANK - clocksFromStartOfScanLine) < (Long)clocksToUpdate ? 
          (HBLANK - clocksFromStartOfScanLine) : clocksToUpdate;
      clocksFromStartOfScanLine += tmp;
      clocksToUpdate -= tmp;

    // TODO: make this cleaner plus is seems that it can be < 0 in some cases?
    if(clocksToUpdate == 0)
      break;
    }

    Word x = clocksFromStartOfScanLine - HBLANK;

    // Calculate the ending frame pointer value
    uByte* ending = myFramePointer + clocksToUpdate;
    
    // See if we're in the vertical blank region
    if(myVBLANK & 0x02)
    {
      memset(myFramePointer, 0, clocksToUpdate);
    }
    // Handle all other possible combinations
    else
    {
      switch(myEnabledObjects)
      {
        case 0:
        {
          memset(myFramePointer, myCOLUBK, clocksToUpdate);
          break;
        }

        case myPFBit:    // Playfield is enabled
        {
          uLong* mask = &myCurrentPFMask[x];

          if(myCTRLPF & 0x02)
          {
            // Update a uByte at a time until reaching a uLong boundary
            for(; ((int)myFramePointer & 0x03) && (myFramePointer < ending); 
                ++myFramePointer, ++mask, ++x)
            {
              *myFramePointer = (myCurrentPF & *mask) ? 
                  (x < 80 ? myCOLUP0 : myCOLUP1) : myCOLUBK;
            }

            // Now, update a uLong at a time
            for(; myFramePointer < ending; 
                myFramePointer += 4, mask += 4, x += 4)
            {
              *((uLong*)myFramePointer) = (myCurrentPF & *mask) ?
                  (x < 80 ? myCOLUP0 : myCOLUP1) : myCOLUBK;
            }
          }
          else
          {
            // Update a uByte at a time until reaching a uLong boundary
            for(; ((int)myFramePointer & 0x03) && (myFramePointer < ending);
                ++myFramePointer, ++mask)
            {
              *myFramePointer = (myCurrentPF & *mask) ? myCOLUPF : myCOLUBK;
            }

            // Now, update a uLong at a time
            for(; myFramePointer < ending; myFramePointer += 4, mask += 4)
            {
              *((uLong*)myFramePointer) = (myCurrentPF & *mask) ? 
                  myCOLUPF : myCOLUBK;
            }
          }
          break;
        }

        case myP0Bit:    // Player 0 is enabled
        {
          uByte* mP0 = &myCurrentP0Mask[x];

          while(myFramePointer < ending)
          {
            if(!((int)myFramePointer & 0x03) && !*(uLong*)mP0)
            {
              *(uLong*)myFramePointer = myCOLUBK;
              mP0 += 4; myFramePointer += 4;
            }
            else
            {
              *myFramePointer = (myCurrentGRP0 & *mP0) ? myCOLUP0 : myCOLUBK;
              ++mP0; ++myFramePointer;
            }
          }
          break;
        }

        case myP1Bit:    // Player 1 is enabled
        {
          uByte* mP1 = &myCurrentP1Mask[x];

          while(myFramePointer < ending)
          {
            if(!((int)myFramePointer & 0x03) && !*(uLong*)mP1)
            {
              *(uLong*)myFramePointer = myCOLUBK;
              mP1 += 4; myFramePointer += 4;
            }
            else
            {
              *myFramePointer = (myCurrentGRP1 & *mP1) ? myCOLUP1 : myCOLUBK;
              ++mP1; ++myFramePointer;
            }
          }
          break;
        }

        case myM0Bit:    // Missle 0 is enabled
        {
          uByte* mM0 = &myCurrentM0Mask[x];

          while(myFramePointer < ending)
          {
            if(!((int)myFramePointer & 0x03) && !*(uLong*)mM0)
            {
              *(uLong*)myFramePointer = myCOLUBK;
              mM0 += 4; myFramePointer += 4;
            }
            else
            {
              *myFramePointer = *mM0 ? myCOLUP0 : myCOLUBK;
              ++mM0; ++myFramePointer;
            }
          }
          break;
        }

        case myM1Bit:    // Missle 1 is enabled
        {
          uByte* mM1 = &myCurrentM1Mask[x];

          while(myFramePointer < ending)
          {
            if(!((int)myFramePointer & 0x03) && !*(uLong*)mM1)
            {
              *(uLong*)myFramePointer = myCOLUBK;
              mM1 += 4; myFramePointer += 4;
            }
            else
            {
              *myFramePointer = *mM1 ? myCOLUP1 : myCOLUBK;
              ++mM1; ++myFramePointer;
            }
          }
          break;
        }

        case myBLBit:    // Ball is enabled
        {
          uByte* mBL = &myCurrentBLMask[x];

          while(myFramePointer < ending)
          {
            if(!((int)myFramePointer & 0x03) && !*(uLong*)mBL)
            {
              *(uLong*)myFramePointer = myCOLUBK;
              mBL += 4; myFramePointer += 4;
            }
            else
            {
              *myFramePointer = *mBL ? myCOLUPF : myCOLUBK;
              ++mBL; ++myFramePointer;
            }
          }
          break;
        }

        case myP0Bit | myP1Bit:    // Player 0 and 1 are enabled
        {
          uByte* mP0 = &myCurrentP0Mask[x];
          uByte* mP1 = &myCurrentP1Mask[x];

          while(myFramePointer < ending)
          {
            if(!((int)myFramePointer & 0x03) && !*(uLong*)mP0 && !*(uLong*)mP1)
            {
              *(uLong*)myFramePointer = myCOLUBK;
              mP0 += 4; mP1 += 4; myFramePointer += 4;
            }
            else
            {
              *myFramePointer = (myCurrentGRP0 & *mP0) ? 
                  myCOLUP0 : ((myCurrentGRP1 & *mP1) ? myCOLUP1 : myCOLUBK);

              if((myCurrentGRP0 & *mP0) && (myCurrentGRP1 & *mP1))
                myCollision |= ourCollisionTable[myP0Bit | myP1Bit];

              ++mP0; ++mP1; ++myFramePointer;
            }
          }
          break;
        }

        case myM0Bit | myM1Bit:    // Missle 0 and 1 are enabled
        {
          uByte* mM0 = &myCurrentM0Mask[x];
          uByte* mM1 = &myCurrentM1Mask[x];

          while(myFramePointer < ending)
          {
            if(!((int)myFramePointer & 0x03) && !*(uLong*)mM0 && !*(uLong*)mM1)
            {
              *(uLong*)myFramePointer = myCOLUBK;
              mM0 += 4; mM1 += 4; myFramePointer += 4;
            }
            else
            {
              *myFramePointer = *mM0 ? myCOLUP0 : (*mM1 ? myCOLUP1 : myCOLUBK);

              if(*mM0 && *mM1)
                myCollision |= ourCollisionTable[myM0Bit | myM1Bit];

              ++mM0; ++mM1; ++myFramePointer;
            }
          }
          break;
        }

        case myBLBit | myM0Bit:    // Ball and Missle 0 are enabled
        {
          uByte* mBL = &myCurrentBLMask[x];
          uByte* mM0 = &myCurrentM0Mask[x];

          while(myFramePointer < ending)
          {
            if(!((int)myFramePointer & 0x03) && !*(uLong*)mBL && !*(uLong*)mM0)
            {
              *(uLong*)myFramePointer = myCOLUBK;
              mBL += 4; mM0 += 4; myFramePointer += 4;
            }
            else
            {
              *myFramePointer = (myCTRLPF & 0x04) ?
                  (*mBL ? myCOLUPF : (*mM0 ? myCOLUP0 : myCOLUBK)) :
                  (*mM0 ? myCOLUP0 : (*mBL ? myCOLUPF : myCOLUBK));

              if(*mBL && *mM0)
                myCollision |= ourCollisionTable[myBLBit | myM0Bit];

              ++mBL; ++mM0; ++myFramePointer;
            }
          }
          break;
        }

        case myBLBit | myM1Bit:    // Ball and Missle 1 are enabled
        {
          uByte* mBL = &myCurrentBLMask[x];
          uByte* mM1 = &myCurrentM1Mask[x];

          while(myFramePointer < ending)
          {
            if(!((int)myFramePointer & 0x03) && !*(uLong*)mBL && !*(uLong*)mM1)
            {
              *(uLong*)myFramePointer = myCOLUBK;
              mBL += 4; mM1 += 4; myFramePointer += 4;
            }
            else
            {
              *myFramePointer = (myCTRLPF & 0x04) ?
                  (*mBL ? myCOLUPF : (*mM1 ? myCOLUP1 : myCOLUBK)) :
                  (*mM1 ? myCOLUP1 : (*mBL ? myCOLUPF : myCOLUBK));

              if(*mBL && *mM1)
                myCollision |= ourCollisionTable[myBLBit | myM1Bit];

              ++mBL; ++mM1; ++myFramePointer;
            }
          }
          break;
        }

        case myPFBit | myP0Bit:    // Playfield and Player 0 are enabled
        {
          uLong* mPF = &myCurrentPFMask[x];
          uByte* mP0 = &myCurrentP0Mask[x];

          if(myCTRLPF & 0x02)
          {
            for(; myFramePointer < ending; ++myFramePointer, ++x, ++mPF, ++mP0)
            {
              *myFramePointer = (myCTRLPF & 0x04) ?
                  ((myCurrentPF & *mPF) ? 
                      (x < 80 ? myCOLUP0 : myCOLUP1) : 
                      ((myCurrentGRP0 & *mP0) ? myCOLUP0 : myCOLUBK)) :
                  ((myCurrentGRP0 & *mP0) ?
                      myCOLUP0 : 
                      ((myCurrentPF & *mPF) ? 
                          (x < 80 ? myCOLUP0 : myCOLUP1) : myCOLUBK));

              if((myCurrentPF & *mPF) && (myCurrentGRP0 & *mP0))
                myCollision |= ourCollisionTable[myPFBit | myP0Bit];
            }
          }
          else
          {
            while(myFramePointer < ending)
            {
              if(!((int)myFramePointer & 0x03) && !*(uLong*)mP0)
              {
                *(uLong*)myFramePointer = (myCurrentPF & *mPF) ? 
                    myCOLUPF : myCOLUBK;
                mPF += 4; mP0 += 4; myFramePointer += 4;
              }
              else
              {
                *myFramePointer = (myCTRLPF & 0x04) ?
                  ((myCurrentPF & *mPF) ? myCOLUPF : 
                      ((myCurrentGRP0 & *mP0) ? myCOLUP0 : myCOLUBK)) :
                  ((myCurrentGRP0 & *mP0) ? 
                      myCOLUP0 : ((myCurrentPF & *mPF) ? myCOLUPF : myCOLUBK));

                if((myCurrentPF & *mPF) && (myCurrentGRP0 & *mP0))
                  myCollision |= ourCollisionTable[myPFBit | myP0Bit];

                ++mPF; ++mP0; ++myFramePointer;
              }
            }
          }
          break;
        }

        case myPFBit | myP1Bit:    // Playfield and Player 1 are enabled
        {
          uLong* mPF = &myCurrentPFMask[x];
          uByte* mP1 = &myCurrentP1Mask[x];

          if(myCTRLPF & 0x02)
          {
            for(; myFramePointer < ending; ++myFramePointer, ++x, ++mPF, ++mP1)
            {
              *myFramePointer = (myCTRLPF & 0x04) ?
                  ((myCurrentPF & *mPF) ? 
                      (x < 80 ? myCOLUP0 : myCOLUP1) : 
                      ((myCurrentGRP1 & *mP1) ? myCOLUP1 : myCOLUBK)) :
                  ((myCurrentGRP1 & *mP1) ?
                      myCOLUP1 : 
                      ((myCurrentPF & *mPF) ? 
                          (x < 80 ? myCOLUP0 : myCOLUP1) : myCOLUBK));

              if((myCurrentPF & *mPF) && (myCurrentGRP1 & *mP1))
                myCollision |= ourCollisionTable[myPFBit | myP1Bit];
            }
          }
          else
          {
            while(myFramePointer < ending)
            {
              if(!((int)myFramePointer & 0x03) && !*(uLong*)mP1)
              {
                *(uLong*)myFramePointer = (myCurrentPF & *mPF) ? 
                    myCOLUPF : myCOLUBK;
                mPF += 4; mP1 += 4; myFramePointer += 4;
              }
              else
              {
                *myFramePointer = (myCTRLPF & 0x04) ?
                  ((myCurrentPF & *mPF) ? myCOLUPF : 
                      ((myCurrentGRP1 & *mP1) ? myCOLUP1 : myCOLUBK)) :
                  ((myCurrentGRP1 & *mP1) ? 
                      myCOLUP1 : ((myCurrentPF & *mPF) ? myCOLUPF : myCOLUBK));

                if((myCurrentPF & *mPF) && (myCurrentGRP1 & *mP1))
                  myCollision |= ourCollisionTable[myPFBit | myP1Bit];

                ++mPF; ++mP1; ++myFramePointer;
              }
            }
          }
          break;
        }

        case myPFBit | myM0Bit:    // Playfield and Missle 0 are enabled
        {
          uLong* mPF = &myCurrentPFMask[x];
          uByte* mM0 = &myCurrentM0Mask[x];

          if(myCTRLPF & 0x02)
          {
            for(; myFramePointer < ending; ++myFramePointer, ++x, ++mPF, ++mM0)
            {
              *myFramePointer = (myCTRLPF & 0x04) ?
                  ((myCurrentPF & *mPF) ? 
                      (x < 80 ? myCOLUP0 : myCOLUP1) : 
                      (*mM0 ? myCOLUP0 : myCOLUBK)) :
                  (*mM0 ? 
                      myCOLUP0 : ((myCurrentPF & *mPF) ? 
                          (x < 80 ? myCOLUP0 : myCOLUP1) : myCOLUBK));

              if((myCurrentPF & *mPF) && *mM0)
                myCollision |= ourCollisionTable[myPFBit | myM0Bit];
            }
          }
          else
          {
            while(myFramePointer < ending)
            {
              if(!((int)myFramePointer & 0x03) && !*(uLong*)mM0)
              {
                *(uLong*)myFramePointer = (myCurrentPF & *mPF) ? 
                    myCOLUPF : myCOLUBK;
                mPF += 4; mM0 += 4; myFramePointer += 4;
              }
              else
              {
                *myFramePointer = (myCTRLPF & 0x04) ?
                  ((myCurrentPF & *mPF) ? 
                      myCOLUPF : (*mM0 ? myCOLUP0 : myCOLUBK)) :
                  (*mM0 ? 
                      myCOLUP0 : ((myCurrentPF & *mPF) ? myCOLUPF : myCOLUBK));

                if((myCurrentPF & *mPF) && *mM0)
                  myCollision |= ourCollisionTable[myPFBit | myM0Bit];

                ++mPF; ++mM0; ++myFramePointer;
              }
            }
          }
          break;
        }

        case myPFBit | myM1Bit:    // Playfield and Missle 1 are enabled
        {
          uLong* mPF = &myCurrentPFMask[x];
          uByte* mM1 = &myCurrentM1Mask[x];

          if(myCTRLPF & 0x02)
          {
            for(; myFramePointer < ending; ++myFramePointer, ++x, ++mPF, ++mM1)
            {
              *myFramePointer = (myCTRLPF & 0x04) ?
                  ((myCurrentPF & *mPF) ? 
                      (x < 80 ? myCOLUP0 : myCOLUP1) : 
                      (*mM1 ? myCOLUP1 : myCOLUBK)) :
                  (*mM1 ? 
                      myCOLUP1 : ((myCurrentPF & *mPF) ? 
                          (x < 80 ? myCOLUP0 : myCOLUP1) : myCOLUBK));

              if((myCurrentPF & *mPF) && *mM1)
                myCollision |= ourCollisionTable[myPFBit | myM1Bit];
            }
          }
          else
          {
            while(myFramePointer < ending)
            {
              if(!((int)myFramePointer & 0x03) && !*(uLong*)mM1)
              {
                *(uLong*)myFramePointer = (myCurrentPF & *mPF) ? 
                    myCOLUPF : myCOLUBK;
                mPF += 4; mM1 += 4; myFramePointer += 4;
              }
              else
              {
                *myFramePointer = (myCTRLPF & 0x04) ?
                  ((myCurrentPF & *mPF) ? 
                      myCOLUPF : (*mM1 ? myCOLUP1 : myCOLUBK)) :
                  (*mM1 ? 
                      myCOLUP1 : ((myCurrentPF & *mPF) ? myCOLUPF : myCOLUBK));

                if((myCurrentPF & *mPF) && *mM1)
                  myCollision |= ourCollisionTable[myPFBit | myM1Bit];

                ++mPF; ++mM1; ++myFramePointer;
              }
            }
          }
          break;
        }

        case myPFBit | myBLBit:    // Playfield and Ball are enabled
        {
          uLong* mPF = &myCurrentPFMask[x];
          uByte* mBL = &myCurrentBLMask[x];

          if(myCTRLPF & 0x02)
          {
            for(; myFramePointer < ending; ++myFramePointer, ++x, ++mPF, ++mBL)
            {
              *myFramePointer = (myCurrentPF & *mPF) ? 
                  (x < 80 ? myCOLUP0 : myCOLUP1) : (*mBL ? myCOLUPF : myCOLUBK);

              if((myCurrentPF & *mPF) && *mBL)
                myCollision |= ourCollisionTable[myPFBit | myBLBit];
            }
          }
          else
          {
            while(myFramePointer < ending)
            {
              if(!((int)myFramePointer & 0x03) && !*(uLong*)mBL)
              {
                *(uLong*)myFramePointer = (myCurrentPF & *mPF) ? 
                    myCOLUPF : myCOLUBK;
                mPF += 4; mBL += 4; myFramePointer += 4;
              }
              else
              {
                *myFramePointer = ((myCurrentPF & *mPF) || *mBL) ?
                    myCOLUPF : myCOLUBK;

                if((myCurrentPF & *mPF) && *mBL)
                  myCollision |= ourCollisionTable[myPFBit | myBLBit];

                ++mPF; ++mBL; ++myFramePointer;
              }
            }
          }
          break;
        }

        default:
        {
          for(; myFramePointer < ending; ++myFramePointer, ++x)
          {
            uByte collision = 0;
 
            *myFramePointer = myCOLUBK;

            if(!(myCTRLPF & 0x04))
            {
              if(myCurrentPF & myCurrentPFMask[x])
              {
                *myFramePointer = (myCTRLPF & 2) ? 
                    (x < 80 ? myCOLUP0 : myCOLUP1) : myCOLUPF;
                collision |= myPFBit;
              }

              // Draw the ball
              if((myEnabledObjects & myBLBit) && myCurrentBLMask[x])
              {
                *myFramePointer = myCOLUPF;
                collision |= myBLBit;
              }
            }

            // Draw player 1
            if(myCurrentGRP1 & myCurrentP1Mask[x])
            {
              *myFramePointer = myCOLUP1;
              collision |= myP1Bit;
            }

            // Draw missle 1
            if((myEnabledObjects & myM1Bit) && myCurrentM1Mask[x])
            {
              *myFramePointer = myCOLUP1;
              collision |= myM1Bit;
            }

            // Draw player 0
            if(myCurrentGRP0 & myCurrentP0Mask[x])
            {
              *myFramePointer = myCOLUP0;
              collision |= myP0Bit;
            }

            // Draw missle 0
            if((myEnabledObjects & myM0Bit) && myCurrentM0Mask[x])
            {
              *myFramePointer = myCOLUP0;
              collision |= myM0Bit;
            }

            if(myCTRLPF & 0x04)
            {
              if(myCurrentPF & myCurrentPFMask[x])
              {
                *myFramePointer = (myCTRLPF & 2) ? 
                    (x < 80 ? myCOLUP0 : myCOLUP1) : myCOLUPF;
                collision |= myPFBit;
              }

              // Draw the ball
              if((myEnabledObjects & myBLBit) && myCurrentBLMask[x])
              {
                *myFramePointer = myCOLUPF;
                collision |= myBLBit;
              }
            }
  
            // Update the collision register
            myCollision = myCollision | ourCollisionTable[collision];
          }
          break;  
        }
      }
    }
    myFramePointer = ending;
  } 
  while(myClockAtLastUpdate < clock);
}

//============================================================================
// Update the current frame buffer to the specified color clock
//============================================================================
inline void TIA::updateFrame(uLong clock)
{
  // See if we're in the nondisplayable portion of the screen
  if(!myDrawFrame || (clock < myClockStartDisplay) || 
      (myClockAtLastUpdate >= myClockStopDisplay))
  {
    return;
  }

  // Truncate the number of cycles to update to the stop display point
  if(clock > myClockStopDisplay)
  {
    clock = myClockStopDisplay;
  }

  Long x = (228 - myClockToEndOfScanLine) - HBLANK;

  // See if we need to handle a latch point
  if(x & 0x03)
  {
    // Yes, we do so calculate latch point
    uLong lp = myClockAtLastUpdate + 4 - (x & 0x03);

    if(lp <= clock)
    {
      updateFrameHelper(lp);
      myCurrentPF = myPF;
    }
  }
  else
  {
    // No need to handle latch point we are at one now so latch
    myCurrentPF = myPF;
  }

  // Set the PF enabled bit if something is enabled in the playfield
  if(myCurrentPF != 0)
    myEnabledObjects |= myPFBit;
  else
    myEnabledObjects &= ~myPFBit;

  // Update the rest of the frame
  updateFrameHelper(clock);
}

//////////////////////////////////////////////////////////////////////////////
// Called to waste CPU cycles until current scanline is finished
//////////////////////////////////////////////////////////////////////////////
inline void TIA::waitHorizontalSync()
{
  uLong cyclesToEndOfLine = 76 - ((mySystem.m6507().cycles() - 
      (myClockWhenFrameStarted / 3)) % 76);

  if(cyclesToEndOfLine < 76)
  {
    mySystem.m6507().wasteCycles(cyclesToEndOfLine);
  }
}

//============================================================================
// Answer byte at the given address
//============================================================================
uByte TIA::peek(uWord addr)
{
  // Update frame to current color clock before we make any changes!
  updateFrame(mySystem.m6507().cycles() * 3);

  switch(addr & 0x000f)
  {
    case 0x00:    // CXM0P
      return ((myCollision & 0x0001) ? 0x80 : 0x00) | 
          ((myCollision & 0x0002) ? 0x40 : 0x00);

    case 0x01:    // CXM1P
      return ((myCollision & 0x0004) ? 0x80 : 0x00) | 
          ((myCollision & 0x0008) ? 0x40 : 0x00) | 0x01;

    case 0x02:    // CXP0FB
      return ((myCollision & 0x0010) ? 0x80 : 0x00) | 
          ((myCollision & 0x0020) ? 0x40 : 0x00) | 0x02;

    case 0x03:    // CXP1FB
      return ((myCollision & 0x0040) ? 0x80 : 0x00) | 
          ((myCollision & 0x0080) ? 0x40 : 0x00) | 0x03;

    case 0x04:    // CXM0FB
      return ((myCollision & 0x0100) ? 0x80 : 0x00) | 
          ((myCollision & 0x0200) ? 0x40 : 0x00) | 0x04;

    case 0x05:    // CXM1FB
      return ((myCollision & 0x0400) ? 0x80 : 0x00) | 
          ((myCollision & 0x0800) ? 0x40 : 0x00) | 0x05;

    case 0x06:    // CXBLPF
      return ((myCollision & 0x1000) ? 0x80 : 0x00) | 0x06;

    case 0x07:    // CXPPMM
      return ((myCollision & 0x2000) ? 0x80 : 0x00) | 
          ((myCollision & 0x4000) ? 0x40 : 0x00) | 0x07;

    case 0x08:    // INPT0
      return (mySystem.controller(0).readTIA() & 0x01) ? 0xff : 0x7f;

    case 0x09:    // INPT1
      return (mySystem.controller(0).readTIA() & 0x02) ? 0xff : 0x7f;

    case 0x0a:    // INPT2
      return (mySystem.controller(1).readTIA() & 0x01) ? 0xff : 0x7f;

    case 0x0b:    // INPT3
      return (mySystem.controller(1).readTIA() & 0x02) ? 0xff : 0x7f;

    case 0x0c:    // INPT4
      return (mySystem.controller(0).readTIA() & 0x04) ? 0xff : 0x7f;

    case 0x0d:    // INPT5
      return (mySystem.controller(1).readTIA() & 0x04) ? 0xff : 0x7f;

    case 0x0e:
      return 0x0e;

    default:
      return 0x0f;
  }
}

//============================================================================
// Write value at the given address.
//============================================================================
void TIA::poke(uWord addr, uByte value)
{
  addr = addr & 0x003f;

  uLong clock = mySystem.m6507().cycles() * 3;
  uLong delay = 0;

  // See if the result of this poke needs to be delayed 
  if((addr == 0x0D) || (addr == 0x0E) || (addr == 0x0F))
  { 
    delay = 2;
  }
  else if((addr == 0x1B) || (addr == 0x1C))
  {
    delay = myPlayerDelay;
  }

  // Update frame to current CPU cycle before we make any changes!
  updateFrame(clock + delay);
 
  switch(addr)
  {
    case 0x00:    // Vertical sync set-clear
    {
      myVSYNC = value;

      if(myVSYNC & 0x02)
      {
        // Indicate when the VSYNC should be finished.  This should
        // really be 3 * 228 according to Atari's documentation, however,
        // some games don't supply the full 3 scanlines of VSYNC.
        myVSYNCFinishClock = clock + 228;
      }
      else if(!(myVSYNC & 0x02) && (clock >= myVSYNCFinishClock))
      {
        // Setup clocks that'll be used for drawing the frame we are starting
        myClockWhenFrameStarted = myClockWhenFrameStarted + 
            228 * ((clock - myClockWhenFrameStarted) / 228);
        myClockStartDisplay = myClockWhenFrameStarted + myStartDisplayOffset;
        myClockStopDisplay = myClockWhenFrameStarted + myStopDisplayOffset;
        myClockAtLastUpdate = myClockStartDisplay;
        myClockToEndOfScanLine = 228;

        // We're no longer interested in myVSYNCFinishClock
        myVSYNCFinishClock = 0xFFFFFFFF;

        // Since we're finished with the frame tell the processor to halt
        mySystem.m6507().halt();
      }
      break;
    }

    case 0x01:    // Vertical blank set-clear
    {
      // Is the dump to ground path being set for I0, I1, I2, and I3?
      if(!(myVBLANK & 0x80) && (value & 0x80))
      {
        // Yes, so notify the controllers
        mySystem.controller(0).configureTIA(Controller::EnableDump);
        mySystem.controller(1).configureTIA(Controller::EnableDump);
      }

      // Is the dump to ground path being removed from I0, I1, I2, and I3?
      if((myVBLANK & 0x80) && !(value & 0x80))
      {
        // Yes, so notify the controllers
        mySystem.controller(0).configureTIA(Controller::DisableDump);
        mySystem.controller(1).configureTIA(Controller::DisableDump);
      }

      myVBLANK = value;
      break;
    }

    case 0x02:    // Wait for leading edge of HBLANK
    {
      // Tell the cpu to waste the necessary amount of time
      waitHorizontalSync();

      // Update frame since we changed the cpu cycles
      updateFrame(mySystem.m6507().cycles() * 3);

      // Do we need to handle the TIA M0 "bug" used by Cosmic Ark?
      if(myM0CosmicArkMotionEnabled)
      {
        myPOSM0 -= 15;

        if(myPOSM0 >= 160)
          myPOSM0 -= 160;
        else if(myPOSM0 < 0)
          myPOSM0 += 160;

        myCurrentM0Mask = &ourMissleMaskTable[myPOSM0 & 0x03]
            [myNUSIZ0 & 0x07][(myNUSIZ0 & 0x30) >> 4][160 - (myPOSM0 & 0xFC)];
      } 
      break;
    }

    case 0x03:    // Reset horizontal sync counter
    {
//      cerr << "TIA Poke: " << hex << addr << endl;
      break;
    }

    case 0x04:    // Number-size of player-missle 0
    {
      myNUSIZ0 = value;
      myCurrentP0Mask = &ourPlayerMaskTable[myPOSP0 & 0x03]
          [myNUSIZ0 & 0x07][160 - (myPOSP0 & 0xFC)];
      myCurrentM0Mask = &ourMissleMaskTable[myPOSM0 & 0x03]
          [myNUSIZ0 & 0x07][(myNUSIZ0 & 0x30) >> 4][160 - (myPOSM0 & 0xFC)];

      break;
    }

    case 0x05:    // Number-size of player-missle 1
    {
      myNUSIZ1 = value;
      myCurrentP1Mask = &ourPlayerMaskTable[myPOSP1 & 0x03]
          [myNUSIZ1 & 0x07][160 - (myPOSP1 & 0xFC)];
      myCurrentM1Mask = &ourMissleMaskTable[myPOSM1 & 0x03]
          [myNUSIZ1 & 0x07][(myNUSIZ1 & 0x30) >> 4][160 - (myPOSM1 & 0xFC)];

      break;
    }

    case 0x06:    // Color-Luminance Player 0
    {
      uLong color = (uLong)value;
      myCOLUP0 = (((((color << 8) | color) << 8) | color) << 8) | color;
      break;
    }

    case 0x07:    // Color-Luminance Player 1
    {
      uLong color = (uLong)value;
      myCOLUP1 = (((((color << 8) | color) << 8) | color) << 8) | color;
      break;
    }

    case 0x08:    // Color-Luminance Playfield
    {
      uLong color = (uLong)value;
      myCOLUPF = (((((color << 8) | color) << 8) | color) << 8) | color;
      break;
    }

    case 0x09:    // Color-Luminance Background
    {
      uLong color = (uLong)value;
      myCOLUBK = (((((color << 8) | color) << 8) | color) << 8) | color;
      break;
    }

    case 0x0A:    // Control Playfield, Ball size, Collisions
    {
      myCTRLPF = value;
      myCurrentPFMask = ourPlayfieldTable[value & 0x01];
      myCurrentBLMask = &ourBallMaskTable[myPOSBL & 0x03]
          [(myCTRLPF & 0x30) >> 4][160 - (myPOSBL & 0xFC)];
      break;
    }

    case 0x0B:    // Reflect Player 0
    {
      // See if the reflection state of the player is being changed
      if(((value & 0x08) && !myREFP0) || (!(value & 0x08) && myREFP0))
      {
        myREFP0 = (value & 0x08);
        myCurrentGRP0 = ourPlayerReflectTable[myCurrentGRP0];
      }
      break;
    }

    case 0x0C:    // Reflect Player 1
    {
      // See if the reflection state of the player is being changed
      if(((value & 0x08) && !myREFP1) || (!(value & 0x08) && myREFP1))
      {
        myREFP1 = (value & 0x08);
        myCurrentGRP1 = ourPlayerReflectTable[myCurrentGRP1];
      }
      break;
    }

    case 0x0D:    // Playfield register byte 0
    {
      myPF = (myPF & 0x000FFFF0) | ((value >> 4) & 0x0F);
      break;
    }

    case 0x0E:    // Playfield register byte 1
    {
      myPF = (myPF & 0x000FF00F) | ((uLong)value << 4);
      break;
    }

    case 0x0F:    // Playfield register byte 2
    {
      myPF = (myPF & 0x00000FFF) | ((uLong)value << 12);
      break;
    }

    case 0x10:    // Reset Player 0
    {
//      int x = ((clock + 3 - myClockWhenFrameStarted) + 1 ) % 228 ;
      int x = ((clock + 3 - myClockWhenFrameStarted) + 2) % 228 ;

      myPOSP0 = x < HBLANK ? 3 : x - HBLANK;
      myCurrentP0Mask = &ourPlayerMaskTable[myPOSP0 & 0x03]
          [myNUSIZ0 & 0x07][160 - (myPOSP0 & 0xFC)];
      break;
    }

    case 0x11:    // Reset Player 1
    {
//      int x = ((clock + 3 - myClockWhenFrameStarted) + 1) % 228 ;
      int x = ((clock + 3 - myClockWhenFrameStarted) + 2) % 228 ;

      myPOSP1 = x < HBLANK ? 3 : x - HBLANK;
      myCurrentP1Mask = &ourPlayerMaskTable[myPOSP1 & 0x03]
          [myNUSIZ1 & 0x07][160 - (myPOSP1 & 0xFC)];
      break;
    }

    case 0x12:    // Reset Missle 0
    {
//      int x = ((clock + 3 - myClockWhenFrameStarted) + 2) % 228 ;
      int x = ((clock + 3 - myClockWhenFrameStarted) + 1 ) % 228;

      myPOSM0 = x < HBLANK ? 3 : x - HBLANK;
      myCurrentM0Mask = &ourMissleMaskTable[myPOSM0 & 0x03]
          [myNUSIZ0 & 0x07][(myNUSIZ0 & 0x30) >> 4][160 - (myPOSM0 & 0xFC)];
      break;
    }

    case 0x13:    // Reset Missle 1
    {
//      int x = ((clock - myClockWhenFrameStarted) + 3 ) % 228;
      int x = ((clock + 3 - myClockWhenFrameStarted) + 1 ) % 228;

      myPOSM1 = x < HBLANK ? 3 : x - HBLANK;
      myCurrentM1Mask = &ourMissleMaskTable[myPOSM1 & 0x03]
          [myNUSIZ1 & 0x07][(myNUSIZ1 & 0x30) >> 4][160 - (myPOSM1 & 0xFC)];
      break;
    }

    case 0x14:    // Reset Ball
    {
//      int x = ((clock - myClockWhenFrameStarted) + 3 ) % 228;
      int x = ((clock + 3 - myClockWhenFrameStarted) + 1 ) % 228;

      myPOSBL = x < HBLANK ? 2 : x - HBLANK;
      myCurrentBLMask = &ourBallMaskTable[myPOSBL & 0x03]
          [(myCTRLPF & 0x30) >> 4][160 - (myPOSBL & 0xFC)];
      break;
    }

    case 0x15:    // Audio control 0
    {
      mySound->set(Sound::AUDC0, value);
      break;
    }
  
    case 0x16:    // Audio control 1
    {
      mySound->set(Sound::AUDC1, value);
      break;
    }
  
    case 0x17:    // Audio frequency 0
    {
      mySound->set(Sound::AUDF0, value);
      break;
    }
  
    case 0x18:    // Audio frequency 1
    {
      mySound->set(Sound::AUDF1, value);
      break;
    }
  
    case 0x19:    // Audio volume 0
    {
      mySound->set(Sound::AUDV0, value);
      break;
    }
  
    case 0x1A:    // Audio volume 1
    {
      mySound->set(Sound::AUDV1, value);
      break;
    }

    case 0x1B:    // Graphics Player 0
    {
      // Set player 0 graphics
      myGRP0 = value;

      // Copy player 1 graphics into its delayed register
      myDGRP1 = myGRP1;

      // Get the "current" data for GRP0 base on delay register and reflect
      uByte grp0 = myVDELP0 ? myDGRP0 : myGRP0;
      myCurrentGRP0 = myREFP0 ? ourPlayerReflectTable[grp0] : grp0; 

      // Get the "current" data for GRP1 base on delay register and reflect
      uByte grp1 = myVDELP1 ? myDGRP1 : myGRP1;
      myCurrentGRP1 = myREFP1 ? ourPlayerReflectTable[grp1] : grp1; 

      // Set enabled object bits
      if(myCurrentGRP0 != 0)
        myEnabledObjects |= myP0Bit;
      else
        myEnabledObjects &= ~myP0Bit;

      if(myCurrentGRP1 != 0)
        myEnabledObjects |= myP1Bit;
      else
        myEnabledObjects &= ~myP1Bit;

      break;
    }

    case 0x1C:    // Graphics Player 1
    {
      // Set player 1 graphics
      myGRP1 = value;

      // Copy player 0 graphics into its delayed register
      myDGRP0 = myGRP0;

      // Copy ball graphics into its delayed register
      myDENABL = myENABL;

      // Get the "current" data for GRP0 base on delay register
      uByte grp0 = myVDELP0 ? myDGRP0 : myGRP0;
      myCurrentGRP0 = myREFP0 ? ourPlayerReflectTable[grp0] : grp0; 

      // Get the "current" data for GRP1 base on delay register
      uByte grp1 = myVDELP1 ? myDGRP1 : myGRP1;
      myCurrentGRP1 = myREFP1 ? ourPlayerReflectTable[grp1] : grp1; 

      // Set enabled object bits
      if(myCurrentGRP0 != 0)
        myEnabledObjects |= myP0Bit;
      else
        myEnabledObjects &= ~myP0Bit;

      if(myCurrentGRP1 != 0)
        myEnabledObjects |= myP1Bit;
      else
        myEnabledObjects &= ~myP1Bit;

      if(myVDELBL ? myDENABL : myENABL)
        myEnabledObjects |= myBLBit;
      else
        myEnabledObjects &= ~myBLBit;

      break;
    }

    case 0x1D:    // Enable Missle 0 graphics
    {
      myENAM0 = value & 0x02;

      if(myENAM0 && !myRESMP0)
        myEnabledObjects |= myM0Bit;
      else
        myEnabledObjects &= ~myM0Bit;
      break;
    }

    case 0x1E:    // Enable Missle 1 graphics
    {
      myENAM1 = value & 0x02;

      if(myENAM1 && !myRESMP1)
        myEnabledObjects |= myM1Bit;
      else
        myEnabledObjects &= ~myM1Bit;
      break;
    }

    case 0x1F:    // Enable Ball graphics
    {
      myENABL = value & 0x02;

      if(myVDELBL ? myDENABL : myENABL)
        myEnabledObjects |= myBLBit;
      else
        myEnabledObjects &= ~myBLBit;

      break;
    }

    case 0x20:    // Horizontal Motion Player 0
    {
      myHMP0 = ourMotionTable[(value >> 4) & 0x0F];
      break;
    }

    case 0x21:    // Horizontal Motion Player 1
    {
      myHMP1 = ourMotionTable[(value >> 4) & 0x0F];
      break;
    }

    case 0x22:    // Horizontal Motion Missle 0
    {
      Byte tmp = ourMotionTable[(value >> 4) & 0x0F];

      // Should we enabled TIA M0 "bug" used for stars in Cosmic Ark?
      if((clock == (myLastHMOVEClock + 21 * 3)) &&
          (myHMM0 == ourMotionTable[7]) && (tmp == ourMotionTable[6]))
      {
        myM0CosmicArkMotionEnabled = true;
      }

      myHMM0 = tmp;
      break;
    }

    case 0x23:    // Horizontal Motion Missle 1
    {
      myHMM1 = ourMotionTable[(value >> 4) & 0x0F];
      break;
    }

    case 0x24:    // Horizontal Motion Ball
    {
      myHMBL = ourMotionTable[(value >> 4) & 0x0F];
      break;
    }

    case 0x25:    // Vertial Delay Player 0
    {
      myVDELP0 = value & 0x01;

      uByte grp0 = myVDELP0 ? myDGRP0 : myGRP0;
      myCurrentGRP0 = myREFP0 ? ourPlayerReflectTable[grp0] : grp0; 

      if(myCurrentGRP0 != 0)
        myEnabledObjects |= myP0Bit;
      else
        myEnabledObjects &= ~myP0Bit;
      break;
    }

    case 0x26:    // Vertial Delay Player 1
    {
      myVDELP1 = value & 0x01;

      uByte grp1 = myVDELP1 ? myDGRP1 : myGRP1;
      myCurrentGRP1 = myREFP1 ? ourPlayerReflectTable[grp1] : grp1; 

      if(myCurrentGRP1 != 0)
        myEnabledObjects |= myP1Bit;
      else
        myEnabledObjects &= ~myP1Bit;
      break;
    }

    case 0x27:    // Vertial Delay Ball
    {
      myVDELBL = value & 0x01;

      if(myVDELBL ? myDENABL : myENABL)
        myEnabledObjects |= myBLBit;
      else
        myEnabledObjects &= ~myBLBit;
      break;
    }

    case 0x28:    // Reset missle 0 to player 0
    {
      if(myRESMP0 && !(value & 0x02))
      {
        uWord middle;

        if((myNUSIZ0 & 0x07) == 0x05)
          middle = 8;
        else if((myNUSIZ0 & 0x07) == 0x07)
          middle = 16;
        else
          middle = 4;

        myPOSM0 = (myPOSP0 + middle) % 160;
        myCurrentM0Mask = &ourMissleMaskTable[myPOSM0 & 0x03]
            [myNUSIZ0 & 0x07][(myNUSIZ0 & 0x30) >> 4][160 - (myPOSM0 & 0xFC)];
      }

      myRESMP0 = value & 0x02;

      if(myENAM0 && !myRESMP0)
        myEnabledObjects |= myM0Bit;
      else
        myEnabledObjects &= ~myM0Bit;

      break;
    }

    case 0x29:    // Reset missle 1 to player 1
    {
      if(myRESMP1 && !(value & 0x02))
      {
        uWord middle;

        if((myNUSIZ1 & 0x07) == 0x05)
          middle = 8;
        else if((myNUSIZ1 & 0x07) == 0x07)
          middle = 16;
        else
          middle = 4;

        myPOSM1 = (myPOSP1 + middle) % 160;
        myCurrentM1Mask = &ourMissleMaskTable[myPOSM1 & 0x03]
            [myNUSIZ1 & 0x07][(myNUSIZ1 & 0x30) >> 4][160 - (myPOSM1 & 0xFC)];
      }

      myRESMP1 = value & 0x02;

      if(myENAM1 && !myRESMP1)
        myEnabledObjects |= myM1Bit;
      else
        myEnabledObjects &= ~myM1Bit;
      break;
    }

    case 0x2A:    // Apply horizontal motion
    {
      myPOSP0 += myHMP0;
      myPOSP1 += myHMP1;
      myPOSM0 += myHMM0;
      myPOSM1 += myHMM1;
      myPOSBL += myHMBL;

      if(myPOSP0 >= 160)
        myPOSP0 -= 160;
      else if(myPOSP0 < 0)
        myPOSP0 += 160;

      if(myPOSP1 >= 160)
        myPOSP1 -= 160;
      else if(myPOSP1 < 0)
        myPOSP1 += 160;

      if(myPOSM0 >= 160)
        myPOSM0 -= 160;
      else if(myPOSM0 < 0)
        myPOSM0 += 160;

      if(myPOSM1 >= 160)
        myPOSM1 -= 160;
      else if(myPOSM1 < 0)
        myPOSM1 += 160;

      if(myPOSBL >= 160)
        myPOSBL -= 160;
      else if(myPOSBL < 0)
        myPOSBL += 160;

      myCurrentBLMask = &ourBallMaskTable[myPOSBL & 0x03]
          [(myCTRLPF & 0x30) >> 4][160 - (myPOSBL & 0xFC)];


      myCurrentP0Mask = &ourPlayerMaskTable[myPOSP0 & 0x03]
          [myNUSIZ0 & 0x07][160 - (myPOSP0 & 0xFC)];
      myCurrentP1Mask = &ourPlayerMaskTable[myPOSP1 & 0x03]
          [myNUSIZ1 & 0x07][160 - (myPOSP1 & 0xFC)];

      myCurrentM0Mask = &ourMissleMaskTable[myPOSM0 & 0x03]
          [myNUSIZ0 & 0x07][(myNUSIZ0 & 0x30) >> 4][160 - (myPOSM0 & 0xFC)];
      myCurrentM1Mask = &ourMissleMaskTable[myPOSM1 & 0x03]
          [myNUSIZ1 & 0x07][(myNUSIZ1 & 0x30) >> 4][160 - (myPOSM1 & 0xFC)];

      // Remember what clock HMOVE occured at
      myLastHMOVEClock = clock;

      // Disable TIA M0 "bug" used for stars in Cosmic ark
      myM0CosmicArkMotionEnabled = false;
      break;
    }

    case 0x2b:    // Clear horizontal motion registers
    {
      myHMP0 = 0;
      myHMP1 = 0;
      myHMM0 = 0;
      myHMM1 = 0;
      myHMBL = 0;
      break;
    }

    case 0x2c:    // Clear collision latches
    {
      myCollision = 0;
      break;
    }

    default:
    {
#ifdef DEBUG_ACCESSES
      cerr << "BAD TIA Poke: " << hex << addr << endl;
#endif
      break;
    }
  }
}

//============================================================================
// Ball mask table (entries are true or false)
//============================================================================
uByte TIA::ourBallMaskTable[4][4][320];

//============================================================================
// Used to set the collision register to the correct value
//============================================================================
uWord TIA::ourCollisionTable[64];

//============================================================================
// Missle mask table (entries are true or false)
//============================================================================
uByte TIA::ourMissleMaskTable[4][8][4][320];

//============================================================================
// Used to convert value written in a motion register into
// its internal representation
//============================================================================
Byte TIA::ourMotionTable[16] = {
    0, -1, -2, -3, -4, -5, -6, -7, 8, 7, 6, 5, 4, 3, 2, 1
};

//============================================================================
// Player mask table
//============================================================================
uByte TIA::ourPlayerMaskTable[4][8][320];

//============================================================================
// Used to reflect a players graphics
//============================================================================
uByte TIA::ourPlayerReflectTable[256];

//============================================================================
// Playfield mask table for reflected and non-reflected playfields
//============================================================================
uLong TIA::ourPlayfieldTable[2][160];

