//============================================================================
//
//    SSSS    tt          lll  lll              
//   SS  SS   tt           ll   ll                
//   SS     tttttt  eeee   ll   ll   aaaa    "An Atari 2600 VCS Emulator"
//    SSSS    tt   ee  ee  ll   ll      aa      
//       SS   tt   eeeeee  ll   ll   aaaaa   Copyright (c) 1995,1996,1997
//   SS  SS   tt   ee      ll   ll  aa  aa         Bradford W. Mott
//    SSSS     ttt  eeeee llll llll  aaaaa    
//
//============================================================================

#ifndef SYSTEM_HXX
#define SYSTEM_HXX

#include <stdlib.h>
#include <iostream.h>

class System;
class M6507;
class M6532;
class TIA;
class Cartridge;
class BasicTerminal;

#include "machine.hxx"
#include "Device.hxx"
#include "Console.hxx"
#include "Control.hxx"
#include "Props.hxx"

/**
  The 2600 System class.

  @author  Bradford W. Mott
  @version $Id: System.hxx,v 1.2 1997/05/17 19:00:08 bwmott Exp $
*/
class System
{
  public:
    /// Constructor
    System(BasicTerminal& terminal, const Properties& props);
 
    /// Destructor
    ~System();

  public:
    /// Reset myself and all my components
    void reset();

  public: 
    struct Map
    {
      uByte* peekBase;
      uWord* peekOffset;
      uByte* pokeBase;
      Device* device;
    };

    /// Map all peeks at address to the given device using base and offset
    void mapPeek(uWord address, Device& device, uByte* base, uWord* offset);

    /// Map all peeks at address to the given device
    void mapPeek(uWord address, Device& device);

    /// Map all pokes at address to the given device using base and offset
    void mapPoke(uWord address, Device& device, uByte* base);

    /// Map all pokes at address to the given device
    void mapPoke(uWord address, Device& device);

    /// Remove any mappings associated with the given device
    void unmap(Device& device);

  public:
    /// Answer the uByte at the given address
    uByte peek(uWord address);

    /// Answer the uByte at the given address
    uByte fastPeek(uWord address);

    /// Answer the uWord at the given address
    uWord fastDPeek(uWord address);

    /// Write value at the given address.
    void poke(uWord address, uByte value);

  public:
    /// Answer the effective number of frames which have been displayed
    uLong numberOfFrames() const { return myNumberOfFrames; }

    /// Answer a reference to my controller (port should be 0 or 1)
    Controller& controller(uByte port) const;

    /// Answer my console switches
    Console& console() { return myConsole; }

    /// Answer my terminal
    BasicTerminal& terminal() const { return myTerminal; }

    /// Answer my 6507
    M6507& m6507() const { return *myM6507; }

    /// Answer my 6532 (PIA)
    M6532& m6532() const { return *myM6532; }

    /// Answer my TIA
    TIA& tia() const { return *myTIA; }

  public:
    /// Answer my properties
    const Properties& properties() const { return *myProperties; }

    /// Change my properties and configure controllers
    void properties(const Properties& props);

    /// Answer my cartridge
    Cartridge& cartridge() const { return *myCartridge; }

    /// Change my cartridge
    void cartridge(Cartridge& cart) { myCartridge = &cart; }

  public:
    /// This method should be called on a regular basis to update the system
    void update();

    /// Throttle the frame rate so games don't run to fast
    void throttle();

    /// Reset the throttle variables based on current system state
    void throttleReset();

  private:
    // Pointer to array of maps
    Map* myMap;

    // Dummy data for fast peeks to bad addresses
    uWord myUnmappedDummyData;
    uWord myUnmappedDummyDataOffset;
 
  private:
    Cartridge* myCartridge;
    Controller* myController[2];
    const Properties* myProperties;
    BasicTerminal& myTerminal;

    M6507* myM6507;
    M6532* myM6532;
    TIA* myTIA; 

    Console myConsole;
    uWord myMinimumFrameRate;
    uWord myFrameRateMultiple;
    uLong myNumberOfFrames;

    double myClocksPerThrottle;
    double myCyclesPerThrottle;
    double myCycleLimit;
    double myClockLimit;
};

inline uByte System::peek(uWord address)
{
  Map* map = &myMap[address & 0x1fff];

  if(map->peekBase != (uByte*)&myUnmappedDummyData)
    return *(map->peekBase + *(map->peekOffset));
  else
    return map->device->peek(address & 0x1fff);
}
 
inline uByte System::fastPeek(uWord address)
{
  Map* map = &myMap[address & 0x1fff];

  return *(map->peekBase + *(map->peekOffset));
}

inline uWord System::fastDPeek(uWord address)
{
  Map* map = &myMap[address & 0x1fff];

#ifdef LSB_FIRST
  return *((uWord*)(map->peekBase + *(map->peekOffset)));
#else
  uWord value = *(map->peekBase + *(map->peekOffset));
  return value | (((uWord)*(map->peekBase + *(map->peekOffset) + 1)) << 8);
#endif
}

inline void System::poke(uWord address, uByte value)
{
  Map* map = &myMap[address & 0x1fff];

  if(map->pokeBase != (uByte*)&myUnmappedDummyData)
    *(map->pokeBase) = value;
  else
    map->device->poke(address & 0x1fff, value);
}

#include "M6507.hxx"
#include "M6532.hxx"
#include "TIA.hxx"
#include "Cart.hxx"

#endif

