//============================================================================
//
//    SSSS    tt          lll  lll              
//   SS  SS   tt           ll   ll                
//   SS     tttttt  eeee   ll   ll   aaaa    "An Atari 2600 VCS Emulator"
//    SSSS    tt   ee  ee  ll   ll      aa      
//       SS   tt   eeeeee  ll   ll   aaaaa   Copyright (c) 1995,1996,1997
//   SS  SS   tt   ee      ll   ll  aa  aa         Bradford W. Mott
//    SSSS     ttt  eeeee llll llll  aaaaa    
//
//============================================================================

/**
  The 8K cartridge class for Tigervision games.  There are four
  2K banks while two 2K segments are visible to the 2600.  The
  last 2K bank is always mapped into the second segment.  Banks
  can be switched by preforming an STA $3F instruction with the
  bank number in the lower two bits of the acuumulator.
  
  @author  Bradford W. Mott
  @version $Id: Cart3F.cxx,v 1.1 1997/05/17 19:04:12 bwmott Exp $
*/

#include "Cart3F.hxx"
#include "System.hxx"

//============================================================================
// Constructor
//============================================================================
Cartridge3F::Cartridge3F(System& system, uByte* image)
    : Cartridge(system)
{
  uWord addr;

  myImageOffset[0] = 0;
  myImageOffset[1] = 3 * 2048;

  // Copy the ROM image into my buffer
  for(addr = 0; addr < 8192; ++addr)
    myImage[addr] = image[addr];

  // Map all of my addresses in the system
  for(addr = 0; addr < 8192; ++addr)
  {
    if(addr & 0x1000)
    {
      // Which segment is this address in?
      if((addr & 0x0FFF) < 0x0800)
      {
        mySystem.mapPeek(addr, *this, 
            &myImage[addr & 0x07ff], &myImageOffset[0]);
      }
      else
      {
        mySystem.mapPeek(addr, *this, 
            &myImage[addr & 0x07ff], &myImageOffset[1]);
      }
      mySystem.mapPoke(addr, *this);
    }
  }

  // Reset myself
  reset();
}
 
//============================================================================
// Destructor
//============================================================================
Cartridge3F::~Cartridge3F()
{
}

//============================================================================
// Answer the byte at the given address
//============================================================================
uByte Cartridge3F::peek(uWord addr)
{
  if((addr & 0x0FFF) < 0x0800)
    return myImage[(addr & 0x07ff) + myImageOffset[0]];
  else
    return myImage[(addr & 0x07ff) + myImageOffset[1]];
}

//============================================================================
// Store value in the given address
//============================================================================
void Cartridge3F::poke(uWord addr, uByte value)
{
  // Are they try to switch banks?
  if(addr == 0x003F)
  {
    myImageOffset[0] = (value & 0x03) * 2048;
  }
}

//============================================================================
// Reset to power on state
//============================================================================
void Cartridge3F::reset()
{
  myImageOffset[0] = 0;
  myImageOffset[1] = 3 * 2048;

  // Still the 0x3F address from the TIA so we can bankswitch 
  // whenever it's written to
  mySystem.mapPoke(0x3F, *this);
}

