/*
 * cw - filter program to output morse code on system console
 */

/* trap for when no valid platform selected */
#if !defined(LINUX) && !defined(SCO) && !defined(UNIXWARE)
#include <-DLINUX,-DSCO,_or_-DUNIXWARE_must_be_defined>
#endif

/* include files */
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <sys/time.h>
#include <sys/param.h>
#include <signal.h>
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>

/* conditional include files */
#if defined(LINUX)
#	include <strings.h>
#	include <getopt.h>
#	include <linux/kd.h>
#endif
#if defined(SCO)
#	include <string.h>
#	include <limits.h>
#	define	MAXPATHLEN	_POSIX_PATH_MAX
#	include <sys/vtkd.h>
#endif
#if defined(UNIXWARE)
#	include <string.h>
#	include <sys/kd.h>
#endif

/* include definitions of CW control parameters */
#include "cw.h"

/* definitions */
#define	VERSION		"cw version 1.1"

#define	bool		int			/* boolean type */
#define	FALSE		0
#define TRUE		(!FALSE)

#define	EOS		'\0'			/* end of string */
#define	ASC_SPACE	' '			/* ASCII space char */
#define	ASC_FNS		'/'			/* ASCII filename sep char */
#define STDIN		0			/* stdin stream */
#define STDOUT		1			/* stdout stream */
#define STDERR		2			/* stderr stream */

#define	SILENT		0			/* 0Hz = silent 'tone' */

#define TONE_DIV 	1190000			/* kernel tone delay */
#define	DOT_LEN(WPM,A)	( 1200 / WPM + ( 1200 / WPM ) * A / 100 )
						/* dot length fm WPM adjusted */
#define	DASH_LEN(WPM,A)	( 3 * DOT_LEN( WPM, A ))/* dash is 3 * dot */
#define EOE_LEN(WPM,A)	DOT_LEN( WPM, A )	/* end-of-element is 1 * dot:
						   synonym for DOT_LEN */
#define EOC_LEN(WPM,A)	( 2 * DOT_LEN( WPM, A ))/* end-of-char is 3 * dot:
						   2 + EOE already done */
#define EOW_LEN(WPM,A)	( 4 * DOT_LEN( WPM, A ))/* end-of-word is 7 * dot:
						   4 + EOC already done */

#define MBUF_LEN	20		/* general messages buffer length */
#define ASC_WHITESPACE	" \t\n\r"	/* characters treated as whitespace */

#define	MAXARGS		128		/* args to exec */
#define	MAXOPTSTR	1024		/* longest _OPTIONS env variable */
#define	ARGS_WHITESPACE	" \t"		/* argument separators */
#define	ENV_OPTIONS	"CW_OPTIONS"	/* env string holding options */

#define	INIT_TONE	800		/* initial tone in Hz */
#define	INIT_WPM	12		/* initially 12 WPM */
#define	INIT_ADJ	0		/* initially no speed adjustment */
#define	INIT_GAP	0		/* initially no 'Farnsworth' */


/* global variables, alterable by embedded commands */
static int console	= STDOUT;	/* initially sound on stdout */
static int hz		= INIT_TONE;	/* initially 800 Hz */
static int wpm		= INIT_WPM;	/* initially 12 WPM */
static int adj		= INIT_ADJ;	/* percentage adjustment to timing */
static int gap		= INIT_GAP;	/* initial extra gap between chars */
static bool doecho	= TRUE;		/* initially echoing characters */
static bool doerrs	= TRUE;		/* print error messages to stderr */
static bool docmds	= TRUE;		/* execute embedded commands */
static bool docombo	= TRUE;		/* execute [...] combinations */
static bool docomment	= TRUE;		/* allow {...} as comments */

/* morse code characters table */
static struct {
	char	key;
	char	*cw;
} cw_chars[] = {
	'A',	".-"	, 'B',	"-..."	, 'C',	"-.-."	,
	'D',	"-.."	, 'E',	"."	, 'F',	"..-."	,
	'G',	"--."	, 'H',	"...."	, 'I',	".."	,
	'J',	".---"	, 'K',	"-.-"	, 'L',	".-.."	,
	'M',	"--"	, 'N',	"-."	, 'O',	"---"	,
	'P',	".--."	, 'Q',	"--.-"	, 'R',	".-."	,
	'S',	"..."	, 'T',	"-"	, 'U',	"..-"	,
	'V',	"...-"	, 'W',	".--"	, 'X',	"-..-"	,
	'Y',	"-.--"	, 'Z',	"--.."	,
	'0',	"-----"	, '1',	".----"	, '2',	"..---"	,
	'3',	"...--"	, '4',	"....-"	, '5',	"....."	,
	'6',	"-...."	, '7',	"--..."	, '8',	"---.."	,
	'9',	"----."	,
	'"',	".-..-.", '\'', ".----.", '$',	"...-..-",
	'(',	"-.--.",  ')',	"-.--.-", '+',	".-.-.",
	',',	"--..--", '-',  "-....-", '.',	".-.-.-",
	'/',	"-..-.",  ':',	"---...", ';',	"-.-.-.",
	'=',	"-...-",  '?',	"..--..", '_',	"..--.-",
	EOS,	""	};

/*
 * print_help - give help information
 */
static void print_help( char *pname ) {
	printf(
#if defined(LINUX)
	"Usage: %s [options...]\n\n\
	-d, --device=DEVICE	use DEVICE for sound ioctl (default stdout)\n\
	-t, --hz,--tone=HZ	set initial tone to HZ (default %d)\n\
				valid HZ values are between %d and %d\n\
	-w, --wpm=WPM		set initial words per minute (default %d)\n\
				valid WPM values are between %d and %d\n\
	-g, --gap=GAP		set extra gap between letters (default %d)\n\
				valid GAP values are between %d and %d\n\
	-a, --adj=ADJ		adjust speed by +/-ADJ %% (default %d)\n\
				valid ADJ values are between %d and %d\n\
	-e, --noecho		don't echo sending to stdout (default echo)\n\
	-m, --nomsgs		don't write messages to stderr (default msgs)\n\
	-c, --nocmds		don't execute embedded commands (default cmds)\n\
	-o, --nocombo		don't allow [...] combinations (default combo)\n\
	-p, --nocomments	don't allow {...} comments (default comments)\n\
	-h, --help		print this message\n\
	-v, --version		output version information and exit\n\n",
#endif
#if defined(SCO) || defined(UNIXWARE)
	"Usage: %s [options...]\n\n\
	-d DEVICE	use DEVICE for sound ioctl (default stdout)\n\
	-t HZ		set initial tone to HZ (default %d)\n\
			valid HZ values are between %d and %d\n\
	-w WPM		set initial words per minute (default %d)\n\
			valid WPM values are between %d and %d\n\
	-g GAP		set extra gap between letters (default %d)\n\
			valid GAP values are between %d and %d\n\
	-a ADJ		adjust speed by +/-ADJ %% (default %d)\n\
			valid ADJ values are between %d and %d\n\
	-e		don't echo sending to stdout (default echo)\n\
	-m		don't write messages to stderr (default msgs)\n\
	-c		don't execute embedded commands (default cmds)\n\
	-o		don't allow [...] combinations (default combo)\n\
	-p		don't allow {...} comments (default comments)\n\
	-h		print this message\n\
	-v		output version information and exit\n\n",
#endif
	pname,
	INIT_TONE, CW_MIN_TONE, CW_MAX_TONE,
	INIT_WPM, CW_MIN_WPM, CW_MAX_WPM,
	INIT_GAP, CW_MIN_GAP, CW_MAX_GAP,
	INIT_ADJ, CW_MIN_ADJ, CW_MAX_ADJ );
}


/*
 * parse_cmdline - parse command line options for initial values
 */
static void parse_cmdline( int argc, char **argv ) {
	int	c;				/* option character */
	char	*argv0;				/* program name */
	int	argind;				/* loop index */
	char	env_options[MAXOPTSTR];		/* env options string */
	char	*sptr;				/* string pointer */
	char	*local_argv[MAXARGS];		/* local argv array */
	int	local_argc = 0;			/* local argc */
	char	device[MAXPATHLEN];		/* path to device file */
#if defined(LINUX)
	int	option_index;			/* option index */
	static struct option long_options[] = {	/* options table */
		{ "device",	1, 0, 0 },
		{ "tone",	1, 0, 0 },
		{ "hz",		1, 0, 0 },
		{ "wpm",	1, 0, 0 },
		{ "gap",	1, 0, 0 },
		{ "adj",	1, 0, 0 },
		{ "noecho",	0, 0, 0 },
		{ "nomsgs",	0, 0, 0 },
		{ "nocmds",	0, 0, 0 },
		{ "nocombo",	0, 0, 0 },
		{ "nocomments",	0, 0, 0 },
		{ "help",	0, 0, 0 },
		{ "version",	0, 0, 0 },
		{ 0, 0, 0, 0 }};
#endif

	/* set argv0 to be the basename part of the program name */
	argv0 = argv[0] + strlen( argv[0] );
	while ( *argv0 != ASC_FNS && argv0 > argv[0] )
		argv0--;
	if ( *argv0 == ASC_FNS ) argv0++;

	/* build a new view of argc and argv by first prepending
	   the strings from ..._OPTIONS, if defined, then putting the
	   command line args on (so they take precedence) */
	local_argv[ local_argc++ ] = argv[0];
	if ( getenv( ENV_OPTIONS ) != NULL ) {
		strcpy( env_options, getenv( ENV_OPTIONS ));
		sptr = env_options;
		while ( local_argc < MAXARGS - 1 ) {
			for ( ; strchr( ARGS_WHITESPACE, *sptr ) != (char *)NULL
					&& *sptr != EOS;
					sptr++ );
			if ( *sptr == EOS )
				break;
			else {
				local_argv[ local_argc++ ] = sptr;
				for ( ; strchr( ARGS_WHITESPACE, *sptr )
						== (char *)NULL && *sptr != EOS;
						sptr++ );
				if ( strchr( ARGS_WHITESPACE, *sptr )
						!= (char *)NULL && *sptr != EOS ) {
					*sptr = EOS; sptr++;
				}
			}
		}
	}
	for ( argind = 1; argind < argc; argind++ ) {
		local_argv[ local_argc++ ] = argv[ argind ];
	}

	/* process every option */
#if defined(LINUX)
	while ( (c=getopt_long( local_argc, local_argv, "d:t:w:g:a:emcohv",
			long_options, &option_index )) != -1 ) {
#endif
#if defined(SCO) || defined(UNIXWARE)
	while ( (c=getopt( local_argc, local_argv, "d:t:w:g:a:emcohv" ))
			!= -1 ) {
#endif

		/* check for -d or --device argument */
		if ( c == 'd'
#if defined(LINUX)
			|| c == 0 && !strcmp( long_options[option_index].name,
				"device" )) {
#endif
#if defined(SCO) || defined(UNIXWARE)
					) {
#endif
			if ( sscanf( optarg, "%s", device ) != 1 ) {
				fprintf( stderr, "%s: invalid device value\n",
						argv0 );
				exit( 1 );
			}
			/* open device */
			if ( (console = open( device, O_RDWR )) == -1 ) {
				fprintf( stderr, "%s: error opening output device\n",
						argv0 );
				perror( device ); exit( 1 );
			}
		}

		/* check for -t or --tone/--hz argument */
		else if ( c == 't'
#if defined(LINUX)
			|| c == 0 && ( !strcmp( long_options[option_index].name,
					"tone" ) ||
				    !strcmp( long_options[option_index].name,
					"hz" ))) {
#endif
#if defined(SCO) || defined(UNIXWARE)
					) {
#endif
			if ( sscanf( optarg, "%d", &hz ) != 1 ||
					hz < CW_MIN_TONE || hz > CW_MAX_TONE ) {
				fprintf( stderr, "%s: invalid tone value\n",
						argv0 );
				exit( 1 );
			}
		}

		/* check for -w or --wpm argument */
		else if ( c == 'w'
#if defined(LINUX)
			|| c == 0 && !strcmp( long_options[option_index].name,
				"wpm" )) {
#endif
#if defined(SCO) || defined(UNIXWARE)
					) {
#endif
			if ( sscanf( optarg, "%d", &wpm ) != 1 ||
					wpm < CW_MIN_WPM || wpm > CW_MAX_WPM ) {
				fprintf( stderr, "%s: invalid wpm value\n",
						argv0 );
				exit( 1 );
			}
		}

		/* check for -g or --gap argument */
		else if ( c == 'g'
#if defined(LINUX)
			|| c == 0 && !strcmp( long_options[option_index].name,
				"gap" )) {
#endif
#if defined(SCO) || defined(UNIXWARE)
					) {
#endif
			if ( sscanf( optarg, "%d", &gap ) != 1 ||
					gap < CW_MIN_GAP || gap > CW_MAX_GAP ) {
				fprintf( stderr, "%s: invalid gap value\n",
						argv0 );
				exit( 1 );
			}
		}

		/* check for -a or --adj argument */
		else if ( c == 'a'
#if defined(LINUX)
			|| c == 0 && !strcmp( long_options[option_index].name,
				"adj" )) {
#endif
#if defined(SCO) || defined(UNIXWARE)
					) {
#endif
			if ( sscanf( optarg, "%d", &adj ) != 1 ||
					adj < CW_MIN_ADJ || adj > CW_MAX_ADJ ) {
				fprintf( stderr, "%s: invalid adj value\n",
						argv0 );
				exit( 1 );
			}
		}

		/* check for -e or --noecho argument */
		else if ( c == 'e'
#if defined(LINUX)
			|| c == 0 && !strcmp( long_options[option_index].name,
				"noecho" )) {
#endif
#if defined(SCO) || defined(UNIXWARE)
					) {
#endif
			doecho = FALSE;
		}

		/* check for -m or --nomsgs argument */
		else if ( c == 'm'
#if defined(LINUX)
			|| c == 0 && !strcmp( long_options[option_index].name,
				"nomsgs" )) {
#endif
#if defined(SCO) || defined(UNIXWARE)
					) {
#endif
			doerrs = FALSE;
		}

		/* check for -c or --nocmds argument */
		else if ( c == 'c'
#if defined(LINUX)
			|| c == 0 && !strcmp( long_options[option_index].name,
				"nocmds" )) {
#endif
#if defined(SCO) || defined(UNIXWARE)
					) {
#endif
			docmds = FALSE;
		}

		/* check for -o or --nocombo argument */
		else if ( c == 'o'
#if defined(LINUX)
			|| c == 0 && !strcmp( long_options[option_index].name,
				"nocombo" )) {
#endif
#if defined(SCO) || defined(UNIXWARE)
					) {
#endif
			docombo = FALSE;
		}

		/* check for -p or --nocomments argument */
		else if ( c == 'p'
#if defined(LINUX)
			|| c == 0 && !strcmp( long_options[option_index].name,
				"nocomments" )) {
#endif
#if defined(SCO) || defined(UNIXWARE)
					) {
#endif
			docomment = FALSE;
		}

		/* check for -h or --help argument */
		else if ( c == 'h'
#if defined(LINUX)
			|| c == 0 && !strcmp( long_options[option_index].name,
				"help" )) {
#endif
#if defined(SCO) || defined(UNIXWARE)
					) {
#endif
			print_help( argv0 );
			exit( 0 );
		}

		/* check for -v or --version argument */
		else if ( c == 'v'
#if defined(LINUX)
			|| c == 0 && !strcmp( long_options[option_index].name,
				"version" )) {
#endif
#if defined(SCO) || defined(UNIXWARE)
					) {
#endif
			printf( "%s\n", VERSION );
			exit( 0 );
		}

		/* check for illegal option */
		else if ( c == '?' ) {
			fprintf( stderr,
#if defined(LINUX)
				"Try '%s --help' for more information.\n",
#endif
#if defined(SCO) || defined(UNIXWARE)
				"Try '%s -h' for more information.\n",
#endif
				argv0 );
			exit( 1 );
		}

		/* nothing else left to do */
		else {
			fprintf( stderr, "%s: getopts returned %c\n",
					argv0, c );
			exit( 1 );
		}

	}
	if ( optind != local_argc ) {
		fprintf( stderr,
#if defined(LINUX)
			"Try '%s --help' for more information.\n",
#endif
#if defined(SCO) || defined(UNIXWARE)
			"Try '%s -h' for more information.\n",
#endif
			argv0 );
		exit( 1 );
	}

	/* finally, check that the console will do sound */
	if ( ioctl( console, KIOCSOUND, 0 ) == -1 ) {
		fprintf( stderr, "%s: output device won't do sound\n",
				argv0 );
		perror( device ); exit( 1 );
	}
}


/*
 * send_tone - send tone for specified number of milliseconds.  This
 *             routine starts an itimer, then exits to allow the program
 *             to get on with something else.  When called next, if
 *             still in the last timer interval it waits.  That way,
 *             the program can keep one delay ahead of what's happening.
 */
static	in_tone		= FALSE;	/* set if already inside a tone */
static	handler_regd	= FALSE;	/* set if a handler is registered */
static void al_handler( int sig ) {	/* largely a dummy routine */
	in_tone = FALSE;
}
static void send_tone( int ms, int hz ) {
	struct itimerval	it;	/* itimer control structure */
	struct sigaction	sa;	/* sigaction structure */

	/* if already in some tone, wait for it to finish */
	while ( in_tone ) {
		pause();
	}

	/* start the tone */
	ioctl( console, KIOCSOUND, (hz > 0) ? (TONE_DIV / hz) : 0 );

	/* if milliseconds is zero, don't do anything - so a call
	   send_tone(0,0) finishes up the current tone, switches
	   sound off, and exits without starting anything new */
	if ( ms > 0 ) {

		/* note that we're inside a tone delay */
		in_tone = TRUE;

		/* register dummy handler as signal catcher if not done yet */
		if ( ! handler_regd ) {
			sa.sa_handler	= al_handler;
			sa.sa_flags	= SA_RESTART;
			sigemptyset( &sa.sa_mask );
			if ( sigaction( SIGALRM, &sa, NULL ) == -1 ) {
				perror( "sigaction" ); exit( 1 );
			}
			handler_regd = TRUE;
		}

		/* set itimer to interrupt ONCE after specified milliseconds */
		it.it_interval.tv_sec	= 0L;
		it.it_interval.tv_usec	= 0L;
		it.it_value.tv_sec	= (long)(ms * 1000L) / 1000000L;
		it.it_value.tv_usec	= (long)(ms * 1000L) % 1000000L;
		if ( setitimer( ITIMER_REAL, &it, (struct itimerval*)0 )
				== -1 ) {
			perror( "setitimer" ); exit( 1 );
		}
	}

	/* rush off to do something else before the timer expires */
}


/*
 * echo_char - echo a character unless echoing suppressed
 */
static void echo_char( char c ) {
	if ( doecho ) {
		fprintf( stdout, "%c", c ); fflush( stdout );
	}
}


/*
 * err_msg - print an error message string on stderr, prefixed as
 *           reqd, and ending with a newline
 */
static void err_msg( char prefix, char *s ) {
	if ( doerrs ) {
		fprintf( stderr, "%c%s\n", prefix, s ); fflush( stderr );
	}
}


/*
 * send_char - routine to send a given ASCII character as cw, and echo
 *             to stdout, and optionally wait for the post-character gap
 */
static void send_char( char c, bool combo_char ) {
	char	ebuf[MBUF_LEN];		/* error message buffer */
	int	tab_index;		/* cw lookup table index */
	int	str_index;		/* cw string index */

	/* ensure uppercase */
	c = toupper( c );

	/* if character is a space simply delay end-of-word then return */
	if ( c == ASC_SPACE ) {
		echo_char( c );
		send_tone( EOW_LEN( wpm, adj )
				+ gap * DOT_LEN( wpm, adj ), SILENT );
		return;
	}

	/* check table to find character to send */
	for ( tab_index = 0; cw_chars[tab_index].key != EOS; tab_index++ ) {

		/* if found, sound character */
		if ( cw_chars[tab_index].key == c ) {

			/* sound the elements of the cw equivalent */
			for ( str_index = 0;
				str_index < strlen( cw_chars[tab_index].cw );
				str_index++ ) {

				/* send a tone of dot or dash length */
				if ( cw_chars[tab_index].cw[str_index] == '-' )
					send_tone( DASH_LEN( wpm, adj ), hz );
				else
					send_tone(  DOT_LEN( wpm, adj ), hz );

				/* delay one dot length */
				send_tone( EOE_LEN( wpm, adj ), SILENT );
			}

			/* echo the character and return */
			echo_char( c );

			/* delay for end of character length, plus any
			   extra 'Farnsworth' delay, unless multi-char item */
			if ( ! combo_char )
				send_tone( EOC_LEN( wpm, adj )
						+ gap * DOT_LEN( wpm, adj ),
						SILENT );
			return;
		}

	}

	/* not found - print an error message */
	sprintf( ebuf, "%c", c );
	err_msg( CW_STATUS_ERR, ebuf );
}


/*
 * send_string - routine to send a given ASCII string as cw, and echo
 *               to stdout; can also send multi-char items
 */
static void send_string( char *s, bool combo_char ) {
	int	str_index;		/* send string index */

	/* simply send every character in the string */
	for ( str_index = 0; str_index < strlen( s ); str_index++ ) {
		send_char( s[str_index], combo_char );
	}
}


/*
 * parse_stdin - read characters from stdin, and either sound them
 *               or interpret controls in them
 */
static void parse_stdin() {
	char	c;			/* character from stdin */
	int	i;			/* param value in embedded commands */
	char	ebuf[MBUF_LEN];		/* error message buffer */
	bool	in_combination = FALSE;	/* if inside [...] combination */
	bool	in_comment = FALSE;	/* if inside {...} comment */

	/* read characters from stdin until end of stream */
	while ( TRUE ) {

		/* get a character, and check for end of file */
		c = getchar();
		if ( c == EOF )
			goto loop_end;
	
		/* check for comment start */
		if ( c == CW_COMMENT_START && docomment ) {
			in_comment = TRUE;
			echo_char( c ); continue;
		}
		if ( c == CW_COMMENT_END && docomment ) {
			in_comment = FALSE;
			echo_char( c ); continue;
		}

		/* if in comment just echo the character */
		if ( in_comment ) {
			echo_char( c ); continue;
		}

		/* not in comment, or beginning or ending a comment, so
		   convert the character to uppercase and look for
		   other options */
		c = toupper( c );

		/* check for command start */
		if ( c == CW_CMD_ESCAPE && docmds ) {

			/* get command identifier */
			c = toupper( getchar() );
			switch ( c ) {
			case EOF:
				goto loop_end;

			case CW_CMDV_TONE:
				if ( scanf( "%d;", &i ) != 1 ) {
					sprintf( ebuf, "%c", c );
					err_msg( CW_STATUS_ERR, ebuf );
					continue;
				}
				if ( i < CW_MIN_TONE || i > CW_MAX_TONE ) {
					sprintf( ebuf, "%c%d", c, i );
					err_msg( CW_STATUS_ERR, ebuf );
					continue;
				}
				hz = i;
				sprintf( ebuf, "%c%d", c, hz );
				err_msg( CW_STATUS_OK, ebuf );
				break;

			case CW_CMDV_WPM:
				if ( scanf( "%d;", &i ) != 1 ) {
					sprintf( ebuf, "%c", c );
					err_msg( CW_STATUS_ERR, ebuf );
					continue;
				}
				if ( i < CW_MIN_WPM || i > CW_MAX_WPM ) {
					sprintf( ebuf, "%c%d", c, i );
					err_msg( CW_STATUS_ERR, ebuf );
					continue;
				}
				wpm = i;
				sprintf( ebuf, "%c%d", c, wpm );
				err_msg( CW_STATUS_OK, ebuf );
				break;

			case CW_CMDV_GAP:
				if ( scanf( "%d;", &i ) != 1 ) {
					sprintf( ebuf, "%c", c );
					err_msg( CW_STATUS_ERR, ebuf );
					continue;
				}
				if ( i < CW_MIN_GAP || i > CW_MAX_GAP ) {
					sprintf( ebuf, "%c%d", c, i );
					err_msg( CW_STATUS_ERR, ebuf );
					continue;
				}
				gap = i;
				sprintf( ebuf, "%c%d", c, gap );
				err_msg( CW_STATUS_OK, ebuf );
				break;

			case CW_CMDV_ADJ:
				if ( scanf( "%d;", &i ) != 1 ) {
					sprintf( ebuf, "%c", c );
					err_msg( CW_STATUS_ERR, ebuf );
					continue;
				}
				if ( i < CW_MIN_ADJ || i > CW_MAX_ADJ ) {
					sprintf( ebuf, "%c%d", c, i );
					err_msg( CW_STATUS_ERR, ebuf );
					continue;
				}
				adj = i;
				sprintf( ebuf, "%c%d", c, adj );
				err_msg( CW_STATUS_OK, ebuf );
				break;

			case CW_CMDV_ECHO:
				if ( scanf( "%d;", &i ) != 1 ) {
					sprintf( ebuf, "%c", c );
					err_msg( CW_STATUS_ERR, ebuf );
					continue;
				}
				doecho = i ? TRUE : FALSE;
				sprintf( ebuf, "%c%d", c, doecho );
				err_msg( CW_STATUS_OK, ebuf );
				break;

			case CW_CMDV_MSGS:
				if ( scanf( "%d;", &i ) != 1 ) {
					sprintf( ebuf, "%c", c );
					err_msg( CW_STATUS_ERR, ebuf );
					continue;
				}
				doerrs = i ? TRUE : FALSE;
				sprintf( ebuf, "%c%d", c, doerrs );
				err_msg( CW_STATUS_OK, ebuf );
				break;

			case CW_CMDV_CMDS:
				if ( scanf( "%d;", &i ) != 1 ) {
					sprintf( ebuf, "%c", c );
					err_msg( CW_STATUS_ERR, ebuf );
					continue;
				}
				docmds = i ? TRUE : FALSE;
				sprintf( ebuf, "%c%d", c, docmds );
				err_msg( CW_STATUS_OK, ebuf );
				break;

			case CW_CMDV_COMBO:
				if ( scanf( "%d;", &i ) != 1 ) {
					sprintf( ebuf, "%c", c );
					err_msg( CW_STATUS_ERR, ebuf );
					continue;
				}
				docombo = i ? TRUE : FALSE;
				sprintf( ebuf, "%c%d", c, docombo );
				err_msg( CW_STATUS_OK, ebuf );
				break;

			case CW_CMDV_COMMENT:
				if ( scanf( "%d;", &i ) != 1 ) {
					sprintf( ebuf, "%c", c );
					err_msg( CW_STATUS_ERR, ebuf );
					continue;
				}
				docomment = i ? TRUE : FALSE;
				sprintf( ebuf, "%c%d", c, docomment );
				err_msg( CW_STATUS_OK, ebuf );
				break;

			case CW_CMD_QUERY:
				c = toupper( getchar() );
				switch ( c ) {
				case EOF:
					goto loop_end;
				case CW_CMDV_TONE:
					sprintf( ebuf, "%c%d", c, hz );
					err_msg( CW_STATUS_OK, ebuf );
					break;
				case CW_CMDV_WPM:
					sprintf( ebuf, "%c%d", c, wpm );
					err_msg( CW_STATUS_OK, ebuf );
					break;
				case CW_CMDV_GAP:
					sprintf( ebuf, "%c%d", c, gap );
					err_msg( CW_STATUS_OK, ebuf );
					break;
				case CW_CMDV_ADJ:
					sprintf( ebuf, "%c%d", c, adj );
					err_msg( CW_STATUS_OK, ebuf );
					break;
				case CW_CMDV_ECHO:
					sprintf( ebuf, "%c%d", c, doecho );
					err_msg( CW_STATUS_OK, ebuf );
					break;
				case CW_CMDV_MSGS:
					sprintf( ebuf, "%c%d", c, doerrs );
					err_msg( CW_STATUS_OK, ebuf );
					break;
				case CW_CMDV_CMDS:
					sprintf( ebuf, "%c%d", c, docmds );
					err_msg( CW_STATUS_OK, ebuf );
					break;
				case CW_CMDV_COMBO:
					sprintf( ebuf, "%c%d", c, docombo );
					err_msg( CW_STATUS_OK, ebuf );
					break;
				case CW_CMDV_COMMENT:
					sprintf( ebuf, "%c%d", c, docomment );
					err_msg( CW_STATUS_OK, ebuf );
					break;
				default:
					sprintf( ebuf, "%c%c",
							CW_CMD_QUERY, c );
					err_msg( CW_STATUS_ERR, ebuf );
					break;
				}
				break;

			case CW_CMD_CWQUERY:
				c = toupper( getchar() );
				switch ( c ) {
				case EOF:
					goto loop_end;
				case CW_CMDV_TONE:
					sprintf( ebuf, "%d HZ ", hz );
					break;
				case CW_CMDV_WPM:
					sprintf( ebuf, "%d WPM ", wpm );
					break;
				case CW_CMDV_GAP:
					sprintf( ebuf, "%d DOTS ", gap );
					break;
				case CW_CMDV_ADJ:
					sprintf( ebuf, "%d PERCENT ", adj );
					break;
				case CW_CMDV_ECHO:
					sprintf( ebuf, "ECHO %s ",
							doecho?"ON":"OFF" );
					break;
				case CW_CMDV_MSGS:
					sprintf( ebuf, "ERRS %s ",
							doerrs?"ON":"OFF" );
					break;
				case CW_CMDV_CMDS:
					sprintf( ebuf, "CMDS %s ",
							docmds?"ON":"OFF" );
					break;
				case CW_CMDV_COMBO:
					sprintf( ebuf, "COMBO %s ",
							docombo?"ON":"OFF" );
					break;
				case CW_CMDV_COMMENT:
					sprintf( ebuf, "COMMENT %s ",
							docomment?"ON":"OFF" );
					break;
				default:
					sprintf( ebuf, "%c%c",
							CW_CMD_CWQUERY, c );
					err_msg( CW_STATUS_ERR, ebuf );
					sprintf( ebuf, "" );
					break;
				}
				send_string( ebuf, FALSE );
				break;

			case CW_CMDV_QUIT:
				send_tone( 0, SILENT );
				close( console );
				exit( 0 );
				break;

			default:
				sprintf( ebuf, "%c%c", CW_CMD_ESCAPE, c );
				err_msg( CW_STATUS_ERR, ebuf );
				break;
			}

		}
		else {
			/* not command - check for [] combination delimiters
			   if we are to honour them */
			if ( c == CW_COMBO_START && docombo ) {
				in_combination = TRUE;
				echo_char( c );
			}
			else if ( c == CW_COMBO_END && docombo) {
				in_combination = FALSE;
				echo_char( c );
			}

			/* not command and not [ or ] - send character */
			else if ( strchr( ASC_WHITESPACE, c ) != (char *)NULL )
				send_char( ASC_SPACE, in_combination );
			else
				send_char( c, in_combination );
		}
	}
loop_end:

	/* end of input file found - return to caller */
	return;
}


/*
 * catcher - signal handler for signals, to stop tone before
 *           exit
 */
static void catcher( int sig ) {

	/* stop any tone and exit */
	ioctl( console, KIOCSOUND, 0 );
	close( console );
	exit( 0 );
}


/*
 * main - parse command line args, then produce cw
 */
int main( int argc, char **argv ) {

	/* parse the command line parameters and arguments */
	parse_cmdline( argc, argv );

	/* install handler for signals */
	signal( SIGALRM, SIG_IGN );
	signal( SIGHUP, catcher );
	signal( SIGINT, catcher );
	signal( SIGQUIT, catcher );
	signal( SIGPIPE, catcher );
	signal( SIGTERM, catcher );

	/* read data from stdin until end of file */
	parse_stdin();

	/* silence speaker - complete any outstanding tones, close
	   console file */
	send_tone( 0, SILENT );
	close( console );

	/* all done */
	exit( 0 );

	return( 0 ); /* for LINT */
}
