//<copyright>
//
// Copyright (c) 1996
// Institute for Information Processing and Computer Supported New Media (IICM),
// Graz University of Technology, Austria.
//
// This file is part of VRweb.
//
// VRweb is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2, or (at your option)
// any later version.
//
// VRweb is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with VRweb; see the file LICENCE. If not, write to the
// Free Software Foundation, Inc., 59 Temple Place - Suite 330,
// Boston, MA 02111-1307, USA.
//
// Note that the GNU General Public License does not permit incorporating
// the Software into proprietary or commercial programs. Such usage
// requires a separate license from IICM.
//
//</copyright>

//<file>
//
// Name:        bsptree.C
//
// Purpose:     Interface to class BSPTree
//
// Created:      8 Feb 1996  Georg Meszaros
//
// Modified:    13 Jun 1996  Georg Meszaros 
//
//
// Description:
//
// 
// 
//
//</file>


//<class>
//
// Name:       BSPTree
//
// Purpose:    
//
//
// Public Interface:
//
//
//
// Description:
//
//
//
//
//</class>


#include "bsptree.h"

#include <hyperg/utils/types.h>

#include "polygon.h"
#include "clipping.h"
#include "svbsptree.h"

#include <iostream.h>


unsigned long BSPTree::node_number_ = 0;
unsigned long BSPTree::face_number_ = 0;

 
BSPTree::BSPTree()
{
  node_number_++;
  parent_ = nil;
  front_ = nil;
  back_ = nil;

  facelist_.allocValues(0);
}

BSPTree::BSPTree(Face* face)
{
  node_number_++;
  parent_ = nil;
  front_ = nil;
  back_ = nil;

  facelist_.allocValues(0);
  connect(face);
}


BSPTree::~BSPTree()
{
//cout << "delete BSPNode: " << node_number_ << "\n";
  if (front_) delete front_;
  if (back_) delete back_;
  

  // delete facelist_ just deletes the pointers to the faces because canUseMalloc was
  // set in QvMFFace, we store an array of pointers and do not want to
  // store an array of faces

  //cout << "Size of facelist: " << facelist_.num << "\n";
  for (int i=0; i < facelist_.num; i++)
  {
      if (facelist_.values[i]) 
      {
        delete facelist_.values[i];
        face_number_--;
      }
  }
  node_number_--;
  //cerr << "BSP Node: " << node_number_ << " deleted\n";
}


void BSPTree::insert(Face* face)
{
//cerr << "*BSPTree::insert " << face->id() << "\n";
//face->print();

  // should only be true for the bsp_root_ 
  if (facelist_.num == 0)
  {
    //cout << "first face in this node\n";
    connect(face);
    return;
  }

  vector3D normal = facelist_.values[0]->normal();
  float plane = facelist_.values[0]->d();

  // clip generates new faces if the face was split, otherwise the
  // pointer to the original face is returned
  // if the face lies on the same plane both front_face and back_face are nil

  Face* front_face = 0;
  Face* back_face = 0; 
  SplitType split_type;

/*
  cerr << "-----------------------------\n";
  cerr << "face at: " << &face << " " << face <<"\n";
  cerr << "front at: " << &front_face << " " << front_face << "\n";
  cerr << "back at: " << &back_face << " " << back_face << "\n";
*/

  Clipping::clip(*face, normal, plane, front_face, back_face, split_type);
 
/*
  cerr << "split type: " << split_type << "\n";
  cerr << "face at: " << &face << " " << face << "\n";
  cerr << "front at: " << &front_face << " " << front_face << "\n";
  cerr << "back at: " << &back_face << " " << back_face << "\n";
  cerr << "++++++++++++++++++++++++++++++\n";
*/  

  //attention: if the face was not split, NO new face was created


  if ((split_type == ONLY_FRONT) || (split_type == SPLIT)) 
  { 
    if (front_)
      front_->insert(front_face);
    else
    {    
      front_ = new BSPTree(front_face);
      front_->setParent(this);
      //cerr << "front inserted into tree: " << front_face->id() << "\n";
      //front_face->print();
    }
  }

  if ((split_type == ONLY_BACK) || (split_type == SPLIT))
  {
    if (back_)  
      back_->insert(back_face); 
    else 
    {  
      back_ = new BSPTree(back_face);
      back_->setParent(this);
      //cerr << "back inserted into tree: " << back_face->id() << "\n";
      //back_face->print();
    } 
  }


  if (split_type == SPLIT) 
  {
    // original face is not needed any more
    // 2 new faces had been created
    //cerr << "*BSPTree::insert ABSOLUTE END, deleting " << face->id() << "\n";  
    delete face;
    return;
  }


  if (split_type == SAME_PLANE) 
  {
    //no clipping, on the same plane:\n";
    //cerr << "*BSPTree::insert ABSOLUTE END, connecting " << face->id() << "\n";  
    connect(face);
    return;
  }
 
  //cerr << "returning from insert:\n";
 
}



void BSPTree::buildSVBSPTree(SVBSPTree* svbsp_root,
                             BSPMode mode)
{
// similar dataflow like in drawFrontToBack(position, lookat)
// runs throught ths BSP-Tree in front to back order

  vector3D n; // viewing dir, scalar: ray with face normal - backface culling 
  point3D any_vertex;
  point3D position;

  position.x = svbsp_root->eye().x;
  position.y = svbsp_root->eye().y;
  position.z = svbsp_root->eye().z;
  
  if (facelist_.num > 0)
  {
    vector3D normal = facelist_.values[0]->normal();
    float plane = facelist_.values[0]->d();  
    float scalar;  
    Face* face_ptr;

    //TODO: take lookat into account to make trivial reject
    //      usefull if camera is "in" the scene

    if (Clipping::inBack(position, normal, plane))
    { 
      if (back_) back_->buildSVBSPTree(svbsp_root, mode);
      
      for (int i=0; i < facelist_.num; i++)
      { 
        face_ptr = facelist_.values[i];
        any_vertex = face_ptr->anyVertex();
        sub3D(any_vertex, position, n);
        scalar = dot3D(n, face_ptr->normal());  
        // opposite direction or hint no backfaceculling
        if ( (scalar < 0) || !face_ptr->backfaceCulling() ) 
        { 
          // cerr << "*buildSVBSPTree: set VIS\n";
          face_ptr->setStatus(VIS);
          if (mode == BSP_SHADOW_VOLUME) svbsp_root->filter(face_ptr, face_ptr); 
 
          // if the status has not been changed to PARTLY_VISIBLE then
          // the polygon has no partly visible parts, e.g. it is
          // completely shadowed, not visible
   
          if ((face_ptr->status() == VIS) && (mode == BSP_SHADOW_VOLUME))
          {
             face_ptr->setStatus(NOT_VISIBLE);
             svbsp_root->hidden_face_count_++;
          }
        }
        else
        {
          face_ptr->setStatus(NOT_VISIBLE);
          svbsp_root->back_face_count_++;
          // cerr << "*buildSVBSPTree: set NOT_VISIBLE\n";
        }
        
      }

      if (front_) front_->buildSVBSPTree(svbsp_root, mode);
    }
    else
    { 
      if (front_) front_->buildSVBSPTree(svbsp_root, mode);
      
      for (int i=0; i < facelist_.num; i++)   
      {
        face_ptr = facelist_.values[i];
        any_vertex = face_ptr->anyVertex();
        sub3D(any_vertex, position, n);
        scalar = dot3D(n, face_ptr->normal());  
        // opposite direction or hint no backfaceculling
        if ( (scalar < 0) || !face_ptr->backfaceCulling() ) 
        {
          // cerr << "*buildSVBSPTree: set VIS\n";
          face_ptr->setStatus(VIS);
          if (mode == BSP_SHADOW_VOLUME) svbsp_root->filter(face_ptr, face_ptr);
          
          // if the status has not been changed to PARTLY_VISIBLE then
          // the polygon has no partly visible parts, e.g. it is
          // completely shadowed, not visible
   
          if ((face_ptr->status() == VIS) && (mode == BSP_SHADOW_VOLUME))
          {    
             face_ptr->setStatus(NOT_VISIBLE); 
             svbsp_root->hidden_face_count_++;
          }
        }
        else 
        {
          face_ptr->setStatus(NOT_VISIBLE);
          svbsp_root->back_face_count_++;
          // cerr << "*buildSVBSPTree: set NOT_VISIBLE\n";
        }        
      }      

      if (back_) back_->buildSVBSPTree(svbsp_root, mode);
    } 
  }  

}



void BSPTree::draw()
{
 for (int i=0; i < facelist_.num; i++)
 {
   if (facelist_.values[i]) facelist_.values[i]->draw(); 
 }
}


void BSPTree::draw(const point3D& position, BSPMode mode)
{
 for (int i=0; i < facelist_.num; i++)
 {
   if (facelist_.values[i]) facelist_.values[i]->draw(position, mode); 
 }
}





void BSPTree::drawBackToFront(const point3D& position, BSPMode mode)
{
if (facelist_.num > 0)
{
  vector3D normal = facelist_.values[0]->normal();
  float plane = facelist_.values[0]->d();  

  if (Clipping::inFront(position, normal, plane))
  {
    if (back_) back_->drawBackToFront(position, mode);
    draw(position,mode);
    if (front_) front_->drawBackToFront(position, mode);
  }
  else
  { 
    if (front_) front_->drawBackToFront(position, mode);
    draw(position,mode);
    if (back_) back_->drawBackToFront(position, mode);
  }
}
}


void BSPTree::drawFrontToBack (const point3D& position, BSPMode mode)
{
if (facelist_.num > 0)
{
  vector3D normal = facelist_.values[0]->normal();
  float plane = facelist_.values[0]->d();  

  if (Clipping::inFront(position, normal, plane))
  {
    if (front_) front_->drawFrontToBack(position, mode);
    draw(position,mode);
    if (back_) back_->drawFrontToBack(position, mode);
  }
  else
  { 
    if (back_) back_->drawFrontToBack(position, mode);
    draw(position,mode);
    if (front_) front_->drawFrontToBack(position, mode);
  }
}
}



void BSPTree::print() 
{
  for (int i=0; i < facelist_.num; i++) 
     if (facelist_.values[i]) facelist_.values[i]->print(); 
}




void BSPTree::connect(Face* face) 
{ 
//cout << "*: connect Face:" << face.id() << "\n";

  // num is being changed by allocValues
  facelist_.allocValues(facelist_.num + 1);

  facelist_.values[facelist_.num - 1] = face;
  face_number_++;
}



void BSPTree::resetAttributes()
{
// cerr << "BSPTree::resetAttributes. facelist_.num: " << facelist_.num << endl;
// if (facelist_.num)
//   cerr << "facelist_.values[0]: " << (void*) facelist_.values[0] << endl;

  if (facelist_.num)
    facelist_.values[0]->resetAttributes();

// cerr << "... returned from resetAttributes" << endl;
}



void BSPTree::reset()
{
  if (facelist_.num) facelist_.values[0]->reset();
}
