#include <stdio.h>
#include <ctype.h>
#include <malloc.h>
#include <string.h>
#include <stdlib.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <unistd.h>
#include <pwd.h>

#include "arena.h"
#include "defs.h"
#ifdef ARENA_DEBUG
#  include "debug.h"
#endif
#include "error.h"
#include "main.h"
#include "status.h"
#include "util.h"

#include "HTList.h"
#include "HTInet.h"
#include "HTLib.h"


static short hexTable[0x100];
static  Bool hexTableInitialized = False;


#ifndef ARENA_USE_RAW_PAINTSTREAM_HANDLING

/*
 * Get `short' interger from stream.
 */
unsigned short _Arena_GetShort(unsigned char **p)
{
 unsigned short c;
 unsigned short c1, c2;
 unsigned char* pp = (*p);

 c1 = *pp++; c2 = *pp++; c  = (c1 | c2<<8);
#if ARENA_SHORT_INT_SIZE >= 4
 c1 = *pp++; c2 = *pp++; c |= (c1 | c2<<8) << 16;
#  if ARENA_SHORT_INT_SIZE >= 6
 c1 = *pp++; c2 = *pp++; c |= (c1 | c2<<8) << 32;
#    if ARENA_SHORT_INT_SIZE >= 8
 c1 = *pp++; c2 = *pp++; c |= (c1 | c2<<8) << 48;
#    endif
#  endif
#endif

 (*p) = pp;

 return c;
}


/*
 * Get `int' interger from stream.
 */
unsigned int _Arena_GetInt(unsigned char **p)
{
 unsigned int c;
 unsigned int c1, c2;
 unsigned char* pp = (*p);

 c1 = *pp++; c2 = *pp++; c  = (c1 | c2<<8);
#if ARENA_INT_SIZE >= 4
 c1 = *pp++; c2 = *pp++; c |= (c1 | c2<<8) << 16;
#  if ARENA_INT_SIZE >= 6
 c1 = *pp++; c2 = *pp++; c |= (c1 | c2<<8) << 32;
#    if ARENA_INT_SIZE >= 8
 c1 = *pp++; c2 = *pp++; c |= (c1 | c2<<8) << 48;
#    endif
#  endif
#endif

 (*p) = pp;

 return c;
}


/*
 * Get `long' interger from stream.
 */
unsigned long _Arena_GetLong(unsigned char **p)
{
 unsigned long c;
 unsigned long c1, c2;
 unsigned char* pp = (*p);

 c1 = *pp++; c2 = *pp++; c  = (c1 | c2<<8);
#if ARENA_LONG_INT_SIZE >= 4
 c1 = *pp++; c2 = *pp++; c |= (c1 | c2<<8) << 16;
#  if ARENA_LONG_INT_SIZE >= 6
 c1 = *pp++; c2 = *pp++; c |= (c1 | c2<<8) << 32;
#    if ARENA_LONG_INT_SIZE >= 8
 c1 = *pp++; c2 = *pp++; c |= (c1 | c2<<8) << 48;
#    endif
#  endif
#endif

 (*p) = pp;

 return c;
}


/*
 * Get `void*' pointer from stream.
 */
void* _Arena_GetPointer(unsigned char **p)
{
 unsigned long c;
 unsigned long c1, c2; /* must hold a 64bit Pointer wm 19.Jan.95 */
 unsigned char* pp = (*p);

 c1 = *pp++; c2 = *pp++; c  = (c1 | c2<<8);
#if ARENA_VOID_POINTER_SIZE >= 4
 c1 = *pp++; c2 = *pp++; c |= (c1 | c2<<8) << 16;
#  if ARENA_VOID_POINTER_SIZE >= 6
 c1 = *pp++; c2 = *pp++; c |= (c1 | c2<<8) << 32;
#    if ARENA_VOID_POINTER_SIZE >= 8
 c1 = *pp++; c2 = *pp++; c |= (c1 | c2<<8) << 48;
#    endif
#  endif
#endif

 *p = pp;

 return (void*)c;
}


/*
 * Put `short' integer to stream.
 */
void _Arena_PutShort(unsigned char **p, unsigned short s)
{
 unsigned int i2;
 unsigned char* pp = (*p);

 i2 = (unsigned int)( s        & 0xFFFF);
 *pp++ = (unsigned char)( i2       & 0xFF);
 *pp++ = (unsigned char)((i2 >> 8) & 0xFF);
#if ARENA_SHORT_INT_SIZE >= 4
 i2 = (unsigned int)((s >> 16) & 0xFFFF);
 *pp++ = (unsigned char)( i2       & 0xFF);
 *pp++ = (unsigned char)((i2 >> 8) & 0xFF);
#  if ARENA_SHORT_INT_SIZE >= 6
 i2 = (unsigned int)((s >> 32) & 0xFFFF);
 *pp++ = (unsigned char)( i2       & 0xFF);
 *pp++ = (unsigned char)((i2 >> 8) & 0xFF);
#    if ARENA_SHORT_INT_SIZE >= 8
 i2 = (unsigned int)((s >> 48) & 0xFFFF);
 *pp++ = (unsigned char)( i2       & 0xFF);
 *pp++ = (unsigned char)((i2 >> 8) & 0xFF);
#    endif
#  endif
#endif

 (*p) = pp; 
}


/*
 * Put `int' integer to stream.
 */
void _Arena_PutInt(unsigned char **p, unsigned int i)
{
 unsigned int i2;
 unsigned char* pp = (*p);

 i2 = (unsigned int)( i        & 0xFFFF);
 *pp++ = (unsigned char)( i2       & 0xFF);
 *pp++ = (unsigned char)((i2 >> 8) & 0xFF);
#if ARENA_INT_SIZE >= 4
 i2 = (unsigned int)((i >> 16) & 0xFFFF);
 *pp++ = (unsigned char)( i2       & 0xFF);
 *pp++ = (unsigned char)((i2 >> 8) & 0xFF);
#  if ARENA_INT_SIZE >= 6
 i2 = (unsigned int)((i >> 32) & 0xFFFF);
 *pp++ = (unsigned char)( i2       & 0xFF);
 *pp++ = (unsigned char)((i2 >> 8) & 0xFF);
#    if ARENA_INT_SIZE >= 8
 i2 = (unsigned int)((i >> 48) & 0xFFFF);
 *pp++ = (unsigned char)( i2       & 0xFF);
 *pp++ = (unsigned char)((i2 >> 8) & 0xFF);
#    endif
#  endif
#endif

 (*p) = pp; 
}


/*
 * Put `long' integer to stream.
 */
void _Arena_PutLong(unsigned char **p, unsigned long l)
{
 unsigned int i2;
 unsigned char* pp = (*p);

 i2 = (unsigned int)( l        & 0xFFFF);
 *pp++ = (unsigned char)( i2       & 0xFF);
 *pp++ = (unsigned char)((i2 >> 8) & 0xFF);
#if ARENA_INT_SIZE >= 4
 i2 = (unsigned int)((l >> 16) & 0xFFFF);
 *pp++ = (unsigned char)( i2       & 0xFF);
 *pp++ = (unsigned char)((i2 >> 8) & 0xFF);
#  if ARENA_INT_SIZE >= 6
 i2 = (unsigned int)((l >> 32) & 0xFFFF);
 *pp++ = (unsigned char)( i2       & 0xFF);
 *pp++ = (unsigned char)((i2 >> 8) & 0xFF);
#    if ARENA_INT_SIZE >= 8
 i2 = (unsigned int)((l >> 48) & 0xFFFF);
 *pp++ = (unsigned char)( i2       & 0xFF);
 *pp++ = (unsigned char)((i2 >> 8) & 0xFF);
#    endif
#  endif
#endif

 (*p) = pp; 
}


/*
 * Put `void*' pointer to stream.
 */
void _Arena_PutPointer(unsigned char **p, void *ptr)
{
 unsigned int ptr16;
 unsigned char* pp = (*p);

 ptr16 = (unsigned int)((long)ptr & 0xFFFF);
 *pp++ = (unsigned char)( ptr16       & 0xFF);
 *pp++ = (unsigned char)((ptr16 >> 8) & 0xFF);
#if ARENA_VOID_POINTER_SIZE >= 4
 ptr16 = (unsigned int)(((long)ptr >> 16) & 0xFFFF);
 *pp++ = (unsigned char)( ptr16       & 0xFF);
 *pp++ = (unsigned char)((ptr16 >> 8) & 0xFF);
#  if ARENA_VOID_POINTER_SIZE >= 6
 ptr16 = (unsigned int)(((long)ptr >> 32) & 0xFFFF);
 *pp++ = (unsigned char)( ptr16       & 0xFF);
 *pp++ = (unsigned char)((ptr16 >> 8) & 0xFF);
#    if ARENA_VOID_POINTER_SIZE >= 8
 ptr16 = (unsigned int)(((long)ptr >> 48) & 0xFFFF);
 *pp++ = (unsigned char)( ptr16       & 0xFF);
 *pp++ = (unsigned char)((ptr16 >> 8) & 0xFF);
#    endif
#  endif
#endif

 (*p) = pp; 
}

#endif /* ARENA_USE_RAW_PAINTSTREAM_HANDLING */


/*
 * Own `malloc' function to wrap the library one.
 */
void* Arena_MAlloc(size_t theBufferSize,
		   Bool ExitArenaOnFail)
{
 void*  theBuffer = NULL;
#ifdef ARENA_DEBUG
 char Iam[] = "Arena_MAlloc";
#endif


 if (theBufferSize > 0)
   {
    theBuffer = malloc(theBufferSize);
   }

 if (theBuffer == NULL)
   {
#ifdef ARENA_DEBUG
    Arena_TracePrint(Iam,
		     " ERROR! Failed to allocate buffer of size %d.\n",
		     theBufferSize);
#endif
   if (ExitArenaOnFail)
     {
#ifdef ARENA_DEBUG
      Arena_TracePrint(Iam, " Exiting.\n");
#endif
      Exit(1);
     }
   }

 return theBuffer;
}


/*
 * Own `calloc' function to wrap the library one.
 */
void* Arena_CAlloc(size_t theMembersNumber, size_t theMemberSize,
		   Bool ExitArenaOnFail)
{
 void*  theBuffer = NULL;
#ifdef ARENA_DEBUG
 char Iam[] = "Arena_CAlloc";
#endif


 if (theMembersNumber > 0 && theMemberSize > 0)
   {
    theBuffer = calloc(theMembersNumber, theMemberSize);
   }

 if (theBuffer == NULL)
   {
#ifdef ARENA_DEBUG
    Arena_TracePrint(Iam,
		     " ERROR! Failed to allocate %d elements of %d.\n",
		     theMembersNumber, theMemberSize);
#endif
   if (ExitArenaOnFail)
     {
#ifdef ARENA_DEBUG
      Arena_TracePrint(Iam, " Exiting.\n");
#endif
      Exit(1);
     }
   }

 return theBuffer;
}


/*
 * Own `strlen' function to wrap around the library one.
 */
size_t Arena_StrLen (const char* theString)
{
 if (theString) if (*theString) return strlen(theString);
 return 0;
}



#ifndef HAVE_STRDUP

char *strdup(char *s)
{
#ifdef ARENA_DEBUG
 char Iam[] = "strdup";
#endif

 if (s)
   {
    int n;
    char *pp;

    n = Arena_StrLen(s);

    if ((pp = (char *)Arena_MAlloc(n + 1, False)))
      {
       strncpy(pp, s, n);
       pp[n] = '\0';
      }
     else
      {
       Arena_PrintError(_("ERROR: malloc failed, memory exhausted!\n"));
      }

    return pp;
   }
  else
   {
#  ifdef ARENA_DEBUG
    Arena_TracePrint(Iam,
		     " WARNING! Has been called with NULL source string.\n");
#  endif
    return NULL;
   }
}

#endif /* HAVE_STRDUP */


#ifndef HAVE_STRNDUP

char *strndup(char *s, int n)
{
#ifdef ARENA_DEBUG
 char Iam[] = "strndup";
#endif


 if (s)
   {
    char *pp;

    if (n < 0)
      {
#  ifdef ARENA_DEBUG
       Arena_TracePrint(Iam,
			" WARNING!"
			" Has been called with negative string length (%i).\n",
			n);
#  endif
       n = 0;
      }

    if ((pp = (char *)Arena_MAlloc(n + 1, False)))
      {
       strncpy(pp, s, n);
       pp[n] = '\0';
      }
     else
      {
       Arena_PrintError(_("ERROR: malloc failed, memory exhausted!\n"));
      }

    return pp;
   }
  else
   {
#  ifdef ARENA_DEBUG
    Arena_TracePrint("strndup",
		     " WARNING! Has been called with NULL source string.\n");
#  endif
    return NULL;
   }
}

#endif /* HAVE_STRNDUP */


/*
 * A reentrant `strtok()'.
 */
char* str_tok(char* a, char* b, char** c)
{
 char* start = NULL;

 if (b)
   {
    char* d;

    if ((d = ((a) ? a : (*c))))
      {
       char* stop;

       while (*d && ARENA_Index(b, (int)*d)) d++;
       start = d;
       while (*d && !ARENA_Index(b, (int)*d)) d++;
       stop = d;
       if (start == stop)
	 start = NULL;
        else
	 {
	  if (*d) *d++ = '\0';

	  (*c) = d;
	 }
      }
   }

 return start;
}


char *chop_str(char *p)
{
 if (p)
   {
    char *p_start = p, *p_end;

    while (isspace((unsigned char)*p_start)) p_start++;

    p_end = p + Arena_StrLen(p) - 1;

    while (isspace((unsigned char)*p_end)) p_end--;

    /* how should we handle empty stuff?? */

    if (p_start > p_end)
      return (strdup(p));
     else
      return((char *)strndup(p_start, p_end - p_start + 1));
   }

 return NULL;
}


/*
 * Strip surrounding "" or '' from the string.
 */
char* unquote(char* theS)
{
 if (theS)
   {
    if (((*theS) == '"') || ((*theS) == '\''))
      {
       int l = Arena_StrLen(theS) - 1;

       if ((*theS) == theS[l])
	{
	 theS[l] = '\0';
	 theS++;
	}
      }
   }

 return theS;
}


/*
 * Initialize hex table, set the flag.
 */
static void hexTableInit(void)
{
 extern short hexTable[];
 extern  Bool hexTableInitialized;

 if (hexTableInitialized != True)
   {
    int i;

   /*
    * hexTable[i] value:
    *       -2    --- funny symbol;
    *       -1    --- delimiter;
    *    0x0--0xf --- hex digit value.
    */

    for (i = 0x100; i--; ) hexTable[i] = -2;

    /* Delimiters of significance */
    hexTable[' ']  = -1;
    hexTable[',']  = -1;
    hexTable['}']  = -1;
    hexTable['\f'] = -1;
    hexTable['\n'] = -1;
    hexTable['\r'] = -1;
    hexTable['\t'] = -1;
    hexTable['\v'] = -1;

    /* Valid hex digits */
    hexTable['0'] = 0x00;
    hexTable['1'] = 0x01;
    hexTable['2'] = 0x02;
    hexTable['3'] = 0x03;
    hexTable['4'] = 0x04;
    hexTable['5'] = 0x05;
    hexTable['6'] = 0x06;
    hexTable['7'] = 0x07;
    hexTable['8'] = 0x08;
    hexTable['9'] = 0x09;
    hexTable['A'] = 0x0A;   hexTable['a'] = 0x0a;
    hexTable['B'] = 0x0B;   hexTable['b'] = 0x0b;
    hexTable['C'] = 0x0C;   hexTable['c'] = 0x0c;
    hexTable['D'] = 0x0D;   hexTable['d'] = 0x0d;
    hexTable['E'] = 0x0E;   hexTable['e'] = 0x0e;
    hexTable['F'] = 0x0F;   hexTable['f'] = 0x0f;
        
    hexTableInitialized = True;
   }

 return;
}


#ifndef HAVE_ISXDIGIT
/*
 * Isn't portable, internationalizable, localizable, etc. Just a quick hack.
 */
Bool isxdigit(char c)
{
 extern short hexTable[];
 extern  Bool hexTableInitialized;
 short cv;

 if (hexTableInitialized != True) hexTableInit();
 cv = hexTable[c];
 return ((0x0 <= cv) && (cv <= 0xf));
}
#endif


/* parse string for hex rgb codes of the form rgb or rrggbb
 * with or without surrounding "".
 */
Bool isrgb(const char *theS, long *rgb)
{
 Bool isRGB = True;
 char* sp;
 char* s = NULL;
 int l = 0;
 long colour = 0;
 long factor = 0, shift = 0;


 if (theS) if (*theS) s = strdup(theS);

 if (s == NULL)
   {
    *rgb = 0;
    return False;
   }


 sp = unquote((char*)s);

 if ((*sp) == '#') sp++;

 l = Arena_StrLen(sp);

 switch (l)
   {
    case 0:
      isRGB = False;
      break;

    case 3:	/* rgb */
      factor = 0x11;
      shift  = 0x08;
      break;

    case 6:	/* rrggbb */
      factor = 0x01;
      shift  = 0x04;
      break;

    default:
      isRGB = False;
      break;
   }

 if (isRGB)
   {
    for (; l-- ; sp++)
      {
       if (isxdigit(*sp))
	 colour = (colour << shift) + (hex2byte('0', (*sp)) * factor) ;
        else
	 {
	  isRGB = False;
	  break;
	 }
      }
   }

 Free(s);

 *rgb = (isRGB ? colour : 0);

 return isRGB;
}


/*
 * Parse two hex digits to the appropriate value.
 */
Byte hex2byte(char hexH, char hexL)
{
 extern short hexTable[];
 extern  Bool hexTableInitialized;


 if (hexTableInitialized != True) hexTableInit();

 if (isxdigit(hexH) && isxdigit(hexL))
   return (hexTable[(unsigned int)hexH] * 0x10 + hexTable[(unsigned int)hexL]);
  else
   return 0;
}


#ifndef HAVE_STRERROR   /* most SVR4 machines */
char *strerror(int a)
{
 return NULL;
}
#endif



/*
 * Counts column number of position in the string.
 */
int ColumnNumber_in_String(char* s, int slen, int pos)
{
 int i, l;
 char* p;

 if (pos < 0) return 0;   /* Invalid value! */
 if (pos == 0) return 0;
 if (slen < 0) return 0;   /* Invalid value! */
 if (slen == 0) return 0;
 if (s == NULL) return 0;   /* Invalid value! */

 if ((l = Arena_StrLen(s)) == 0) return 0;

 if (l > slen) l = slen;
 if (pos > l) pos = l;

 for (i = 0, p = s + pos - 1; (pos > 0) && (*p != '\n'); p--, pos--) i++;

 return i;
}


/*
 * Counts line number of position in the string.
 */
int LineNumber_in_String(char* s, int slen, int pos)
{
 int i, j, l;

 if (pos < 0) return 0;   /* Invalid value! */
 if (pos == 0) return 0;
 if (slen < 0) return 0;   /* Invalid value! */
 if (slen == 0) return 0;
 if (s == NULL) return 0;   /* Invalid value! */

 if ((l = Arena_StrLen(s)) == 0) return 0;

 if (l > slen) l = slen;
 if (pos > l) pos = l;

 for (i = 0, j = 0; j < pos; j++, s++) if (*s == '\n') i++;

 return i;
}


/*
 * Find out username. The result string is allocated in the function.
 */
char* GetArenaUser(void)
{
 char* theUser;
#ifdef ARENA_DEBUG
 char Iam[] = "GetArenaUser";
#endif


 if ((theUser = getenv("USER")))
   {
#ifdef ARENA_DEBUG      /* QingLong.14-02-97 */
    if (VERBOSE_TRACE)
      Arena_TracePrint(Iam, " User is \"%s\" (got from $USER).\n", theUser);
#endif
    theUser = strdup(theUser);   /* make malloc'ed copy of the variable */
   }
  else
   {
    struct passwd * theauthority;

    if ((theauthority = getpwuid(getuid())))
      {
       if (theauthority->pw_name)
	 {
	  theUser = strdup(theauthority->pw_name);

#ifdef ARENA_DEBUG      /* QingLong.14-02-97 */
	  if (VERBOSE_TRACE)
	    Arena_TracePrint(Iam,
			     " User is \"%s\" (got from passwd).\n", theUser);
#endif
	 }
#ifdef ARENA_DEBUG	/* QingLong.30-06-97 */
        else
	 {
	  Arena_TracePrint(Iam, " ERROR! Got NULL pw_name in passwd entry.\n");
	 }
#endif
       /*
	* How should passwd struct be freed?
	* Luc St-Louis <lucs@cam.org> wrote:
	*   The answer is: Don't free it.
	*   The structure is allocated statically by
	*   functions in the C library and the memory cost is
	*   constant and negligeable.
	*/
      }
#ifdef ARENA_DEBUG	/* QingLong.30-06-97 */
     else
      {
       Arena_TracePrint(Iam, " Failed to get passwd entry.\n");
      }
#endif
   }

 if (theUser)
   {
    if (*theUser)
      return theUser;
     else
      {
       Free(theUser);
#ifdef ARENA_DEBUG
       if (BRIDGE_TRACE)
	 Arena_TracePrint(Iam, " WEIRD! Got dummy (empty) username.\n");
#endif
      }
   }

 return NULL;
}


/* Find my HOME location... The result string is allocated in the function.
 * No attempt to do some security stuff here, but...
 */
char *ALibHomePath(void)
{
 char* theHome;
#ifdef ARENA_DEBUG
 char Iam[] = "ALibHomePath";
#endif


 if ((theHome = getenv("HOME")))
   {
    theHome = strdup(theHome);   /* make malloc'ed copy of the variable */
#ifdef ARENA_DEBUG      /* QingLong.14-02-97 */
    if (VERBOSE_TRACE)
      Arena_TracePrint(Iam,
		       " Home dir is \"%s\" (got from $HOME).\n",
		       theHome);
#endif
   }
  else
   {
    struct passwd * theauthority;

    if ((theauthority = getpwuid(getuid())))
      {
       if (theauthority->pw_dir)
	 {
	  theHome = strdup(theauthority->pw_dir);

#ifdef ARENA_DEBUG      /* QingLong.14-02-97 */
	  if (VERBOSE_TRACE)
	    Arena_TracePrint(Iam,
			     " Home dir is \"%s\" (got from passwd).\n",
			     theHome);
#endif
	 }
#ifdef ARENA_DEBUG	/* QingLong.30-06-97 */
        else
	 {
	  Arena_TracePrint(Iam, " ERROR! Got NULL pw_dir in passwd entry.\n");
	 }
#endif

       Free(theauthority);   /* How should passwd struct be freed? */
      }
#ifdef ARENA_DEBUG	/* QingLong.30-06-97 */
     else
      {
       Arena_TracePrint(Iam, " Failed to get passwd entry.\n");
      }
#endif
   }

 if (theHome)
   {
    if (*theHome)
      {
       if (theHome[Arena_StrLen(theHome) - 1] != '/') StrAllocCat(theHome, "/");
       return theHome;
      }
     else
      {
       Free(theHome);
#ifdef ARENA_DEBUG
       if (BRIDGE_TRACE)
	 Arena_TracePrint(Iam, " WEIRD! Got dummy (empty) home directory.\n");
#endif
      }
   }

 return NULL;
}


/* Routine used to create a directory limb!
 * Will create the entire limb if necessary.
 * The basis for this routine is from HTCacheInit with a few extra checks!
 */
Bool ALibCreatePath(const char* droot, mode_t protect)
{
 int istat, mstat;
 struct stat st;
 char *loc = NULL;
 char *cur = NULL;

/* Make sure we are looking at a reasonably valid path specification
 * and force this to be a limb!
 */
 if (!droot || *droot != '/' || !protect) return NO;
 StrAllocCopy(loc, droot);
 if (*(loc + Arena_StrLen(loc) - 1) != '/') StrAllocCat(loc, "/");

/* Create (if necessary) each part of the limb.
 */
 cur = loc + 1;
 while ((cur = ARENA_Index(cur, '/'))) /* progress down the limb             */
   {
    *cur = '\0';
    istat = mstat = stat(loc, &st);    /* see if already exists              */
    if (istat)
      {
       mstat = mkdir(loc, protect);
       istat = stat(loc, &st);
      }

    if (mstat || !S_ISDIR(st.st_mode)) /* force it to be a DIR               */
      {
       Free(loc);
       return NO;
      }
    *cur++ = '/';
   }                                   /* end while                          */
 Free(loc);
 return YES;
}


/* duplicate  string s and ensure it is zero terminated
 * where n is the maximum number of chars copied from s
 */
void HTList_destroy(HTList* me)
{
 HTList* current;

 while ((current = me))
   {
    if (me->object) { Free(me->object); }
    me = me->next;
    Free (current);
   }
}
