/*
 * Preliminary:
 *
 * i do not care about little pieces of memory, that are
 * allocated once during program execution and not used any
 * longer from some certain time on. E.g. if a string is
 * allocated via strdup:
 *
 *  char * str = strdup("some stuff");
 *
 * i'm not always freeing it the next time this pointer will
 * be assigned a new value. If wasting 100 Bytes is a problem,
 * buy a new machine. Call me, what you like.
 */

#include <stdio.h>
#include <string.h>
#include <signal.h>
#include <fcntl.h>
#include <sys/wait.h>
#include <unistd.h>
#include <stdarg.h>
#include <utils.h>
#include <fileutil.h>
#include <x_regex.h>
#include <backup.h>

#define	CLEANUP		{ goto cleanup; }

#define	ES		((UChar *) "")

#define	BACKUP_RE	"[Bb][Aa][Cc][Kk][Uu][Pp]"
#define	CARTRIDGE_RE	"[Cc][Aa][Rr][Tt][Rr][Ii][Dd][Gg][Ee]"
#define	FILE_RE		"[Ff][Ii][Ll][Ee]"

#define	EMREC_PREFIX	"~~"
#define	EMREC_PREFIX_RE	"^[ \t]*" EMREC_PREFIX "[ \t]*"

#define	BU_NO_LOCK		0
#define	BU_LOCKED		1
#define	BU_GOT_LOCK		2
#define	BU_CANT_LOCK		99

#define	MODE_FULL_BACKUP	0
#define	MODE_INCR_BACKUP	1
#define	MODE_PRINT_ERRORS	2
#define	MODE_RESTORE		3
#define	MODE_VERIFYBU		4

UChar	*cmd_res[] = {
	"[Ff][Uu][Ll][Ll]",
	"[Ii][Nn][Cc][Rr]",
	"[Ee][Rr][Rr][Oo][Rr]",
	"\\([Bb][Aa][Cc][Kk][Oo][Uu][Tt]\\|[Rr][Ee][Ss][Tt][Oo][Rr][Ee]\\)",
	"[Vv][Ee][Rr][Ii][Ff][Yy]",
		};

UChar	*option2prefix[] = {	FILECONTOPTION " " FILECONTPREFIX,
				FILECONTZOPTION " " FILECONTZPREFIX,
				LOCALDEVOPTION " " LOCALDEVPREFIX,
				NULL };

UChar	*default_paramfiles[] = { DEFAULT_CLIENT_CONFIGFILES , NULL };

UChar	*bindir = NULL;
UChar	*vardir = NULL;
UChar	*libdir = NULL;
UChar   *logdir = NULL;
UChar	*paramfile = NULL;
UChar	*partfile = NULL;
UChar	*numfile = NULL;
UChar	*oldmarkfile = NULL;
UChar	*orgoldmarkfile = NULL;
UChar	*newmarkfile = NULL;
UChar	*indexfile = NULL;
UChar	*cryptfile = NULL;
UChar	*clientprogram = NULL;

UChar	*backuphost = NULL;
Int32	backupport = -1;
UChar	*rootdir = NULL;
UChar	*dirstobackupraw = NULL;
UChar	**dirstobackup = NULL;
UChar	*filestoskipraw = NULL;
UChar	**filestoskip = NULL;
UChar	*dirstoskipraw = NULL;
UChar	**dirstoskip = NULL;
UChar	*indexfilepart = NULL;
Int32	numindexestostore = -1;
UChar	*compresscmd = NULL;
UChar	*uncompresscmd = NULL;
Int32	compressbu = 1;
UChar	*logfile = NULL;
Int32	numparts = 1;
UChar	*startinfoprog = NULL;
UChar	*exitprog = NULL;
UChar	reportfile[100] = "\0";
UChar	*filelist = NULL;
Int32	compresslogfiles = 1;
UChar	**filecontentstobu = NULL;
UChar	**dont_compress = NULL;
Int32	num_dont_compress = 0;
UChar	*dont_compress_str = NULL;
UChar	*exclude_filename = NULL;

Int32	mode = -1;
int	part = 1;
int	num = 1;
FILE	*lfp = NULL;
UChar	interrupted = 0;
UChar	logf_lost = 0;
int	insfileno = 0;
int	inscartno = 0;
Int32	cartset = 1;
UChar	detach = 0;

/* commandline-arguments */
Int32	c_cartset = -1;

/* for restore */
UChar		do_counting = 0;
UChar		do_listing = 0;
UChar		*restorehost = NULL;
UChar		*restoreroot = NULL;
UChar		restore_all = 0;
UChar		restore_em = 0;
UChar		restore_emlist = 0;
UChar		*beforestr = NULL;
UChar		*afterstr = NULL;
time_t		time_older = 0;
time_t		time_newer = 0;
time_t		current_bu_time = 0;
time_t		prev_bu_time = 0;
UChar		have_timerange = 0;
Int32		num_restore_paths = 0;
UChar		**restore_paths = NULL;
Int32		num_prev_backup = 0;
Int32		num_prev_run = 0;
UChar		**indexfiles = NULL, **zindexfiles = NULL;
Int32		numidx, num_indexfiles;
Int32		curuid;

RE_cmp_buffer	backup_re, cartridge_re, file_re;

typedef	struct _scandir_args_ {
  int		fd;
}	ScanDirArgs;
ScanDirArgs	scandir_args;

UChar		*lockfile = NULL;
int		lockfd;
UChar		locked = BU_NO_LOCK;
struct flock	lockb;


#if	MAXPATHLEN < 1000
#define	TMPBUFSIZ	2000
#else
#define	TMPBUFSIZ	(MAXPATHLEN * 2)
#endif

UChar	tmpbuf[TMPBUFSIZ];

UChar	*lockdirs[] = {
		"/var/locks", "/var/lock", "/var/spool/locks",
		"/var/spool/lock", "/var/tmp", "/tmp", NULL,
};

#define	LOCKFILENAME	"Lck.afbu_client"

UChar	*tmpdirs[10] = { "pwd - dummy",
		FN_DIRSEPSTR "tmp" FN_DIRSEPSTR "%N",
		FN_DIRSEPSTR "var" FN_DIRSEPSTR "tmp" FN_DIRSEPSTR "%N",
		FN_DIRSEPSTR "temp" FN_DIRSEPSTR "%N",
		NULL};
UChar	*tmpfiles[10];

ParamFileEntry	entries[] = {
	{ &dirstobackupraw, NULL,
	(UChar *) "^[ \t]*[Dd]ire?c?t?o?r?i?e?s[ \t_-]*[Tt]o[ \t_-]*[Bb]ackup:?[ \t]*",
		TypeUCharPTR	},
	{ &backuphost, NULL,
	(UChar *) "^[ \t]*[Bb]ackup[ \t_-]*[Hh]ost:?[ \t]*",
		TypeUCharPTR	},
	{ &backupport, NULL,
	(UChar *) "^[ \t]*[Bb]ackup[ \t_-]*[Pp]ort:?[ \t]*",
		TypeInt32	},
	{ &rootdir, NULL,
	(UChar *) "^[ \t]*[Rr]oot[ \t_-]*[Dd]ire?c?t?o?r?y?:?[ \t]*",
		TypeUCharPTR	},
	{ &filestoskipraw, NULL,
	(UChar *) "^[ \t]*[Ff]iles[ \t_-]*[Tt]o[ \t_-]*[Ss]kip:?[ \t]*",
		TypeUCharPTR	},
	{ &dirstoskipraw, NULL,
	(UChar *) "^[ \t]*[Dd]ire?c?t?o?r?i?e?s[ \t_-]*[Tt]o[ \t_-]*[Ss]kip:?[ \t]*",
		TypeUCharPTR	},
	{ &indexfilepart, NULL,
	(UChar *) "^[ \t]*[Ii]ndex[ \t_-]*[Ff]ile[ \t_-]*[Pp]art:?[ \t]*",
		TypeUCharPTR	},
	{ &numindexestostore, NULL,
	(UChar *) "^[ \t]*[Nn]um[-_ \t]*[Ii]nd\\(ic\\|ex\\)es[-_ \t]*[Tt]o[ \t_-]*[Ss]tore:?",
		TypeInt32	},
	{ &compresscmd, NULL,
	(UChar *) "^[ \t]*[Cc]ompress[-_ \t]*[Cc]o?m?ma?n?d:?[ \t]*",
		TypeUCharPTR	},
	{ &uncompresscmd, NULL,
	(UChar *) "^[ \t]*[Uu]ncompress[-_ \t]*[Cc]o?m?ma?n?d:?[ \t]*",
		TypeUCharPTR	},
	{ &dont_compress_str, NULL,
	(UChar *) "^[ \t]*[Dd]o[-_ \t]*[Nn][\'o]?t[-_ \t]*[Cc]ompress:?[ \t]*",
		TypeUCharPTR	},
	{ &exclude_filename, NULL,
	(UChar *) "^[ \t]*[Ee]xclu?d?e?[-_ \t]*[Ll]ist[-_ \t]*[Ff]ile[-_ \t]*[Nn]?a?m?e?:?[ \t]*",
		TypeUCharPTR	},
	{ &logfile, NULL,
	(UChar *) "^[ \t]*[Ll]ogg?i?n?g?[-_ \t]*[Ff]ile:?[ \t]*",
		TypeUCharPTR	},
	{ &numparts, NULL,
	(UChar *) "^[ \t]*[Nn]um[-_ \t]*[Bb]ackup[-_ \t]*[Pp]arts:?",
		TypeInt32	},
	{ &cartset, NULL,
	(UChar *) "^[ \t]*[Cc]artr?i?d?g?e?[-_ \t]*[Ss]et:?",
		TypeInt32	},
	{ &startinfoprog, NULL,
	(UChar *) "^[ \t]*[Ss]tartu?p?[-_ \t]*[Ii]nfo[-_ \t]*[Pp]rogram:?[ \t]*",
		TypeUCharPTR	},
	{ &exitprog, NULL,
	(UChar *) "^[ \t]*[Ee]xit[-_ \t]*[Pp]rogram:?[ \t]*",
		TypeUCharPTR	},
	{ &compresslogfiles, NULL,
	(UChar *) "^[ \t]*[Cc]ompress[-_ \t]*[Ll]ogg?i?n?g?[-_ \t]*[Ff]iles:?",
		TypeInt32	},
	{ &compressbu, NULL,
	(UChar *) "^[ \t]*[Cc]ompress[-_ \t]*[Bb]ackupe?d?[-_ \t]*\\([Ff]iles\\)?:?",
		TypeInt32	},
	{ &cryptfile, NULL,
	(UChar *) "^[ \t]*\\([Ee]n\\)?[Cc]rypti?o?n?[ \t_-]*[Kk]ey[ \t_-]*[Ff]ile:?[ \t]*",
		TypeUCharPTR	},
	{ &vardir, NULL,
	(UChar *) "^[ \t]*[Vv][Aa][Rr][-_ \t]*[Dd]ire?c?t?o?r?y?:?[ \t]*",
		TypeUCharPTR	},
};

static void	print_errors(int, char **);
static void	verifybu(int, char **);
static void	restore(int, char **);
static void	restoreall(int, char **);
static void	restoreem(int, char **);
static Int32	get_restore_args(int, char **);
static Int32	get_perror_args(int, char **);
static Int32	get_verify_args(int, char **);
static int	get_arg_num(int, char **, Int32 *);
static Int32	write_to_restore(UChar **, Int32, UChar);
static UChar	*repl_substring_safe(UChar *, UChar *, UChar *);
static void	matches_check_timerange(UChar **, int *, time_t *,
					time_t *, time_t, time_t);

static void
do_exit(int s)
{
  int	fd;

  if(locked == BU_GOT_LOCK){
    lockb.l_type = F_UNLCK;
    fcntl(lockfd, F_SETLK, &lockb);

    close(lockfd);

    unlink(lockfile);
  }

  if(exitprog){
    sprintf(tmpbuf, "%d", s);

    exitprog = repl_substring_safe(exitprog, "%r", reportfile);
    exitprog = repl_substring_safe(exitprog, "%e", tmpbuf);
    exitprog = repl_substring_safe(exitprog, "%l", filelist ? filelist :
		(UChar *) "<filename-logfile not yet determined");

    if((fd = open(reportfile, O_WRONLY | O_APPEND | O_CREAT, 0644)) >= 0)
	close(fd);

    system(exitprog);

    if(reportfile[0])
	unlink(reportfile);
  }

  exit(s);
}

static void
errmsg(UChar * msg, ...)
{
  va_list	args;

  va_start(args, msg);

  if(lfp){
    fprintf(lfp, "%s, ", actimestr());
    vfprintf(lfp, msg, args);
    fprintf(lfp, ".\n");
    fflush(lfp);
  }

  vfprintf(stderr, msg, args);
  fprintf(stderr, ".\n");

  va_end(args);
}

static void
logmsg(UChar * msg, ...)
{
  va_list	args;

  va_start(args, msg);

  if(lfp){
    fprintf(lfp, "%s, ", actimestr());
    vfprintf(lfp, msg, args);
    fprintf(lfp, ".\n");
    fflush(lfp);
  }

  va_end(args);
}

static void
nomemerrexit()
{
  fprintf(stderr, "Error: No memory.\n");
  do_exit(3);
}

UChar *
repl_substring_safe(UChar * org, UChar * to_subst, UChar * subst_by)
{
  UChar		*newstr;

  newstr = repl_substring(org, to_subst, subst_by);
  if(!newstr)
    nomemerrexit();

  return(newstr);
}

static void
usage(UChar * pname)
{
  pname = FN_BASENAME(pname);

  switch(mode){
   case MODE_RESTORE:
    fprintf(stderr,
        "Usage: %s [ -nl ] [ -<past-backup-no> ] [ -R <root-directory> ] \\\n",
                        pname);
    fprintf(stderr,
        "               [ -h <backuphost> ] [ -c <configuration-file> ] \\\n");
    fprintf(stderr,
        "               [ -A \"<after-date>\" ] [ -B \"<before-date>\" ] \\\n");
    fprintf(stderr,
        "               [ -p ] <path-pattern> [ [ -p ] <path-patterns> [ ... ] ]\n");
    fprintf(stderr,
        "       %s -a [ -<past-backup-no> ] [ -R <root-directory> ] \\\n",
                        pname);
    fprintf(stderr,
        "               [ -h <backuphost> ] [ -c <configuration-file> ]\n");
    fprintf(stderr,
		"       %s -{ef} [ -R <root-directory> ] [ -h <backuphost> ] \\\n",
                        pname);
    fprintf(stderr,
        	"               [ -c <configuration-file> ] < <startup-info-file>\n");
    break;

   case MODE_PRINT_ERRORS:
    fprintf(stderr,
	"Usage: %s [ -c <configuration-file> ] [ -<past-backup-no> ]\n",
			pname);
    break;

   case MODE_VERIFYBU:
    fprintf(stderr,
	"Usage: %s [ -c <configuration-file> ] \\\n", pname);
    fprintf(stderr, "              [ -<past-run-no>[.<past-backup-no>] ]\n");
    break;

   default:
    fprintf(stderr, "Usage: %s [ -S <cartridge-set> ] [ -d ]"
			" [ -c <configuration-file> ]\n", pname);
  }

  do_exit(1);
}

static void
sig_handler(int s)
{
  if(mode != MODE_INCR_BACKUP && mode != MODE_FULL_BACKUP){
    kill(getpid(), s);
    do_exit(2);
  }

  signal(s, sig_handler);

  if(s != SIGPIPE){
    if(interrupted)
	return;

    fprintf(stderr, "Interrupted. Cleanup in progress, please stand by.\n");
  }
  else{
    if(!interrupted)
	errmsg("Connection to client process lost, exiting");

    logf_lost = 1;
  }

  interrupted = 1;
}

static void
exitcleanup()
{
  switch(mode){
   case MODE_FULL_BACKUP:
    if(!access(orgoldmarkfile, R_OK)){
      unlink(oldmarkfile);
      rename(orgoldmarkfile, oldmarkfile);
    }
    break;

   case MODE_INCR_BACKUP:
    unlink(newmarkfile);

   default:
    break;
  }
}

static void
intrpt_exit()
{
  exitcleanup();

  if(lfp){
    switch(mode){
     case MODE_FULL_BACKUP:
      if(numparts > 1)
	logmsg("Full backup part %d interrupted", part);
      else
	logmsg("Full backup interrupted");
      break;

     case MODE_INCR_BACKUP:
	logmsg("Incremental backup interrupted");
      break;

     default:
      break;
    }

    fclose(lfp);
  }

  do_exit(2);
}

static void
failed_exit(Int32 s)
{
  exitcleanup();

  if(lfp){
    if(mode == MODE_FULL_BACKUP){
      if(numparts > 1)
	logmsg("Full backup part %d failed", part);
      else
	logmsg("Full backup failed");
    }
    if(mode == MODE_INCR_BACKUP){
      logmsg("Incremental backup failed");
    }
    fclose(lfp);
  }

  do_exit(s);
}

Int32
set_lock()
{
  struct stat	statb;
  int		i;
  char		buf[20];

  if(locked == BU_GOT_LOCK)
    return((Int32) locked);

  if(!lockfile)
    return((Int32) BU_CANT_LOCK);

  i = stat(lockfile, &statb);
  if(!i){
    lockfd = open(lockfile, O_WRONLY | O_APPEND);
    if(lockfd < 0){
      errmsg("Warning: Lock file \"%s\" exists, but can't open it",
		lockfile);
      return( (Int32) (locked = BU_LOCKED) );
    }

    lockb.l_type = F_WRLCK;
    i = fcntl(lockfd, F_SETLK, &lockb);
    if(i){
      close(lockfd);
      return( (Int32) (locked = BU_LOCKED) );
    }

    lockb.l_type = F_UNLCK;
    fcntl(lockfd, F_SETLK, &lockb);
    close(lockfd);

    i = unlink(lockfile);
    if(i){
      errmsg("Warning: Lock file \"%s\" exists without lock, but can't remove it",
		lockfile);
      return( (Int32) (locked = BU_LOCKED) );
    }
  }

  lockfd = open(lockfile, O_WRONLY | O_CREAT | O_SYNC, 0644);
  if(lockfd < 0){
    errmsg("Error: Cannot create lock file \"%s\"", lockfile);
    return( (Int32) (locked = BU_CANT_LOCK) );
  }

  sprintf(buf, "%d\n", getpid());
  write(lockfd, buf, strlen(buf));

  lockb.l_type = F_WRLCK;
  i = fcntl(lockfd, F_SETLK, &lockb);
  if(i){
    errmsg("Error: Cannot create lock file \"%s\"", lockfile);
    return( (Int32) (locked = BU_CANT_LOCK) );
  }

  return( (Int32) (locked = BU_GOT_LOCK) );
}

static Int32
read_uns_file(UChar * filename, int * num)
{
  FILE	*fp;

  *num = 0;
  fp = fopen(filename, "r");
  if(fp){
    if(fscanf(fp, "%d", num) < 1)
	*num = 0;

    fclose(fp);
  }

  return(*num <= 0);
}

static Int32
write_uns_file(UChar * filename, int num)
{
  FILE		*fp;
  Int32	ret = 0;

  fp = fopen(filename, "w");
  if(fp){
    if(fprintf(fp, "%d\n", num) < 1)
	ret = 1;

    fclose(fp);
  }
  else
    ret = 1;

  return(ret);
}

static Int32
write_filename(UChar * name, void * data)
{
  int		fd;
  UChar		*cptr = NULL;
  Int32	ret = 0, needed_size;
  UChar		allocated = 0;
  ScanDirArgs	*scandirargs;

  scandirargs = (ScanDirArgs *) data;

  fd = scandirargs->fd;

  if(strchr(name, '\\')){
    name = repl_substring_safe(name, "\\", "\\\\");
  }
  if(strchr(name, '"')){
    name = repl_substring_safe(name, "\"", "\\\"");
  }
  needed_size = strlen(name) + 5;
  if(needed_size > TMPBUFSIZ){
    cptr = NEWP(UChar, needed_size);
    if(!cptr){
      ret = errno;
      CLEANUP;
    }

    allocated = 1;
  }
  else{
    cptr = tmpbuf;
  }

  strcpy(cptr + 1, name);
  cptr[0] = '\"';
  strcat(cptr, "\"\n");
  name = cptr;

  write(fd, name, strlen(name));

 cleanup:
  if(cptr && allocated)
    free(cptr);

  if(interrupted)
    return(1000);

  return(ret);
}

static Int32
err_write_filename(UChar * name, void * data)
{
  int		fd;
  ScanDirArgs	*scandirargs;

  scandirargs = (ScanDirArgs *) data;

  fd = scandirargs->fd;

  scandirargs->fd = 1;

  /* don't: write_filename(name, data));*/

  scandirargs->fd = fd;

  return(0);
}

main(int argc, char ** argv)
{
  UChar		*backuphome, *cptr, *cptr2, **cpptr, **cpptr2, **cpptr3;
  UChar		*zippedidxf, *tmpidxf, *tmperrf;
  UChar		*dirsarr[2], must_read_fnamlog = 0;
  UChar		*filename, *uncompcmd;
  Int32	i, j, n, lck, num_inlines = 0, num_outlines = 0;
  FILE		*fp;
  int		pid, pst, pp[3], lfd, *errflp, fd = -1;
  int		ifd, opid, ipid, errfd, i1, i2, i3;
  FindParams	findparams;
  UChar		success = 1, *infile, *outfile, inunzip, outzip;
  UChar		*rename_from = NULL, *rename_to = NULL;
  struct stat	statb, statb2;

  curuid = getuid();

  errflp = pp + 2;

  cptr = FN_BASENAME((UChar *) argv[0]);
  for(i = 0; i < sizeof(cmd_res) / sizeof(*cmd_res); i++)
    if(re_find_match_once(cmd_res[i], cptr, NULL, NULL) >= 0){
	mode = i;
	break;
    }
  if(mode < 0){
    fprintf(stderr, "Error: Command name %s not recognized.\n", cptr);
    do_exit(1);
  }

  /* check usage */
  switch(mode){
   case MODE_RESTORE:
    get_restore_args(argc, argv);
    break;

   case MODE_PRINT_ERRORS:
    get_perror_args(argc, argv);
    break;

   case MODE_VERIFYBU:
    get_verify_args(argc, argv);
    break;

   default:
    if(goptions(-argc, (UChar **) argv, "s:c;i:S;b:d", &paramfile,
			&c_cartset, &detach))
      usage(argv[0]);
  }

  if(detach){
    if(fork()){
	exit(0);
    }
    else{
	detach_from_tty();

	ms_sleep(1000 * 30);
    }
  }

  /* set signal handlers */
  signal(SIGTERM, sig_handler);
  signal(SIGHUP, sig_handler);
  signal(SIGINT, sig_handler);
  signal(SIGSEGV, sig_handler);
  signal(SIGBUS, sig_handler);
  signal(SIGPIPE, sig_handler);

  /* compile some regular expressions */
  memset(&backup_re, 0, sizeof(cartridge_re));
  memset(&cartridge_re, 0, sizeof(cartridge_re));
  memset(&file_re, 0, sizeof(file_re));
  if(re_compile_pattern(CARTRIDGE_RE, strlen(CARTRIDGE_RE), &cartridge_re)
	|| re_compile_pattern(FILE_RE, strlen(FILE_RE), &file_re)
	|| re_compile_pattern(BACKUP_RE, strlen(BACKUP_RE), &backup_re))
    nomemerrexit();

  /* determine home directory */
  backuphome = getenv("BACKUP_HOME");

#ifdef	ORIG_DEFAULTS

  if(!backuphome){
    backuphome = find_program(argv[0]);
    if(!backuphome){
      errmsg("Error: Cannot find program");
      do_exit(4);
    }
    cleanpath(backuphome);
    if(!FN_ISDIRSEP(backuphome[0]) || backuphome[0] == '.'){
	getcwd(tmpbuf, TMPBUFSIZ - 1);
	strcat(tmpbuf, FN_DIRSEPSTR);
	strcat(tmpbuf, backuphome);

	free(backuphome);
	backuphome = strdup(tmpbuf);
	if(!backuphome)
	  nomemerrexit();
    }

    cptr = FN_LASTDIRDELIM(backuphome);
    if(!cptr){
      errmsg("Strange error: cannot separate binary directory");
      do_exit(5);
    }
    *cptr = '\0';
    cptr = FN_LASTDIRDELIM(backuphome);
    if(!cptr){
      errmsg("Strange error: cannot separate client directory");
      do_exit(5);
    }
    *cptr = '\0';
  }
  else
    backuphome = strdup(backuphome);

  if(!backuphome)
    nomemerrexit();

  /* construct file- and dirnames */
  bindir = strapp(backuphome, FN_DIRSEPSTR "bin");
  vardir = strapp(backuphome, FN_DIRSEPSTR "var");
  libdir = strapp(backuphome, FN_DIRSEPSTR "lib");
  logdir = strdup(vardir);

  if(!bindir || !vardir || !libdir || !logdir)
    nomemerrexit();

  if(!paramfile)
    paramfile = strapp(libdir, FN_DIRSEPSTR "backup.conf");

#else	/* defined(ORIG_DEFAULTS) */

  if(!backuphome){
     /* construct file- and dirnames */
     bindir = DEFBINDIR;
     vardir = DEFVARDIR;
     libdir = DEFLIBDIR;
     logdir = DEFLOGDIR;
  }
  else {
     backuphome = strdup(backuphome);
     /* construct file- and dirnames */
     bindir = strapp(backuphome, FN_DIRSEPSTR "bin");
     vardir = strapp(backuphome, FN_DIRSEPSTR "var");
     libdir = strapp(backuphome, FN_DIRSEPSTR "lib");
     logdir = strapp(backuphome, FN_DIRSEPSTR "var");
  }

  if(!bindir || !vardir || !libdir || !logdir)
     nomemerrexit();

  if(!paramfile)
    paramfile = strapp(libdir, FN_DIRSEPSTR DEFCLIENTCONF);

#endif	/* if else defined(ORIG_DEFAULTS) */

  if(!stat(paramfile, &statb) && access(paramfile, R_OK)){
    errmsg("Error: Cannot read parameter file \"%s\"", paramfile);
    do_exit(6);
  }
  if(stat(paramfile, &statb)){
    for(cpptr = default_paramfiles; *cpptr; cpptr++){
      paramfile = *cpptr;
      if(!stat(paramfile, &statb) && access(paramfile, R_OK)){
	errmsg("Error: Cannot read parameter file \"%s\"", paramfile);
	do_exit(6);
      }
      if(!stat(paramfile, &statb))
	break;
    }

    if(!*cpptr){
	errmsg("Error: No parameter file found");
	do_exit(6);
    }
  }

  /* read parameter file */
  i = read_param_file(paramfile, entries,
		sizeof(entries) / sizeof(entries[0]), NULL, NULL);
  if(i){
    errmsg("Error: Cannot read parameter file");
    do_exit(6);
  }

  /* override by commandline-arguments */
  if(c_cartset > 0)
    cartset = c_cartset;

  massage_string(vardir);
  partfile = strapp(vardir, FN_DIRSEPSTR "part");
  numfile = strapp(vardir, FN_DIRSEPSTR "num");
  oldmarkfile = strapp(vardir, FN_DIRSEPSTR "oldmark");
  newmarkfile = strapp(vardir, FN_DIRSEPSTR "newmark");
  orgoldmarkfile = strapp(vardir, FN_DIRSEPSTR "oldmark.org");
  clientprogram = strapp(bindir, FN_DIRSEPSTR "afbackup");
  if(!dont_compress_str)
    dont_compress_str = "";
  if(!paramfile || !partfile || !oldmarkfile || !orgoldmarkfile
		|| !clientprogram || !newmarkfile
		|| str2wordsq(&dont_compress, dont_compress_str))
    nomemerrexit();

  for(cpptr = dont_compress, num_dont_compress = 0;
			*cpptr; cpptr++, num_dont_compress++);

  if(!cryptfile)
    cryptfile = strapp(libdir, FN_DIRSEPSTR "cryptkey");
  massage_string(cryptfile);

  /* read encryption key */
  if(!stat(cryptfile, &statb)){
    if(statb.st_mode & 0044){
	fprintf(stderr,
		"Error: Encryption key file \"%s\" is readable by unprivileged users.\n",
		cryptfile);
	do_exit(11);
    }

    if(set_cryptkey(cryptfile)){
	fprintf(stderr,
		"Warning: Cannot read enough characters from encryption key file \"%s\".\n",
		cryptfile);
	fprintf(stderr,
		"	  Ignoring file, using compiled-in key.\n");
	ZFREE(cryptfile);
    }
  }
  else
    ZFREE(cryptfile);

  /* correct parameter values, if unreasonable or set defaults */
  if(!dirstobackupraw)
    dirstobackupraw = strdup("*");
  if(!backuphost)
    backuphost = strdup(DEFAULT_SERVER);
  if(!rootdir)
    rootdir = strdup(FN_DIRSEPSTR);
  if(!indexfilepart)
    indexfilepart = strapp(logdir, FN_DIRSEPSTR "backup_log.");
  if(numparts < 1)
    numparts = 1;
  if(numindexestostore < 1)
    numindexestostore = 1;
  if(!compresscmd || !uncompresscmd){
    uncompresscmd = compresscmd = "";
    compresslogfiles = compressbu = 0;
  }
  if(!compresscmd[0] || !uncompresscmd[0]){
    uncompresscmd = compresscmd = "";
    compresslogfiles = compressbu = 0;
  }
  if(!logfile)
    logfile = "";
  massage_string(logfile);
  if(!logfile[0])
    logfile = strapp(logdir, FN_DIRSEPSTR "backup.log");
  if(!logfile)
    nomemerrexit();
  if(startinfoprog){
    if(empty_string(startinfoprog)){
	startinfoprog = NULL;
    }
  }
  if(exitprog){
    if(empty_string(exitprog) ||
		(mode != MODE_INCR_BACKUP
			&& mode != MODE_FULL_BACKUP)){
	ZFREE(exitprog);
    }
    else{
	if(strstr(exitprog, "%r"))
	  tmpnam(reportfile);
    }
  }
  if(exclude_filename){
    if(empty_string(exclude_filename)){
	exclude_filename = NULL;
    }
    else
	massage_string(exclude_filename);
  }
  massage_string(rootdir);
  massage_string(backuphost);
  massage_string(indexfilepart);

  if(!lockfile){
    for(cpptr = lockdirs; *cpptr; cpptr++){
      i = stat(*cpptr, &statb);
      cptr = strchain(*cpptr, FN_DIRSEPSTR, LOCKFILENAME, NULL);
      if(!i){
	if(! access(*cpptr, W_OK) || ! stat(cptr, &statb2)){
	  lockfile = cptr;
	  break;
	}
	else{
	  i = setgid(statb.st_gid);

	  if(! access(*cpptr, W_OK)){
	    lockfile = cptr;
	    break;
	  }
	}
      }
    }
  }
  massage_string(lockfile);

  lck = set_lock();
  if(lck != BU_GOT_LOCK){
    j = 0;
    pid = -1;
    i = read_uns_file(lockfile, &pid);
    if(!i){
	j = ! kill(pid, 0);
    }
    else{
	j = 1;
    }

    if(j){
	errmsg("Error: An application seems to hold a lock on this functionality");
	if(pid >= 0){
	  sprintf(tmpbuf, "%d", pid);
	  errmsg("	  The process ID is %s", tmpbuf);
	}
	else{
	  errmsg("	  Cannot determine process-ID");
	}

	fprintf(stderr, "Please check if this process is an obstacle to continue.\n");
	fprintf(stderr, "Do you want to continue anyway ? (y/N) ");
	tmpbuf[0] = '\0';
	fgets(tmpbuf, 10, stdin);
	cptr = tmpbuf;
	while(isspace(*cptr) && *cptr)
	  cptr++;
	if(*cptr != 'y' && *cptr != 'Y')
	  exit(99);
    }

    if(unlink(lockfile)){
	errmsg("Warning: Cannot remove lockfile \"%s\"", lockfile);
    }
    lck = set_lock();
  }

  if(lck != BU_GOT_LOCK){
	fprintf(stderr, "Warning: Cannot set lock. Continue anyway ? (y/N) ");
	tmpbuf[0] = '\0';
	fgets(tmpbuf, 10, stdin);
	cptr = tmpbuf;
	while(isspace(*cptr) && *cptr)
	  cptr++;
	if(*cptr != 'y' && *cptr != 'Y')
	  exit(99);
  }

  if(mode == MODE_INCR_BACKUP || mode == MODE_PRINT_ERRORS
		|| mode == MODE_RESTORE || mode == MODE_VERIFYBU){
    if( (i = read_uns_file(numfile, &num)) )
	cptr = numfile;

    if(mode == MODE_PRINT_ERRORS || mode == MODE_RESTORE
		|| mode == MODE_VERIFYBU){
      if(i){
	errmsg("Error: Cannot read file \"%s\"", numfile);
	do_exit(9);
      }
    }
    else{		/* MODE_INCR_BACKUP */
      if(access(oldmarkfile, R_OK)){
	i = 1;
	cptr = oldmarkfile;
      }

      if(i){
	errmsg("Warning: Cannot read file \"%s\", switching to full backup",
				cptr);
	mode = MODE_FULL_BACKUP;
      }
    }
  }

  if(mode == MODE_PRINT_ERRORS)
    print_errors(argc, argv);

  if(mode == MODE_RESTORE)
    restore(argc, argv);

  /* change to the root directory */
  if(chdir(rootdir)){
    errmsg("Error: Cannot change to directory \"%s\"", rootdir);
    do_exit(7);
  }

  if(mode == MODE_VERIFYBU)
    verifybu(argc, argv);

  part = 1;
  /* if backup is split in pieces, get again the directories to backup */
  if(numparts > 1){
    switch(mode){
     case MODE_FULL_BACKUP:
     {
      ParamFileEntry	dirsparam;
      UChar		*partdirstobackupraw = NULL;

      read_uns_file(partfile, &part);

      part++;
      if(part > numparts)
	part = 1;
      sprintf(tmpbuf, "%d:", part);

      dirsparam.pattern = repl_substring_safe(entries[0].pattern, ":", tmpbuf);
      dirsparam.entry_ptr = &partdirstobackupraw;
      dirsparam.num_entries = NULL;
      dirsparam.type = TypeUCharPTR;

      i = read_param_file(paramfile, &dirsparam, 1, NULL, NULL);
      if(i){
	errmsg("Error: Cannot read parameter file");
	do_exit(6);
      }
      free(dirsparam.pattern);

      if(partdirstobackupraw){
	if(dirstobackupraw)
	  free(dirstobackupraw);

	dirstobackupraw = partdirstobackupraw;
      }

      break;
     }

     case MODE_INCR_BACKUP:
     {
      ParamFileEntry	*dirsparams;
      UChar		**partdirstobackupraw = NULL, *cptr;

      dirsparams = NEWP(ParamFileEntry, numparts);
      partdirstobackupraw = NEWP(UChar *, numparts);
      if(!dirsparams || !partdirstobackupraw)
	nomemerrexit();

      for(i = 0; i < numparts; i++){
	sprintf(tmpbuf, "%d:", i + 1);

	dirsparams[i].pattern =
			repl_substring_safe(entries[0].pattern, ":", tmpbuf);
	dirsparams[i].entry_ptr = partdirstobackupraw + i;
	dirsparams[i].num_entries = NULL;
	dirsparams[i].type = TypeUCharPTR;
      }

      i = read_param_file(paramfile, dirsparams, numparts, NULL, NULL);
      if(i){
	errmsg("Error: Cannot read parameter file");
	do_exit(6);
      }

      if(dirstobackupraw)
	free(dirstobackupraw);
      dirstobackupraw = strdup("");

      for(i = 0; i < numparts; i++){
	cptr = strchain(dirstobackupraw, " ",
					partdirstobackupraw[i], NULL);
	free(dirstobackupraw);
	dirstobackupraw = cptr;
	free(partdirstobackupraw[i]);
	free(dirsparams[i].pattern);
      }
      free(partdirstobackupraw);
      free(dirsparams);

      break;
     }
     default:
      break;
    }
  }

  /* convert the strings (containing lists) to string arrays */
  if(str2wordsq(&dirstobackup, dirstobackupraw ? dirstobackupraw : ES)
	|| str2wordsq(&dirstoskip, dirstoskipraw ? dirstoskipraw : ES)
	|| str2wordsq(&filestoskip, filestoskipraw ? filestoskipraw : ES))
    nomemerrexit();

  for(cpptr = dirstobackup; *cpptr; cpptr++){
    if(!strncmp(*cpptr, FILECONTPREFIX, i = strlen(FILECONTPREFIX))){
	cleanpath(*cpptr + i);
    }
    else if(!strncmp(*cpptr, FILECONTZPREFIX, i = strlen(FILECONTZPREFIX))){
	cleanpath(*cpptr + i);
    }
    else{
	cleanpath(*cpptr);
    }
  }
  for(cpptr = dirstoskip; *cpptr; cpptr++)
    cleanpath(*cpptr);
  for(cpptr = filestoskip; *cpptr; cpptr++)
    cleanpath(*cpptr);

#if 0	/* don't do this: it's dangerous, when paths are symlinked or cross-filesystem */
  /* resolve glob patterns */
  alldirs = NEWP(UChar *, 1);
  *alldirs = NULL;
  num_alldirs = 0;
  for(cpptr = dirstobackup; *cpptr; cpptr++){
    cpptr2 = fnglob(*cpptr);
    if(cpptr2){
      for(n = 0, cpptr3 = cpptr2; *cpptr3; cpptr3++, n++);

      alldirs = RENEWP(alldirs, UChar *, n + num_alldirs + 1);
      memcpy(alldirs + num_alldirs, cpptr2, sizeof(UChar *) * n);
      alldirs[n + num_alldirs] = NULL;
      num_alldirs += n;

      free(cpptr2);
    }

    free(*cpptr);
  }
  free(dirstobackup);
  dirstobackup = alldirs;
      
  n = num_alldirs;

  {	/* throw out redundant paths (e.g. /home/alb when /home is present) */
    for(i = 0; i < n - 1; i++){
      for(j = i + 1; j < n; j++){
	if(strlen(dirstobackup[i]) < strlen(dirstobackup[j])
		&& !strncmp(dirstobackup[i], dirstobackup[j],
						strlen(dirstobackup[i]))){
	  for(cpptr = dirstobackup + j; *cpptr; cpptr++){
	    *cpptr = *(cpptr + 1);
	  }
	  j--;
	  n--;
	}
	else if(strlen(dirstobackup[j]) < strlen(dirstobackup[i])
		&& !strncmp(dirstobackup[j], dirstobackup[i],
						strlen(dirstobackup[j]))){
	  for(cpptr = dirstobackup + i; *cpptr; cpptr++){
	    *cpptr = *(cpptr + 1);
	  }
	  i--;
	  n--;
	  break;
	}
      }
    }
  }
#endif

  for(cpptr = dirstobackup; *cpptr; cpptr++){
    for(cpptr2 = option2prefix; *cpptr2; cpptr2++){	/* substitute options */
      cptr = sscanword(*cpptr2, tmpbuf);		/* by prefix */
      if(!strcmp(tmpbuf, *cpptr)){
	sscanword(cptr, tmpbuf);
	free(*cpptr);

	if(*(cpptr + 1)){
	  *cpptr = strapp(tmpbuf, *(cpptr + 1));

	  for(cpptr3 = cpptr + 1; *cpptr3; cpptr3++)
	    *(cpptr3) = *(cpptr3 + 1);
	  cpptr--;
	}
	else
	  *cpptr = NULL;

	break;
      }
    }
  }

  for(cpptr = dirstobackup, n = 0; *cpptr; cpptr++, n++);

  for(i = 0; i < n - 1; i++){		/* remove duplicate entries */
    for(j = i + 1; j < n; j++){
      if(! strcmp(dirstobackup[j], dirstobackup[i])){
	for(cpptr2 = dirstobackup + j; *cpptr2; cpptr2++)
	  *(cpptr2) = *(cpptr2 + 1);
	j--;
	n--;
      }
    }
  }

  filecontentstobu = NEWP(UChar *, 1);
  if(! filecontentstobu)
    nomemerrexit();
  *filecontentstobu = NULL;
  i = 0;
  for(cpptr = dirstobackup; *cpptr; cpptr++){
    if(!strncmp(*cpptr, FILECONTPREFIX, strlen(FILECONTPREFIX))
		|| !strncmp(*cpptr, FILECONTZPREFIX, strlen(FILECONTZPREFIX))){
	filecontentstobu = RENEWP(filecontentstobu, UChar *, i + 2);
	if(!filecontentstobu)
	  nomemerrexit();

	filecontentstobu[i] = *cpptr;
	filecontentstobu[i + 1] = NULL;
	if(filecontentstobu[i][0])
	  i++;

	for(cpptr2 = cpptr; *cpptr2; cpptr2++)
	  *(cpptr2) = *(cpptr2 + 1);
	cpptr--;
    }
  }

  /* open logfile */
  if(mode == MODE_FULL_BACKUP || mode == MODE_INCR_BACKUP)
    lfp = fopen(logfile, "a");

  if(lfp){
    switch(mode){
     case MODE_FULL_BACKUP:
      if(numparts > 1)
	logmsg("Starting full backup part %d", part);
      else
	logmsg("Starting full backup");
      break;

     case MODE_INCR_BACKUP:
      logmsg("Starting incremental backup");
      break;

     default:
      break;
    }
  }

  /* read the number of the backup, if file present */
  if(mode == MODE_FULL_BACKUP){
    if(read_uns_file(numfile, &num))
      part = 1;

    if(part == 1)
      num++;
  }

  /* construct the filename logfile */
  sprintf(tmpbuf, "%d", num);
  indexfile = strapp(indexfilepart, tmpbuf);
  if(!indexfile)
    nomemerrexit();

  /* if it's a new backup, create the timestamp-file and save the old one */
  if(part == 1 && mode == MODE_FULL_BACKUP){
    if(!access(oldmarkfile, R_OK)){
      unlink(orgoldmarkfile);
      if(rename(oldmarkfile, orgoldmarkfile)){
	errmsg("Error: Cannot rename file \"%s\"", oldmarkfile);
	failed_exit(9);
      }
    }

    fp = fopen(oldmarkfile, "w");
    if(!fp){
      errmsg("Error: Cannot create file \"%s\"", oldmarkfile);
      failed_exit(10);
    }
    fclose(fp);
  }

  if(mode == MODE_INCR_BACKUP){
    fp = fopen(newmarkfile, "w");
    if(!fp){
      errmsg("Error: Cannot create file \"%s\"", newmarkfile);
      failed_exit(10);
    }
    fclose(fp);
  }

  if(interrupted)
    intrpt_exit();

  /* get startup information (actual writing position) and log it */
  sprintf(tmpbuf, "%s -Q -h %s", clientprogram, backuphost);

  if(backupport >= 0)
    sprintf(tmpbuf + strlen(tmpbuf), " -p %d", backupport);

  if(cryptfile)
    sprintf(tmpbuf + strlen(tmpbuf), " -k %s", cryptfile);

  if(cartset > 1)
    sprintf(tmpbuf + strlen(tmpbuf), " -S %d", cartset);

  fd = open_from_pipe(tmpbuf, NULL, 1 + 2, &pid);
  if(fd >= 0){
    i = read(fd, tmpbuf, 999);
    tmpbuf[i] = '\0';

    waitpid_forced(pid, &pst, 0);
    close(fd);
    if(WEXITSTATUS(pst)){
      errmsg("Warning: Cannot get startup information");
    }
    else{
	cptr = tmpbuf;
	forever{
	  if( (cptr2 = strchr(cptr, '\n')) )
	    *cptr2 = '\0';

	  if(re_find_match(&cartridge_re, cptr, NULL, NULL) >= 0){
	    cptr = cptr + strlen(cptr) - 1;
	    while(!isspace(*cptr))
		cptr--;
	    sscanf(cptr, "%d", &inscartno);
	  }
	  if(re_find_match(&file_re, cptr, NULL, NULL) >= 0){
	    cptr = cptr + strlen(cptr) - 1;
	    while(!isspace(*cptr))
		cptr--;
	    sscanf(cptr, "%d", &insfileno);
	  }

	  if(!cptr2)
	    break;
	  cptr = cptr2 + 1;
	}

	if(startinfoprog){
	  fp = popen(startinfoprog, "w");
	  fprintf(fp, EMREC_PREFIX "Backup:    %8d\n", num);
	  fprintf(fp, EMREC_PREFIX "Cartridge: %8d\n", inscartno);
	  fprintf(fp, EMREC_PREFIX "File:      %8d\n", insfileno);
	  pclose(fp);
	}
    }
  }

  if(interrupted)
    intrpt_exit();

  zippedidxf = strapp(indexfile, COMPRESS_SUFFIX);
  if(!access(zippedidxf, R_OK) && !access(indexfile, R_OK)){
    stat(indexfile, &statb);
    stat(zippedidxf, &statb2);

    if(statb.st_size == 0){
	unlink(indexfile);
    }
    if(statb2.st_size == 0){
	unlink(zippedidxf);
    }
  }

	/* If there are both files, this must be the result of a
	 * crash of this program (maybe, of course, due to a
	 * kill -9 or other mean kill-s). So the newer file MUST
	 * be the incomplete one and we can remove it, agree ?
	 */
  if(!access(zippedidxf, R_OK) && !access(indexfile, R_OK)){
    if(statb2.st_mtime > statb.st_mtime){
      unlink(zippedidxf);
    }
    else{
      unlink(indexfile);
    }
  }

  tmpidxf = strapp(indexfile, TMP_SUFFIX);
  unlink(tmpidxf);
  tmperrf = strapp(logfile, TMP_SUFFIX);
  unlink(tmperrf);

  infile = outfile = rename_from = rename_to = NULL;
  inunzip = outzip = 0;
  ifd = lfd = ipid = opid = errfd = -1;

  if(!access(indexfile, R_OK)){
    infile = (compresslogfiles ? indexfile : NULL);
    inunzip = 0;
  }

  if(!access(zippedidxf, R_OK)){
    inunzip = 1;

    if(compresslogfiles){
	infile = rename_to = zippedidxf;
    }
    else{
	infile = zippedidxf;
    }
  }
	
  outzip = (compresslogfiles && compresscmd[0] ? 1 : 0);
  outfile = (outzip ? (inunzip ? tmpidxf : zippedidxf) : indexfile);
  filelist = outfile;

  if(infile){
    cptr = inunzip ? (uncompresscmd[0] ? uncompresscmd : (UChar *) "gunzip") : NULL;
    ifd = open_from_pipe(cptr, infile, 1, &ipid);
    if(ifd < 0){
	errmsg("Warning: Cannot read filename logfile \"%s\"", infile);
    }
  }
  lfd = open_to_pipe(outzip ? compresscmd : NULL, outfile, 1 + 2, &opid, 0600);
  if(lfd < 0){
    errmsg("Error: Cannot write filename logfile \"%s\". Logging to stderr",
					outfile);
    lfd = 2;
    outfile = NULL;
  }

  num_inlines = 0;

  if(ifd >= 0 && lfd >= 0){
    while((i = read(ifd, tmpbuf, TMPBUFSIZ - 1)) > 0){
	write(lfd, tmpbuf, i);

	tmpbuf[i] = '\0';	/* count lines */
	cptr = tmpbuf;
	while( (cptr = strchr(cptr, '\n')) ){
	  num_inlines++;
	  cptr++;
	}
    }

    close(ifd);
    if(ipid >= 0){
      waitpid_forced(ipid, &pst, 0);

      if(WEXITSTATUS(pst)){
	errmsg("Warning: Errors occurred reading \"%s\"", infile);
      }
    }
  }

#if 0	/* This is nonsense. Filenames go to stderr !!! */
  /* open file for child (client) error log */
  errfd = open(tmperrf, O_WRONLY | O_CREAT | O_BINARY, 0600);
  if(errfd < 0){
    errmsg("Error: Cannot open file \"%s\" for client process error log.\n",
			tmperrf);
  }
#endif

  sprintf(tmpbuf, "%s backup at %s, current number: %d\n%d.%d:\n",
		mode == MODE_INCR_BACKUP ? "\n## Incremental" : "## Full",
		(char *) actimestr(), (int) num,
		(long int) inscartno, (long int) insfileno);
  write(lfd, tmpbuf, strlen(tmpbuf));

  /* now start the backup client subprocess */
  i = pipe(pp);
  if(!i){
    pid = fork_forced();
    if(pid < 0){
	errmsg("Error: Cannot start backup client side subprocess");
	failed_exit(12);
    }
    if(!pid){		/* child */
	char	**clargv;
	char	portnostr[10],cartsetnostr[10];

	close(pp[1]);
	dup2(pp[0], 0);
	dup2(lfd, 1);
	dup2(errfd >= 0 ? errfd : lfd, 2);

	signal(SIGTERM, SIG_IGN);
	signal(SIGINT, SIG_IGN);

	clargv = NEWP(char *, 18 + num_dont_compress * 2);

	cpptr = (UChar **) clargv;
	*(cpptr++) = FN_BASENAME(clientprogram);
	*(cpptr++) = "-cvnOE";
	*(cpptr++) = "-h";
	*(cpptr++) = backuphost;
	if(backupport >= 0){
	  *(cpptr++) = "-p";
	  sprintf(portnostr, "%d", backupport);
	  *(cpptr++) = portnostr;
	}
	if(compressbu){
	  *(cpptr++) = "-z";
	  *(cpptr++) = compresscmd;
	  *(cpptr++) = uncompresscmd;
	}
	if(cryptfile){
	  *(cpptr++) = "-k";
	  *(cpptr++) = cryptfile;
	}
	if(dont_compress){
	  for(cpptr2 = dont_compress; *cpptr2; cpptr2++){
	    *(cpptr++) = "-s";
	    *(cpptr++) = *cpptr2;
	  }
	}
	if(cartset > 1){
	  sprintf(cartsetnostr, "%d", cartset);
	  *(cpptr++) = "-S";
	  *(cpptr++) = cartsetnostr;
	}
	if(reportfile[0]){
	  *(cpptr++) = "-V";
	  *(cpptr++) = reportfile;
	}
	*cpptr = NULL;
	execv(clientprogram, clargv);

	do_exit(24);
    }
  }
  else{
    errmsg("Error: Cannot create pipe to subprocess");
    failed_exit(13);
  }

  close(pp[0]);
  if(outfile)
    close(lfd);
  if(errfd >= 0)
    close(errfd);

  *errflp = 0;

  SETZERO(findparams);
  findparams.excl_dirs = dirstoskip;
  findparams.excl_names = filestoskip;
  findparams.errfunc = err_write_filename;
  findparams.errfunc_param = (void *) (pp + 1);
  findparams.excl_filename = exclude_filename;
  if(mode == MODE_INCR_BACKUP){
    stat(oldmarkfile, &statb);
    findparams.newer_than = statb.st_mtime;
  }

  success = 1;

  forever{	/* pseudo-loop */
    dirsarr[1] = NULL;

    for(cpptr = dirstobackup; *cpptr; cpptr++){
      dirsarr[0] = *cpptr;

      if(!strncmp(dirsarr[0], LOCALDEVPREFIX, i = strlen(LOCALDEVPREFIX))){
	findparams.options |= FIND_LOCAL_DEV;
	dirsarr[0] += i;
      }
      else{
	findparams.options &= ~FIND_LOCAL_DEV;
      }

      if(!dirsarr[0][0])
	continue;

      SETZERO(scandir_args);
      scandir_args.fd = pp[1];

      i = find(dirsarr, &findparams, write_filename, (void *) (& scandir_args));
      success = (success && (! i));

      if(interrupted)
	break;
    }

    if(interrupted)
    	break;

    i = 0;
    for(cpptr = filecontentstobu; *cpptr; cpptr++){
      if(! cpptr[0][0])
	continue;

      i |= write_filename(*cpptr, (void *) (pp + 1));

      if(interrupted)
	break;
    }

    success = (success && (! i));

    break;
  }

  close(pp[1]);
  waitpid_forced(pid, &pst, 0);
  *errflp |= WEXITSTATUS(pst);
  if(opid >= 0){
    waitpid_forced(opid, &pst, 0);
    *errflp |= WEXITSTATUS(pst);
  }

#if 0	/* This is nonsense */
  /* move error messages from the temporary error file to the logfile */
  errfp = fopen(tmperrf, "r");
  if(errfp){
    while(!feof(errfp) && (cptr = fget_alloc_str(errfp)) ){
	errmsg(cptr);
	free(cptr);
    }

    fclose(errfp);
  }
  unlink(tmperrf);
#endif

  if(logf_lost || *errflp)	/* This happens, when the child dies */
    must_read_fnamlog = 1;

  if(must_read_fnamlog){
    num_outlines = 0;
    if(inunzip || outzip){
      ipid = -1;			/* count written lines */
      uncompcmd = outzip ? (uncompresscmd[0] ? uncompresscmd : (UChar *) "gunzip") : NULL;
      filename = (inunzip && outzip) ? tmpidxf : outfile;
      i = stat(filename, &statb);
      if(i){
	errmsg("Warning: Cannot read filename logfile \"%s\"", filename);
      }
      if(!i && statb.st_size > 0){
	ifd = open_from_pipe(uncompcmd, filename, 1, &ipid);
	if(ifd < 0){
	  errmsg("Warning: Cannot read filename logfile \"%s\"", filename);
	}
	else{
	  fp = fdopen(ifd, "r");

	  while(!feof(fp)){
	    cptr = fget_alloc_str(fp);
	    if(!cptr)
		continue;

	    num_outlines++;

	    if(num_outlines > num_inlines && !empty_string(cptr)
			&& sscanf(cptr, "%d.%d:", &i1, &i2) != 2
			&& sscanf(cptr, "%d.%d" UIDSEP "%d:",
						&i1, &i2, &i3) != 3
			&& *(first_nospace(cptr)) != '#'){
		fprintf(stderr, "%s", cptr);
	    }			/* print new error messages in log */

	    free(cptr);
	  }

	  fclose(fp);
	  if(ipid >= 0){
	    waitpid_forced(ipid, &pst, 0);
	    if(WEXITSTATUS(pst)){
		errmsg("Warning: Cannot find out number of lines written to \"%s\"",
				filename);
		num_outlines = 0;
	    }
	  }
	}
      }
    }

    if(num_outlines > num_inlines){	/* That's what it's all about: */
	logf_lost = 0;			/* new file is longer, so nothing */
    }					/* is lost. */
  }

  if(logf_lost){
    filelist = infile;

    if(inunzip && outzip){
	unlink(tmpidxf);
    }
    else{
      if(inunzip || outzip)
	unlink(outfile);
    }
  }
  else{
    if(inunzip && outzip){
	unlink(zippedidxf);
	rename(tmpidxf, zippedidxf);
	filelist = zippedidxf;
    }
    else{
      if(inunzip || outzip)
	unlink(infile);
    }
  }

  if(interrupted)
    intrpt_exit();

  if(! success){
    errmsg("Error: An error occurred during full backup");
    errmsg("The logfile \"%s\" and the command %s" FN_DIRSEPSTR "print_errors should tell details", logfile, bindir);
    failed_exit(14);
  }

  if(!success)
    failed_exit(15);

  if(mode == MODE_FULL_BACKUP){
    /* write the backup part number to file */
    if(numparts > 1){
      if(write_uns_file(partfile, part)){
	errmsg("Error: Cannot write to file \"%s\"", partfile);
	failed_exit(16);
      }

      logmsg("Full backup part %d finished", part);
    }
    else{
      logmsg("Full backup finished");
    }

    /* write the backup number to file */
    if(part == 1){
      if(write_uns_file(numfile, num)){
	errmsg("Error: Cannot write to file \"%s\"", numfile);
	failed_exit(8);
      }
    }

    unlink(orgoldmarkfile);

    /* remove dated filename logfiles */
    if(numindexestostore >= 0){
      num -= (numindexestostore + 1);

      i = strlen(indexfile) + 30;
      if(i < TMPBUFSIZ)
	cptr = tmpbuf;
      else
	cptr = NEWP(UChar, i);
      if(cptr){
	while(num > 0){
	  sprintf(cptr, "%s%d", indexfilepart, num);
	  unlink(cptr);
	  strcat(cptr, COMPRESS_SUFFIX);
	  unlink(cptr);
	  num--;
	}
      }
    }
  }

  if(mode == MODE_INCR_BACKUP){
    unlink(oldmarkfile);
    rename(newmarkfile, oldmarkfile);

    logmsg("Incremental backup finished");
  }

  if(*errflp){
    errmsg("Warning: Minor errors occurred during backup. Try %s" FN_DIRSEPSTR "print_errors for details", bindir);
    errmsg("Also the file \"%s\" should supply additional information",
			logfile);
  }

  do_exit(0);
}

static int
open_idxfile_read(int * pid)
{
  UChar		*zippedname = NULL;
  int		fd, lpid;

  sprintf(tmpbuf, "%d", num);
  indexfile = strapp(indexfilepart, tmpbuf);

  lpid = fd = -1;
  if(!access(indexfile, R_OK)){
    fd = open(indexfile, O_RDONLY);
  }
  else{
    zippedname = strapp(indexfile, COMPRESS_SUFFIX);
    if(access(zippedname, R_OK))
	errmsg("Error: No filename logfile found");
    else
	fd = open_from_pipe(uncompresscmd[0] ? uncompresscmd : (UChar*)"gunzip",
			zippedname, 1, &lpid);
  }

  if(fd >= 0 && lpid >= 0)
    *pid = lpid;

  return(fd);
}

static Int32
get_perror_args(int argc, char ** argv)
{
  int		i;
  UChar		have_prev = 0;

			/* option -p is handled in the filelist processing */
  i = goptions(-argc, (UChar **) argv,
    "b:0;b:1;b:2;b:3;b:4;b:5;b:6;b:7;b:8;b:9;s:c;i:S",
	&have_prev, &have_prev, &have_prev, &have_prev, &have_prev,
	&have_prev, &have_prev, &have_prev, &have_prev, &have_prev,
	&paramfile, &c_cartset);

  if(i)
    usage(argv[0]);

  if(have_prev)
    num_prev_backup = get_arg_num(argc, argv, &numidx);

  return(0);
}

static Int32
get_verify_args(int argc, char ** argv)
{
  int		i, backupno, runno, n;
  UChar		have_prev = 0;

			/* option -p is handled in the filelist processing */
  i = goptions(-argc, (UChar **) argv,
    "b:0;b:1;b:2;b:3;b:4;b:5;b:6;b:7;b:8;b:9;b:.;s:c;i:S",
	&have_prev, &have_prev, &have_prev, &have_prev, &have_prev,
	&have_prev, &have_prev, &have_prev, &have_prev, &have_prev,
	&have_prev, &paramfile, &c_cartset);

  if(i)
    usage(argv[0]);

  if(have_prev){
    for(i = 1; i < argc; i++){
	n = sscanf(argv[i] + 1, "%d.%d", &runno, &backupno);
	if(n > 1)
	  num_prev_backup = backupno;
	if(n > 0){
	  num_prev_run = runno;
	  break;
	}
    }
  }

  return(0);
}

static void
print_errors(int c, char ** v)
{
  UChar		*line;
  int		fd, pid = -1, pst, i, j, k;
  FILE		*fp;

  if(num_prev_backup > 0)
    num -= num_prev_backup;

  fd = open_idxfile_read(&pid);
  fp = fdopen(fd, "r");
  if(!fp){
    errmsg("Error: Cannot read file \"%s\"", indexfile);
    do_exit(8);
  }

  fprintf(stdout, "Errors in backup # %d:\n\n", num);

  while(!feof(fp)){
    line = fget_alloc_str(fp);
    if(!line)
	break;

    if(sscanf(line, "%d.%d:", &i, &j) < 2 
		&& sscanf(line, "%d.%d" UIDSEP "%d:", &i, &j, &k) < 3
		&& !empty_string(line) && *(first_nospace(line)) != '#')
      fprintf(stdout, "%s", line);

    free(line);
  }

  fclose(fp);

  if(pid >= 0){
    waitpid_forced(pid, &pst, 0);
    if(WEXITSTATUS(pst)){
      errmsg("Warning: Some errors occurred uncompressing \"%s\"", indexfile);
    }
  }

  do_exit(0);
}

static void
verifybu(int c, char ** v)
{
  UChar		*line;
  int		fd, pid = -1, pst, i, j, k, n;
  int		*cnos = NULL, *fnos = NULL, listidx;
  FILE		*fp;

  if(num_prev_backup > 0)
    num -= num_prev_backup;

  cnos = NEWP(int, num_prev_run + 1);
  fnos = NEWP(int, num_prev_run + 1);
  if(!cnos || !fnos)
    nomemerrexit();

  for(i = 0; i <= num_prev_run; i++)
    cnos[i] = fnos[i] = -1;

  fd = open_idxfile_read(&pid);
  fp = fdopen(fd, "r");
  if(!fp){
    errmsg("Error: Cannot read file \"%s\"", indexfile);
    do_exit(8);
  }

  listidx = 0;
  while(!feof(fp)){
    line = fget_alloc_str(fp);
    if(!line)
	break;

    k = sscanf(line, "%d.%d" UIDSEP "%d:%n", &i, &j, &k, &n);
    if(k < 3)
	k = sscanf(line, "%d.%d:%n", &i, &j, &n);
    if(k >= 2){
      if(empty_string(line + n)){
	cnos[listidx] = i;
	fnos[listidx] = j;
	listidx = (listidx + 1) % (num_prev_run + 1);
      }
    }

    free(line);
  }
  fclose(fp);

  if(pid >= 0){
    waitpid_forced(pid, &pst, 0);
    if(WEXITSTATUS(pst)){
      errmsg("Warning: Some errors occurred uncompressing \"%s\"", indexfile);
    }
  }

  if(cnos[listidx] < 0 && fnos[listidx] < 0){
    errmsg("Error: Did not find start mark of previous backup in \"%s\"",
		indexfile);
    do_exit(9);
  }

  sprintf(tmpbuf, "%s -d -C %d -F %d -h %s", clientprogram,
		cnos[listidx], fnos[listidx], backuphost);

  if(cryptfile)
    sprintf(tmpbuf + strlen(tmpbuf), " -k %s", cryptfile);

  if(backupport >= 0)
    sprintf(tmpbuf + strlen(tmpbuf), " -p %d", backupport);

  if(cartset > 1)
    sprintf(tmpbuf + strlen(tmpbuf), " -S %d", cartset);

  system(tmpbuf);

  do_exit(0);
}

static int
get_arg_num(int argc, char ** argv, Int32 * idx)
{
  int	i, n, r = -1;

  for(i = 1; i < argc; i++){
    if(argv[i][0] == '-'){
	sscanf(argv[i] + 1, "%d%n", &r, &n);
	if(strlen(argv[i] + 1) == n)
	  break;
    }
  }

  if(i < argc && idx)
    *idx = i;

  return(r);
}

static Int32
get_restore_args(int argc, char ** argv)
{
  int		i;
  UChar		have_prev = 0;

			/* option -p is handled in the filelist processing */
  i = goptions(-argc, (UChar **) argv,
    "b:0;b:1;b:2;b:3;b:4;b:5;b:6;b:7;b:8;b:9;b:n;b:l;s:h;s:R;b:a;b:e;b:f;"
		"s:B;s:A;s:c;i:S;*",
	&have_prev, &have_prev, &have_prev, &have_prev, &have_prev,
	&have_prev, &have_prev, &have_prev, &have_prev, &have_prev,
	&do_counting, &do_listing, &restorehost, &restoreroot,
	&restore_all, &restore_em, &restore_emlist, &beforestr, &afterstr,
	&paramfile, &c_cartset, &num_restore_paths, &restore_paths);

  if(i)
    usage(argv[0]);

  if(num_restore_paths == 0 &&
		!restore_all && !restore_em && !restore_emlist)
    usage(argv[0]);

  i = 0;
  if(restore_all)
    i++;
  if(do_listing || do_counting)
    i++;
  if(restore_em || restore_emlist)
    i++;
  if(i > 1)
    usage(argv[0]);
  if(restore_em && restore_emlist)
    usage(argv[0]);
  if(have_prev && (restore_emlist || restore_em))
    usage(argv[0]);
  if(num_restore_paths > 0 && (restore_emlist || restore_em || restore_all))
    usage(argv[0]);

  if(have_prev)
    num_prev_backup = get_arg_num(argc, argv, NULL);

  if(afterstr){
    time_newer = time_from_datestr(afterstr);
    if(time_newer == (time_t) -1){
	fprintf(stderr, "Error: Cannot read date and/or time from \"%s\".\n",
			afterstr);
	usage(argv[0]);
    }

    fprintf(stdout, "Restoring files not older than: %s\n", ctime(&time_newer));
  }
  if(beforestr){
    time_older = time_from_datestr(beforestr);
    if(time_older == (time_t) -1){
	fprintf(stderr, "Error: Cannot read date and/or time from \"%s\".\n",
			beforestr);
	usage(argv[0]);
    }

    fprintf(stdout, "Restoring files not newer than: %s\n", ctime(&time_older));
  }

  return(0);
}

#define	startpattern	"^## [a-zA-Z]+ backup at [a-zA-Z0-9: ]+, "	\
				"current number: [1-9][0-9]*$"

static void
restore(int c, char ** v)
{
  static UChar		re_initialized = 0;
  static RE_cmp_buffer	startre;

  UChar		**cpptr, **cpptr2, **matches = NULL, *line, *cptr;
  UChar		**newmatches = NULL, added;
  int		i, n, num_matches = 0, fd, cartno, fileno, allocated;
  int		pst, pid, fileuid;
  FILE		*fp;
  struct stat	statb;
  time_t	newtim, *starttimes = NULL, *endtimes = NULL;
  time_t	*newstarttimes = NULL, *newendtimes = NULL;

  if(!re_initialized){
    SETZERO(startre);
    re_compile_pattern(startpattern, strlen(startpattern), &startre);
    re_initialized = 1;
  }

  if(curuid){
    if(setuid(0)){
	errmsg("Error: Cannot setuid root, sorry.");
	do_exit(17);
    }
  }

  if(restorehost)
    backuphost = restorehost;
  if(restoreroot){
    i = 0;

    if(curuid){
	i = access(restoreroot, R_OK);
	if(i){
	  errmsg("Error: Cannot read directory \"%s\". No read permission",
			restoreroot);
	  do_exit(20);
	}
	i = stat(restoreroot, &statb);
	if(i){
	  errmsg("Error: Cannot stat directory \"%s\"", restoreroot);

	  do_exit(21);
	}
	if(statb.st_uid != curuid){
	  errmsg("Error: The directory \"%s\" is not owned by you",
			restoreroot);

	  do_exit(22);
	}
    }

    rootdir = restoreroot;
  }

  /* change to the root directory */
  if(chdir(rootdir)){
    errmsg("Error: Cannot change to directory \"%s\"", rootdir);
    do_exit(7);
  }

  if(restore_em || restore_emlist)
    restoreem(c, v);

  for(i = 0; i < num_restore_paths; i++){
    if(!strcmp(restore_paths[i], "-p")){
	num_restore_paths--;
	free(restore_paths[i]);
	for(n = i; n < num_restore_paths; n++)
	  restore_paths[n] = restore_paths[n + 1];
    }
    else{
	cptr = strchain("*", restore_paths[i], "*", NULL);
	free(restore_paths[i]);
	restore_paths[i] = cptr;
    }
  }

  have_timerange = (time_older > 0 || time_newer > 0);

  indexfiles = NEWP(UChar *, 2);
  zindexfiles = NEWP(UChar *, 2);
  if(!indexfiles || !zindexfiles)
    nomemerrexit();
  memset(indexfiles, 0, 2 * sizeof(UChar *));
  memset(zindexfiles, 0, 2 * sizeof(UChar *));

  if(!have_timerange){
    if(num_prev_backup > 0){
      num -= num_prev_backup;
      sprintf(tmpbuf, "%d", num);
      indexfiles[0] = strapp(indexfilepart, tmpbuf);
    }
    else{
      if(read_uns_file(partfile, &part))
	part = 1;

      if(part < numparts && num > 1){
	sprintf(tmpbuf, "%d", num - 1);
	indexfiles[0] = strapp(indexfilepart, tmpbuf);
	sprintf(tmpbuf, "%d", num);
	indexfiles[1] = strapp(indexfilepart, tmpbuf);
	zindexfiles[1] = strapp(indexfiles[1], COMPRESS_SUFFIX);
      }
      else{
	sprintf(tmpbuf, "%d", num);
	indexfiles[0] = strapp(indexfilepart, tmpbuf);
      }
    }
    zindexfiles[0] = strapp(indexfiles[0], COMPRESS_SUFFIX);
  }
  else{
    num_indexfiles = 0;

    sprintf(tmpbuf, "%s%d", indexfilepart, num);
    forever{
      cptr = tmpbuf + strlen(tmpbuf) + 2;
      sprintf(cptr, "%s%s", tmpbuf, COMPRESS_SUFFIX);

      num_indexfiles++;

      indexfiles = RENEWP(indexfiles, UChar *, num_indexfiles + 1);
      zindexfiles = RENEWP(zindexfiles, UChar *, num_indexfiles + 1);
      indexfiles[num_indexfiles - 1] = strdup(tmpbuf);
      zindexfiles[num_indexfiles - 1] = strdup(cptr);
      indexfiles[num_indexfiles] = zindexfiles[num_indexfiles] = NULL;

      num--;
      sprintf(tmpbuf, "%s%d", indexfilepart, num);
      if(!access(tmpbuf, R_OK))
	continue;
      cptr = tmpbuf + strlen(tmpbuf) + 2;
      sprintf(cptr, "%s%s", tmpbuf, COMPRESS_SUFFIX);
      if(!access(cptr, R_OK))
	continue;

      break;
    }

    for(i = 0; i < num_indexfiles / 2; i++){
	memswap(indexfiles + i, indexfiles + num_indexfiles - 1 - i,
				sizeof(UChar *));
	memswap(zindexfiles + i, zindexfiles + num_indexfiles - 1 - i,
				sizeof(UChar *));
    }
  }

  for(cpptr = indexfiles, i = 0; *cpptr; cpptr++, i++){
    if(access(*cpptr, R_OK) && access(zindexfiles[i], R_OK)){
	errmsg("Serious warning: Cannot read filename logfile \"%s\"", *cpptr);
	free(*cpptr);
	free(zindexfiles[i]);

	for(cpptr2 = cpptr, n = i; *cpptr2; cpptr2++, n++){
	  *cpptr2 = *(cpptr2 + 1);
	  zindexfiles[n] = zindexfiles[n + 1];
	}

	cpptr--;
	i--;
    }
  }

  if(restore_all)
    restoreall(c, v);

  line = NULL;
  allocated = 0;
  current_bu_time = 0;
  prev_bu_time = 0;

  for(cpptr = indexfiles; *cpptr; cpptr++){
    pid = -1;
    fd = open(*cpptr, O_RDONLY);
    if(fd < 0){
      if((fd = open_from_pipe(uncompresscmd[0] ? uncompresscmd : (UChar*)"gunzip",
		zindexfiles[cpptr - indexfiles], 1, &pid)) < 0)
	continue;
    }
    fp = fdopen(fd, "r");
    if(!fp){
	errmsg("Error: Cannot read file \"%s\"", *cpptr);
	continue;
    }
    while(!feof(fp)){
      if(line)
	free(line);
      line = fget_alloc_str(fp);
      if(!line)
	continue;

      cptr = line + strlen(line) - 1;
      if(*cptr == '\n')
	*cptr = '\0';

      i = sscanf(line, "%d.%d" UIDSEP "%d:%n",
				&cartno, &fileno, &fileuid, &n);
      if(i < 3)
        i = sscanf(line, "%d.%d:%n", &cartno, &fileno, &n);
      if(i < 2){
	if(!strncmp(cptr = first_nospace(line), "##", 2)){
	  if(re_find_match(&startre, cptr, NULL, NULL) == 0){
	    for(i = 0; i < 4; i++){
		while(!isspace(*cptr))
		  cptr++;
		cptr = first_nospace(cptr);
	    }
	    *(strchr(cptr, ',')) = '\0';

	    newtim = time_from_datestr(cptr);
	    if(newtim != (time_t) -1){
		prev_bu_time = current_bu_time;
		current_bu_time = newtim;

		if(have_timerange){
		  for(i = num_matches - 1; i >= 0; i--){
		    if(endtimes[i] != (time_t) -1)
			break;

		    endtimes[i] = current_bu_time;
		  }
		}
	    }
	  }
        }
	continue;
      }

      cptr = line + n;
      if(empty_string(cptr))
	continue;

      while(isspace(*cptr))
	cptr++;

      for(i = 0; i < num_restore_paths; i++){
	if(!fnmatch(restore_paths[i], cptr, FNM_PATHNAME)){
	  added = 0;

	  newmatches = NEWP(UChar *, num_matches + 1);
	  if(have_timerange){
	    newstarttimes = NEWP(time_t, num_matches + 1);
	    newendtimes = NEWP(time_t, num_matches + 1);
	  }
	  if(!newmatches || (have_timerange
				&& (!newstarttimes || !newendtimes))){
	    ZFREE(newmatches);
	    ZFREE(newstarttimes);
	    ZFREE(newendtimes);

	    if(num_matches < 10)
		nomemerrexit();

	    if(have_timerange)
		matches_check_timerange(matches, &num_matches, starttimes,
					endtimes, time_newer, time_older);

	    write_to_restore(matches, num_matches, 0);

	    matches = NULL;
	    num_matches = allocated = 0;
	    ZFREE(starttimes);
	    ZFREE(endtimes);

	    newmatches = NEWP(UChar *, 1);
	    if(have_timerange){
		newstarttimes = NEWP(time_t, 1);
		newendtimes = NEWP(time_t, 1);
	    }
	    if(!newmatches || (have_timerange
				 && (!newstarttimes || !newendtimes)))
		nomemerrexit();
	  }

	  if(matches){
	    memcpy(newmatches, matches, num_matches * sizeof(UChar *));

	    free(matches);
	  }
	  if(have_timerange && starttimes && endtimes){
	    memcpy(newstarttimes, starttimes, num_matches * sizeof(time_t));
	    memcpy(newendtimes, endtimes, num_matches * sizeof(time_t));

	    free(starttimes);
	    free(endtimes);
	  }

	  matches = newmatches;
	  matches[num_matches] = strdup(line);
	  if(have_timerange){
	    starttimes = newstarttimes;
	    starttimes[num_matches] = prev_bu_time;
	    endtimes = newendtimes;
	    endtimes[num_matches] = (time_t) -1;
	  }
	  if(matches[num_matches]){
	    num_matches++;
	    added = 1;
	    allocated += strlen(line) + 1;
	  }
	  if(allocated > 200000 || !added){
	    if(have_timerange)
		matches_check_timerange(matches, &num_matches, starttimes,
					endtimes, time_newer, time_older);

	    write_to_restore(matches, num_matches, 0);
	    ZFREE(starttimes);
	    ZFREE(endtimes);

	    matches = NEWP(UChar *, 1);
	    if(have_timerange){
		starttimes = NEWP(time_t, 1);
		endtimes = NEWP(time_t, 1);
	    }
	    if(!matches || (have_timerange
				&& (!starttimes || !endtimes)))
		nomemerrexit();
	    if(!added){
	      matches[0] = strdup(line);
	      if(!matches[0])
		nomemerrexit();
	    }
	    num_matches = added ? 0 : 1;
	    allocated = added ? 0 : strlen(line) + 1;
	  }

	  break;
	}
      }
    }

    fclose(fp);
    if(pid >= 0){
      waitpid_forced(pid, &pst, 0);

      if(WEXITSTATUS(pst)){
	errmsg("Warning: Uncompressing \"%s\" returned bad exit status",
			*(cpptr + 3));
      }
    }
  }

  if(have_timerange){
    current_bu_time = time(NULL);

    for(i = num_matches - 1; i >= 0; i--){
	if(endtimes[i] != (time_t) -1)
	  break;

	endtimes[i] = current_bu_time;
    }
  }

  if(have_timerange)
    matches_check_timerange(matches, &num_matches, starttimes,
				endtimes, time_newer, time_older);

  write_to_restore(matches, num_matches, 1);

  ZFREE(starttimes);
  ZFREE(endtimes);

  do_exit(0);
}

static	Int32	total_num_matches = 0, total_nc = 0, total_nf = 0;

static Int32
write_to_restore(UChar ** matches, Int32 num_matches, UChar last_one)
{
  int		i, j, cartno, fileno, *cartnos, *filenos, *efilenos;
  int		fd, pid, pst, n, actcart, actfile, nf, nc, num_sections;
  int		fileuid;
  UChar		*cptr, **cpptr, showit, warned_old_log = 0;

  if(num_matches < 1)
    return(0);

  cartnos = NEWP(int, 1);
  filenos = NEWP(int, 1);
  efilenos = NEWP(int, 1);
  cpptr = NEWP(UChar *, num_matches);
  if(!cartnos || !filenos || !efilenos || !cpptr)
    nomemerrexit();

  if(!have_timerange){
    for(i = 0; i < num_matches; i++){	/* eliminate duplicates */
      j = sscanf(matches[i], "%d.%d" UIDSEP "%d:%n",
				&cartno, &fileno, &fileuid, &n);
      if(j < 3)
	sscanf(matches[i], "%d.%d:%n", &cartno, &fileno, &n);
      cpptr[i] = matches[i] + n;
      while(isspace(*(cpptr[i])))
	cpptr[i]++;
    }
    for(i = num_matches - 1; i > 0; i--){
      for(j = i - 1; j >= 0; j--){
	if(!strcmp(cpptr[i], cpptr[j])){
	  num_matches--;
	  free(matches[j]);
	  for(n = j; n < num_matches; n++){
	    matches[n] = matches[n + 1];
	    cpptr[n] = cpptr[n + 1];
	  }
	  i--;
	}
      }
    }
  }

  free(cpptr);

  cpptr = matches;

  num_sections = nc = nf = 0;
  actcart = actfile = -1;
  for(i = 0; i < num_matches; i++){
    j = sscanf(matches[i], "%d.%d" UIDSEP "%d:", &cartno, &fileno, &fileuid);
    if(j < 3)
	j = sscanf(matches[i], "%d.%d:", &cartno, &fileno);

    if(actcart != cartno){
	nc++;
	if(actfile == fileno)		/* e.g. 3.1 -> 4.1 may happen ... */
	  nf++;
    }
    if(actfile != fileno)
	nf++;
    if(actcart != cartno || (actfile != fileno && actfile + 0 != fileno)){
	cartnos = RENEWP(cartnos, int, num_sections + 1);
	filenos = RENEWP(filenos, int, num_sections + 1);
	efilenos = RENEWP(efilenos, int, num_sections + 1);
	if(!cartnos || !filenos || !efilenos)
	  nomemerrexit();

	cartnos[num_sections] = cartno;
	filenos[num_sections] = fileno;
	efilenos[num_sections] = fileno;

	num_sections++;
    }

    actcart = cartno;
    actfile = fileno;
    efilenos[num_sections - 1] = fileno;
  }

  if(do_counting || do_listing){
    total_num_matches += num_matches;
    total_nf += nf;
    total_nc += nc;

    if(do_listing){
      for(i = 0; i < num_matches; i++){
	fileuid = 0;

	j = sscanf(matches[i], "%d.%d" UIDSEP "%d:%n",
				&cartno, &fileno, &fileuid, &n);
	if(j < 3)
	  sscanf(matches[i], "%d.%d:%n", &cartno, &fileno, &n);

	cptr = matches[i] + n;
	while(isspace(*cptr))
	  cptr++;
	if(!empty_string(cptr)){
	  showit = 1;

	  if(curuid){
	    if(!fileuid){
	      if(!warned_old_log){
		errmsg("Warning: Filelist without user-ID information. Not displaying contents");
		warned_old_log = 1;
	      }
	      showit = 0;
	    }
	    else{
	      if(fileuid != curuid)
		showit = 0;
	    }
	  }

	  if(showit)
	    fprintf(stdout, "%s\n", cptr);
	}

	free(matches[i]);
      }
    }

    if(last_one){
      if(do_counting)
	fprintf(stdout, "Found %d matches in %d files on %d tapes.\n",
			total_num_matches, total_nf, total_nc);

      total_num_matches = total_nf = total_nc = 0;
    }

    free(matches);

    return(0);
  }

  fprintf(stdout, "Found %d matches in %d files on %d tapes.\n",
			num_matches, nf, nc);

  if(!last_one)
    fprintf(stdout,
	"(This is not the last set of files, please be patient)\n");

  actfile = filenos[0];
  actcart = cartnos[0];
  for(n = 0; n < num_sections; n++){
    sprintf(tmpbuf, "%s -xvgu -C %d -F %d -h %s", clientprogram,
				cartnos[n], filenos[n], backuphost);

    if(backupport >= 0)
	sprintf(tmpbuf + strlen(tmpbuf), " -p %d", backupport);

    if(restoreroot)
      strcat(tmpbuf, " -r");

    if(cryptfile)
	sprintf(tmpbuf + strlen(tmpbuf), " -k %s", cryptfile);

    if(cartset > 1)
	sprintf(tmpbuf + strlen(tmpbuf), " -S %d", (int) cartset);

    if(curuid > 1)
	sprintf(tmpbuf + strlen(tmpbuf), " -o %d", (int) curuid);

    if(time_older > 0)
	sprintf(tmpbuf + strlen(tmpbuf), " -B %lu", (unsigned long) time_older);

    if(time_newer > 0)
	sprintf(tmpbuf + strlen(tmpbuf), " -A %lu", (unsigned long) time_newer);

    if( (fd = open_to_pipe(tmpbuf, NULL, 1 + 2, &pid, 0)) < 0){
      errmsg("Error: Cannot start client subprocess");
      do_exit(18);
    }

    if(filenos[n] == efilenos[n])
	fprintf(stdout, "Restoring files from cartridge %d, file %d\n",
				cartnos[n], filenos[n]);
    else
	fprintf(stdout, "Restoring files from cartridge %d, file %d - %d\n",
				cartnos[n], filenos[n], efilenos[n]);

    while(cpptr < matches + num_matches){
	i = sscanf(*cpptr, "%d.%d" UIDSEP "%d:%n",
				&cartno, &fileno, &fileuid, &j);
	if(i < 3)
	  i = sscanf(*cpptr, "%d.%d:%n", &cartno, &fileno, &j);

	if(actcart != cartno || (fileno != actfile && fileno != actfile + 0))
	  break;
	actfile = fileno;
	actcart = cartno;

	cptr = *cpptr + j;
	while(isspace(*cptr))
	  cptr++;

	if(!empty_string(cptr))
	  write_filename(cptr, (void *) (&fd));

	free(*cpptr);
	cpptr++;
    }
    actfile = fileno;
    actcart = cartno;

    close(fd);
    waitpid_forced(pid, &pst, 0);

    if(WEXITSTATUS(pst)){
	fprintf(stderr, "Errors occurred during restore. See previous output for details.\n");
    }

    if(filenos[n] == efilenos[n])
	fprintf(stdout, "Done with cartridge %d, file %d\n",
				cartnos[n], filenos[n]);
    else
	fprintf(stdout, "Done with cartridge %d, file %d - %d\n",
				cartnos[n], filenos[n], efilenos[n]);
  }

  free(matches);

  return(0);
}

static void
restoreall(int c, char ** v)
{
  UChar		**cpptr, *cptr, *line = NULL, *prevlast = NULL, found;
  UChar		*filename, *line2 = NULL;
  FILE		*fp, *cfp;
  int		fd, pid, pst, cartno, fileno, n, i, cpid, cfd;
  int		oldcart, oldfile, fileuid;

  oldcart = oldfile = -1;

  for(cpptr = indexfiles; *cpptr; cpptr++){
    pid = -1;
    fd = open(*cpptr, O_RDONLY);
    if(fd < 0){
      if((fd = open_from_pipe(uncompresscmd[0] ? uncompresscmd : (UChar*)"gunzip",
			zindexfiles[cpptr - indexfiles], 1, &pid)) < 0)
	continue;
    }
    fp = fdopen(fd, "r");
    if(!fp){
	errmsg("Error: Cannot read file \"%s\"", *cpptr);
	continue;
    }
    found = 0;
    while(!feof(fp)){	/* search input logfile lines */
      if(line)
	free(line);
      line = fget_alloc_str(fp);
      if(!line)
	continue;

      i = sscanf(line, "%d.%d" UIDSEP "%d:%n",
				&cartno, &fileno, &fileuid, &n);
      if(i < 3)
	i = sscanf(line, "%d.%d:%n", &cartno, &fileno, &n);

      if(i < 2 || empty_string(line + n))
	continue;

	/* in this case error messages appear at end of client output */
      if(found && cartno == oldcart && fileno == oldfile)
	continue;

      oldcart = cartno;
      oldfile = fileno;

      cptr = line + strlen(line) - 1;
      if(*cptr == '\n')
	*cptr = '\0';
      filename = line + n;
      while(isspace(*filename))
	filename++;

      if(!prevlast || found){	/* if found of first line at all: restore */
	sprintf(tmpbuf, "%s -xvnau -h %s -C %d -F %d",
				clientprogram, backuphost, cartno, fileno);

	if(backupport >= 0)
	  sprintf(tmpbuf + strlen(tmpbuf), " -p %d", backupport);

	if(restoreroot)
	  strcat(tmpbuf, " -r");

	if(cartset > 1)
	  sprintf(tmpbuf + strlen(tmpbuf), " -S %d", cartset);

	if(cryptfile)
	  sprintf(tmpbuf + strlen(tmpbuf), " -k %s", cryptfile);

	if(curuid)
	  sprintf(tmpbuf + strlen(tmpbuf), " -o %d", (int) curuid);

	if(time_older > 0)
	  sprintf(tmpbuf + strlen(tmpbuf), " -B %lu", (unsigned long) time_older);

	if(time_newer > 0)
	  sprintf(tmpbuf + strlen(tmpbuf), " -A %lu", (unsigned long) time_newer);

	if( (cfd = open_from_pipe(tmpbuf, NULL, 1 + 2, &cpid)) < 0){
	  errmsg("Error: Cannot start client subprocess");
	  continue;
	}

	cfp = fdopen(cfd, "r");
	if(!cfp){
	  errmsg("Error: Cannot read from restore subprocess");
	  do_exit(34);
	}

	fprintf(stdout, "Going to restore from cartridge %d, file %d\n",
					cartno, fileno);
	while(!feof(cfp)){
	  if(line2)
	    free(line2);
	  line2 = fget_alloc_str(cfp);
	  if(!line2)
	    continue;

	  fprintf(stdout, "%s", line2);

	  i = sscanf(line2, "%d.%d" UIDSEP "%d:%n",
				&cartno, &fileno, &fileuid, &n);
	  if(i < 3)
	    i = sscanf(line2, "%d.%d:%n", &cartno, &fileno, &n);

	  if(i < 2 || empty_string(line2 + n))
	    continue;

	  cptr = line2 + strlen(line2) - 1;
	  if(*cptr == '\n')
	  *cptr = '\0';

	  cptr = line2 + n;
	  while(isspace(*cptr))
	    cptr++;

	  if(prevlast)
	    free(prevlast);
	  prevlast = strdup(cptr);
	}

	fclose(cfp);
	waitpid_forced(cpid, &pst, 0);
	if(WEXITSTATUS(pst))
	  fprintf(stderr, "Errors occurred during restore. See previous output for details.\n");

	found = 0;
      }

      if(!prevlast)
	break;

      if(!strcmp(prevlast, filename))
	found = 1;
    }

    fclose(fp);
    if(pid >= 0){
      waitpid_forced(pid, &pst, 0);
      if(WEXITSTATUS(pst)){
	fprintf(stderr,
		"Warning: Uncompressing \"%s\" returned bad exit status.\n",
			*(cpptr + 3));
      }
    }

    if(prevlast)
	free(prevlast);
    prevlast = NULL;
  }

  do_exit(0);
}

void
matches_check_timerange(
  UChar		**matches,
  int		*num_matches,
  time_t	*starttimes,
  time_t	*endtimes,
  time_t 	time_newer,
  time_t	time_older)
{
  int		idx1, idx2;

  if(! time_newer && ! time_older)
    return;

  for(idx1 = idx2 = 0; idx2 < *num_matches; idx1++, idx2++){
    if((time_newer && endtimes[idx2] < time_newer
				&& endtimes[idx2] != (time_t) -1)
		|| (time_older && starttimes[idx2] > time_older
				&& starttimes[idx2] != (time_t) -1)){
	free(matches[idx2]);
	idx1--;
    }
    else{
	matches[idx1] = matches[idx2];
	starttimes[idx1] = starttimes[idx2];
	endtimes[idx1] = starttimes[idx2];
    }
  }

  *num_matches = idx1;
}

static int
trailingint(UChar * str)
{
  int	mul = 1, r = 0;

  str += strlen(str) - 1;
  while(isspace(*str))
    str--;
  while(isdigit(*str)){
    r += mul * (*str - '0');
    mul *= 10;
    str--;
  }

  return(r);
}

static void
restoreem(int c, char ** v)
{
  UChar		**lines = NULL, *cptr, *line;
  int		i, j, l, num_lines = 0, prev_bunum = -1, act_bunum = -1;
  FILE		*fp;
  int		actcart, actfile, fd, pid, opid, pst, ofd;
  struct stat	statb;

  if(curuid){
    errmsg("Error: Emergency restore may only be performed by root");
    do_exit(19);
  }

  if(re_compile_pattern(EMREC_PREFIX_RE CARTRIDGE_RE,
			strlen(EMREC_PREFIX_RE CARTRIDGE_RE), &cartridge_re)
	|| re_compile_pattern(EMREC_PREFIX_RE FILE_RE,
			strlen(EMREC_PREFIX_RE FILE_RE), &file_re)
	|| re_compile_pattern(EMREC_PREFIX_RE BACKUP_RE,
			strlen(EMREC_PREFIX_RE BACKUP_RE), &backup_re))
    nomemerrexit();

  fp = stdin;

  lines = NEWP(UChar *, 1);
  while(!feof(fp)){
    line = fget_alloc_str(fp);
    if(!line)
	continue;
    if(!strstr(line, EMREC_PREFIX))
	continue;

    lines = RENEWP(lines, UChar *, num_lines + 1);
    if(!lines)
	nomemerrexit();

    lines[num_lines] = line;
    num_lines++;

    if(re_find_match(&backup_re, line, NULL, NULL) >= 0){
      if(act_bunum != (i = trailingint(line))){
	prev_bunum = act_bunum;
	act_bunum = i;
	for(i = num_lines - 2; i >= 0; i--){
	  if(re_find_match(&backup_re, lines[i], NULL, NULL) >= 0
			&&  trailingint(lines[i]) < prev_bunum){
	    i++;
	    while(re_find_match(&backup_re, lines[i], NULL, NULL) < 0)
		i++;
	    for(j = i - 1; j >= 0; j--)
		free(lines[j]);
	    for(j = 0; i < num_lines; j++, i++)
		lines[j] = lines[i];
	    num_lines = j;
	    break;
	  }
	}
      }
    }
  }

  forever{
    sprintf(tmpbuf, "%d", act_bunum);
    indexfile = strapp(indexfilepart, tmpbuf);
    cptr = strapp(indexfile, COMPRESS_SUFFIX);

    if(!stat(indexfile, &statb) || !stat(cptr, &statb)){
      fprintf(stdout,
	"Warning: Filename logfile for backup number %d exists, trying next.\n",
			act_bunum);

      act_bunum++;
      free(cptr);
      free(indexfile);
      continue;
    }
    else
	break;
  }

  opid = -1;
  ofd = -1;
  if(compresslogfiles){
    ofd = open_to_pipe(compresscmd, cptr, 1, &opid, 0600);
  }
  if(ofd < 0){
    ofd = open_to_pipe(NULL, indexfile, 1, &i, 0600);
    if(ofd < 0){
	fprintf(stderr, "Error: Cannot open filename logfile.\n");
    }
  }

  for(l = 0; l < num_lines;){
    if(re_find_match(&backup_re, lines[l], NULL, NULL) < 0){
	l++;
	continue;
    }

    while(re_find_match(&cartridge_re, lines[l], NULL, NULL) < 0
							&& l < num_lines)
	l++;
    if(l >= num_lines)
	break;

    actcart = trailingint(lines[l]);

    while(re_find_match(&file_re, lines[l], NULL, NULL) < 0
							&& l < num_lines)
	l++;
    if(l >= num_lines)
	break;

    actfile = trailingint(lines[l]);

    if(!restore_emlist){
      fprintf(stdout,
		"Going to restore files from cartridge %d, file %d.\n",
			actcart, actfile);

      sprintf(tmpbuf, "%s -xvnau -h %s -F %d -C %d",
				clientprogram, backuphost, actfile, actcart);

      if(restoreroot)
	strcat(tmpbuf, " -r");
    }
    else{
      fprintf(stdout,
		"Going to restore filenames from cartridge %d, file %d.\n",
			actcart, actfile);

      sprintf(tmpbuf, "%s -tn -h %s -F %d -C %d",
				clientprogram, backuphost, actfile, actcart);
    }

    if(backupport >= 0)
	sprintf(tmpbuf + strlen(tmpbuf), " -p %d", backupport);

    if(cartset > 1)
	sprintf(tmpbuf + strlen(tmpbuf), " -S %d", cartset);

    if(cryptfile)
	sprintf(tmpbuf + strlen(tmpbuf), " -k %s", cryptfile);

    if(time_older > 0)
	sprintf(tmpbuf + strlen(tmpbuf), " -B %lu", (unsigned long) time_older);

    if(time_newer > 0)
	sprintf(tmpbuf + strlen(tmpbuf), " -A %lu", (unsigned long) time_newer);

    pid = -1;
    fd = open_from_pipe(tmpbuf, NULL, 1 + 2, &pid);
    if(fd < 0){
      fprintf(stderr, "Error: Cannot restore files.\n");
      if(pid >= 0)
	waitpid_forced(pid, &pst, 0);

      continue;
    }
    j = 0;
    while((i = read(fd, tmpbuf, 50)) > 0){
      if(ofd >= 0){
	if(write(ofd, tmpbuf, i) < i){
	  j++;
	  if(j < 10)
	    errmsg("Warning: Writing to filename logfile failed");
	  if(j == 9)
	    errmsg("Further warnings suppressed");
	}
      }
      write(1, tmpbuf, i);
    }
    close(fd);
    waitpid_forced(pid, &pst, 0);

    if(WEXITSTATUS(pst)){
      errmsg("Warning: Something went wrong during restore. See previous output for details");
    }
  }

  if(ofd >= 0)
    close(ofd);
  if(opid >= 0){
    waitpid_forced(opid, &pst, 0);
    if(WEXITSTATUS(pst)){
	fprintf(stderr, "Warning: Compressing \"%s\" returned bad status.\n",
		indexfile);
    }
  }

  if(write_uns_file(numfile, act_bunum)){
    fprintf(stderr, "Warning: Cannot write file \"%s\".\n", numfile);
  }

  do_exit(0);
}
