/*
 *  Copyright (c) by Jaroslav Kysela (Perex soft)
 *  Routines for control of GF1 chip (synthesizer things)
 */

#include "driver.h"

/*
 *  queue routines
 */

static void queue_total_clear( struct GUS_STRU_GF1_QUEUE *queue )
{
  queue -> size = 0;
  queue -> used = queue -> head = queue -> tail = 0;
  queue -> threshold = 0;
}
      
static void queue_clear( struct GUS_STRU_GF1_QUEUE *queue )
{
  queue -> used = queue -> head = queue -> tail = 0;
}
        
static void queue_append_item( struct GUS_STRU_GF1_QUEUE *queue )
{
  queue -> head++;
  queue -> head %= queue -> size;
  queue -> used++;
}
      
static void queue_remove_item( struct GUS_STRU_GF1_QUEUE *queue )
{
  queue -> tail++;
  queue -> tail %= queue -> size;
  queue -> used--;
}

/*
 *  process all commands to GUS_CMD_WAIT
 */

void gus_gf1_process_events( gus_card_t *card )
{
  int end = 0;
  unsigned char *command, voice, channel;
  struct GUS_STRU_VOICE *pvoice;
  struct GUS_STRU_CHANNEL *pchannel;
    
  if ( card -> gf1.syn_voices_change ) return;
  while ( !end && card -> gf1.wqueue.used > 0 )
    {
      command = &card -> gf1.wqueue.ptr[ 8 * card -> gf1.wqueue.tail ];
#if 0
      printk( "gf1 update voice - command 0x%x/0x%x\n", gus_get_byte( command, 0 ), gus_get_byte( command, 1 ) );
#endif
      pvoice = NULL; pchannel = NULL;
#if 0
      printk( "1> -%i- command = 0x%x\n", gus_get_byte( command, 1 ), gus_get_byte( command, 0 ) );
#endif
      if ( gus_get_byte( command, 0 ) < 0x40 )
        {
          voice = gus_get_byte( command, 1 );
          if ( !card -> gf1.voice_ranges[ GF1_VOICE_RANGE_SYNTH ].voices ) goto __skip;
          voice += card -> gf1.voice_ranges[ GF1_VOICE_RANGE_SYNTH ].min;
          if ( voice > card -> gf1.voice_ranges[ GF1_VOICE_RANGE_SYNTH ].max ) goto __skip;
#if 0
          if ( voice < 1 || voice > 1 ) goto __skip;
#endif
          pvoice = &card -> gf1.syn_voices[ voice ];
          if ( pvoice -> flags & VFLG_DYNAMIC ) goto __skip;
          if ( !pvoice -> commands &&
               gus_get_byte( command, 0 ) != GUS_CMD_VOICE_PROGRAM ) goto __skip;
        }
       else
      if ( gus_get_byte( command, 0 ) < 0x60 )
        {
          channel = gus_get_byte( command, 1 );
          if ( channel >= GUS_CHANNELS ) goto __skip;
          pchannel = &card -> gf1.syn_channels[ channel ];
          if ( !pchannel -> commands && gus_get_byte( command, 0 ) != GUS_CMD_CHN_PROGRAM ) goto __skip;
        }
#if 0
      printk( "2> -%i- command = 0x%x\n", gus_get_byte( command, 1 ), gus_get_byte( command, 0 ) );
#endif
      switch ( gus_get_byte( command, 0 ) ) {
        case GUS_CMD_VOICE_PROGRAM:
          gus_engine_voice_program( card, pvoice, gus_get_dword( command, 2 ) );
          break;
        case GUS_CMD_VOICE_START:
          pvoice -> start_command = gus_get_byte( command, 2 );
          if ( pvoice -> start_command != GUS_CMD_VOICE_PROGRAM )
            {
              if ( pvoice -> commands )
                pvoice -> commands -> voice_start( card, pvoice );
              pvoice -> flags |= VFLG_WAIT_FOR_START1;
            }
          break;
        case GUS_CMD_VOICE_STOP:
          pvoice -> commands -> voice_stop( card, pvoice, gus_get_byte( command, 2 ) );
          break;
        case GUS_CMD_VOICE_CONTROL:
          pvoice -> commands -> voice_control( card, pvoice, gus_get_byte( command, 2 ) );
          break;
        case GUS_CMD_VOICE_FREQ:
          pvoice -> commands -> voice_freq( card, pvoice, gus_get_dword( command, 2 ) );
          break;
        case GUS_CMD_VOICE_VOLUME:
          pvoice -> commands -> voice_volume( card, pvoice, gus_get_word( command, 2 ) );
          break;
        case GUS_CMD_VOICE_SLOOP:
          pvoice -> sloop = gus_get_dword( command, 2 );
          break;
        case GUS_CMD_VOICE_ELOOP:
          pvoice -> commands -> voice_loop(
          		card, pvoice,
          		pvoice -> sloop,
          		gus_get_dword( command, 2 ) );
        case GUS_CMD_VOICE_RAMP:
          pvoice -> commands -> voice_ramp(
          		card, pvoice,
          		gus_get_byte( command, 2 ),
          		gus_get_byte( command, 3 ),
          		gus_get_byte( command, 4 ),
          		gus_get_byte( command, 5 ) );
          break;
        case GUS_CMD_VOICE_POS:
          pvoice -> commands -> voice_pos( card, pvoice, gus_get_dword( command, 2 ) );
          break;
        case GUS_CMD_VOICE_PAN:
          pvoice -> commands -> voice_pan( card, pvoice, gus_get_word( command, 2 ) );
          break;
        case GUS_CMD_VOICE_LFO:
          pvoice -> commands -> voice_lfo( card, pvoice, command + 2 );
          break;
        case GUS_CMD_VOICE_PRIVATE1:
          pvoice -> commands -> voice_private1( card, pvoice, command + 2 );
          break;
        /* --- */
        case GUS_CMD_CHN_PROGRAM:
          gus_engine_channel_program_drum( card, pchannel, gus_get_word( command, 2 ) );
          break;
        case GUS_CMD_CHN_NOTE_ON:
          gus_engine_midi_note_on( card, pchannel,
					gus_get_byte( command, 2 ),
					gus_get_byte( command, 3 ),
					gus_get_byte( command, 4 ) );
          break;
        case GUS_CMD_CHN_NOTE_OFF:
          gus_engine_midi_note_off( card, pchannel,
					gus_get_byte( command, 2 ),
   				        gus_get_byte( command, 3 ) );
          break;
	case GUS_CMD_CHN_PITCHBEND:
	  gus_engine_midi_pitchbend( card, pchannel, gus_get_word( command, 2 ) );
	  break;
	case GUS_CMD_CHN_CONTROL:
	  gus_engine_midi_control( card, pchannel, gus_get_byte( command, 2 ), gus_get_byte( command, 3 ) );
	  break;
        /* --- */
        case GUS_CMD_TEMPO:
          gus_timer_tempo( card, gus_get_dword( command, 2 ) );
          gus_timer_ioctl( card, GUS_IOCTL_TIMER_START, 0 );
          break;
        case GUS_CMD_WAIT:
          card -> gf1.timer_wait_ticks = gus_get_dword( command, 2 );
          if ( ( GETLOCK( card, wqueue ) & WK_SLEEP ) &&
               (
                 ( !card -> gf1.syn_flush_flag && card -> gf1.wqueue.used < card -> gf1.wqueue.threshold ) ||
                 card -> gf1.rqueue.used > 0
               )
             )
            {
              GETLOCK( card, wqueue ) &= ~WK_SLEEP;
              WAKEUP( card, wqueue );
            }
          if ( card -> gf1.timer_wait_ticks > 0 )
            {
              card -> gf1.timer_wait_ticks--;
              end++;
            }
          break;
        case GUS_CMD_STOP:
          break;
        case GUS_CMD_ECHO:
          if ( card -> gf1.rqueue.size > 0 )
            {
              MEMCPY( &card -> gf1.rqueue.ptr[ 8 * card -> gf1.rqueue.head ], command, 8 );
              queue_append_item( &card -> gf1.rqueue );
              if ( GETLOCK( card, rqueue ) & WK_SLEEP )
                {
                  GETLOCK( card, rqueue ) &= ~WK_SLEEP;
                  WAKEUP( card, rqueue );
                }
            }
          break;
      }
      if ( gus_get_byte( command, 0 ) < 0x40 )
        {
          if ( pvoice -> flags & VFLG_WAIT_FOR_START1 )
            {
              if ( gus_get_byte( command, 0 ) == pvoice -> start_command )
                {
                  pvoice -> commands -> voice_go( card, pvoice );
                  pvoice -> flags &= ~VFLG_WAIT_FOR_START1;
                }
#if 0
              printk( "wait for start completed!!! - start_command = 0x%x, cmd = 0x%x, commands = 0x%lx\n",
              		pvoice -> start_command, gus_get_byte( command, 0 ), (long)pvoice -> commands );
#endif
            }
        }
      __skip:
      queue_remove_item( &card -> gf1.wqueue );
    }
  if ( card -> gf1.syn_flush_flag && 
       !card -> gf1.wqueue.used &&
       ( GETLOCK( card, wqueue ) & WK_SLEEP ) )
    {
      GETLOCK( card, wqueue ) &= ~WK_SLEEP;
      WAKEUP( card, wqueue );
    }
}

static void gus_synth_voices_init( gus_card_t *card )
{
  unsigned long flags;

  gus_engine_voices_init( card, 0 );
  CLI( &flags );
  queue_clear( &card -> gf1.wqueue );
  queue_clear( &card -> gf1.rqueue );
  card -> gf1.syn_flush_flag = 0;
  card -> gf1.syn_stop_flag = 0;
  STI( &flags );
}

static void gus_synth_abort( gus_card_t *card, int flags )
{
#if 0
  printk( "abort - 0x%x\n", flags );
#endif
  card -> gf1.syn_abort_flag = flags & 2 ? 2 : 1;
  gus_timer_ioctl( card, GUS_IOCTL_TIMER_STOP, 0 );
  if ( flags & 1 )
    gus_gf1_clear_voices( card,
                          card -> gf1.voice_ranges[ GF1_VOICE_RANGE_SYNTH ].min,
                          card -> gf1.voice_ranges[ GF1_VOICE_RANGE_SYNTH ].max );
   else
    gus_gf1_stop_voices( card,
                         card -> gf1.voice_ranges[ GF1_VOICE_RANGE_SYNTH ].min,
                         card -> gf1.voice_ranges[ GF1_VOICE_RANGE_SYNTH ].max );
  gus_synth_voices_init( card );
  if ( GETLOCK( card, wqueue ) & WK_SLEEP )
    {
      GETLOCK( card, wqueue ) &= ~WK_SLEEP;
      WAKEUP( card, wqueue );
    }
  if ( flags & 2 )
    gus_timer_ioctl( card, GUS_IOCTL_TIMER_CONTINUE, 0 );
#if 0
  printk( "abort - end\n" );
#endif
}

static int gus_synth_stop( gus_card_t *card )
{
  card -> gf1.syn_stop_flag = 1;
  gus_timer_ioctl( card, GUS_IOCTL_TIMER_STOP, 0 );
  gus_gf1_stop_voices( card,
                       card -> gf1.voice_ranges[ GF1_VOICE_RANGE_SYNTH ].min,
                       card -> gf1.voice_ranges[ GF1_VOICE_RANGE_SYNTH ].max );
  return 0;
}

static int gus_synth_continue( gus_card_t *card )
{
  if ( !card -> gf1.syn_stop_flag ) return -EACCES;
#if 0
  PRINTK( "gf1: continue, timer = %i\n", stop_timer_count );
#endif
  gus_timer_ioctl( card, GUS_IOCTL_TIMER_CONTINUE, 0 );
  card -> gf1.syn_stop_flag = 0;
  return 0;
}

static int gus_synth_flush( gus_card_t *card )
{
  int res = 0;

  card -> gf1.syn_flush_flag = 1;
  while ( !card -> gf1.syn_abort_flag && card -> gf1.wqueue.used > 0 )
    {
      GETLOCK( card, wqueue ) |= WK_SLEEP;
      SLEEP( card, wqueue, 60 * HZ );
      GETLOCK( card, wqueue ) &= ~WK_SLEEP;
      if ( TABORT( card, wqueue ) ) break;
      if ( TIMEOUT( card, wqueue ) )
        {
          printk( "gus: gf1 flush timeout?\n" );
          res = -EIO;
          break;
        }
    }
  card -> gf1.syn_flush_flag = 0;
  return res;
}

static int gus_synth_reset( gus_card_t *card, unsigned int cmd, struct GUS_STRU_RESET *a_reset, int space )
{
  int res;
  struct GUS_STRU_RESET reset;

  gus_timer_ioctl( card, GUS_IOCTL_TIMER_STOP, 0 );
  card -> gf1.timer_wait_ticks = 0;
  if ( space == SP_USER )
    {
      if ( VERIFY_AREA( VERIFY_READ, a_reset, sizeof( *a_reset ) ) ||
           VERIFY_AREA( VERIFY_WRITE, a_reset, sizeof( *a_reset ) ) ) return -EIO;
      MEMCPY_FROMFS( &reset, a_reset, sizeof( reset ) );
    }
   else
    MEMCPY( &reset, a_reset, sizeof( reset ) );
  gus_synth_voices_init( card );
  if ( cmd == GUS_IOCTL_RESET )
    if ( ( res = gus_engine_reset( card, reset.voices, reset.channel_voices ) ) < 0 ) return res;
  card -> gf1.syn_abort_flag = 0;
  if ( space == SP_USER )
    put_fs_word( card -> gf1.active_voices, (short *)&a_reset -> voices );
   else
    a_reset -> voices = card -> gf1.active_voices;
  queue_clear( &card -> gf1.rqueue );
  return 0;
}

static int gus_synth_init_queue( gus_card_t *card, struct GUS_STRU_GF1_QUEUE *queue, int items )
{
  int i;

  if ( items < -1 || items > 8 * 1024 ) return -EINVAL;
  if ( items >= 0 && queue -> used ) return -EBUSY;	/* I cann't work with echo queue */
  if ( items > 0 && queue -> size == items )
    {
      queue_clear( queue );
      queue -> threshold = ( card -> gf1.wqueue.size / 3 ) * 2;
      return 0;
    }
  if ( queue -> size > 0 )
    {
      i = queue -> size;
      queue -> used = queue -> size = 0;
      gus_free( queue -> ptr, queue -> item_size * i );
      queue -> ptr = NULL;
    }
  queue_total_clear( queue );
  if ( items <= 0 ) return 0;
  if ( ( queue -> ptr = (unsigned char *)gus_malloc( queue -> item_size * items ) ) != NULL )
    {
      queue -> size = items;
      queue -> threshold = ( card -> gf1.wqueue.size / 3 ) * 2;
    }
  return queue -> ptr ? 0 : -ENOMEM;
}

static int gus_open_gf1_card( gus_card_t *card )
{
  int res;

  if ( card -> gf1.mode & GF1_MODE_PCM_PLAY ) return -EBUSY;
  if ( ( res = gus_timer_open( card, "synth" ) ) < 0 ) return res;
  card -> gf1.timer_midi = 0;
  card -> gf1.timer_old_count2 = 1;
  gus_timer_tempo( card, 60 );	/* default tempo = 60Hz */
  if ( ( res = gus_engine_open( card ) ) < 0 )
    {
      gus_timer_close( card );
      return res;
    }
  card -> gf1.syn_abort_flag = 0;
  queue_total_clear( &card -> gf1.wqueue ); card -> gf1.wqueue.item_size = 8;
  queue_total_clear( &card -> gf1.rqueue ); card -> gf1.rqueue.item_size = 8;
  GETLOCK( card, wqueue ) = GETLOCK( card, rqueue ) = 0;
  card -> gf1.mode |= GF1_MODE_SYNTH;
  return 0;
}

void gus_close_gf1_card( gus_card_t *card )
{
  if ( !card ) return;
  gus_timer_ioctl( card, GUS_IOCTL_TIMER_STOP, 0 );
  gus_gf1_stop_voices( card,
                       card -> gf1.voice_ranges[ GF1_VOICE_RANGE_SYNTH ].min,
                       card -> gf1.voice_ranges[ GF1_VOICE_RANGE_SYNTH ].max );
  gus_timer_close( card );
  gus_engine_close( card );
  gus_synth_init_queue( card, &card -> gf1.wqueue, -1 );
  gus_synth_init_queue( card, &card -> gf1.rqueue, -1 );
  card -> gf1.mode &= ~GF1_MODE_SYNTH;
}

int gus_ioctl_gf1( struct file *file, unsigned int cmd, unsigned long arg )
{  
  gus_card_t *card = (gus_card_t *)file -> private_data;

#if 0
  printk( "cmd = 0x%x\n", cmd );
#endif
  if ( ( cmd & 0xff00 ) != ( 'g' << 8 ) ) return -EINVAL;
  switch ( cmd ) {
    case GUS_IOCTL_CARDS:
      return IOCTL_OUT( arg, gus_cards_count );
    case GUS_IOCTL_VERSION:
      return IOCTL_OUT( arg, GUS_SYNTH_VERSION );
    case GUS_IOCTL_RESET0:
    case GUS_IOCTL_RESET:
      return gus_synth_reset( card, cmd, (struct GUS_STRU_RESET *)arg, SP_USER );
    case GUS_IOCTL_INFO:
      return gus_info( card, &card -> gf1.mem_alloc, (struct GUS_STRU_INFO *)arg );

    case GUS_IOCTL_GET_ULTRACLICK:
      return gus_engine_get_ultraclick( card, (gus_ultraclick_t *)arg, SP_USER );
    case GUS_IOCTL_SET_ULTRACLICK:
      return gus_engine_set_ultraclick( card, (gus_ultraclick_t *)arg, SP_USER );
    case GUS_IOCTL_GET_MIDI_EMUL:
      return IOCTL_OUT( arg, card -> gf1.midi_emul );
    case GUS_IOCTL_SET_MIDI_EMUL:
      {
        int new_emul = IOCTL_IN( arg );

        if ( new_emul == GUS_MIDI_EMUL_AUTO )
          new_emul = card -> gf1.default_midi_emul;
        if ( new_emul > GUS_MIDI_EMUL_MT32 ) return -EINVAL;
        if ( new_emul != card -> gf1.midi_emul )
          {
            card -> gf1.midi_emul = new_emul;
            gus_gf1_daemon_midi_emul_change( card );
          }
      }
      return 0;
    case GUS_IOCTL_EFFECT_RESET:
      return gus_effects_reset( card );
    case GUS_IOCTL_EFFECT_SETUP:
      return gus_effects_setup( card, (struct GUS_STRU_EFFECT *)arg, SP_USER );

    case GUS_IOCTL_TIMER_START:
    case GUS_IOCTL_TIMER_STOP:
    case GUS_IOCTL_TIMER_CONTINUE:
    case GUS_IOCTL_TIMER_TEMPO:
    case GUS_IOCTL_TIMER_MASTER:
    case GUS_IOCTL_TIMER_SLAVE:
      return gus_timer_ioctl( card, cmd, arg );
    
    case GUS_IOCTL_MEMORY_RESET:
      return gus_memory_reset(
      			card,
      			&card -> gf1.mem_alloc,
                        IOCTL_IN( arg ) ? GUS_ALLOC_MODE_TEST : GUS_ALLOC_MODE_NORMAL,
                        0
                        );
    case GUS_IOCTL_MEMORY_TEST:
      return gus_memory_alloc( card, &card -> gf1.mem_alloc, (struct GUS_STRU_INSTRUMENT *)arg, MTST_ONE, SP_USER );
    case GUS_IOCTL_MEMORY_ALLOC:
      return gus_memory_alloc( card, &card -> gf1.mem_alloc, (struct GUS_STRU_INSTRUMENT *)arg, MTST_NONE, SP_USER );
    case GUS_IOCTL_MEMORY_FREE:
      return gus_memory_free( card, &card -> gf1.mem_alloc, (struct GUS_STRU_INSTRUMENT *)arg, SP_USER );
    case GUS_IOCTL_MEMORY_PACK:
      return gus_memory_pack( card, &card -> gf1.mem_alloc );
    case GUS_IOCTL_MEMORY_LIST:
      return gus_memory_list( card, &card -> gf1.mem_alloc, (struct GUS_STRU_MEMORY_LIST *)arg, SP_USER );
    case GUS_IOCTL_MEMORY_BALLOC:
      return gus_memory_block_alloc( card, (struct GUS_STRU_MEMORY_BLOCK *)arg, GUS_MEMORY_LOCK_USER, SP_USER );
    case GUS_IOCTL_MEMORY_BFREE:
      return gus_memory_block_free( card, (struct GUS_STRU_MEMORY_BLOCK *)arg, GUS_MEMORY_LOCK_USER, SP_USER );
    case GUS_IOCTL_MEMORY_GET_NAME:
      return gus_memory_get_name( card, &card -> gf1.mem_alloc, (struct GUS_STRU_INSTRUMENT_NAME *)arg, SP_USER );
    case GUS_IOCTL_MEMORY_DUMP:
      return gus_memory_dump( card, (struct GUS_STRU_MEMORY_DUMP *)arg, SP_USER );

    case GUS_IOCTL_WQUEUE_SET_SIZE:
      return gus_synth_init_queue( card, &card -> gf1.wqueue, IOCTL_IN( arg ) );
    case GUS_IOCTL_WQUEUE_GET_SIZE:
      return IOCTL_OUT( arg, card -> gf1.wqueue.size );
    case GUS_IOCTL_WQUEUE_FREE:
      return IOCTL_OUT( arg, card -> gf1.wqueue.size - card -> gf1.wqueue.used );
    case GUS_IOCTL_WQUEUE_THRESHOLD:
      if ( IOCTL_IN( arg ) >= 1 && IOCTL_IN( arg ) <= card -> gf1.wqueue.size )
        card -> gf1.wqueue.threshold = IOCTL_IN( arg );
       else
        return -EINVAL;
      return 0;
    case GUS_IOCTL_RQUEUE_SET_SIZE:
      return gus_synth_init_queue( card, &card -> gf1.rqueue, IOCTL_IN( arg ) );
    case GUS_IOCTL_RQUEUE_GET_SIZE:
      return IOCTL_OUT( arg, card -> gf1.rqueue.size );
    case GUS_IOCTL_RQUEUE_USED:
      return IOCTL_OUT( arg, card -> gf1.rqueue.used );
    case GUS_IOCTL_FLUSH:
      return gus_synth_flush( card );
    case GUS_IOCTL_ABORT:
    case GUS_IOCTL_QABORT:
      gus_synth_abort( card, cmd == GUS_IOCTL_QABORT ? 2 : 0 );
      return 0;
    case GUS_IOCTL_STOP:
      return gus_synth_stop( card );
    case GUS_IOCTL_CONTINUE:
      return gus_synth_continue( card );
    default:
      PRINTK( "gus: unknown gus_ioctl_gf1 command 0x%x\n", cmd );
  }
  return -EIO;
}

int gus_open_gf1( unsigned short minor, struct file *file )
{
  int res;

  minor >>= 4;
  minor &= 7;
  if ( minor >= gus_cards_count ) return -ENODEV;
  if ( ( res = gus_open_gf1_card( gus_cards[ minor ] ) ) < 0 ) return res;
  file -> private_data = gus_cards[ minor ];
  MOD_INC_USE_COUNT;
  return 0;
}
        
void gus_release_gf1( struct file *file )
{
  gus_close_gf1_card( (gus_card_t *)file -> private_data );
  file -> private_data = NULL;
  MOD_DEC_USE_COUNT;
}

int gus_read_gf1( struct file *file, char *buf, int count )
{
  gus_card_t *card = (gus_card_t *)file -> private_data;
  int filled;

  if ( !card ) return -EIO;		/* no card selected */
#if 0
  printk( "read start - 0x%x\n", count );
#endif
  filled = 0;
  while ( card -> gf1.rqueue.used && count > 7 )
    {
      MEMCPY_TOFS( buf, &card -> gf1.rqueue.ptr[ 8 * card -> gf1.rqueue.tail ], 8 );
      queue_remove_item( &card -> gf1.rqueue );
      buf += 8;
      filled += 8;
      count -= 8;
    }
#if 0
  printk( "read end - 0x%x\n", filled );
#endif
  return filled;
}

int gus_write_gf1( struct file *file, char *buf, int count )
{
  gus_card_t *card = (gus_card_t *)file -> private_data;
  int result, old_count;

  if ( !card ) return -EIO;	/* no card selected */
  if ( !card -> gf1.wqueue.ptr ) return -ENOSPC;
#if 0
  printk( "write start - 0x%x\n", count );
#endif
  result = 0;
  old_count = count;
  while ( count > 7 )
    {
      __repeat:
      switch ( card -> gf1.syn_abort_flag ) {
        case 0:
          if ( card -> gf1.wqueue.used >= card -> gf1.wqueue.size &&
               ( file -> f_flags & O_NONBLOCK ) ) return old_count - count;
          while ( card -> gf1.wqueue.used >= card -> gf1.wqueue.size )
            {
              GETLOCK( card, wqueue ) |= WK_SLEEP;
              SLEEP( card, wqueue, HZ * 60 );
              GETLOCK( card, wqueue ) &= ~WK_SLEEP;
              if ( TABORT( card, wqueue ) )
                {
                  if ( !card -> gf1.syn_abort_flag )
                    gus_synth_abort( card, 0 );
                  goto __repeat;
                }
              if ( TIMEOUT( card, wqueue ) )
                {
                  PRINTK( "gus: gf1 timeout\n" );
                  return -EIO;
                }
            }
          if ( !card -> gf1.syn_abort_flag )
            {
              MEMCPY_FROMFS( &card -> gf1.wqueue.ptr[ 8 * card -> gf1.wqueue.head ], buf, 8 );
              queue_append_item( &card -> gf1.wqueue );
              buf += 8;
              count -= 8;
            }
          break;
        case 1:
          return old_count - ( count & 7 );
        case 2:
	  if ( get_fs_byte( buf ) == GUS_CMD_STOP )
            card -> gf1.syn_abort_flag = 0;
          buf += 8;
          count -= 8;
      }
    }
#if 0
  printk( "write end\n" );
#endif
  return old_count - count;
}

#ifdef GUS_POLL
unsigned int gus_poll_gf1( struct file *file, poll_table *wait )
{
  gus_card_t *card;
  unsigned long flags;
  unsigned int mask;
  
  card = (gus_card_t *)file -> private_data;
  if ( !card ) return 0;		/* no card selected */

  CLI( &flags );
  GETLOCK( card, rqueue ) |= WK_SLEEP;
  SLEEP_POLL( card, rqueue, wait );
  GETLOCK( card, wqueue ) |= WK_SLEEP;
  SLEEP_POLL( card, wqueue, wait );
  STI( &flags );

  mask = 0;
  if ( card -> gf1.rqueue.used )
    mask |= POLLIN | POLLRDNORM;
  if ( card -> gf1.wqueue.used < card -> gf1.wqueue.threshold )
    mask |= POLLOUT | POLLWRNORM;
    
  return mask;
}
#else
int gus_select_gf1( struct file *file, int sel_type, select_table *wait )
{
  gus_card_t *card = (gus_card_t *)file -> private_data;
  unsigned long flags;
  
  if ( !card ) return -EIO;		/* no card selected */
  switch ( sel_type ) {
    case SEL_IN:
      CLI( &flags );
      if ( !card -> gf1.rqueue.used )
        {
          GETLOCK( card, rqueue ) |= WK_SLEEP;
          SLEEPS( card, rqueue, wait );
          STI( &flags );
          return 0;
        }
      GETLOCK( card, rqueue ) &= ~WK_SLEEP;
      STI( &flags );
      return 1;
    case SEL_OUT:
      CLI( &flags );
      if ( card -> gf1.wqueue.used >= card -> gf1.wqueue.threshold )
        {
          GETLOCK( card, wqueue ) |= WK_SLEEP;
          SLEEPS( card, wqueue, wait );
          STI( &flags );
          return 0;
        }
      GETLOCK( card, wqueue ) &= ~WK_SLEEP;
      STI( &flags );
      return 1;
    case SEL_EX:
      return 0;
  }
  return 0;
}
#endif
