/*
 *  Copyright (c) by Jaroslav Kysela (Perex soft)
 */

#include "driver.h"
#include "pcm.h"

/*
 *  ok.. default interrupt handlers...
 */

static void gus_default_interrupt_handler_midi_out( gus_card_t *card )
{
#ifdef GUSCFG_MIDI_DEVICES
  gf1_uart_cmd( card, card -> gf1.uart_cmd &= ~0x20 );
#else
  OUTB( 0x03, GUSP( card, MIDICTRL ) );			/* reset */
#endif
}

static void gus_default_interrupt_handler_midi_in( gus_card_t *card )
{
#ifdef GUSCFG_MIDI_DEVICES
  gf1_uart_cmd( card, card -> gf1.uart_cmd &= ~0x80 );
#else
  OUTB( 0x03, GUSP( card, MIDICTRL ) );			/* reset */
#endif
}

static void gus_default_interrupt_handler_timer1( gus_card_t *card )
{
  gus_i_write8( card, 0x45, card -> gf1.timer_enabled &= ~4 );
}

static void gus_default_interrupt_handler_timer2( gus_card_t *card )
{
  gus_i_write8( card, 0x45, card -> gf1.timer_enabled &= ~8 );
}

static void gus_default_interrupt_handler_wave_and_volume( gus_card_t *card, int voice )
{
  gus_i_ctrl_stop( card, 0x00 );
  gus_i_ctrl_stop( card, 0x0d );
}

static void gus_default_interrupt_handler_dma_write( gus_card_t *card )
{
  gus_i_write8( card, 0x41, 0x00 );
}

static void gus_default_interrupt_handler_dma_read( gus_card_t *card )
{
  gus_i_write8( card, 0x49, 0x00 );
}

void gus_set_default_handlers( gus_card_t *card, unsigned int what )
{
  if ( what & GF1_HANDLER_MIDI_OUT )
    card -> gf1.interrupt_handler_midi_out = gus_default_interrupt_handler_midi_out;
  if ( what & GF1_HANDLER_MIDI_IN )
    card -> gf1.interrupt_handler_midi_in = gus_default_interrupt_handler_midi_in;
  if ( what & GF1_HANDLER_TIMER1 )
    card -> gf1.interrupt_handler_timer1 = gus_default_interrupt_handler_timer1;
  if ( what & GF1_HANDLER_TIMER2 )
    card -> gf1.interrupt_handler_timer2 = gus_default_interrupt_handler_timer2;
  if ( what & GF1_HANDLER_RANGE )
    {
      unsigned int what1 = what & 0xffff;
      card -> gf1.voice_ranges[ what1 ].interrupt_handler_wave =
      card -> gf1.voice_ranges[ what1 ].interrupt_handler_volume =
        gus_default_interrupt_handler_wave_and_volume;
      card -> gf1.voice_ranges[ what1 ].voices_change_start =
      card -> gf1.voice_ranges[ what1 ].voices_change_stop =
      card -> gf1.voice_ranges[ what1 ].volume_change = NULL;
    }
  if ( what & GF1_HANDLER_SYNTH_DMA_WRITE )
    card -> gf1.interrupt_handler_synth_dma_write = gus_default_interrupt_handler_dma_write;
  if ( what & GF1_HANDLER_PCM_DMA_WRITE )
    card -> gf1.interrupt_handler_pcm_dma_write = gus_default_interrupt_handler_dma_write;
  if ( what & GF1_HANDLER_PCM_DMA_READ )
    card -> gf1.interrupt_handler_pcm_dma_read = gus_default_interrupt_handler_dma_read;
}

/*
 *
 */

static void clear_regs( gus_card_t *card )
{
  unsigned long flags;

  CLI( &flags );
  INB( GUSP( card, IRQSTAT ) );
  gus_write8( card, 0x41, 0 );		/* DRAM DMA Control Register */
  gus_write8( card, 0x45, 0 );		/* Timer Control */
  gus_write8( card, 0x49, 0 ); 		/* Sampling Control Register */
  STI( &flags );
}

static void look_regs( gus_card_t *card )
{
  unsigned long flags;

  CLI( &flags );
  gus_look8( card, 0x41 );	/* DRAM DMA Control Register */
  gus_look8( card, 0x49 );	/* Sampling Control Register */
  INB( GUSP( card, IRQSTAT ) );
  gus_read8( card, 0x0f );	/* IRQ Source Register */
  STI( &flags );
}

/*
 *  put selected GF1 voices to initial stage...
 */

void gus_gf1_smart_stop_voice( gus_card_t *card, short voice )
{
  unsigned long flags;
  
  CLI( &flags );
  gf1_select_voice( card, voice );
#if 0
  printk( " -%i- smart stop voice - volume = 0x%x\n", voice, gus_i_read16( card, GF1_VW_VOLUME ) );
#endif
  gus_ctrl_stop( card, GF1_VB_ADDRESS_CONTROL );
  gus_ctrl_stop( card, GF1_VB_VOLUME_CONTROL );
  STI( &flags );
}

void gus_gf1_stop_voice( gus_card_t *card, short voice )
{
  unsigned long flags;
  
  CLI( &flags );
  gf1_select_voice( card, voice );
#if 0
  printk( " -%i- stop voice - volume = 0x%x\n", voice, gus_i_read16( card, GF1_VW_VOLUME ) );
#endif
  gus_ctrl_stop( card, GF1_VB_ADDRESS_CONTROL );
  gus_ctrl_stop( card, GF1_VB_VOLUME_CONTROL );
  gus_write8( card, GF1_VB_ACCUMULATOR, 0 );
  STI( &flags );
  gus_lfo_shutdown( card, voice, GUS_LFO_VIBRATO );
  gus_lfo_shutdown( card, voice, GUS_LFO_TREMOLO );
}

void gus_gf1_clear_voices( gus_card_t *card, short v_min, short v_max )
{
  unsigned long flags;
  unsigned int daddr;
  unsigned short i, w_16;

  daddr = card -> gf1.default_voice_address << 4;
  for ( i = v_min; i <= v_max; i++ )
    {
      if ( card -> gf1.syn_voices )
        card -> gf1.syn_voices[ i ].flags = ~VFLG_DYNAMIC;
      CLI( &flags );
      gf1_select_voice( card, i );
      gus_ctrl_stop( card, GF1_VB_ADDRESS_CONTROL );	/* Voice Control Register = voice stop */
      gus_ctrl_stop( card, GF1_VB_VOLUME_CONTROL ); 	/* Volume Ramp Control Register = ramp off */
#ifdef GUSCFG_INTERWAVE
      if ( card -> gf1.enh_mode )
        gus_write8( card, GF1_VB_MODE, card -> gf1.memory ? 0x02 : 0x82 );	/* Deactivate voice */
#endif
      w_16 = gus_read8( card, GF1_VB_ADDRESS_CONTROL ) & 0x04;
      gus_write16( card, GF1_VW_FREQUENCY, 0x400 );
      gus_write_addr( card, GF1_VA_START, daddr, w_16 );
      gus_write_addr( card, GF1_VA_END, daddr, w_16 );
      gus_write8( card, GF1_VB_VOLUME_START, 0 );
      gus_write8( card, GF1_VB_VOLUME_END, 0 );
      gus_write8( card, GF1_VB_VOLUME_RATE, 0 );
      gus_write16( card, GF1_VW_VOLUME, 0 );
      gus_write_addr( card, GF1_VA_CURRENT, daddr, w_16 );
      gus_write8( card, GF1_VB_PAN, 7 );
#ifdef GUSCFG_INTERWAVE
      if ( card -> gf1.enh_mode )
        {
          gus_write8( card, GF1_VB_ACCUMULATOR, 0 );
          gus_write16( card, GF1_VW_EFFECT_VOLUME, 0 );
          gus_write16( card, GF1_VW_EFFECT_VOLUME_FINAL, 0 );
        }
#endif
      STI( &flags );
      gus_lfo_shutdown( card, i, GUS_LFO_VIBRATO );
      gus_lfo_shutdown( card, i, GUS_LFO_TREMOLO );
    }
}

void gus_gf1_stop_voices( gus_card_t *card, short v_min, short v_max )
{
  unsigned long flags;
  short i, ramp_ok;
  unsigned short ramp_end;

  if ( !in_interrupt() )	/* this can't be done in interrupt */
    {
      for ( i = v_min, ramp_ok = 0; i <= v_max; i++ )
        {
          CLI( &flags );
          gf1_select_voice( card, i );
          ramp_end = gus_i_read16( card, 9 ) >> 8;
          if ( ramp_end > MIN_OFFSET )
            {
              ramp_ok++;
              gus_write8( card, GF1_VB_VOLUME_RATE, 20 );		/* ramp rate */
              gus_write8( card, GF1_VB_VOLUME_START, MIN_OFFSET );	/* ramp start */
              gus_write8( card, GF1_VB_VOLUME_END, ramp_end );		/* ramp end */
              gus_write8( card, GF1_VB_VOLUME_CONTROL, 0x40 );		/* ramp down */
              if ( card -> gf1.enh_mode )
                {
                  gus_delay( card );
                  gus_write8( card, GF1_VB_VOLUME_CONTROL, 0x40 );
                }
            }
          STI( &flags );
        }
      if ( ramp_ok )
        {
          SLEEP( card, synth_tmp, 2 );
        }
    }
  gus_gf1_clear_voices( card, v_min, v_max );
}

/*
 *  call this function only by start of driver
 */

int gus_gf1_start( gus_card_t *card )
{
  unsigned long flags;
  unsigned int i;

  gus_i_write8( card, GF1_GB_RESET, 0 );	/* reset GF1 */
  gus_delay1( 1 );
  gus_i_write8( card, GF1_GB_RESET, 1 );	/* disable IRQ & DAC */
  gus_delay1( 1 );
  gus_i_write8( card, GF1_GB_JOYSTICK_DAC_LEVEL, card -> joystick_dac );

  gus_set_default_handlers( card, GF1_HANDLER_ALL );
  gus_set_default_handlers( card, GF1_HANDLER_RANGE | GF1_VOICE_RANGE_SYNTH );
  gus_set_default_handlers( card, GF1_HANDLER_RANGE | GF1_VOICE_RANGE_PCM );
  gus_set_default_handlers( card, GF1_HANDLER_RANGE | GF1_VOICE_RANGE_EFFECT );

#ifdef GUSCFG_MIDI_DEVICES
  gf1_uart_cmd( card, 0x03 );		/* huh.. this cleanup take me some time... */
#else
  OUTB( 0x03, GUSP( card, MIDICTRL ) );	/* reset */
#endif
  if ( card -> gf1.enh_mode )		/* enhanced mode !!!! */
    {
      gus_i_write8( card, GF1_GB_GLOBAL_MODE, gus_i_look8( card, GF1_GB_GLOBAL_MODE ) | 0x01 );
      gus_i_write8( card, GF1_GB_MEMORY_CONTROL, 0x01 );
    }
  clear_regs( card );
  card -> gf1.voice_ranges[ GF1_VOICE_RANGE_SYNTH ].rvoices = 0;
  card -> gf1.voice_ranges[ GF1_VOICE_RANGE_PCM ].rvoices = 0;
  card -> gf1.voice_ranges[ GF1_VOICE_RANGE_EFFECT ].rvoices = 0;
  gus_reselect_active_voices( card );
  gus_delay( card );
  card -> gf1.default_voice_address = card -> gf1.memory > 0 ? 0 : 512 - 8;
#ifdef GUSCFG_INTERWAVE
  /* initialize LFOs & clear LFOs memory */
  if ( card -> gf1.enh_mode && card -> gf1.memory )
    {
      card -> gf1.hw_lfo = 1;
      card -> gf1.default_voice_address += 1024;
    }
   else
#endif
    card -> gf1.sw_lfo = 1;
  gus_lfo_init( card );
  if ( card -> gf1.memory > 0 )
    for ( i = 0; i < 4; i++ )
      gus_poke( card, card -> gf1.default_voice_address + i, 0 );
  clear_regs( card );
  gus_gf1_clear_voices( card, 0, 31 );
  look_regs( card );
  gus_delay1( 1 );
  gus_i_write8( card, GF1_GB_RESET, 7 ); 	/* Reset Register = IRQ enable, DAC enable */
  gus_delay1( 1 );
  gus_i_write8( card, GF1_GB_RESET, 7 );	/* Reset Register = IRQ enable, DAC enable */
  if ( card -> gf1.enh_mode )			/* enhanced mode !!!! */
    {
      gus_i_write8( card, GF1_GB_GLOBAL_MODE, gus_i_look8( card, GF1_GB_GLOBAL_MODE ) | 0x01 );
      gus_i_write8( card, GF1_GB_MEMORY_CONTROL, 0x01 );
    }
  
  while ( ( gus_i_read8( card, GF1_GB_VOICES_IRQ ) & 0xc0 ) != 0xc0 );

  CLI( &flags );
  OUTB( card -> gf1.active_voice = 0, GUSP( card, GF1PAGE ) );
  OUTB( card -> mixer.mix_ctrl_reg, GUSP( card, MIXCNTRLREG ) );
  STI( &flags );

  look_regs( card );

  gus_memory_init( card, &card -> gf1.mem_alloc );

#ifdef GUSCFG_INTERWAVE
  if ( card -> pnp_flag )
    {
      if ( card -> codec.playback_fifo_size > 0 )
       gus_i_write16( card, GF1_GW_FIFO_RECORD_BASE_ADDR, card -> codec.playback_fifo_block -> ptr >> 8 );
      if ( card -> codec.record_fifo_size > 0 )
       gus_i_write16( card, GF1_GW_FIFO_PLAY_BASE_ADDR, card -> codec.record_fifo_block -> ptr >> 8 );
      gus_i_write16( card, GF1_GW_FIFO_SIZE, card -> codec.interwave_fifo_reg );
    }
#endif

  return 0;
}

/*
 *  call this function only by shutdown of driver
 */

void gus_gf1_stop( gus_card_t *card )
{
#ifdef GUSCFG_CODEC
  if ( card -> max_flag && card -> codec.mode == CODEC_MODE_NONE )
  					/* clear out MAX DMA latches */
    {
      unsigned long flags;
      
      CLI( &flags );
      OUTB( card -> max_cntrl_val & ~0x30, GUSP( card, MAXCNTRLPORT ) );
      OUTB( card -> max_cntrl_val, GUSP( card, MAXCNTRLPORT ) );
      STI( &flags );
    }  
#endif

  gus_i_write8( card, 0x45, 0 );	/* stop all timers */
  gus_gf1_stop_voices( card, 0, 31 );	/* stop all voices */
  gus_i_write8( card, 0x4c, 1 );	/* disable IRQ & DAC */
  gus_memory_done( card, &card -> gf1.mem_alloc );
  gus_lfo_done( card );
}

/*
 *  standard open/close function
 */

static void gus_gf1_set_reset_flags( gus_card_t *card )
{
  unsigned short mode;
  unsigned char reset;
  
  mode = card -> gf1.mode;
  reset = 1;				/* IRQ & DAC disable, no reset */
  if ( mode & (GF1_MODE_ENGINE|GF1_MODE_PCM) )
    reset |= 7;				/* IRQ & DAC enable */
  if ( mode & GF1_MODE_TIMER )
    reset |= 5;				/* IRQ enable */
  gus_i_write8( card, 0x4c, reset );  
}
 
void gus_gf1_open( gus_card_t *card, unsigned short mode )
{
  if ( card -> gf1.mode & mode ) return; /* already */
  card -> gf1.mode |= mode;
  gus_gf1_set_reset_flags( card );
}

void gus_gf1_close( gus_card_t *card, unsigned short mode )
{
  if ( !( card -> gf1.mode & mode ) ) return;
  card -> gf1.mode &= ~mode;
  gus_gf1_set_reset_flags( card );
}
