// framegraph.C

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/


#ifdef __GNUG__
#pragma implementation
#endif

#include "localdefs.h"
#include <InterViews/brush.h>
#include <InterViews/cursor.h>
#include <InterViews/event.h>
#include <InterViews/rubline.h>
#ifdef index
#define was_index index
#undef index
#define mxv_index_undefined
#endif
#include <InterViews/font.h>
#ifdef mxv_index_undefined
#undef mxv_index_undefined
#define index was_index
#endif
#include <InterViews/painter.h>
#include <InterViews/perspective.h>
#include <InterViews/sensor.h>
#include <InterViews/shape.h>
#include "framegraph.h"
#include "data.h"
#include "graphplot.h"
#include "scale.h"
#include "selector.h"

FrameGraph::FrameGraph(Controller *c, Data *data, const Range& vertRange) 
		: Graph(c, data, vertRange), graphplots(nil), vScaleUnits(Amplitude) {
	Init();
}

FrameGraph::FrameGraph(Controller *c, Data *data)
		: Graph(c, data, data->limits()), graphplots(nil),
		  vScaleUnits(Amplitude) {
	Init();
}

FrameGraph::~FrameGraph() {
	for(int plot=0; plot < maxPlots; plot++)
		Resource::unref(graphplots[plot]);
	delete [] graphplots;
	delete [] frameNumbers;
	Resource::unref(dashBrush);
}

void
FrameGraph::Init() {
	BUG("FrameGraph::Init()");
	SetClassName("FrameGraph");
	oldFramesVisible = 0;
	frameNumbers = nil;
	dashBrush = nil;
	maxPlots = 0;
	vertGrain = 1.0;
	hshift = 50;		// for now, until variable
	vshift = 200;		// for now, until variable
}

void
FrameGraph::Reconfig() {
	BUG("FrameGraph::Reconfig()");
    static int pattern[] = {2, 2};
	if(!isConfigured()) {
		const char *a = GetAttribute("VerticalPlotOffset");
		if (a != nil) vshift = atoi(a);
		a = GetAttribute("HorizontalPlotOffset");
		if (a != nil) hshift = atoi(a);
		a = GetAttribute("VerticalScaleUnits");
		if (a != nil && !strcmp(a, "Decibels"))
			setVerticalScaleUnits(Decibels);
		SetCanvasType(CanvasSaveContents);	// to avoid redrawing
		dashBrush = new Brush(pattern, 2, 0);
		dashBrush->ref();	// to keep it around
	}
	Graph::Reconfig();			// default or attribute-set size done here
}

inline double
FrameGraph::currentVGrain() {
	return vertGrain;
}

inline int
FrameGraph::vertOriginLocation() {
	return Graph::vertOriginLocation() + vertShift();
}

inline const char *
FrameGraph::verticalScaleLabel() {
	return scaleInAmplitude() ? "Magnitude" : "Decibels";		
}

inline int
FrameGraph::framesVisible() {
	return min(perspective->curheight, maxPlots);
}

inline int
FrameGraph::channelsVisible() {
	return perspective->curwidth;
}

inline int
FrameGraph::frameSize() {
	return graphData->frameLength();
}

void
FrameGraph::setShape() {	// called from Graph::Reconfig()
	shape->width += horizShift();
	shape->height += vertShift();
	shape->Rigid(max(0, shape->width - (horizShift() * 2)), hfil,
				 max(0, shape->height/2 - vertShift()), vfil);
}

void
FrameGraph::Redraw(Coord left, Coord bottom, Coord right, Coord top) {
	BUG("FrameGraph::Redraw()");
	output->Clip(canvas, left, bottom, right, top);
	output->ClearRect(canvas, left, bottom, right, top);
	int totalHorizOffset = horizShift();
	int hOrigin = horizOriginLocation();
	int totalVertOffset = vertShift();
	int vOrigin = vertOriginLocation();
	int frameEnd = framesVisible();
	const Brush* oldBrush = output->GetBrush();
	oldBrush->ref();	// to keep it around
	output->SetBrush(dashBrush);
    // draw dotted z origin
	output->Line(canvas, hOrigin, ymax - topBorder(),
		hOrigin, ymax - topBorder() - currentHeight()); 
	output->SetBrush(oldBrush);
	char frameLabel[16];
	const char *label = nil;
	const Font* font = output->GetFont();
	int voffset = -(font->Height() / 2);
	for(int frame=0; frame < frameEnd; frame++) {	// draw plots and labels
		double fraction = double(frame)/frameEnd;
		int horizLoc = hOrigin + int(totalHorizOffset * fraction);
		int vertLoc = vOrigin - int(totalVertOffset * fraction);
		output->SetOrigin(horizLoc, vertLoc);
		// check each plot to see if it needs to be redrawn
		if(graphplots[frame]->contains(left - horizLoc, bottom - vertLoc,
				right - horizLoc, top - vertLoc)) {
			if((label = getLabel(frameLabel, frame)) != NULL) {
				output->Text(canvas, label,
					(-2 - font->Width(label)), voffset);
			}
			graphplots[frame]->draw(canvas, output);
			output->SetBrush(dashBrush);
			output->Line(canvas, 0, 0, currentWidth(), 0);	// base line
			output->SetBrush(oldBrush);
		}
	}
	output->SetOrigin(0, 0);
    // draw dotted y origin
	output->SetBrush(dashBrush);
	output->Line(canvas, hOrigin, vOrigin,
		hOrigin + totalHorizOffset, vOrigin - totalVertOffset); 
	output->SetBrush(oldBrush);
	output->NoClip();
	Resource::unref(oldBrush);
}

// if more than two areas need to be redrawn, it is more efficient to just
// redraw the entire window.  This will be further optimized in the future.

void FrameGraph::RedrawList(
	int n, IntCoord left[], IntCoord bottom[], IntCoord right[], IntCoord top[]
	) {
	if(n > 2)
		Draw();
	else
		Graph::RedrawList(n, left, bottom, right, top);
}

// this is called by Graph::Resize() prior to loading plots

void
FrameGraph::doResize() {
	BUG("FrameGraph::doResize()");
	int newMax = vertShift()/2; // plots can be spaced every 2nd vertical pixel
	int oldMax = maxPlots;
	if(newMax > oldMax) {
		GraphPlot **newplots = new GraphPlot *[newMax];	// create slots
		for(int plot = 0; plot < newMax; plot++) {
			if(plot < oldMax && graphplots[plot] != nil)
				newplots[plot] = graphplots[plot];	// copy plots to new slots
			else
				newplots[plot] = nil;				// and zero new slots
		}
		delete [] graphplots;		// delete old slot array
		graphplots = newplots;		// and copy new array to old
		maxPlots = newMax;
	}
}

void
FrameGraph::Update() {
	BUG("FrameGraph::Update()");
	Graph::Update();
}

boolean
FrameGraph::isViewChanged(Perspective &np) {
	register Perspective *p = perspective;
	return (
		np.curwidth != p->curwidth
		|| np.width != p->width
		|| np.curx != p->curx
		|| np.curheight != p->curheight
		|| np.height != p->height
		|| np.cury != p->cury
	);
}

// compiler sez "sorry, not implemented: aggregate value in COND_EXPR" -- UGH!
// so cant use () ? : here.

void
FrameGraph::setReferenceRange() {
	if(scaleInAmplitude())
		Graph::setReferenceRange(graphData->limits());
	else
		Graph::setReferenceRange(Range(0.0, 90.31));
}

void
FrameGraph::setVerticalGrain(double newgrain) {
	BUG("FrameGraph::setVerticalGrain()");
	if(newgrain != vertGrain) {
		vertGrain = newgrain;
		doAdjust();
		Draw();
		if(vscale) {
			vscale->setRange(verticalScaleRange());
			vscale->Update();
		}
	}
}

void
FrameGraph::setVerticalScaleUnits(VerticalScaleUnits units) {
	if(units != vScaleUnits) {
		vScaleUnits = units;
		Update();
		if(vscale) {
			vscale->setLabel(verticalScaleLabel());
			vscale->Update();
		}
	}
}

void
FrameGraph::createPlot() {
	BUG("FrameGraph::createPlot()");
	for(int frame = 0; frame < maxPlots; frame++) {
		Resource::unref(graphplots[frame]);
		graphplots[frame] = new SolidPlot(this, currentWidth());
		graphplots[frame]->ref();
	}
	delete [] frameNumbers;
	frameNumbers = new int[maxPlots];
	oldFramesVisible = maxPlots;
}

void
FrameGraph::loadPlot() {
	BUG("FrameGraph::loadPlot()");
	register Perspective *p = GetPerspective();
	int framesize = min(frameSize(), channelsVisible());
	int frameoffset = p->curx;
	double frameIncrement = currentFrameGrain();
	int startFrame = p->cury;
	for(int plot = 0, frame = 0; plot < framesVisible(); plot++) {
		GraphPlot* graphplot = graphplots[plot];
		graphplot->setPlotLength(min(framesize, currentWidth()));
		frame = startFrame + int(plot * frameIncrement);
		frameNumbers[plot] = frame;		// for later display
		Data* frameData = graphData->cloneFrame(
			frame, Range(frameoffset, frameoffset + framesize - 1)
		);
		frameData->ref();
		graphplot->load(frameData,
			scaleInAmplitude() ? PeakToPeakAmplitude : ::Decibels);
		Resource::unref(frameData);
	}
}

void
FrameGraph::doPlot() {
	for(int frame=0; frame < framesVisible(); frame++)
		graphplots[frame]->plot();
}

// currentWidth redefined to represent the width of each plot rather than
// the width of the canvas

int
FrameGraph::currentWidth() {
	return Graph::currentWidth() - horizShift();
}

// currentHeight redefined to represent the height of rearmost plot

int
FrameGraph::currentHeight() {
	return Graph::currentHeight() - vertShift();
}

void
FrameGraph::grab(Event &e) {
	if(!e.control_is_down())
		return;
	SetCursor(crosshairs);
    static int pattern[] = {2, 2};
	const Brush* oldBrush = output->GetBrush();
	oldBrush->ref();	// to keep it around
	output->SetBrush(new Brush(pattern, 2, 2));
	Listen(allEvents);
	int horigin = horizOriginLocation();
	int vorigin = vertOriginLocation();
	RubberLine line(output, canvas, horigin, vorigin,
		horigin + horizShift(), vorigin + vertShift());
	line.Track(e.x, e.y);
	int xloc = e.x;
	int yloc = e.y;
    do {
		switch (e.eventType) {
			  case MotionEvent:
				e.target->GetRelative(e.x, e.y, this);
				constrainOffsets(e.x, e.y);
				line.Track(e.x, e.y);
				xloc = e.x;
				yloc = e.y;
				break;
			  default:
				break;
		}
		Read(e);
    } while (e.eventType != UpEvent);
    Listen(input);
	output->SetBrush(oldBrush);
	Resource::unref(oldBrush);
	SetCursor(defaultCursor);
	setAxisOffsets(xloc - horigin, vorigin - yloc);
}

void
FrameGraph::constrainOffsets(int& x, int& y) {
	x = max(horizOriginLocation(), min(xmax / 2, x));
	y = min(vertOriginLocation(), y);
}

void
FrameGraph::setAxisOffsets(int horiz, int vert) {
	setHorizShift(horiz);
	setVertShift(vert);
	doResize();
	createPlot();
	Update();
	Parent()->Change(this);
}
	
void
FrameGraph::setVertShift(int vsh) {
	static const int minVertOffset = 2;
	vshift = min(Graph::currentHeight() - 50, 
		max(vsh, minVertOffset * framesVisible()));
}

void
FrameGraph::setHorizShift(int hsh) {
	hshift = hsh;
}

double
FrameGraph::currentFrameGrain() {
	return max(1.0, double(perspective->curheight)/framesVisible());
}

const char *
FrameGraph::getLabel(char *label, int frame) {
	int frameNumber = frameNumbers[frame];
	int visible = framesVisible();
	double grain = currentFrameGrain();
	int thresh = int(grain);
	boolean retval;
	int increment = 1;
	if(grain == 1.0 && visible < 10)
		retval = true;
	else if(grain <= 1.9 && visible < 50)
		retval = (frameNumber % (increment=5) == 0);
	else if(grain < 3.0 && visible < 100)
		retval = (frameNumber % (increment=10) < thresh);
	else
		retval = (frameNumber % (increment=50) < thresh);
	if(retval) {
		toString(label, int(roundDownToNearest(frameNumber, increment)));
		return label;
	}
	return NULL;
}
