// application.C

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/


#ifdef __GNUG__
#pragma implementation
#endif

#include <InterViews/banner.h>
#include <InterViews/event.h>
#include <InterViews/frame.h>
#include <InterViews/world.h>
#include "application.h"
#include "controller.h"
#include "converter.h"
#include "datafile.h"
#include "sound.h"
#include "vmessage.h"
#include <String.h>
#include <iostream.h>
#include "localdefs.h"

#ifndef NeXT
#include <errno.h>
extern char *sys_errlist[];
char *strerror(int errn) {
	return(sys_errlist[errn]);
}
#else
extern "C" {
	char *strerror(int);
	int sleep(unsigned);
}
#endif /* NeXT */

class GlobalResourceList {
	friend Application;
private:
	class ResourceMember {
	public:
		ResourceMember(const char* resname, const char* resval)
			: next(nil), resourceName(resname), resourceValue(resval) {}
		~ResourceMember() {}
		boolean nameMatches(const char* name) { return resourceName == name; }
		void setValue(const char* value) { resourceValue = value; }
		const char* value() { return resourceValue; }
		ResourceMember* next;
	private:
		String resourceName;
		String resourceValue;
	};
	void add(const char* resname, const char* resval);
	const char* retrieve(const char* resname);
protected:
	GlobalResourceList() {}
	~GlobalResourceList();
	static ResourceMember* head;
};

GlobalResourceList::ResourceMember* GlobalResourceList::head = nil;

GlobalResourceList::~GlobalResourceList() {
	ResourceMember* member = head;
	while(member != nil) {
		ResourceMember* next = member->next;
		delete member;
		member = next;
	}
}

void
GlobalResourceList::add(const char* resname, const char* resval) {
	if(head == nil) {
		head = new ResourceMember(resname, resval);
	}
	else {
		ResourceMember* member = head;
		while(member->next != nil) {
			if(member->nameMatches(resname)) {
				member->setValue(resval);	// if resource in list, just reset
				return;
			}
			member = member->next;
		}
		// otherwise add new member at end of list
		member->next = new ResourceMember(resname, resval);
	}
}	

const char *
GlobalResourceList::retrieve(const char* resname) {
	ResourceMember* member = head;
	while(member != nil) {
		if(member->nameMatches(resname))
			return member->value();
		member = member->next;
	}
	return nil;
}

int Application::worldVisible = 0;
World* Application::world = nil;
StatusBar* Application::statusBar = nil;
GlobalResourceList* Application::resourceList = nil;
Controller* Application::global_controller = nil;

extern void testit(World *);
extern void read_mxvrc();

Application::Application(int argc, char** argv,
		struct OptionDesc options[], struct PropertyData properties[]) {
#if defined(NeXT)
	malloc_singlethreaded();
#endif
	world = new World("MiXViews", argc, argv, options, properties);
	resourceList = new GlobalResourceList;
	read_mxvrc();
	Controller* controller = nil;
	if(argc == 1) {				// default when no args
		Sound* temp = new Sound(1.0, Sound::defaultSampleRate(), 1,
				Sound::defaultDataType());
		temp->modified(false);	// to avoid save warning when closed
		controller = new Controller(temp);
		controller->display(world);
	}
	else for(int argnum = 1; argnum < argc; ) {
		double skip = 0, dur = 0;
		char* filename = argv[argnum++];
		if(argnum < argc)
			argnum = extractSkipAndDurationArgs(argv, argc, argnum, skip, dur);
		DataFile* file = new DataFile(filename, "r", skip, dur);
		controller = Controller::create(file);
		if(controller)
			controller->display(world);
	}
	set_File_error_handler(error_handler);	// wait until after windows up
	createStatusBar(world);
	inform();
	worldVisible = true;
	testit(world);
}

int
Application::extractSkipAndDurationArgs(char** args, int nargs, int indx,
		double& skip, double& dur) {
	int maxarg = nargs - 1;
	while(indx <= maxarg) {
		char* arg = args[indx];
		if(strcmp(arg, "-skip") == 0 && indx <= maxarg) {
			indx++;
			skip = double(atof(args[indx++]));
			if(skip <= 0.0) {
				cerr << "Inskip arguments must be greater than zero -- ";
				cerr << "setting to " << (skip = -skip) << endl;
			}
		}
		else if(strcmp(arg, "-duration") == 0 && indx <= maxarg) {
			indx++;
			dur = double(atof(args[indx++]));
			if(dur <= 0.0) {
				cerr << "Duration arguments must be greater than zero -- ";
				cerr << "setting to " << (dur = -dur) << endl;
			}
		}
		else if(*arg == '-') {
			cerr << "Unknown flag: `" << arg << "` ignored." << endl;
			indx++;
		}
		else break;
	}
	return indx;
}

Application::~Application() {
	Converter::destroyInstance();	// in case of program exit
	delete resourceList;
	cout << "Exiting mxv." << endl;
}

int
Application::Run() {
	if(world != nil)
		world->Run();
	return 0;
}

void
Application::setGlobalController(Controller* c) { global_controller = c; }

boolean
Application::isGlobalController(Controller* c) {
	return c == global_controller;
}

void
Application::busy(boolean bsy) {
	global_controller->busy(bsy);
}

void
Application::inform(const char* string, boolean wait) {
	if(string == nil)
		return;
	static char msg[256];
	sprintf(msg, "%s%s", "Status:  ", string);
	if(statusBar != nil)
		statusBar->setText(msg);
	else
		cerr << msg << "\n";
	if(wait)
		sleep(1);
}

void
Application::inform() {
	inform("Ready");
}

void
Application::die(const char* string) {
	cerr << string << "\n";
	cerr.flush();
	abort();
}

void
Application::error(const char* msg1, const char* msg2) {
	const char* errmsg = errno ? strerror(errno) : nil;
	const char* str2 = msg2 ? msg2 : errmsg;
	const char* str3 = msg2 ? errmsg : nil;
	alert(msg1, str2, str3);
	errno = 0;
}

void
Application::error_handler(const char* msg) {
	error(msg);
}

void
Application::alert(const char* msg1, const char* msg2,
		const char* msg3, const char* btn) {
	if(global_controller)
		global_controller->alert(msg1, msg2, msg3, btn);
	else
		cerr << msg1 << endl << (msg2 ? msg2 : "") << endl << (msg3 ? msg3 : "") << endl;
}

boolean
Application::confirm(const char* msg1, const char* msg2,
		const char* msg3, Response r, const char* btn1, const char* btn2) {
	return global_controller->confirm(msg1, msg2, msg3, r, btn1, btn2);
}

Response
Application::choice(const char* msg1, const char* msg2, const char* msg3,
		Response r, const char* btn1, const char* btn2, const char* btn3) {
	return global_controller->choice(msg1, msg2, msg3, r, btn1, btn2, btn3);
}

int
Application::handle(Request &r) {
	return global_controller->handleRequest(r);
}

void
Application::createStatusBar(World *world) {
	statusBar = new StatusBar("MXV Status", 
		"MMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMM");
	world->Insert(
		new TitleFrame(new Banner("", "MXV Status", ""), statusBar, 1), 
		300,
		world->Height() - 80
	);
}

void
Application::setGlobalResource(const char* resName, const char* resValue) {
	resourceList->add(resName, resValue);
}

// look in global resource list to see if value has been set
// if not there, attempt to retrieve from global attribute list

const char *
Application::getGlobalResource(const char* resName) {
	const char* gRes = resourceList->retrieve(resName);
	return (gRes != nil) ? gRes : world->GetAttribute(resName);
}

// short cut for checking if resource is true or set to yes

boolean
Application::globalResourceIsTrue(const char* resName) {
	static Regex Yes("YES\\|[Yy]es\\|TRUE\\|[Tt]rue");
	const char* res = getGlobalResource(resName);
	return (res != nil && Yes.match(res, strlen(res)) != -1);
}

boolean
Application::isLittleEndian() {
	int testInt = 1;
	char* endbyte = (char *) &testInt;
	return (*endbyte);
}
