/*----------------------------------------------------------------
 * partial loading mechanism
 *----------------------------------------------------------------*/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include "util.h"
#include "aweseq.h"
#include "dynload.h"

static void set_preset(SFPatchRec *pat, char *arg);

/*----------------------------------------------------------------*/

LoadList *add_loadlist(LoadList *list, SFPatchRec *pat, SFPatchRec *map)
{
	LoadList *rec;
	rec = (LoadList*)safe_malloc(sizeof(LoadList));
	rec->pat = *pat;
	if (map)
		rec->map = *map;
	else
		rec->map = *pat;
	rec->loaded  = FALSE;
	rec->next = list;
	return rec;
}

/* add the elements in the latter list to the former one */
LoadList *merge_loadlist(LoadList *list, LoadList *old)
{
	LoadList *p;
	for (p = old; p; p = p->next) {
		list = add_loadlist(list, &p->pat, &p->map);
		list->loaded  = p->loaded;
	}
	return list;
}

/* free all the elements in the list */
void free_loadlist(LoadList *p)
{
	LoadList *next;
	for (; p; p = next) {
		next = p->next;
		safe_free(p);
	}
}

/* parse source and mapping presets and return the remaining string;
 * the original argument will be broken
 */
int awe_parse_loadlist(char *arg, SFPatchRec *pat, SFPatchRec *map, char **strp)
{
	char *next;

	if ((next = strschr(arg, ":=")) != NULL)
		*next++ = 0;
	set_preset(pat, arg);
	if (next == NULL) {
		*map = *pat;
		if (strp) *strp = NULL;
		return TRUE;
	}
	arg = next;
	if ((next = strschr(arg, ":=")) != NULL)
		*next++ = 0;
	set_preset(map, arg);
	if (strp) *strp = next;
	return TRUE;
}

/* parse preset/bank/keynote */
static void set_preset(SFPatchRec *pat, char *arg)
{
	char *next;
	if ((next = strschr(arg, "/: \t\n")) != NULL)
		*next++ = 0;
	pat->preset = atoi(arg);
	pat->bank = 0;
	pat->keynote = -1;
	arg = next;
	if (arg) {
		if ((next = strschr(arg, "/: \t\n")) != NULL)
			 *next++ = 0;
		if (isdigit(*arg))
			pat->bank = atoi(arg);
		else if (*arg == '*' || *arg == '-')
			pat->bank = -1;
		if (next) {
			if (isdigit(*next))
				pat->keynote = atoi(next);
			else if (*next == '-' || *next == '*')
				pat->keynote = -1;
		}
	}
}


/*----------------------------------------------------------------*/

/* linked list of sample record */
typedef struct _DynSample {
	int id, marked;
	struct _DynSample *next;
} DynSample;

static int nsamples;
static DynSample *dynsample;

/* initialize lists */
void awe_init_marks(void)
{
	nsamples = 0;
	dynsample = NULL;
}

/* free allocated lists */
void awe_free_marks(void)
{
	DynSample *sp, *sp_next;

	for (sp = dynsample; sp; sp = sp_next) {
		sp_next = sp->next;
		safe_free(sp);
	}
}

/* check the preset matches to the given pattern (rec) */
int awe_match_preset(SFPatchRec *rec, SFPatchRec *pat)
{
	if (rec->preset != -1 && pat->preset != -1 &&
	    rec->preset != pat->preset)
		return FALSE;
	if (rec->bank != -1 && pat->bank != -1 &&
	    rec->bank != pat->bank)
		return FALSE;
	if (rec->keynote != -1 && pat->keynote != -1 &&
	    rec->keynote != pat->keynote)
		return FALSE;
	return TRUE;
}

/* mark the sample */
static void add_sample(int sample, int marked)
{
	DynSample *sp;
	for (sp = dynsample; sp; sp = sp->next) {
		if (sp->id == sample)
			return;
	}

	sp = (DynSample*)safe_malloc(sizeof(DynSample));
	sp->id = sample;
	sp->marked = marked;
	sp->next = dynsample;
	dynsample = sp;
	nsamples++;
}

void awe_add_sample(int sample)
{
	add_sample(sample, FALSE);
}

void awe_add_marked_sample(int sample)
{
	add_sample(sample, TRUE);
}

/* search the sample element with the specified id */
static DynSample *search_sample(int id)
{
	DynSample *sp;
	for (sp = dynsample; sp; sp = sp->next) {
		if (sp->id == id)
			return sp;
	}
	return NULL;
}

/* check if the specified sample matches the sample list */
int awe_is_listed_sample(int id)
{
	if (search_sample(id))
		return TRUE;
	return FALSE;
}

/* mark up the sample */
void awe_mark_sample(int id)
{
	DynSample *sp;
	if ((sp = search_sample(id)) != NULL)
		sp->marked = TRUE;
}

/* check if the specified sample is loaded on the driver */
int awe_is_marked(int id)
{
	DynSample *sp;
	if ((sp = search_sample(id)) != NULL)
		return sp->marked;
	return FALSE;
}

