/*================================================================
 * event.c
 *	MIDI event processing routine
 *================================================================*/

#include <stdio.h>
#include "channel.h"
#include "midievent.h"
#include "controls.h"
#include "util.h"
#include "seq.h"
#include "awe_effect.h"

/*----------------------------------------------------------------*/

static void channel_reset(MidiInfo *mp, int ch, int init);
static void Nothing(MidiEvent *ev, MidiInfo *minfo, int mode);
static void NoteOn(MidiEvent *ev, MidiInfo *minfo, int mode);
static void NoteOff(MidiEvent *ev, MidiInfo *minfo, int mode);
static void KeyPressure(MidiEvent *ev, MidiInfo *minfo, int mode);
static void ChanPressure(MidiEvent *ev, MidiInfo *minfo, int mode);
static void PitchSens(MidiEvent *ev, MidiInfo *minfo, int mode);
static void PitchWheel(MidiEvent *ev, MidiInfo *minfo, int mode);
static void Program(MidiEvent *ev, MidiInfo *minfo, int mode);
static void ControlChange(MidiEvent *ev, MidiInfo *minfo, int mode);
static void ResetControl(MidiEvent *ev, MidiInfo *minfo, int mode);
static void AllNotesOff(MidiEvent *ev, MidiInfo *minfo, int mode);
static void AllSoundsOff(MidiEvent *ev, MidiInfo *minfo, int mode);
static void ToneBank(MidiEvent *ev, MidiInfo *minfo, int mode);
static void PrintLyric(MidiEvent *ev, MidiInfo *minfo, int mode);
static void Tempo(MidiEvent *ev, MidiInfo *minfo, int mode);
static void NRPNEvent(MidiEvent *ev, MidiInfo *minfo, int mode);
static void GSNRPNEvent(MidiEvent *ev, MidiInfo *minfo, int mode);
static void FineTune(MidiEvent *ev, MidiInfo *minfo, int mode);
static void CoarseTune(MidiEvent *ev, MidiInfo *minfo, int mode);
static void MasterVolume(MidiEvent *ev, MidiInfo *minfo, int mode);

/*----------------------------------------------------------------*/

/* global variables for channel status */

int channel_drums;
ChannelStat channels[MAX_MIDI_CHANNELS];

/*----------------------------------------------------------------*/

/* MT-32 emulation translate table (copied from playmidi) */
static int mt32pgm[128] =
{
   0,   1,   2,   4,   4,   5,   5,   3,  16,  16,  16,  16,  19,
  19,  19,  21,   6,   6,   6,   7,   7,   7,   8,   8,  62,  57,
  63,  58,  38,  38,  39,  39,  88,  33,  52,  35,  97, 100,  38,
  39,  14, 102,  68, 103,  44,  92,  46,  80,  48,  49,  51,  45,
  40,  40,  42,  42,  43,  46,  46,  24,  25,  28,  27, 104,  32,
  32,  34,  33,  36,  37,  39,  35,  79,  73,  76,  72,  74,  75,
  64,  65,  66,  67,  71,  71,  69,  70,  60,  22,  56,  59,  57,
  63,  60,  60,  58,  61,  61,  11,  11,  99, 100,   9,  14,  13,
  12, 107, 106,  77,  78,  78,  76, 111,  47, 117, 127, 115, 118,
 116, 118, 126, 121, 121,  55, 124, 120, 125, 126, 127
};

/*----------------------------------------------------------------*/
  
typedef void (*EventFunc)(MidiEvent *ev, MidiInfo *minfo, int mode);

static EventFunc midi_events[] = {
	Nothing,	/* ME_NONE */
	NoteOn,		/* ME_NOTEON */
	NoteOff,	/* ME_NOTEOFF */
	KeyPressure,	/* ME_KEYPRESSURE */
	ControlChange,	/* ME_CONTROL */
	Program,	/* ME_PROGRAM */
	ChanPressure,	/* ME_CHNPRESSURE */
	PitchWheel,	/* ME_PITCHWHEEL */
	Tempo,		/* ME_TEMPO */
	PitchSens,	/* ME_PITCH_SENS */
	ToneBank,	/* ME_TONE_BANK */
	AllSoundsOff,	/* ME_ALL_SOUNDS_OFF */
	ResetControl,	/* ME_RESET_CONTROLLERS */
	AllNotesOff,	/* ME_ALL_NOTES_OFF */
	PrintLyric,	/* ME_LYRIC */
	MasterVolume,	/* ME_MASETER_VOLUME */
	FineTune,	/* ME_FINETUNE */
	CoarseTune,	/* ME_COARSETUNE */
	NRPNEvent,	/* ME_AWE_FX */
	GSNRPNEvent,	/* ME_GS_FX */
};


/*
 * process one midi event
 */
int do_midi_event(MidiEvent *ev, MidiInfo *minfo, int mode)
{
	if (ev->type == ME_EOT)
		return RC_TUNE_END;
	else if (ev->type < ME_EOT)
		midi_events[ev->type](ev, minfo, mode);

	return RC_NONE;
}


/*
 * initialize all midi channels (not call sequencer controls)
 */
void channel_init(MidiInfo *mp)
{
	int i;
	mp->master_volume = 100;
	channel_drums = mp->drumflag;
	for (i = 0; i < MAX_MIDI_CHANNELS; i++) {
		if (CHAN_ISDRUM(i)) {
			channels[i].defbank = 128;
			if (mp->midi_mode == MODE_MT32)
				channels[i].defpreset = 127;
			else
				channels[i].defpreset = 0;
			channels[i].bankmode = 127;
		} else {
			channels[i].defpreset = 0;
			if (mp->midi_mode == MODE_MT32)
				channels[i].defbank = 127;
			else
				channels[i].defbank = 0;
			channels[i].bankmode = 0;
		}
		if (!(mp->chn_volflag & DRUMBIT(i)))
			channels[i].volscale = 100;
		channel_reset(mp, i, TRUE);
	}
}

/*
 * reset channel status to default (not call sequencer)
 */
static void channel_reset(MidiInfo *mp, int i, int init)
{
	ChannelStat *ch = &channels[i];
	if (init) {
		ch->preset = ch->defpreset;
		ch->bank = ch->defbank;
		ch->controls[CTL_MAIN_VOLUME] = 100;
		ch->pitchsense = 2 * 128;
		ch->controls[CTL_CHORUS_DEPTH] = 0;
		ch->controls[CTL_REVERB_DEPTH] = 0;
		ch->controls[CTL_PAN] = 63;
		ch->controls[CTL_CHORUS_DEPTH] = mp->chorusdepth;
		ch->controls[CTL_REVERB_DEPTH] = mp->reverbdepth;
	}
	ch->controls[CTL_MODWHEEL] = 0;
	ch->controls[CTL_EXPRESSION] = 127;
	ch->controls[CTL_SUSTAIN] = 0;
	ch->controls[CTL_SOSTENUTO] = 0;
	ch->pitchbend = 0x2000;
	ch->chan_press = 0;
	ch->finetune = 0;
	ch->coarsetune = 0;
}

/*
 * send all the channel status to sequencer device
 */

#define CONTROL(v,type)  seq_control(v, type, channels[v].controls[type])

void channel_set(MidiInfo *mp)
{
	int i;

	seq_set_drumchannels(channel_drums);
	for (i = 0; i < MAX_MIDI_CHANNELS; i++) {
		ChannelStat *ch = &channels[i];
		seq_reset_channel(i);
		seq_set_bank(i, ch->bank);
		if (mp->midi_mode == MODE_XG && mp->xg_mapping &&
		    CHAN_ISDRUM(i))
			seq_set_program(i, ch->preset + 64);
		else
			seq_set_program(i, ch->preset);
			
		CONTROL(i, CTL_MODWHEEL);
		CONTROL(i, CTL_PAN);
		CONTROL(i, CTL_MAIN_VOLUME);
		CONTROL(i, CTL_EXPRESSION);
		CONTROL(i, CTL_SUSTAIN);
		CONTROL(i, CTL_SOSTENUTO);
		if (ch->controls[CTL_CHORUS_DEPTH] >= 0)
			CONTROL(i, CTL_CHORUS_DEPTH);
		if (ch->controls[CTL_REVERB_DEPTH] >= 0)
			CONTROL(i, CTL_REVERB_DEPTH);
		seq_pitchsense(i, ch->pitchsense);
		seq_pitchbend(i, ch->pitchbend);
		seq_chan_pressure(i, ch->chan_press);
		seq_detune(i, ch->coarsetune, ch->finetune);
	}
}

/*----------------------------------------------------------------*/

static void Nothing(MidiEvent *ev, MidiInfo *minfo, int mode) {}

/* note on */
static void NoteOn(MidiEvent *ev, MidiInfo *minfo, int mode)
{
	int key, vol;
	if (mode == EV_PRELOAD) {
		/* only for check of dynamic loading of drum sounds */
		if (CHAN_ISDRUM(ev->channel) ||
		    channels[ev->channel].bankmode == 126) {
			int preset = channels[ev->channel].preset;
			if (minfo->midi_mode == MODE_XG && minfo->xg_mapping &&
			    CHAN_ISDRUM(ev->channel))
				preset += 64;
			add_preload(ev->channel, preset,
				    channels[ev->channel].bank, KeyOf(ev));
		}
		return;
	} else if (mode != EV_PLAY && mode != EV_CONTROL)
		return;
	if (VelOf(ev) == 0) {
		NoteOff(ev, minfo, mode);
		return;
	}
	vol = VelOf(ev);
#ifndef NO_VOLSCALE
	vol = (vol * minfo->volscale) / 100;
	vol = (vol * channels[ev->channel].volscale) / 100;
	if (vol <= 0) vol = 0;
	else if (vol > 127) vol = 127;
#endif
	key = KeyOf(ev);
	if (!CHAN_ISDRUM(ev->channel)) {
		key += minfo->base_offset;
		if (key < 0 || key >= 127) return;
	}
	if (mode == EV_PLAY)
		seq_start_note(ev->channel, key, vol);
	else if (mode == EV_CONTROL)
		ctl->note(ev->channel, key, vol);
}

/* note off */
static void NoteOff(MidiEvent *ev, MidiInfo *minfo, int mode)
{
	int key;
	if (mode != EV_PLAY && mode != EV_CONTROL) return;
	key = KeyOf(ev);
	if (!CHAN_ISDRUM(ev->channel)) {
		key += minfo->base_offset;
		if (key < 0 || key >= 127) return;
	}
	if (mode == EV_PLAY)
		seq_stop_note(ev->channel, key, VelOf(ev));
	else if (mode == EV_CONTROL)
		ctl->note(ev->channel, key, 0);
}

/* key pressure change (aftertouch) */
static void KeyPressure(MidiEvent *ev, MidiInfo *minfo, int mode)
{
	int vol, key;
	if (mode != EV_PLAY && mode != EV_CONTROL) return;
	vol = VelOf(ev);
#ifndef NO_VOLSCALE
	vol = (vol * minfo->volscale) / 100;
	vol = (vol * channels[ev->channel].volscale) / 100;
	if (vol <= 0) vol = 0;
	else if (vol > 127) vol = 127;
#endif
	key = KeyOf(ev);
	if (!CHAN_ISDRUM(ev->channel)) {
		key += minfo->base_offset;
		if (key < 0 || key >= 127) return;
	}
	if (mode == EV_PLAY)
		seq_aftertouch(ev->channel, key, vol);
	else if (mode == EV_CONTROL)
		ctl->note(ev->channel, key, vol);
}

/* pitch sense change (RPN control) */
static void PitchSens(MidiEvent *ev, MidiInfo *minfo, int mode)
{
	channels[ev->channel].pitchsense = ev->p.val;
	if (mode == EV_PLAY)
		seq_pitchsense(ev->channel, ev->p.val);
	else if (mode == EV_CONTROL)
		ctl->pitch_sense(ev->channel, ev->p.val);
}	

/* pitch wheel change */
static void PitchWheel(MidiEvent *ev, MidiInfo *minfo, int mode)
{
	channels[ev->channel].pitchbend = ev->p.val;
	if (mode == EV_PLAY)
		seq_pitchbend(ev->channel, ev->p.val);
	else if (mode == EV_CONTROL)
		ctl->pitch_bend(ev->channel, ev->p.val);
}

/* channel pressure change */
static void ChanPressure(MidiEvent *ev, MidiInfo *minfo, int mode)
{
#if 0
	int vol;
	vol = ev->p.val;
#ifndef NO_VOLSCALE
	vol = (vol * minfo->volscale) / 100;
	vol = (vol * channels[ev->channel].volscale) / 100;
	if (vol <= 0) vol = 0;
	else if (vol > 127) vol = 127;
#endif
	channels[ev->channel].chan_press = vol;
	if (mode == EV_PLAY)
		seq_chan_pressure(ev->channel, vol);
#endif
}

/* program change */
static void Program(MidiEvent *ev, MidiInfo *minfo, int mode)
{
	ChannelStat *ch = &channels[ev->channel];
	int preset;
	if (minfo->midi_mode == MODE_MT32_INT)
		ch->preset = mt32pgm[ev->p.val];
	else
		ch->preset = ev->p.val;
	if (minfo->midi_mode == MODE_XG && minfo->xg_mapping &&
	    CHAN_ISDRUM(ev->channel))
		preset = ch->preset + 64;
	else
		preset = ch->preset;
	if (mode == EV_PRELOAD) {
		if (! CHAN_ISDRUM(ev->channel) && ch->bankmode != 126)
			add_preload(ev->channel, preset, ch->bank, -1);
		return;
	}
	else if (mode == EV_PLAY)
		seq_set_program(ev->channel, preset);
	else if (mode == EV_CONTROL)
		ctl->program(ev->channel, ch->preset);
}

/* midi control change */
static void ControlChange(MidiEvent *ev, MidiInfo *minfo, int mode)
{
	channels[ev->channel].controls[CtrlOf(ev)] = CValOf(ev);
	if (minfo->midi_mode == MODE_XG) {
		/* not implemented in driver */
		int i;
		if (! minfo->use_fx) return;
		for (i = 0; i < num_xg_effects; i++) {
			if (CtrlOf(ev) == xg_effects[i].control) {
				int cval = xg_effects[i].convert(minfo, CValOf(ev));
				if (mode == EV_PLAY)
					seq_add_effect(ev->channel, xg_effects[i].awe_effect, cval);
				return;
			}
		}
	}
	if (CtrlOf(ev) == CTL_SOFT_PEDAL) {
		/* not implemented in driver */
		if (! minfo->use_fx) return;
		seq_soft_pedal(ev->channel, CValOf(ev));
		return;
	}

	if (mode == EV_PLAY)
		CONTROL(ev->channel, CtrlOf(ev));
	else if (mode == EV_CONTROL)
		ctl->control_change(ev->channel, CtrlOf(ev), CValOf(ev));
}

/* reset controllers */
static void ResetControl(MidiEvent *ev, MidiInfo *minfo, int mode)
{
	seq_reset_control(ev->channel);
}

/* all notes off; send note-off to all channels */
static void AllNotesOff(MidiEvent *ev, MidiInfo *minfo, int mode)
{
	if (mode == EV_PLAY && minfo->accept_all_off)
		seq_note_off_all();
}

/* all sounds off; terminate all sounds immediately */
static void AllSoundsOff(MidiEvent *ev, MidiInfo *minfo, int mode)
{
	if (mode == EV_PLAY && minfo->accept_all_off)
		seq_sound_off_all();
}

/* tone bank change */
static void ToneBank(MidiEvent *ev, MidiInfo *minfo, int mode)
{
	ChannelStat *ch = &channels[ev->channel];
	if (minfo->midi_mode == MODE_XG && IsMSB(ev)) {
		ch->bankmode = BankOf(ev);
		/* XG MSB value; not normal bank selection */
		switch (BankOf(ev)) {
		case 127: /* remap to drum channel */
			if (! CHAN_ISDRUM(ev->channel)) {
				BITON(channel_drums, DRUMBIT(ev->channel));
				ch->bank = 128;
				if (mode == EV_PLAY)
					seq_set_drumchannels(channel_drums);
			}
			break;
		default: /* remap to normal channel */
			if (CHAN_ISDRUM(ev->channel)) {
				BITOFF(channel_drums, DRUMBIT(ev->channel));
				if (mode == EV_PLAY)
					seq_set_drumchannels(channel_drums);
			}
			ch->bank = BankOf(ev);
			if (mode == EV_PLAY && ch->bankmode != 0)
				seq_set_bank(ev->channel, ch->bank);
			break;
		}
		return;
	} else if (minfo->midi_mode == MODE_GS && !IsMSB(ev))
		return; /* ignore LSB bank in GS mode (used for mapping) */
	/* normal bank controls; accept both MSB and LSB */
	if (! CHAN_ISDRUM(ev->channel)) {
		if (ch->bankmode == 0) {
			if (minfo->midi_mode == MODE_XG &&
			    (BankOf(ev) == 64 || BankOf(ev) == 126))
				channels[ev->channel].bank = 0;
			else
				channels[ev->channel].bank = BankOf(ev);
			if (mode == EV_PLAY)
				seq_set_bank(ev->channel, BankOf(ev));
		}
	}
	if (mode == EV_CONTROL)
		ctl->bank(ev->channel, channels[ev->channel].bank);
}

/* print a text (SYSEX) */
static void PrintLyric(MidiEvent *ev, MidiInfo *minfo, int mode)
{
	if (mode != EV_CONTROL && mode != EV_STARTUP) return;
	if (ev->p.text)
		ctl->cmsg(CMSG_TEXT, 0, ev->p.text);
}

/* tempo change */
static void Tempo(MidiEvent *ev, MidiInfo *minfo, int mode)
{
	minfo->tempo = ev->p.val;
}

/* sysex master volume change */
static void MasterVolume(MidiEvent *ev, MidiInfo *minfo, int mode)
{
	minfo->master_volume = ev->p.val * 100 / 127;
	if (mode == EV_PLAY)
		seq_change_volume(minfo->volume_base * minfo->master_volume / 127);
}

/* fine tune (RPN control) */
static void FineTune(MidiEvent *ev, MidiInfo *minfo, int mode)
{
	if (!minfo->do_tuning) return;
	channels[ev->channel].finetune = ev->p.val - 8192;
	if (mode == EV_PLAY)
		seq_detune(ev->channel, channels[ev->channel].coarsetune,
			   channels[ev->channel].finetune);
}

/* coarse tune (RPN control) */
static void CoarseTune(MidiEvent *ev, MidiInfo *minfo, int mode)
{
	if (!minfo->do_tuning) return;
	channels[ev->channel].coarsetune = ev->p.val - 8192;
	if (mode == EV_PLAY)
		seq_detune(ev->channel, channels[ev->channel].coarsetune,
			   channels[ev->channel].finetune);
}

/* AWE32 NRPN events */
static void NRPNEvent(MidiEvent *ev, MidiInfo *minfo, int mode)
{
	int type, val, cval;

	if (! minfo->use_fx)
		return;
	type = ev->p.par[0];
	val = ev->p.par[1] - 8192;
	if (type < 0 || type >= num_awe_effects) {
		ctl->cmsg(CMSG_INFO, 2, "[nrpn unknown ch=%d type=%d val=%d]",
			  ev->channel, type, val);
		return;
	}

	cval = awe_effects[type].convert(val);
	ctl->cmsg(CMSG_INFO, 2, "[nrpn ch=%d %s: %d -> %x]",
		  ev->channel, awe_effects[type].name, val, (unsigned short)cval);
	if (mode == EV_PLAY)
		seq_send_effect(ev->channel, awe_effects[type].awe_effect, cval);
	else if (mode == EV_CONTROL)
		ctl->effects_change(ev->channel, type, val);
}

/* SC88 NRPN events */
static void GSNRPNEvent(MidiEvent *ev, MidiInfo *minfo, int mode)
{
	int i, type, val, cval;

	if (! minfo->use_fx)
		return;
	type = ev->p.par[0];
	val = ev->p.par[1];

	for (i = 0; i < num_gs_effects; i++) {
		if (gs_effects[i].control == ev->p.par[0]) {
			cval = gs_effects[i].convert(minfo, val);
			if (mode == EV_PLAY)
				seq_add_effect(ev->channel, gs_effects[i].awe_effect, cval);
			break;
		}
	}
}
