/*
 * main.c  -  Utility program to create a bootrom image
 *
 * Copyright (C) 1997,1998 Gero Kuhlmann <gero@gkminix.han.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#define NO_BINARY 1	/* No need to include binary support here */
#include "common.h"
#include "makerom.h"
#ifdef HAVE_GETOPT_H
#include <getopt.h>
#endif


/*
 * Set some defaults required for this module
 */
#ifndef DEFAULT_DIR
#define DEFAULT_DIR	"."
#endif
#ifndef DEFAULT_CONFIG
#define DEFAULT_CONFIG	"//makerom.config"
#endif
#define DATABASE	"//makerom.db"
#define KERNELDIR	"//binaries"
#define PKTDRVDIR	"//pktdrvr"
#define UTILSDIR	"//utils"

#define ENV_MAKEROM	"MAKEROM"
#define ENV_CONFIG	"MAKEROMCONFIG"



/*
 * Global variables
 */
char *progname;				/* program name */
char *netbootdir = NULL;		/* netboot directory */
char *dbname = NULL;			/* database file name */
char *kerneldir = NULL;			/* directory with kernel binaries */
char *loaderdir = NULL;			/* directory with loader binaries */
char *pktdrvdir = NULL;			/* directory with packet drivers */
char *utilsdir = NULL;			/* directory with utility binaries */
struct desclist *kerneldesc = NULL;	/* list of kernel descriptions */
struct desclist *pktdrvdesc = NULL;	/* list of packet driver descriptions */
unsigned short chksum = 0;		/* output file checksum */
int verbose = 0;			/* verbosity flag */



/*
 * Private variables
 */
static char *configname = NULL;		/* config file name */
static int accesserror = 0;		/* non-zero if unable to access a file */



/*
 * Read one block of data
 */
int doread(buf, bufsize, infile)
unsigned char *buf;
unsigned int bufsize;
int infile;
{
  int ofs;

  if ((ofs = read(infile, (char *)buf, bufsize)) < 0) {
	perror(progname);
	exit(EXIT_READ);
  }
  return(ofs);
}



/*
 * Write one block of data to the output file.
 */
int dowrite(buf, bufsize, outfile)
unsigned char *buf;
unsigned int bufsize;
int outfile;
{
  int j, ofs;

  if ((ofs = write(outfile, (char *)buf, bufsize)) < 0) {
	perror(progname);
	exit(EXIT_WRITE);
  }

  /* Compute the checksum */
  for (j = 0; j < ofs; j++)
	chksum += buf[j];
  return(ofs);
}



/*
 * Set a file or directory name with a default value, if it has no
 * name yet. If the name starts with two slashes, replace the first
 * slash with the netboot directory name.
 */
static void setname(fname, defaultname)
char **fname;
char  *defaultname;
{
  char *newname;
  size_t len;

  if (*fname == NULL) {
	if (defaultname == NULL)
		return;
	if ((*fname = strdup(defaultname)) == NULL) {
		perror(progname);
		exit(EXIT_MEMORY);
	}
  }

  if (!strncmp(*fname, "//", 2)) {
	len = strlen(*fname) + strlen(netbootdir) + 1;
	if ((newname = malloc(len)) == NULL) {
		perror(progname);
		exit(EXIT_MEMORY);
	}
	sprintf(newname, "%s%s", netbootdir, (*fname)+1);
	free(*fname);
	*fname = newname;
  }
}



/*
 * Make a local copy of a string
 */
static void copystr(dest, src)
char **dest;
char  *src;
{
  if (*dest != NULL)
	free(*dest);
  if ((*dest = strdup(src)) == NULL) {
	perror(progname);
	exit(EXIT_MEMORY);
  }
}



/*
 * Check that we have read access to the specified filed
 */
static void checkaccess(fname)
char *fname;
{
  if (access(fname, R_OK) < 0) {
	fprintf(stderr, "%s: unable to access file %s\n", progname, fname);
	accesserror++;
  }
}



/*
 * Print boot definition record for debugging purposes
 */
static void printbootdef(bp)
struct bootdef *bp;
{
  int i;

  if (bp->name != NULL)
	printf("Bootrom building definitions for %s\n", bp->name);
  else
	printf("Bootrom building definitions\n");
  printf("Kernel file        = %s (int %dh)\n", bp->kernelname,
			bp->use_int18 ? 18 : 19);

  for (i = 0; i < MAXLOADERS; i++)
	if (bp->loadernames[i] != NULL) {
		printf("Loader binary %d    = %s\n", i, bp->loadernames[i]);
		printf("Output file %d      = %s\n", i, bp->outnames[i]);
	}
  printf("\n");

  for (i = 0; i < MAXDRIVERS; i++)
	if (bp->drivernames[i] != NULL) {
		printf("Driver %d program   = %s\n", i, bp->drivernames[i]);
		printf("Driver %d arguments = %s\n", i, bp->driverargs[i]);
	}
  printf("\n");
}



/*
 * Print the usage
 */
static void usage()
{
  fprintf(stderr,
	"Usage: %s [-b <batch name>] [-c <config file>]\n"
	"\t\t\t\t[-n <netboot directory>] [-x]\n"
	"       %s -h\n"
	"       %s -v\n",
		progname, progname, progname);
  exit(EXIT_USAGE);
}



/*
 * Main program
 */
void main(argc, argv)
int argc;
char **argv;
{
  struct bootdef *bp;
  char *batchname = NULL;
  char *cmdldir = NULL;
  char *tmpfname;
  char *cp;
  int i, tmpfile;

  /* Determine my own name for error output */
  if ((cp = strrchr(argv[0], '/')) == NULL)
	progname = argv[0];
  else
	progname = ++cp;

  /* Set names for netboot directory and configuration file from environment */
  if ((cp = getenv(ENV_MAKEROM)) != NULL)
	copystr(&netbootdir, cp);
  if ((cp = getenv(ENV_CONFIG)) != NULL)
	copystr(&configname, cp);

  /* Parse options */
  opterr = 0;
  while ((i = getopt(argc, argv, "b:c:hn:vx")) != EOF)
	switch (i) {
		case 'b': copystr(&batchname, optarg);
		          break;
		case 'c': copystr(&configname, optarg);
		          break;
		case 'n': copystr(&netbootdir, optarg);
		          copystr(&cmdldir, optarg);
		          break;
		case 'v': fprintf(stderr, VERSION"\n");
		          exit(EXIT_SUCCESS);
		case 'x': verbose++;
		          break;
		case 'h': /* fall thru */
		default:  usage();
	}

  if (optind != argc)
	usage();

  if (verbose > 0)
	printf("Bootrom configuration program, " VERSION "\n");

  /* Read configuration file */
  if (netbootdir == NULL)
	copystr(&netbootdir, DEFAULT_DIR);
  setname(&configname, DEFAULT_CONFIG);
  readconfig(configname);
  free(configname);

  /* For the netboot dir the command line parameter overrides config file */
  if (cmdldir != NULL) {
	free(netbootdir);
	netbootdir = cmdldir;
  }

  /* Set file and directory names */
  setname(&dbname, DATABASE);
  setname(&kerneldir, KERNELDIR);
  setname(&loaderdir, KERNELDIR);
  setname(&pktdrvdir, PKTDRVDIR);
  setname(&utilsdir, UTILSDIR);
  if (verbose > 0) {
	printf("Netboot directory       = %s\n", netbootdir);
	printf("Database file           = %s\n", dbname);
	printf("Kernel directory        = %s\n", kerneldir);
	printf("Loader directory        = %s\n", loaderdir);
	printf("Packet driver directory = %s\n", pktdrvdir);
	printf("Utilities directory     = %s\n\n", utilsdir);
  }

  /* Get bootrom specification either from the user or the database */
  if (batchname != NULL) {
	if (dbname == NULL) {
		fprintf(stderr, "%s: no database specified for -b option\n", progname);
		exit(EXIT_MAKEROM_NODB);
	}
	bp = getdb(batchname);
	if (bp == NULL) {
		fprintf(stderr, "%s: <%s> not in database\n", progname, batchname);
		exit(EXIT_MAKEROM_NOSYS);
	}
  } else {
	printf("\nBootrom configuration program, " VERSION "\n\n");
	bp = getuser();
  }

  /* Expand filenames */
  setname(&(bp->kernelname), NULL);
  for (i = 0; i < MAXLOADERS; i++)
	setname(&(bp->loadernames[i]), NULL);
  for (i = 0; i < MAXDRIVERS; i++)
	setname(&(bp->drivernames[i]), NULL);
  if (verbose > 1)
	printbootdef(bp);

  /* Check that all input files are readable */
  checkaccess(bp->kernelname);
  for (i = 0; i < bp->loadernum; i++)
	checkaccess(bp->loadernames[i]);
  for (i = 0; i < bp->drivernum; i++)
	checkaccess(bp->drivernames[i]);
  if (accesserror > 0)
	exit(EXIT_ACCESS);

  /* Open temporary file and call the different passes successively */
  if ((tmpfname = tempnam("/usr/tmp", "mkrom")) == NULL) {
	fprintf(stderr, "%s: unable to generate temporary file name\n",
								progname);
	exit(EXIT_TEMPNAME);
  }
  if ((tmpfile = open(tmpfname, O_CREAT | O_RDWR | O_TRUNC, 0644)) < 0) {
	fprintf(stderr, "%s: unable to open temporary file %s\n",
							progname, tmpfname);
	exit(EXIT_CREATE);
  }
  if (verbose < 2)
	unlink(tmpfname);

  pass1(bp, tmpfile);
  for (i = 0; i < bp->loadernum; i++)
	pass2(bp, tmpfile, i);

  close(tmpfile);
  exit(EXIT_SUCCESS);
}

