/*
   Siag, Scheme In A Grid
   Copyright (C) 1996  Ulric Eriksson <ulric@edu.stockholm.se>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
   Module name:    selection.c

   This module handles selection: grabbing, releasing, cutting, pasting.

   The selection uses a custom target SIAG_BLOCK which has this format:

   <HEIGHT>\0<WIDTH>\0<TYPE><TEXT>\0<TYPE><TEXT>\0...

   HEIGHT = numbers of rows in selection
   WIDTH = number of columns in selection
   TYPE = type char where:
	'"' = label
	'#' = empty
	'=' = expression
	'$' = string
	'+' = expression with named interpreter
	'm' = image
   TEXT = verbatim text from the cell

   This is pretty similar to the file format but different enough to
   be confusing.
*/

#include <stdio.h>
#include <string.h>

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Xatom.h>
#include <X11/Xmu/Atoms.h>
#include <X11/Xmu/StdSel.h>

#include "../siag/types.h"
#include "../siag/calc.h"
#include "xsiag.h"

Atom target_atom;	/* used for selection */

int pack_selection(buffer *b, char *data, int r1, int c1, int r2, int c2)
{
	int row, col;
	char *q = data;

	sprintf(data, "%d", r2-r1+1);
	data += strlen(data)+1;
	sprintf(data, "%d", c2-c1+1);
	data += strlen(data)+1;

	for (row = r1; row <= r2; row++) {
		if (data-q > 30000) goto toolong;
		for (col = c1; col <= c2; col++) {
			switch (ret_type(b, row, col)) {
			case EMPTY:
				sprintf(data, "#");
				break;
			case LABEL:
				sprintf(data, "\"%s",
					ret_text(b, row, col));
				break;
			case EMBED:
				sprintf(data, "m%s",
					ret_text(b, row, col));
				break;
			default:	/* incl. STRING and ERROR */
				sprintf(data, "+%s,%s",
					interpreter2name(ret_interpreter(b,
								row, col)),
					ret_text(b, row, col));
			}
			data += strlen(data)+1;
		}
	}
toolong:
	*data = '\0';
	return 0;
}

int pack_string_selection(buffer *b, char *data,
				int r1, int c1, int r2, int c2)
{
	int r, c;
	char *p, *q;

	q = data;
	for (r = r1; r <= r2; r++) {
		for (c = c1; c <= c2; c++) {
			if (data-q > 30000) goto toolong;
			p = ret_pvalue(NULL, b, r, c, -1);
			strcpy(data, p);
			data += strlen(data);
			if (c < c2) *data++ = ',';
		}
		*data++ = '\n';
	}
toolong:
	*data = '\0';
	return 0;
}

Boolean convert_proc(Widget w,
	Atom *selection, Atom *target, Atom *type_return,
	XtPointer *value_return, unsigned long *length_return,
	int *format_return)
{
	window *wl = find_window_by_widget(w);
	XSelectionRequestEvent *req = XtGetSelectionRequest(w,
		*selection, (XtRequestId) NULL);

	/* handle all required atoms, and the one that we use */
	/* Xt already handles MULTIPLE, no branch necessary */
	if (*target == XA_TARGETS(XtDisplay(w))) {
		Atom *targetP;
		Atom *std_targets;
		unsigned long std_length;
		XmuConvertStandardSelection(w, req->time, selection,
			target, type_return,
			(XPointer *)&std_targets,
			&std_length, format_return);
		*value_return = XtMalloc(sizeof(Atom)*(std_length+1));
		targetP = *(Atom **)value_return;
		*length_return = std_length+1;
		*targetP++ = target_atom;
		bcopy((char *)std_targets, (char *)targetP,
			sizeof(Atom)*std_length);
		XtFree((char *)std_targets);
		*type_return = XA_ATOM;
		*format_return = sizeof(Atom)*8;
		return True;
	} else if (*target == target_atom) {
		/* handle normal selection */
		char *data;

		/* Use a fixed size buffer for now */
		*length_return = 32000;

		data = XtMalloc(*length_return);
		pack_selection(wl->buf, data,
				wl->blku.row, wl->blku.col,
				wl->blkl.row, wl->blkl.col);

		*value_return = data;

		*type_return = target_atom;

		*format_return = 8;
		return True;
	} else if (*target == XA_STRING) {
		/* handle string selections (from outside Siag) */
		char *data;
		*length_return = 32000;
		data = XtMalloc(*length_return);
		pack_string_selection(wl->buf, data,
				wl->blku.row, wl->blku.col,
				wl->blkl.row, wl->blkl.col);
		*value_return = data;
		*length_return = strlen(data);
		*type_return = XA_STRING;
		*format_return = 8;
		return True;
	} else {
		if (XmuConvertStandardSelection(w, CurrentTime, selection,
				target, type_return, (XPointer *)value_return,
				length_return, format_return))
			return True;
		else {
			XtWarning("Siag: unsupported selection type\n");
			return False;
		}
	}
	/*NOTREACHED*/
}

void lose_ownership_proc(Widget w, Atom *selection)
{
	window *wl = find_window_by_widget(w);

	wl->blku.row = wl->blku.col = -1;
	wl->blkl.row = wl->blkl.col = -1;
	pr_scr_flag = TRUE;	/* make sure it's redrawn */
	show_cur(wl);
}

int unpack_selection(buffer *b, char *data, int row, int col)
{
	int height, width, i, j, intp;
	char *texti, *comma;
	cval value;
	value.number = 0;

	sscanf(data, "%d", &height);
	data += strlen(data)+1;
	sscanf(data, "%d", &width);
	data += strlen(data)+1;

	undo_save(b, row, col, row+height-1, col+width-1);

	for (i = 0; i < height; i++) {
		for (j = 0; j < width; j++) {
			switch (data[0]) {
			case '"':
				texti = (data+1);
				ins_data(b,
					siod_interpreter,
					texti, value,
					LABEL, row+i, col+j);
				break;
			case '=':
				texti = (data+1);
				ins_data(b,
					siod_interpreter,
					texti, value,
					EXPRESSION, row+i, col+j);
				break;
			case '$':
				texti = (data+1);
				ins_data(b,
					siod_interpreter,
					texti, value,
					STRING, row+i, col+j);
				break;
			case '+':
				comma = strchr(data, ',');
				if (comma) {
					*comma = '\0';
				}
				intp = name2interpreter(data+1);
				if (intp < 0) intp = siod_interpreter;
				data = comma+1;
				ins_data(b, intp, data, value,
					EXPRESSION, row+i, col+j);
				break;
			default:
				ins_data(b,
					siod_interpreter,
					NULL, value,
					EMPTY, row+i, col+j);
			}
			data += strlen(data)+1;
		}
	}
	pr_scr_flag = TRUE;
	show_cur(w_list);
	return 0;
}

int unpack_string_selection(buffer *b, char *data, int row, int col)
{
	char *p;
	int n = 0;
	cval value;
	value.number = 0;

	for (p = data; *p; p++)
		if (*p == '\n') n++;

	undo_save(b, row, col, row+n, col);

	while ((p = strchr(data, '\n')) != NULL) {
		*p = '\0';
		ins_data(b, siod_interpreter, data,
			value, LABEL, row, col);
		row++;
		data = p+1;
	}
	if (*data) {
		ins_data(b, siod_interpreter, data,
			value, LABEL, row, col);
	}
	pr_scr_flag = TRUE;
	show_cur(w_list);
	return 0;
}

void string_requestor_callback(Widget w, XtPointer client_data,
	Atom *selection, Atom *type, XtPointer value,
	unsigned long *length, int *format)
{
	if ((value == NULL) && (*length == 0)) {
		XBell(XtDisplay(w), 100);
		XtWarning("Siag: no selection or selection timed out\n");
	} else {
		unpack_string_selection(w_list->buf, (char *)value,
			get_point(w_list).row, get_point(w_list).col);
		w_list->buf->change = TRUE;
		calc_matrix(w_list->buf);
		XtFree((char *)value);
		pr_scr_flag = TRUE;
	}
}

void requestor_callback(Widget w, XtPointer client_data,
	Atom *selection, Atom *type, XtPointer value,
	unsigned long *length, int *format)
{
	if ((value == NULL) && (*length == 0)) {
	/* if we asked for a SIAG_BLOCK and got a null response,
	   we'll ask again, this time for an XA_STRING */
		XtGetSelectionValue(w, XA_PRIMARY, XA_STRING,
			string_requestor_callback,
			NULL, CurrentTime);	/* NULL is bogus event */
	} else {
		unpack_selection(w_list->buf, (char *)value,
			get_point(w_list).row, get_point(w_list).col);
		w_list->buf->change = TRUE;
		calc_matrix(w_list->buf);
		XtFree((char *)value);
		pr_scr_flag = TRUE;
	}
}

