// This file is part of Moonlight Creator
//   Copyright (C) 1996-1998  Stephane Rehel
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
//
// You should have received a copy of the GNU Library General Public
// License along with this library; if not, write to the Free
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/*
   Array2.cc

   Creation: SR, September 5th, 1995
   Revisions:

*/

#ifndef __Array2_h
#define __Array2_h

#include <assert.h>
#include "IBOOL.h"

/////////////////////////////////////////////////////////////////////////////

template<class T>
class Array2
{
protected:
  int nX, nY;
  T** array;

public:
  Array2();
  virtual ~Array2()
    {
    destroy();
    }

  void destroy();
  void create( int _nX, int _nY );
  IBOOL allocated() const
    {
    return array != 0;
    }

  int getI() const { return nX; }
  int getJ() const { return nY; }

  // 0 <= i < nX
  // 0 <= j < nY
  T get( int i, int j ) const;

  // 0 <= j < nY
  const T* line( int j ) const;

  // 0 <= i < nX
  // 0 <= j < nY
  T& operator () ( int i, int j );

  IBOOL inArray( int i, int j ) const
    {
    return i >= 0 && i < nX &&
           j >= 0 && j < nY ;
    }

  void fillRegion( int ni1, int ni2,
                   int nj1, int nj2,
                   const T t );

  void fill( const T t )
    {
    assert( array != 0 );
    fillRegion( 0, nX-1, 0, nY-1, t );
    }
};

/////////////////////////////////////////////////////////////////////////////

template<class T>
inline
Array2<T>::Array2()
{
  array= 0;
  nX= nY= 0;
}

/////////////////////////////////////////////////////////////////////////////

template<class T>
inline
void Array2<T>::destroy()
{
  if( array == 0 )
    return;

  for( int j= 0; j < nY; ++j )
    {
    assert( array[j] != 0 );
    delete [] array[j];
    }

  delete array;
  array= 0;
}

/////////////////////////////////////////////////////////////////////////////

template<class T>
inline
void Array2<T>::create( int _nX, int _nY )
{
  assert( array == 0 );
  assert( _nX >= 1 );
  assert( _nY >= 1 );

  nX= _nX;
  nY= _nY;

  array= new T* [ nY ];
  for( int j= 0; j < nY; ++j )
    array[j]= new T [ nX ];
}

/////////////////////////////////////////////////////////////////////////////

// 0 <= i < nX
// 0 <= j < nY
template<class T>
inline
T Array2<T>::get( int i, int j ) const
{
  assert( i >= 0 );
  assert( i < nX );
  assert( j >= 0 );
  assert( j < nY );
  assert( array != 0 );

  return array[j][i];
}

/////////////////////////////////////////////////////////////////////////////

// 0 <= j < nY
template<class T>
inline
const T* Array2<T>::line( int j ) const
{
  assert( j >= 0 );
  assert( j < nY );
  assert( array != 0 );

  return array[j];
}

/////////////////////////////////////////////////////////////////////////////

// 0 <= i < nX
// 0 <= j < nY
template<class T>
inline
T& Array2<T>::operator () ( int i, int j )
{
  assert( i >= 0 );
  assert( i < nX );
  assert( j >= 0 );
  assert( j < nY );
  assert( array != 0 );

  return array[j][i];
}

/////////////////////////////////////////////////////////////////////////////

template<class T>
inline
void Array2<T>::fillRegion( int ni1, int ni2,
                            int nj1, int nj2,
                            const T t )
{
  assert( array != 0 );

  assert( ni1 >= 0 );
  assert( ni1 < nX );
  assert( ni2 >= 0 );
  assert( ni2 < nX );
  assert( ni1 <= ni2 );

  assert( nj1 >= 0 );
  assert( nj1 < nY );
  assert( nj2 >= 0 );
  assert( nj2 < nY );
  assert( nj1 <= nj2 );

  T** array_j= array+nj1;
  for( int j= nj1; j <= nj2; ++j, ++array_j )
    {
    T* array_ji= (*array_j)+ni1;
    for( int i= ni1; i <= ni2; ++i, ++array_ji )
      *array_ji= t;
    }
}


#endif // ifndef __Array2_h

