// This file is part of Moonlight Creator
//   Copyright (C) 1996-1998  Stephane Rehel
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
//
// You should have received a copy of the GNU Library General Public
// License along with this library; if not, write to the Free
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/*
  ml.C

  Stephane Rehel

  June 29 1996
*/

#include <stdio.h>

#include "tools/Command.h"

#include "graphics/InitGraphicsSystem.h"
#include "graphics/MLEventManager.h"

#include "interface/Interface.h"
#include "interface/MLInterfaceVersion.h"

#ifdef MODULE_SCENE
#include "scene/convert/import/MLCmdLineConvert.h"
#include "model/ModelModule.h"
#endif

#ifndef __WINDOWS__
#ifdef MODULE_KBUILDER
#include "kbuilder/KBModule.h"
#endif
#endif

#ifdef MODULE_PHOTO
#include "photo/PhotoModule.h"
#endif

#ifdef MODULE_OOM
#include "oom/OOMModule.h"
#endif

#ifdef MODULE_TMPL
#include "tmpl/TmplModule.h"
#endif

/////////////////////////////////////////////////////////////////////////////

void usage();

int main( int argc, char* argv[] )
{

extern void toto();
//toto();

  Command command(argc,(const char**)argv);

  if( command.findOption("-h") ||
      command.findOption("-help") ||
      command.findOption("--h") ||
      command.findOption("--help") ||
      command.findOption("-version") ||
      command.findOption("--version") )
    usage();

#ifdef MODULE_SCENE
  if( command.findOption("-convert") )
    {
    MLCmdLineConvert* cvt= new MLCmdLineConvert;
    int outcome= cvt->run(command);
    if( outcome == -1 )
      usage();
    return outcome;
    }
#endif

  if( ! InitGraphicsSystem(command) )
    {
    fprintf( stderr, "Error: unable to open graphics system\n" );
    return 1;
    }

  interface= new Interface;

  interface->init(command);

  MLModule* module= 0;
  MLModule* defaultModule= 0;

#define new_module(name) \
             module= new name##Module(interface); \
             module->init(command); \
             interface->newModule(module->getID());

#ifdef MODULE_TMPL
  new_module(Tmpl)
#endif

#ifdef MODULE_OOM
  new_module(OOM)
#endif

#ifndef __WINDOWS__
#ifdef MODULE_KBUILDER
  new_module(KB)
#endif
#endif

#ifdef MODULE_PHOTO
  new_module(Photo)
#endif

#ifdef MODULE_SCENE
  new_module(Model)
  defaultModule= module;
#endif

#undef new_module

  if( module == 0 )
    exit(1); // no module!?

  if( defaultModule == 0 )
    defaultModule= module;

  interface->activateModule(defaultModule->getID());

  MLEventManager* eventManager= MLEventManager::getEventManager();

  eventManager->removeAnyRedraw();
  interface->refreshInterface();

  interface->mainloop();

  return 0;
}

/////////////////////////////////////////////////////////////////////////////

void usage()
{
  char* program= "moonlight";
#define fp fprintf
#define f stdout
  fp( f, "Moonlight Creator %s %s-%s\n",
         MLInterfaceVersion::version(),
         MLInterfaceVersion::build_sysname(),
         MLInterfaceVersion::build_machine() );
  fp( f, "Copyright (C) 1997-1998 By Stephane C.F. Rehel\n" );
  fp( f, "Email: rehel@worldnet.fr\n" );
  fp( f, "  WWW: http://home.worldnet.fr/~rehel/ml/ml.html\n" );
  fp( f, "Build #%d, %s\n", MLInterfaceVersion::build_number(),
                            MLInterfaceVersion::build_time() );
#ifndef NDEBUG
  fp( f, "*** DEBUG VERSION ***\n" );
#endif
  fp( f, "\n" );

  fp( f, "General options:\n" );
  fp( f, "  -h                This message\n" );
  fp( f, "  -convert          Run the converter (command line)\n" );
  fp( f, "  -mlrc <file>      Specify ML resource file (default is ~/.mlrc)\n" );
  fp( f, "\n" );

  fp( f, "Interface options:\n" );
  fp( f, "  -gamma <gamma>    Specify gamma correction (default is 1)\n" );
#ifndef __WINDOWS__
  fp( f, "  -display <server> Run on this X server (default is :0)\n" );
#endif
  fp( f, "  -fs               Run full-screen (default)\n" );
  fp( f, "  -debug            Don't run full-screen\n" );
  fp( f, "\n" );

  fp( f, "Converter: %s -convert [options] <infile>\n",program);
  fp( f, "  Options:\n");
  fp( f, "  -outname <out_scene_name> Specify output scene name\n" );
  fp( f, "  -db <database_path>       Specify a custom database (optional)\n" );
  fp( f, "  -v                        Verbose mode\n" );
  fp( f, "  -rv                       Reverse polygons\n" );
  fp( f, "  -dxfl2o                   Convert DXF layers to objects\n" );
  fp( f, "  -nw                       Don't write. For testing.\n" );
  fp( f, "\n");

  exit(1);

#undef fp
#undef f
}

/////////////////////////////////////////////////////////////////////////////
