;;; deb-view.el: View Debian package archive files with tar-mode.

;; Author:  Rick Macdonald (rickm@vsl.com)
;; Version: 1.4
;; Latest version: http://www.cuug.ab.ca/~macdonal/deb-view.el

;; This file is not part of GNU Emacs.

;; GNU Emacs is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; GNU Emacs is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs; see the file COPYING.  If not, write to the
;; Free Software Foundation, Inc., 59 Temple Place - Suite 330,
;; Boston, MA 02111-1307, USA.

;;; Commentary:

;; deb-view presents the contents of debian package archive files for
;; viewing. The viewing is done with emacs tar-mode, with a few
;; enhancements for viewing compressed files and HTML files, and
;; formatting man pages.
;; The normal editing and saving features of tar-mode are not supported
;; by deb-view. Don't even try (I didn't).

;; deb-view extracts the control.tar.gz and data.tar.gz files from debian
;; package and presents two buffers in tar-mode. See tar-mode for info.

;; Required programs: ar, gzip.
;; Optionally required programs: nroff for formatting man pages.
;; Optionally required programs: dpkg-deb for old-style binary .deb files.
;; Optionally required programs: w3-mode for viewing HTML pages.

;; For new-style .deb files (2.0), dpkg-deb isn't used. Therefore
;; deb-view should work on any platform with the ar command, although
;; "ar -p" doesn't seem to work for .deb files on Solaris 2.4 and 2.5.
;; It works on SGI's IRIX 6.1 and 6.2, and Linux, of course.

;; Old-style .deb files require the dpkg-deb program. I don't know how to
;; extract control.tar.gz from these deb files, so you only get to see
;; the package control file, but nothing else such as the install scripts.
;; If you know how to get the control.tar.gz file out, let me know!
;; The data file is still viewable thanks to the "dpkg-deb --fsys-tarfile"
;; option.

;;; Installation:

;; Put this file somewhere where Emacs can find it (i.e., in one of the paths
;; in your `load-path'), `byte-compile-file' it, and put in your ~/.emacs:
;; (load "deb-view")
;;
;; If you're not very familiar with emacs customization, try this simpler
;; approach:
;; Put this file in your home directory, and call it deb-view.el.
;; Start up emacs and do the following:
;;      ESCAPE x load-file RETURN ~/deb-view.el RETURN
;; If you like it, and want it available always, then add this line to your
;; ~/.emacs file (or create ~/.emacs if you don't have one):
;; (load "~/deb-view.el")

;; deb-view tries to be unobtrusive, but does the following:
;; binds ^d in dired to deb-view-dired-view.
;; binds q, N, W, and re-binds v in tar-mode. These functions are then
;;       available in tar-mode on real tar files.


;;; Usage:

;; In dired, press ^d on the dired line of the .deb file to view.
;; Or, execute: ESC x deb-view RET, and enter the .deb file name
;; at the prompt.

;; You are shown two tar files in tar-mode (see tar-mode for help).
;; In the case of old .deb format files, the control info is shown
;; but not the other files of control.tar, such as install scripts.
;; Additional features that deb-view adds to tar-mode:
;; q - kill both view buffers (INFO and DATA) and return to the
;;     dired buffer if that's where you executed deb-mode.
;; v - executes deb-view-tar-view instead of tar-view, with the
;;     additional smarts to uncompress .gz and .Z files for viewing.
;; N - Like in dired, formats man pages for viewing, with the
;;     additional smarts to uncompress .gz and .Z man files for viewing.
;; W - use w3-mode to view an HTML file.
;; These functions are also available in tar-mode on real tar files
;; when deb-view is loaded.

;; To view files not supported by deb-view, such as graphics, use the
;; copy command in tar-mode ("c") to copy the file to a temp directory.
;; You can then do what you want to the file.


;;; Changelog:

;; 1.3  - modified logic that determines old or new style Debian packages.
;;        On systems where the file command recognizes debian files, it
;;        wrongly always came up with old format.

;; 1.4  - added missing semicolons in the comments for Changelog 1.3.
;;      - fixed various spacing issues in doc strings.
;;      - disabled tar-mode keys that are not applicable to deb-view.


;;; Code:

(defvar deb-view-dired-view-return-buffer ""
  "Return to this buffer after deb-view-dired-view-cleanup is called.")
(make-variable-buffer-local 'deb-view-dired-view-return-buffer)

;; User variables:

(defvar deb-view-tar-uncompress-program "gzip -cd"
  "*Program to use for uncompression of .gz and .Z files in deb-view.")

;; Note the following useful variable from tar-mode:
;; (defvar tar-mode-show-date nil
;;  "*Non-nil means Tar mode should show the date/time of each subfile.
;; This information is useful, but it takes screen space away from file names.")

;;; User functions:

;; You might not like the key bindings that I chose:
(add-hook
 'tar-mode-hook
 (function (lambda ()
             (define-key tar-mode-map "\C-d"   'undefined)
             (define-key tar-mode-map "G"      'undefined)
             (define-key tar-mode-map "M"      'undefined)
             (define-key tar-mode-map "O"      'undefined)
             (define-key tar-mode-map "d"      'undefined)
             (define-key tar-mode-map "g"      'undefined)
             (define-key tar-mode-map "r"      'undefined)
             (define-key tar-mode-map "u"      'undefined)
             (define-key tar-mode-map "x"      'undefined)
             (define-key tar-mode-map ""     'undefined)
             
             (define-key tar-mode-map "q"    'deb-view-dired-view-cleanup)
             (define-key tar-mode-map "N"    'deb-view-tar-man)
             (define-key tar-mode-map "W"    'deb-view-tar-w3)
             (define-key tar-mode-map "v"    'deb-view-tar-view)
             (define-key tar-mode-map [up]   'tar-previous-line)
             (define-key tar-mode-map [down] 'tar-next-line)
             (define-key tar-mode-map "\eOA" 'tar-previous-line)
             (define-key tar-mode-map "\eOB" 'tar-next-line)
             (define-key tar-mode-map "\e[A" 'tar-previous-line)
             (define-key tar-mode-map "\e[B" 'tar-next-line))))

(defun deb-view-dired-view ()
  "View the control and data files of the Debian package archive file on this line.
Press \"q\" in either window to kill both buffers and return to the dired buffer. See deb-view."
  (interactive)
  (deb-view (dired-get-filename)))

(defun deb-view (debfile)
  "View the control and data files of a Debian package archive file. Press \"q\"
in either window to kill both buffers.

In dired, press ^d on the dired line of the .deb file to view.
Or, execute: ESC x deb-view RET, and enter the .deb file name
at the prompt.

You are shown two tar files in tar-mode (see tar-mode for help).
In the case of old .deb format files, the control info is shown
but not the other files of control.tar, such as install scripts.
Additional features that deb-view adds to tar-mode:
q - kill both view buffers (INFO and DATA) and return to the
    dired buffer if that's where you executed deb-mode.
v - executes deb-view-tar-view instead of tar-view, with the
    additional smarts to uncompress .gz and .Z files for viewing.
N - Like in dired, formats man pages for viewing, with the
    additional smarts to uncompress .gz and .Z man files for viewing.
W - use w3-mode to view an HTML file.
These functions are also available in tar-mode on real tar files
when deb-view is loaded.

To view files not supported by deb-view, such as graphics, use the
copy command in tar-mode (\"c\") to copy the file to a temp directory.
You can then do what you want to the file. "
  (interactive "fdeb file to view:")
  (require 'view)
  (let* ((info-buffer-name (concat (file-name-nondirectory debfile) "-INFO"))
         (data-buffer-name (concat (file-name-nondirectory debfile) "-DATA"))
         (info-buffer (progn (and (get-buffer info-buffer-name)
                                  (kill-buffer (get-buffer info-buffer-name)))
                             (get-buffer-create info-buffer-name)))
         (data-buffer (progn (and (get-buffer data-buffer-name)
                                  (kill-buffer (get-buffer data-buffer-name)))
                             (get-buffer-create data-buffer-name)))
         (return-buffer (current-buffer))
	 file-buffer
         new-archive-format)
    ;; info
    (setq file-buffer (get-buffer-create " *file-data*"))
    (setq new-archive-format
          (save-excursion
            (set-buffer file-buffer)
            (erase-buffer)
            (call-process shell-file-name nil t nil shell-command-switch
                          (concat "file " debfile))
            (goto-char 1)
            (if (string-match "archive" (buffer-string))
                t
              (goto-char 1)
              (if (string-match "old debian" (buffer-string))
                  nil
                t))))
    (kill-buffer file-buffer)
    (set-buffer info-buffer)
    (if new-archive-format
        ;; New deb format (archive)
        (progn
          (call-process shell-file-name nil t nil shell-command-switch
                        (concat "ar -p " debfile
                                " control.tar.gz | gzip -cd"))
          (goto-char 1)
          (setq buffer-file-name (concat debfile "-INFO"))
          (tar-mode)
          (tar-next-line 1)
          (switch-to-buffer info-buffer t))
      ;; Old deb format
      (message "old dpkg binary format")
      (call-process shell-file-name nil t nil shell-command-switch
                    (concat "dpkg-deb -I " debfile))
      (setq buffer-read-only t)
      (set-buffer-modified-p nil)
      (goto-char 1)
      (switch-to-buffer info-buffer t)
      (view-mode-enter return-buffer 'deb-view-dired-view-cleanup))
    (set-buffer-modified-p nil)
    (setq buffer-read-only t)
    (setq deb-view-dired-view-return-buffer return-buffer)
    (delete-other-windows)
    ;; data
    (set-buffer data-buffer)
    (call-process shell-file-name nil t nil shell-command-switch
                  (if new-archive-format
                      (concat "ar -p " debfile " data.tar.gz | gzip -cd")
                    (concat "dpkg-deb --fsys-tarfile " debfile)))
    (goto-char 1)
    (setq buffer-file-name (concat debfile "-DATA"))
    (tar-mode)
    (tar-next-line 1)
    (setq deb-view-dired-view-return-buffer return-buffer)
    (set-buffer-modified-p nil)
    (setq buffer-read-only t)
    (switch-to-buffer-other-window data-buffer)
    (if new-archive-format (other-window 1))))


;;; Internal functions:

(defvar deb-view-version "1.4"
  "The version of deb-view.")

(defun deb-view-version ()
  "Return string describing the version of deb-view.
When called interactively, displays the version."
  (interactive)
  (if (interactive-p)
      (message "deb-view version %s" (deb-view-version))
    deb-view-version))

(defun deb-view-dired-view-cleanup (&optional buffer)
  "Delete the buffers created by deb-view-dired-view."
  (interactive)
  (let* ((quit-buffer (or buffer (current-buffer)))
         (bufname (buffer-name quit-buffer))
         (debfile (substring bufname 0 (- (length bufname) 5)))
         (info-buffer (get-buffer (concat debfile "-INFO")))
         (data-buffer (get-buffer (concat debfile "-DATA")))
         (ddir-buffer (save-excursion
                        (set-buffer quit-buffer)
                        deb-view-dired-view-return-buffer)))
    (delete-other-windows)
    (and (buffer-live-p info-buffer)
         (kill-buffer info-buffer))
    (and (buffer-live-p data-buffer)
         (kill-buffer data-buffer))
    (and (buffer-live-p quit-buffer)
         (kill-buffer quit-buffer))
    (and (buffer-live-p ddir-buffer)
         (switch-to-buffer ddir-buffer))))

(defun deb-view-tar-man ()
  "*In Tar mode, view the tar file entry on this line as a man page."
  (interactive)
  (require 'man)
  (let ((auto-mode-alist
         (append '(("\\.gz$" . deb-view-tar-uncompress-while-visiting)
                   ("\\.Z$"  . deb-view-tar-uncompress-while-visiting)
                   ) auto-mode-alist)))
    (tar-extract 'view)
    (setq buffer-read-only nil)
    (shell-command-on-region (point-min) (point-max) "nroff -man -h " t t)
    (Man-cleanup-manpage)
    (setq buffer-read-only t)
    (set-buffer-modified-p nil)
    (message "")))

(defun deb-view-tar-uncompress-while-visiting ()
  "Temporary \"major mode\" used for .Z and .gz files, to uncompress them.
It then selects a major mode from the uncompressed file name and contents.
\(Modifed uncompress-while-visiting from uncompress.el\)"
  (interactive)
  (message "Uncompressing...")
  (let ((buffer-read-only nil))
    (shell-command-on-region (point-min) (point-max)
                             deb-view-tar-uncompress-program t))
  (message "Uncompressing...done")
  (set-buffer-modified-p nil)
  (goto-char 1))

(defun deb-view-tar-view ()
  "*In Tar mode, view the tar file entry on this line."
  (interactive)
  (let ((auto-mode-alist
         (append '(("\\.gz$" . deb-view-tar-uncompress-while-visiting)
                   ("\\.Z$"  . deb-view-tar-uncompress-while-visiting)
                   ) auto-mode-alist)))
    (tar-extract 'view)))

(defun deb-view-tar-w3 ()
  "*In Tar mode, view the tar file entry on this line as HTML with w3-mode."
  (interactive)
  (if (fboundp 'w3-preview-this-buffer)
      (let ((auto-mode-alist
             (append '(("\\.gz$" . deb-view-tar-uncompress-while-visiting)
                       ("\\.Z$"  . deb-view-tar-uncompress-while-visiting)
                       ) auto-mode-alist)))
        (tar-extract 'view)
        (rename-buffer (concat " " (buffer-name)))
        (w3-preview-this-buffer)
        (define-key w3-mode-map "q"	'deb-view-tar-w3-quit))
    (error "Sorry, you don't seem to have w3 loaded.")))

(defun deb-view-tar-w3-quit ()
  "Quit WWW mode in a buffer from deb-view."
  (interactive)
  (let ((x w3-current-last-buffer))
    (and (fboundp 'w3-mpeg-kill-processes) (w3-mpeg-kill-processes))
    (kill-buffer (current-buffer))
    (if (and (bufferp x) (buffer-name x))
	(if w3-mutable-windows (pop-to-buffer x) (switch-to-buffer x))))
  (view-exit))

;;; deb-view.el ends here
