-- This file is  free  software, which  comes  along  with  SmallEiffel. This
-- software  is  distributed  in the hope that it will be useful, but WITHOUT 
-- ANY  WARRANTY;  without  even  the  implied warranty of MERCHANTABILITY or
-- FITNESS  FOR A PARTICULAR PURPOSE. You can modify it as you want, provided
-- this header is kept unaltered, and a notification of the changes is added.
-- You  are  allowed  to  redistribute  it and sell it, alone or as a part of 
-- another product.
--          Copyright (C) 1994-98 LORIA - UHP - CRIN - INRIA - FRANCE
--            Dominique COLNET and Suzanne COLLIN - colnet@loria.fr 
--                       http://www.loria.fr/SmallEiffel
--
deferred class COLLECTION2[E]
   -- 
   -- Abstract definition of a 2 dimensional collection of elements
   -- of type E. Each element can be accessed using a couple <i1,i2>
   -- of INTEGER.
   -- 
   -- The SmallEiffel standard library (lib_std) provides three
   -- implementations : ARRAY2[E], FIXED_ARRAY2[E] and LINK_ARRAY2[E].
   -- All implementations have exactly the same behavior. Switching 
   -- from one implementation to another only change the memory used
   -- and the execution time.
   --

inherit
   ANY
      undefine copy 
      redefine is_equal, fill_tagged_out_memory
      end;

feature -- Indexing :

   lower1, lower2: INTEGER is
	 -- Lower index bounds.
      deferred
      end;

   upper1, upper2: INTEGER is
	 -- Upper index bounds.
      deferred
      end;

feature -- Reading :

   item(i1, i2: INTEGER): E is
      require
	 valid_index(i1,i2);
      deferred
      end;

feature -- Writing :

   put(x: like item; i1, i2: INTEGER) is
      require
	 valid_index(i1,i2);
      deferred
      ensure
	 item(i1,i2) = x
      end;
   
feature -- Index validity :

   valid_index1(i1: INTEGER): BOOLEAN is
      do
	 Result := lower1 <= i1 and then i1 <= upper1;
      ensure
	 Result implies lower1 <= i1 and i1 <= upper1
      end;
      
   valid_index2(i2: INTEGER): BOOLEAN is
      do
	 Result := lower2 <= i2 and then i2 <= upper2;
      ensure
	 Result implies lower2 <= i2 and i2 <= upper2
      end;
      
   valid_index(i1, i2: INTEGER): BOOLEAN is
      do
	 Result := lower1 <= i1
	             and then
		   i1 <= upper1 
                     and then
		   lower2 <= i2 
                     and then
		   i2 <= upper2; 
      ensure
	 Result implies valid_index1(i1) and valid_index2(i2)
      end;

feature -- Counting :

   count1: INTEGER is
	 -- Size of the first dimension.
      deferred
      ensure
	 Result = upper1 - lower1 + 1;
      end;

   count2: INTEGER is
	 -- Size of the second dimension.
      deferred
      ensure
	 Result = upper2 - lower2 + 1;
      end;

   count: INTEGER is
      -- Total number of elements.
      do
	 Result := count1 * count2;
      end;
   
feature 

   swap(i1, j1, i2, j2: INTEGER) is
      require
	 valid_index(i1,j1);
	 valid_index(i2,j2)
      local
	 tmp: like item;
      do
	 tmp := item(i1,j1);
	 put(item(i2,j2),i1,j1);
	 put(tmp,i2,j2);
      end;

   set_all_with(v: like item) is
	 -- Set all item with value `v'.
      deferred
      ensure
	 count = old count
      end;
   
    clear_all is
	 -- Set all items to default values.
      local
	 value: like item;
      do
	 set_all_with(value);
      ensure
	 count = old count;
      end;
   
feature -- Looking and comparison :

   is_equal(other: like Current): BOOLEAN is
      local
	 i1, i2: INTEGER;
      do
	 if Current = other then
	    Result := true;
	 elseif lower1 /= other.lower1 then
	 elseif upper1 /= other.upper1 then
	 elseif lower2 /= other.lower2 then
	 elseif upper2 /= other.upper2 then
	 else
	    from
	       Result := true;
	       i1 := upper1;
	    until
	       not Result or else i1 < lower1
	    loop
	       from
		  i2 := upper2;
	       until
		  not Result or else i2 < lower2
	       loop
		  Result := equal_like(item(i1,i2),other.item(i1,i2));
		  i2 := i2 - 1;
	       end;
	       i1 := i1 - 1;
	    end;
	 end;
      end;
   
   all_cleared: BOOLEAN is
	 -- Are all items set to default values?
      local
	 value: like item;
	 i, j: INTEGER;
      do
	 from  
	    Result := true;
	    i := lower1;
	 until
	    not Result or else i > upper1
	 loop
	    from
	       j := upper2;
	    until
	       not Result or else j > upper2
	    loop
	       Result := value = item(i,j);
	       j := j + 1;
	    end;
	    i := i + 1;
	 end;
      end;

feature -- Printing :

   fill_tagged_out_memory is
      local
	 i1, i2: INTEGER;
	 v: like item;
      do
	 tagged_out_memory.append("lower1: "); 
	 lower1.append_in(tagged_out_memory);
	 tagged_out_memory.append(" upper1: "); 
	 upper1.append_in(tagged_out_memory);
	 tagged_out_memory.append(" lower2: "); 
	 lower2.append_in(tagged_out_memory);
	 tagged_out_memory.append(" upper2: "); 
	 upper2.append_in(tagged_out_memory);
	 tagged_out_memory.append(" [");
	 from  
	    i1 := lower1;
	    i2 := lower2;
	 until
	    not valid_index(i1,i2) or else 
	    tagged_out_memory.count > 2048
	 loop
	    v := item(i1,i2);
	    if v = Void then
	       tagged_out_memory.append("Void");
	    else
	       v.out_in_tagged_out_memory;
	    end;
	    i2 := i2 + 1;
	    if i2 > upper2 then
	       i2 := lower2;
	       i1 := i1 + 1;
	    end;
	    if valid_index(i1,i2) then
	       tagged_out_memory.extend(' ');
	    end;
	 end;
	 if valid_index(i1,i2) then
	    tagged_out_memory.append(" ..."); 
	 end;
	 tagged_out_memory.extend(']'); 
      end;

feature -- The Guru section :

   free is
	 -- Free the memory used by the Current COLLECTION (objects
	 -- pointed by the Current COLLECTION are not affected). 
	 -- Assume you don't use Current any more. 
      obsolete "Replaced by automatic Garbage Collection. %
             %Will be removed in the next release."
      deferred
      end;
   
feature {NONE}

   frozen equal_like(e1, e2: like item): BOOLEAN is
	 -- Note: this feature is called to avoid calling `equal'
	 -- on expanded types (no automatic conversion to 
	 -- corresponding reference type).
      do
	 if e1.is_basic_expanded_type then
	    Result := e1 = e2;
	 elseif e1.is_expanded_type then
	    Result := e1.is_equal(e2);
	 elseif e1 = e2 then
	    Result := true;
	 elseif e1 = Void or else e2 = Void then
	 else
	    Result := e1.is_equal(e2);
	 end;
      end;

end -- COLLECTION2[E]

