#|------------------------------------------------------------*-Scheme-*--|
 | File:    modules/paths/locate.scm
 |
 |          Copyright (C)1997 Donovan Kolbly <d.kolbly@rscheme.org>
 |          as part of the RScheme project, licensed for free use.
 |          See <http://www.rscheme.org/> for the latest information.
 |
 | File version:     1.5
 | File mod date:    1997.11.29 23:10:37
 | System build:     v0.7.2, 97.12.21
 | Owned by module:  paths
 |
 | Purpose:          Look along a search path for a specified file
 |------------------------------------------------------------------------|
 | Notes:
 |      These functions make use of the notion of a current directory,
 |      which is stored in the fluid binding for *current-dir*.
 |      The current search path is in the fluid binding for *search-path*
 `------------------------------------------------------------------------|#

(define-fluid *search-path*)

(define (make-try-extensions exit extensions fn)
  (let ((basename (filename fn))
	(ext (extension fn)))
    (let-syntax ((try-out (syntax-form (extn dir)
			    (let ((try (make <file-name>
					     filename: basename
					     extension: extn
					     file-directory: dir)))
			      (if (file-exists? try)
				  (exit try)
				  #f)))))
      (if ext
	  ;; an extension was specified, so try it only
	  ;; (ie, don't try any of the default extensions)
	  (lambda (dir)
	    (try-out ext dir))
	  ;;
	  ;; no extension was specified, so return a closure
	  ;; that will try each of the extensions in the given
	  ;; list (in addition to no extension)
	  ;;
	  (lambda (dir)
	    ;;
	    ;; try the extensions -- ext == #f ==> no extension
	    ;;
	    (for-each
	     (lambda (ext)
	       ;; try a particular extension
	       (try-out ext dir))
	     extensions))))))

;;
;; basic-locate is a dumber, but more controllable, version of locate
;; it takes a search path argument, and doesn't automatically
;; try no extension
;;

(define (basic-locate filename search-path extensions)
  (let ((current-dir (or *current-dir* $dot-dir)))
    (call-with-current-continuation
     (lambda (exit)
       (let* ((fn (string->file filename)))
	 (for-each
	  ;
	  ; look in a particular directory for the file
	  ; (make-try-extensions returns a procedure of
	  ;  one argument that will search the directory
	  ;  which is the argument)
	  ;
	  (make-try-extensions exit extensions fn)
	  ;
	  ; decide what directories to look in
	  ;
	  (if (file-directory fn)
	      ; there IS a path specification, so interpret it
	      ; relative to the current-dir
	      (list (append-dirs current-dir (file-directory fn)))
	      ; there is no dir specification, so use the search
	      ; path starting with the current dir.
	      (cons current-dir search-path)))
	 #f)))))

(define (locate filename . extensions)
  (basic-locate filename 
		(fluid-ref *search-path* '())
		(cons #f extensions)))
