;; -*-Mode: Emacs-Lisp;-*-
;; PRCS - The Project Revision Control System
;; Copyright (C) 1997  Josh MacDonald
;;
;; This program is free software; you can redistribute it and/or
;; modify it under the terms of the GNU General Public License
;; as published by the Free Software Foundation; either version 2
;; of the License, or (at your option) any later version.
;;
;; This program is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.
;;
;; You should have received a copy of the GNU General Public License
;; along with this program; if not, write to the Free Software
;; Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
;;
;; $Id: prcs.el 1.3 Fri, 16 May 1997 17:51:28 -0700 jmacd $

;; PRCS project file parser

(defconst prcs::prj-ignore-regex "\\(\\(;.*\\)\\|[ \t\n]\\)+"
  "Whitespace and comments")
(defconst prcs::prj-string-regex "\"\\([^\\\"]\\|\\(\\\\\\(.\\|\n\\)\\)\\)*\""
  "String literals")
(defconst prcs::prj-label-regex  "\\([^\\\"() \t\n]\\|\\(\\\\\\(.\\|\n\\)\\)\\)+"
  "Labels")

(defun prcs::parse-prj-file(buffer)
  (set-buffer buffer)  (goto-char (point-min))
  (let ((prj-sexps nil))
    (while (< (point) (point-max))
      (let ((s  (prcs::read-sexp)))
	(if s (setq prj-sexps (cons s prj-sexps)))))
    (nreverse prj-sexps)))

(defun prcs::read-sexp()
  (if (looking-at prcs::prj-ignore-regex)
      (goto-char (match-end 0)))
  (let ((beg-point (point))
	(cur-char (following-char)))
    (cond ((eq cur-char 0) nil)  ; EOF
	  ((eq cur-char ?\()     ; read a list
	   (forward-char)
	   (let ((l (prcs::read-list)))
	     (list l beg-point (point))))
	  ((eq cur-char ?\))     ; unbalanced parens
	   (error "unbalanced parens"))
	  ((eq cur-char ?\")     ; start a string
	   (if (not (looking-at prcs::prj-string-regex))
	       (error "unterminated string literal"))
	   (goto-char (match-end 0))
	   (list (buffer-substring-no-properties beg-point (point)) beg-point (point)))
	  (t                     ; start a label
	   (if (not (looking-at prcs::prj-label-regex))
	       (error "illegal label"))
	   (goto-char (match-end 0))
	   (list (buffer-substring-no-properties beg-point (point)) beg-point (point))))))

(defun prcs::read-list()
  (if (looking-at prcs::prj-ignore-regex)
      (goto-char (match-end 0)))
  (let ((cur-char (following-char)))
    (cond ((eq cur-char ?\))      ; end a list
	   (forward-char)
	   nil)
	  ((eq cur-char 0)        ; EOF in list
	   (error "EOF while scanning prj file"))
	  (t                      ; another elt
	   (let ((s (prcs::read-sexp)))
	     (cons s (prcs::read-list)))))))

;; PRCS-Emerge
;;
;; These functions allow the script 'emerge', distributed with PRCS, to
;; invoke emerge during merge.  It uses a recursive edit.  Probably should
;; know what you're doing.

(require 'emerge)

(defun prcs::emerge-files (file-A file-B file-out quit-hooks)
  "Run Emerge on two files."
  (emerge-files-internal file-A file-B nil quit-hooks file-out))

(defun prcs::emerge-files-with-ancestor (file-A file-B file-ancestor file-out quit-hooks)
  "Run Emerge on two files, giving another file as the ancestor."
  (emerge-files-with-ancestor-internal file-A file-B file-ancestor nil quit-hooks file-out))

(defun prcs::emerge(work com sel out)
  (message "prcs::emerge with files %s %s %s %s" work com sel out)
  (let (quit-hooks)
    (add-hook 'quit-hooks (function (lambda () (exit-recursive-edit))))
    (add-hook 'quit-hooks (` (lambda () (emerge-files-exit (, out)))))
    (if (equal com "/dev/null")
	(prcs::emerge-files work sel out quit-hooks)
      (prcs::emerge-files-with-ancestor work sel com out quit-hooks))
    (recursive-edit)
    (message "prcs::emerge finished")
    )
  )

(provide 'prcs)
