/* IluDebug.java */
/* Chris Jacobi, September 11, 1997 11:01 am PDT */

/*
 * Copyright (c) 1996, 1997 Xerox Corporation.  All Rights Reserved.  
 * Unlimited use, reproduction, and distribution of this software is
 * permitted.  Any copy of this software must include both the above
 * copyright notice of Xerox Corporation and this paragraph.  Any
 * distribution of this software must comply with all applicable United
 * States export control laws.  This software is made available AS IS,
 * and XEROX CORPORATION DISCLAIMS ALL WARRANTIES, EXPRESS OR IMPLIED,
 * INCLUDING WITHOUT LIMITATION THE IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS FOR A PARTICULAR PURPOSE, AND NOTWITHSTANDING ANY OTHER
 * PROVISION CONTAINED HEREIN, ANY LIABILITY FOR DAMAGES RESULTING FROM
 * THE SOFTWARE OR ITS USE IS EXPRESSLY DISCLAIMED, WHETHER ARISING IN
 * CONTRACT, TORT (INCLUDING NEGLIGENCE) OR STRICT LIABILITY, EVEN IF
 * XEROX CORPORATION IS ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.
 */
 
/* $Id: IluDebug.java,v 1.24 1997/09/11 18:02:58 jacobi Exp $ */
/* See native code in IluJava_IluDebug.c */

package xerox.ilu;

/**
 * This class helps in debugging IluJava. <p>
 * Applications better don't rely on this as it is likely
 * to be changed frequently. <p>
 *
 * @see IluDebugHooks
 * @see xerox.basics.IluEnvironment
 */
public class IluDebug {

    /**
     * Constant values for diverse debug flags
     */
    public static final int dont = 0;
    public static final int basic = 1;
    public static final int detailed = 2; 
    
    private static final int gip(java.lang.String key, int deflt) {
        return IluEnvironment.getIntProp(key, deflt);
    }
    
    //default value for most flags
    private static int def = gip("ilu.debug.default", dont);
    
    //generic debugging
    //    set to "dont" to run quiet
    //    set to "basic" to provide debug logs
    //    set to "detailed" to block on client as well as ilu errors
    private static int genericFlag = gip("ilu.debug", (def>dont ? def : basic));
    public static final int traceGeneric() {return genericFlag;}
    
    //spare to allow access without recompiling
    private static int spare1Flag = gip("ilu.debug.spare1", def);
    public static final int traceSpare1() {return spare1Flag;}
    
    //spare to allow access without recompiling
    private static int spare2Flag = gip("ilu.debug.spare2", def);
    public static final int traceSpare2() {return spare2Flag;}
    
    //initialization
    //Friendly because of initialization order problems.
    /*friendly*/ static int initsFlag = gip("ilu.debug.init", dont);
    public static final int traceInits() {return initsFlag;}

    //creation and destruction of objects
    private static int objectsFlag = gip("ilu.debug.object", def);
    public static final int traceObjects() {return objectsFlag;}

    //gc of objects (remote and local)
    /*friendly*/ static int gcFlag = gip("ilu.debug.gc", def);
    public static final int traceGC() {return gcFlag;}

    //Pickling and any
    private static int pickleFlag = gip("ilu.debug.pickle", def);
    public static final int tracePickle() {return pickleFlag;}
    
    //Calls, marshalling
    /*friendly*/ static int callsFlag = gip("ilu.debug.calls", def);
    public static final int traceCalls() {return callsFlag;}

    //classes, methods etc.
    private static int definitionsFlag = gip("ilu.debug.defs", def);
    public static final int traceDefs() {return definitionsFlag;}

    //ports, connections, transports...
    private static int pctFlag = gip("ilu.debug.trans", def);
    public static final int tracePCT() {return pctFlag;}

    //Monitor locks. Possibly expensive...
    private static int lockFlag = gip("ilu.debug.locks", dont); 
    public static final int traceLocks() {return lockFlag;}
    
    //Alarm tech.  Possibly expensive...
    private static int alarmFlag = gip("ilu.debug.alarms", dont); 
    public static final int traceAlarms() {return alarmFlag;}
    
    //Set this to a nonzero value to disable garbage collection.
    /*friendly*/ static int dontGCFlag = gip("ilu.debug.dontgc", 0);
    public static final int dontGC() {return dontGCFlag;}


    /**
     * Debugging aid to capture the java stack. 
     */
    public static java.lang.String captureStackTrace() {
        java.io.CharArrayWriter cawr = new java.io.CharArrayWriter();
        java.io.PrintWriter pwr = new java.io.PrintWriter(cawr);
        java.lang.Exception e = new java.lang.Exception();
        e.printStackTrace(pwr);
        return cawr.toString();
    } //captureStackTrace
    
    
    /*
     * "Constructor" for a debug/log stream.
     */
    private static final java.io.PrintWriter makeLog() {
        if (traceGeneric() > dont) {
            return IluDebugWriter.getIluDebugWriter();
        }
        return new java.io.PrintWriter(new xerox.basics.NowhereWriter());
    } //makeLog
    
    /**
     * A stream whose output can be redirected and is used for debugging
     * purposes. While this may be redirected, it is still visible somewhere.
     */
    public static java.io.PrintWriter log = makeLog();
    
     
    
    /**
     * Notification that ilu is about to raise some client
     * error soon.
     * 
     * - In Ilu debugging mode this will stop the thread and
     *   might even try to stop other threads to ease debugging.  
     * - In Ilu production mode this is either a no-op. It relies 
     *    on the caller to raise necessary errors.  
     *
     * clientPreError's are normal production features to Ilu. 
     */
    public static final void clientPreError(java.lang.String s) {
        if (traceGeneric() > basic) {
            block("clientPreError: " + s);
        }
    }
    
    
    /**
     * Notification that ilu is about to raise some internal
     * error soon.   
     * 
     * - In Ilu debugging mode this will stop the thread and
     *   might even try to stop other threads to ease debugging.  
     * - In Ilu production mode this might cause a log entry
     *   but will proceed and rely on the caller to raise 
     *   necessary errors.  
     */
    public static final void iluPreError(java.lang.String s) {
        System.err.println("** xerox.ilu.IluDebug iluPreError: " + s);
        if (traceGeneric() > dont) {
            block("iluError: " + s);
        }
    }
    
    
    public static final void loopForEver() {
        for (;;) {
            try {
                java.lang.Thread.sleep(10000);
            } catch (java.lang.InterruptedException e) {
            }
        }
    }
    
    /**
     * Ilu internal panic.
     * 
     * Does not return; No recovery possible.  
     * Severe ilu problem is encountered which makes
     * any further progress meaningless. Could as well
     * exit the program, but this way more debugging
     * is possible.
     * Trusted callers only.  
     */
    public static final void panic(java.lang.String s) {
        System.err.println("**** xerox.ilu.IluDebug panic");
        System.err.println("     severe internal ilu problem");
        System.err.println(s);
        System.err.println(captureStackTrace());
        loopForEver();
    }
    
    /**
     * block's the thread and blocks as much as conveniant
     * from Ilu..
     * 
     * Does not return; No recovery possible.  
     * This is used to stop ilu in situations which warrant
     * serious debugging in debugging mode; this shouldn't
     * be called in production mode.
     * Trusted callers only.  
     */
    static final void block(java.lang.String s) {
        System.err.println("**** xerox.ilu.IluDebug block");
        System.err.println("     blocking thread in ilu debugging mode");
        System.err.println(s);
        System.err.println(captureStackTrace());
        loopForEver();
    }
    
    /**
     * halt's the thread and blocks as much as conveniant
     * from Ilu.
     * Does not return; no recovery possible.  <p>
     *
     * This is used by test programs only; the difference
     * compared to "panic" is the users perception that  
     * the test program and not Ilu is initiating the action.
     * Trusted callers only.
     */
    public static final void halt(java.lang.String s) {
        System.err.println("**** xerox.ilu.IluDebug halt");
        System.err.println("     halt invoked by test program");
        System.err.println(s);
        System.err.println(captureStackTrace());
        loopForEver();
    }
    
    
    /*
     * Reports value of actual named debug flags into the native world...
     */
    private static native void reportFlags(
    	int genericFlag, int spare1Flag, int spare2Flag,
    	int objectsFlag, int pickleFlag, int callsFlag, 
    	int definitionsFlag, int pctFlag, int gcFlag, 
    	int lockFlag, int alarmFlag
    	);
    	
    	
    /**
     * Reports the current state of most debug flags down
     * to the native world...
     */
    public static void reportSomeFlags() {
        reportFlags(
            genericFlag, spare1Flag, spare2Flag, 
            objectsFlag, pickleFlag, callsFlag, 
            definitionsFlag, pctFlag, gcFlag, 
            lockFlag, alarmFlag
            );
    }
    
    
    /**
     * Can be called to assert initialization.
     * Internally called; ilu clients do not have to.
     */
    public static void init() {
		//java.lang.System.err.println("IluDebug::init entered");
        if (IluInit.doInit()) {
			//java.lang.System.err.println("IluDebug::init calling new IluDebug()");
            iluDebugInstance = new IluDebug();
			//java.lang.System.err.println("IluDebug::init calling registerTheInst");
            iluDebugInstance.registerTheInst();
			//java.lang.System.err.println("IluDebug::init calling reportSomeFlags");
            reportSomeFlags();
            
        }
		//java.lang.System.err.println("IluDebug::init exiting");

    }
    
    
    /**
     * We need an instance on the native side
     * to be able to invoke methods.
     */
    private native void registerTheInst();
    private static IluDebug iluDebugInstance = null;
    static {
       IluInit.init();
       /* init(); - done by IluInit.init() ... IluInit2()... IluDebug.Init() */
    }
    
    /**
     * Dynamic version of captureStackTrace for impoverished
     * calls from the native side 
     */
    public java.lang.String dynamicCaptureStackTrace() {
        return IluDebug.captureStackTrace(); 
    }
    
    /**
     * Dynamic version of getIntProp for impoverished
     * calls from the native side 
     */
    public int dynamicGetIntProp(java.lang.String key, int deflt) {
        return IluEnvironment.getIntProp(key, deflt); 
    }
    
    /**
     * Wait for interactive go-ahead typed into System.in.
     * Useful to stop ilu for setting break points after the
     * native code is loaded.
     */
    public static void interactiveWait() {
        java.io.InputStreamReader ir = new java.io.InputStreamReader(System.in);
        java.io.BufferedReader br = new java.io.BufferedReader(ir);
        boolean goAhead = false;
        java.lang.String line = null;
        while (!goAhead) {
            try {
                System.out.print("interactive wait g<return>:");
                System.out.flush();
                line = br.readLine(); 
            } catch (java.io.IOException e) { 
                line = null;
            }
            if (line != null & line.length() > 0)  {
                if (line.charAt(0)=='g') goAhead = true;
            }
        }
    } //interactiveWait


} // IluDebug
