// QWeb - An SGML Web Browser
// Copyright (C) 1997  Sean Vyain
// svyain@mail.tds.net
// smvyain@softart.com
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#ifndef _Cache_h_
#define _Cache_h_

#include <qlist.h>
#include <qobject.h>
#include <qstring.h>
#include "Url.h"

class Connection;
class Request;

//: The CacheData class stores all information associated with an active or completed URL request.
struct CacheData {
    //. The open connection that is retrieving the data.
    Connection* conn;

    //. The URL that the data was retrieved from.
    Url         url;

    //. The data's Internet media type.
    QString     mediaType;

    //. The data's Internet media subtype.
    QString     mediaSubtype;

    //. The total size, as reported when the connection was opened.
    int         totalSize;

    //. The actual data.
    QByteArray  data;

    //. TRUE if the data has been received in its entirety.
    bool        complete;

    //. The number of Requests that are connected to this CacheData.  This is
    //. not used once the data is complete.  If this count goes to zero while
    //. retrieving data, the connection is aborted and the CacheData is
    //. deleted.
    int         refcount;

    //. Create a new CacheData for the given URL.
    CacheData( const Url& _url ) : conn( 0 ), url( _url ), complete( FALSE ), refcount( 0 ) {}
};

//: The Cache class provides a global memory cache of previously retrieved URLs.
//. The Cache is a global object that is used by the Request class to retrieve
//. data, given a URL.  As data is retrieved, it is stored in an in-memory
//. cache for quick retrieval by future Requests.  In addition, if multiple
//. Requests ask for a URL that is in the process of beign retrieved, the Cache
//. will automatically connect all of the Requests to a single Connection to
//. improve efficiency.
//. <P>
//. The in-memory cache can grow to any size, and it is currently not limited.
//. However, it can be manually cleared.
class Cache : public QObject {
    Q_OBJECT
    
    struct Req {
        Request*   req;
        CacheData* data;
    };

    QList<CacheData> _data;
    QList<CacheData> _active;
    QList<Req>       _pending;
    QList<Req>       _reqs;

    //. Find the CacheData in the cache list associated with the given URL.
    CacheData* findData( const Url& url );

    //. Find the CacheData in the active list associated with the given URL.
    CacheData* findActive( const Url& url );

    //. Find the CacheData in the active list associated with the given
    //. connection.
    CacheData* findActive( Connection* conn );
protected:
    //. This is where Requests are actually connected with open Connections.
    //. For each pending Request, locate its associated CacheData.  If the
    //. startOfData signal has already been received by the CacheData, forward
    //. the information to the Request now.  If there is any existing data,
    //. forward it to the Request now.  If the CacheData is complete, forward
    //. the endOfData signal to the Request now.
    void timerEvent( QTimerEvent* e );
public:
    //. Create a new Cache.
    Cache();

    //. Delete all cached data.
    ~Cache();

    //. Process a request from a Request for a URL.  If the requested URL has
    //. already been retrieved or is being retrieved, connect the Request to
    //. the existing CacheData.  Otherwise, create a new CacheData and
    //. connection, and connect them to the Request.
    bool getUrl( Request* request, const Url& url, bool reload );

    //. Free all cached data.
    void       clear();
public slots:
    //. The open connection has been destroyed for some reason.  Remove the
    //. associated CacheData, as it is not complete.
    void connDestroyed();

    //. Store the media type, and the total data length.
    void startOfData( QString mediaType, QString mediaSubtype, int totalSize );

    //. Append the incoming data to the associated CacheData object.
    void data( const char* bytes, int length );

    //. Mark the data as complete, and store it in the cache.  Data retrieved
    //. through the "file" method is not cached.
    void endOfData();

    //. Process the notification from a request that it has been aborted.  If
    //. no other requests are waiting on the same data, abort the connection
    //. and delete the associated CacheData.
    void requestDone( Request* req );
};

extern Cache* cache;

#endif
