/*  Copyright (c) 1995 John E. Davis (davis@space.mit.edu)
 *  All rights reserved.
 */
#include <config.h>

#include <stdio.h>
#include <string.h>
#include <ctype.h>

#include <sys/types.h>
#include <sys/stat.h>

#ifdef HAVE_UNISTD_H
# include <unistd.h>
#endif

#ifdef HAVE_STDLIB_H
# include <stdlib.h>
#endif

#ifdef STANDALONE
# define SLRN_ERROR(x) fprintf(stderr, (x))
# define SLRN_MESSAGE(x) fprintf(stdout, (x))
# define SLRN_ERROR_1(x,y) fprintf(stderr, (x), (y))
# define SLRN_MESSAGE_1(x,y) fprintf(stdout, (x), (y))
# define SLRN_POPEN popen
# define SLRN_PCLOSE pclose
#define slrn_fclose slrn_fclose_standalone
static int slrn_fclose_standalone (FILE *fp)
{
   if (0 == fclose (fp)) return 0;
   SLRN_ERROR ("Error closing file.  File system full?");
   return -1;
}
   
#else
# include "slrn.h"
# include "misc.h"
# include "uudecode.h"
# define SLRN_ERROR slrn_error
# define SLRN_ERROR_1 slrn_error
# define SLRN_MESSAGE(x) slrn_message(x); slrn_refresh ()
# define SLRN_MESSAGE_1(x,y) slrn_message((x), (y)); slrn_smg_refresh ()
# define SLRN_POPEN slrn_popen
# define SLRN_PCLOSE slrn_pclose
char *Slrn_Decode_Directory;
#endif


static int unpack_as_shell_archive (FILE *fp, char *buf, int size)
{
   FILE *pp;
   
   size--;
   pp = SLRN_POPEN ("/bin/sh", "w");
   if (pp == NULL)
     {
	SLRN_ERROR ("Unable to open /bin/sh\n");
	return -1;
     }
   
   while (fgets (buf, size, fp) != NULL)
     {
	fputs (buf, pp);
	if (!strcmp (buf, "exit 0\n"))
	  break;
     }
   
   if (-1 == SLRN_PCLOSE (pp))
     {
	SLRN_ERROR ("Error encountered while processing shell archive.\n");
	return -1;
     }
   return 0;
}


int uudecode_file (char *file, char *newdir)
{
   FILE *fp, *outfp = NULL;
   char buf[4096];
#ifdef __os2__
   char olddir_buf[256];
#endif
     
   if (newdir == NULL)
     {
#ifndef STANDALONE
	char newdir_buf[256];
#ifdef __os2__
	if (getwd(olddir_buf) == 0) 
	  {
	     SLRN_ERROR_1 ("Unable to save old directory name: %s\n", olddir_buf);
	     return -1;
	  }
#endif
	if (Slrn_Decode_Directory == NULL)
	  Slrn_Decode_Directory = "News";
	newdir = Slrn_Decode_Directory;
	slrn_make_home_dirname (newdir, newdir_buf);
	newdir = newdir_buf;
#endif
     }
   
   if ((newdir != NULL)
       && (-1 == chdir(newdir)))
     {
	SLRN_ERROR_1 ("Unable to chdir to %s.\n", newdir);
	return -1;
     }
   
   if (file == NULL) fp = stdin;
   else fp = fopen (file, "r");
   
   if (fp == NULL)
     {
	SLRN_ERROR_1("Unable to open %s.\n", file);
	return -1;
     }
   
   while (fgets ( buf, sizeof(buf) - 1, fp) != NULL)
     {
	unsigned char *b, *bmax;
	int mode = 0;
	
	/* We are looking for 'begin ddd filename' */
	if (strncmp (buf, "begin ", 6))
	  {
	     if ((buf[0] == '#') && (buf[1] == '!'))
	       {
		  char *binsh = buf + 2;
		  if (*binsh == ' ') binsh++;
		  if (!strcmp (binsh, "/bin/sh\n"))
		    {
		       unpack_as_shell_archive (fp, buf, sizeof (buf));
		    }
	       }
	     continue;
	  }
	
	b = (unsigned char *) buf + 6;
	while (*b == ' ') b++;
	if (0 == isdigit (*b)) continue;
	while (isdigit (*b))
	  {
	     mode = mode * 8 + (*b - '0');
	     b++;
	  }
	
	if (*b != ' ') continue;
	while (*b == ' ') b++;
	if (*b < ' ') continue;
	
	bmax = b + strlen ((char *) b);
	bmax--;
	
	if (*bmax == '\n') *bmax = 0;
	
	outfp = fopen ((char *) b, "wb");
	
	if (outfp == NULL)
	  {
	     SLRN_ERROR_1("Unable to create %s\n", (char *)b);
	     slrn_fclose (fp);
	     return -1;
	  }
	chmod ((char *) b, mode);
	SLRN_MESSAGE_1("creating %s\n", (char *) b);
	
	/* Now read parts of the file in. */
	while (fgets (buf, sizeof(buf) - 1, fp) != NULL)
	  {
	     int len;
	     int buflen;
	     unsigned char out[60], *outp, *outmax;
	     
	     if (((*buf & 0x3F) == ' ') && (*(buf + 1) == '\n'))
	       {
		  /* we may be on the last line before the end.  Lets check */
		  if (NULL == fgets (buf, sizeof(buf) - 1, fp))
		    {
		       SLRN_ERROR ("Unexpected end of file.\n");
		       slrn_fclose (fp);
		       return -1;
		    }
		  
		  if (!strcmp (buf, "end\n"))
		    {
		       slrn_fclose (outfp);
		       outfp = NULL;
		       break;
		    }
	       }
	     
	     /* Now perform some sanity checking */
	     len = *buf - ' ';
	     if (len <= 0) continue;
	     
	     buflen = strlen (buf) - 2;
	     if (buflen > 60) continue;
	     if ((buflen % 4) != 0) continue;
	     
	     /* In general, I cannot make the test:
	      * @ if ((3 * buflen) != (len * 4)) continue;
	      * The problem is that the last line may be padded.  Instead, do
	      * it just for normal length lines and handle padding for others.
	      */
	     
	     if (*buf == 'M')
	       {
		  if ((3 * buflen) != (len * 4)) continue;
	       }
	     else
	       {
		  if (buflen != ((len + 2) / 3) * 4) continue;
	       }
	     
	     b = (unsigned char *) buf + 1;
	     bmax = b + buflen;
	     
	     while (b < bmax)
	       {
		  *b = (*b - 32) & 0x3F;
		  *(b + 1) = (*(b + 1) - 32) & 0x3F;
		  *(b + 2) = (*(b + 2) - 32) & 0x3F;
		  *(b + 3) = (*(b + 3) - 32) & 0x3F;
		  b += 4;
	       }
	     
	     b = (unsigned char *) buf + 1;
	     
	     outp = out;
	     outmax = outp + len;
	     while (outp < outmax)
	       {
		  register unsigned char b1, b2;
		  b1 = *(b + 1);
		  b2 = *(b + 2);
		  
		  
		  *outp++ = (*b << 2) | (b1 >> 4);
		  
		  if (outp < outmax)
		    {
		       *outp++ = (b1 << 4) | (b2 >> 2);
		       if (outp < outmax)
			 *outp++ = (b2 << 6) | *(b + 3);
		    }
		  b += 4;
	       }
	     
	     if (len != fwrite ((char *) out, 1, len, outfp))
	       {
		  SLRN_ERROR ("write to file failed.\n");
		  slrn_fclose (fp);
		  slrn_fclose (outfp);
		  return -1;
	       }
	  }
	/* end of part reading */
	
	/* back to looking for something else */
     }
   slrn_fclose (fp);
   if (outfp != NULL) 
     {
	slrn_fclose (outfp);
	outfp = NULL;
     }

#ifdef __os2__
   if (-1 == chdir(olddir_buf))
     {
	SLRN_ERROR_1 ("Unable to chdir back to %s.\n", olddir_buf);
	return -1;
     }
#endif   
#ifdef STANDALONE
   SLRN_MESSAGE ("No more files found.\n");
#endif
   return 0;
}

#ifdef STANDALONE
int main (int argc, char **argv)
{
   int i;
   
   if (argc <= 1)
     {
	if (!isatty (fileno(stdin)))
	  {
	     return uudecode_file (NULL, NULL);
	  }
	fprintf (stderr, "Usage: uudecode filename ...\n");
	return -1;
     }
   
   for (i = 1; i < argc; i++)
     {
	uudecode_file (argv[i], NULL);
     }
   return 0;
}
#endif




