/* This software is Copyright 1995 by Karl-Johan Johnsson
 *
 * Permission is hereby granted to copy, reproduce, redistribute or otherwise
 * use this software as long as: there is no monetary profit gained
 * specifically from the use or reproduction of this software, it is not
 * sold, rented, traded or otherwise marketed, and this copyright notice is
 * included prominently in any copy made. 
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. ANY USE OF THIS
 * SOFTWARE IS AT THE USER'S OWN RISK.
 */
#include "global.h"
#include "../Widgets/ArtText.h"
#include "../Widgets/ArtTree.h"
#include "../Widgets/Notice.h"
#include "../Widgets/Sash.h"
#include "../Widgets/ScrBar.h"
#include "../Widgets/ScrList.h"
#include "ahead.h"
#include "cache.h"
#include "codes.h"
#include "connect.h"
#include "kedit.h"
#include "kill.h"
#include "newsrc.h"
#include "partial.h"
#include "read.h"
#include "resource.h"
#include "save.h"
#include "search.h"
#include "server.h"
#include "tag.h"
#include "thread.h"
#include "util.h"
#include "widgets.h"
#include "xutil.h"

/* Quit   Done */
void knapp0_callback(Widget gw, XtPointer client_data, XtPointer call_data)
{
    if (global.busy) return;

    switch (global.mode) {
    case NewsModeDisconnected: /* Quit */
	exit(0);
	break;
    case NewsModeConnected: /* Quit */
	quit(True);
	break;
    case NewsModeGroup: /* Done */
	set_busy(False, True);

	popdown_save();
	popdown_search();
	partial_clear_cache();
	clear_tagged_articles();
	clear_history();

	free_read_arts_list(global.curr_group);
	global.curr_group->read_arts = create_read_arts_list();

	cache_leave_group();
	thread_ahead_leave_group(global.curr_group);
	global.curr_art = NULL;
	global.curr_subj = NULL;

	if (global.curr_group->subscribed)
	    setNewsModeConnected();
	else
	    setNewsModeAllgroups();

	unset_busy();
	if (global.mode == NewsModeConnected)
	    check_if_rescan_due();

	res_enter_group("none");
	break;
    case NewsModeAllgroups: /* Done */
	popdown_search();
	setNewsModeConnected();
	check_if_rescan_due();
	break;
    case NewsModeNewgroups: /* Done */
	XtFree((char *)global.new_groups);
	global.new_groups = NULL;
	global.no_new_groups = 0;
	sort_groups();
	setNewsModeConnected();
	check_if_rescan_due();
	break;
    case NewsModeThread:
	break;
    }
}

/* Connect   Disconnect   Subscribe   View Thread  Back */
void knapp1_callback(Widget gw, XtPointer client_data, XtPointer call_data)
{
    long	i, j;

    switch (global.mode) {
    case NewsModeConnected: /* Disconnect */
	if (global.busy)
	    return;

	set_busy(False, False);

	(void)update_newsrc(); /* should check return value */
	(void)update_kill_file();
	server_write_raw(main_server, "QUIT\r\n", 6);
	server_close(main_server);
	global_cleanup(False, False);
	set_standard_message();

	unset_busy();
	break;
    case NewsModeDisconnected: /* Connect */
	if (global.busy)
	    return;

	set_standard_message();
	popup_connect_dialogue();
	break;
    case NewsModeThread: /* View Thread */
	setNewsModeGroup(False /* not used */);
	break;
    case NewsModeGroup: /* Back */
	if (global.curr_subj)
	    setNewsModeThread();
	break;
    case NewsModeAllgroups: /* Subscribe */
	if (global.busy)
	    return;

	i = ScrListGetFirstSelected(main_widgets.group_list);

	if (i < 0) {
	    set_message("No selected groups!", True);
	    return;
	}

	set_busy(False, True);
	while (i >= 0) {
	    global.groups[i]->subscribed = True;
	    i = ScrListGetNextSelected(main_widgets.group_list, i);
	}
	sort_groups();
	global.curr_group = NULL;
	setNewsModeAllgroups();
	set_standard_message();
	unset_busy();
	return;
    case NewsModeNewgroups: /* Subscribe */
	if (global.busy)
	    return;

	i = ScrListGetFirstSelected(main_widgets.group_list);

	if (i < 0) {
	    set_message("No selected groups!", True);
	    return;
	}

	set_busy(False, True);
	while (i >= 0) {
	    for (j = 0 ; j < global.no_groups ; j++) {
		if (global.groups[j]->disp == i) {
		    global.groups[j]->subscribed = True;
		    break;
		}
	    }
	    i = ScrListGetNextSelected(main_widgets.group_list, i);
	}
	setNewsModeNewgroups();
	unset_busy();
	break;
    }
}

/* Unsubscribe   All threads   All groups */
void knapp2_callback(Widget gw, XtPointer client_data, XtPointer call_data)
{
    long	i, j, n;
    SUBJECT	*subj;

    if (global.busy) return;

    switch (global.mode) {
    case NewsModeConnected: /* All groups */
	set_busy(False, True);
	global.curr_group = NULL;
	setNewsModeAllgroups();
	unset_busy();
	break;
    case NewsModeGroup: /* All threads */
	for (subj = get_subjects(main_thr) ;
	     subj ; subj = subj->next)
	    if (subj->disp < 0)
		break;

	set_curr_art(NULL, False);
	set_curr_subj(NULL);
	setNewsModeGroup(subj != NULL);
	set_standard_message();
	break;
    case NewsModeAllgroups: /* Unsubscribe */
	n = ScrListGetFirstSelected(main_widgets.group_list);

	if (n < 0) {
	    set_message("No selected groups!", True);
	    return;
	}

	set_busy(False, True);
	while (n >= 0) {
	    global.groups[n]->subscribed = False;
	    n = ScrListGetNextSelected(main_widgets.group_list, n);
	}
	sort_groups();
	global.curr_group = NULL;
	setNewsModeAllgroups();
	set_standard_message();
	unset_busy();
	break;
    case NewsModeNewgroups: /* Unsubscribe */
	i = ScrListGetFirstSelected(main_widgets.group_list);

	if (i < 0) {
	    set_message("No selected groups!", True);
	    return;
	}

	set_busy(False, True);
	while (i >= 0) {
	    for (j = 0 ; j < global.no_groups ; j++) {
		if (global.groups[j]->disp == i) {
		    global.groups[j]->subscribed = False;
		    break;
		}
	    }
	    i = ScrListGetNextSelected(main_widgets.group_list, i);
	}
	setNewsModeNewgroups();
	unset_busy();
	break;
    case NewsModeDisconnected:
    case NewsModeThread:
	break;
    }
}

/* Kill... */
void knapp5_callback(Widget gw, XtPointer client_data, XtPointer call_data)
{
    if (global.mode != NewsModeDisconnected)
	popup_kill_editor();
}

/* Update   Catchup */
void knapp6_callback(Widget gw, XtPointer client_data, XtPointer call_data)
{
    ART_LIST_NODE	*temp;
    int			kill_flag;

    if (global.busy) return;

    switch (global.mode) {
    case NewsModeConnected: /* Update */
    case NewsModeAllgroups:
	set_busy(False, True);

	kill_flag = update_kill_file();
	if (update_newsrc()) {
	    if (kill_flag)
		set_message("Newsrc and kill files updated.", False);
	    else
		set_message("Failed to update kill file! "
			    "  Newsrc file updated.", True);
	} else {
	    if (kill_flag)
		set_message("Failed to update newsrc! "
			    "  Kill file updated.", True);
	    else
		set_message("Failed to update newsrc and kill files!", True);
	}

	unset_busy();
	break;
    case NewsModeGroup: /* Catchup */
	set_busy(False, True);

	popdown_save();
	popdown_search();
	clear_tagged_articles();
	partial_clear_cache();
	clear_history();

	free_read_arts_list(global.curr_group);
	temp = global.curr_group->read_arts =
	    (ART_LIST_NODE *)XtMalloc(sizeof(ART_LIST_NODE));
	temp->next = NULL;
	temp->first = 1;
	temp->last = global.curr_group->last_art;

	if (res_process_xrefs()) {
	    ARTICLE	*art;

	    for (art = get_articles(main_thr) ; art ; art = art->next)
		if (!art->read)
		    process_xref(art);

	}

	cache_leave_group();
	thread_ahead_leave_group(global.curr_group);
	global.curr_art = NULL;
	global.curr_subj = NULL;
	res_enter_group("none");

	setNewsModeConnected();

	unset_busy();
	break;
    case NewsModeDisconnected:
    case NewsModeThread:
    case NewsModeNewgroups:
	break;
    }
}

/* Read/goto group   Next unread */
void knapp7_callback(Widget gw, XtPointer client_data, XtPointer call_data)
{
    ARTICLE	*art;
    Boolean	new_thread;

    if (global.busy) return;

    switch (global.mode) {
    case NewsModeConnected: /* Read group */
	set_curr_group();
	/* fall through */
    case NewsModeAllgroups:
	if (!global.curr_group) {
	    set_message("No selected group!", True);
	    return;
	}

	popdown_find_group();
	popdown_search();
	read_group(NULL, True, 0);
	break;
    case NewsModeGroup:  /* Next unread */
    case NewsModeThread:
	art = global.curr_art;
	new_thread = False;

	if (!art) {
	    if (global.curr_subj)
		art = global.curr_subj->thread;
	    new_thread = True;
	}

	while (art) {
	    if (art->from && !art->read)
		break;
	    art = next_in_thread_preorder(art);
	}

	if (!art) {
	    SUBJECT	*subj = NULL;
	    ARTICLE	*stop = NULL;

	    new_thread = True;
	    if (global.curr_subj) {		
		stop = global.curr_subj->thread;
		subj = global.curr_subj->next;
		while (subj && subj->thread == stop)
		    subj = subj->next;
	    }

	    if (!subj) {
		subj = get_subjects(main_thr);
		stop = NULL;
	    }

	    if (subj)
		while (subj->thread != stop) {
		    for (art = subj->thread ; art ;
			 art = next_in_thread_preorder(art))
			if (art->from && !art->read)
			    break;
		    if (art)
			break;

		    do {
			subj = subj->next;
		    } while (subj && subj->prev->thread == subj->thread);
		    if (!subj)
			if (stop)
			    subj = get_subjects(main_thr);
			else
			    break;
		}
	}

	set_curr_art(art, True);
	if (art) {
	    if (read_article(art, False, NULL)) {
		if (global.mode == NewsModeThread) {
		    if (new_thread)
			setNewsModeThread();
		} else
		    set_curr_art(art, False);
	    }
	} else {
	    if (global.mode == NewsModeThread)
		setNewsModeGroup(False);
	    else
		knapp0_callback(NULL, NULL, NULL); /* Done */
	}

	break;
    case NewsModeDisconnected:
    case NewsModeNewgroups:
	break;
    }
}

/*  Rescan   Previous */
void knapp8_callback(Widget gw, XtPointer client_data, XtPointer call_data)
{
    ARTICLE	*art, *old;
    Boolean	new_thread;
    char	*buffer;

    if (global.busy) return;

    switch (global.mode) {
    case NewsModeGroup:  /* Previous */
    case NewsModeThread:
	art = history_pop();
	old = global.curr_art;
	if (old && art == old) {
	    art = history_pop();
	    if (!art)
		history_push(old);
	}

	if (!art) {
	    set_message("No previous article!", True);
	    return;
	}

	if (old && old->from && old->read) {
	    old->read = False;
	    old->subject->no_unread++;
	    global.curr_group->no_unread++;
	    if (old->hot > 0) {
		global.n_hot++;
		update_subj_hot_value(old->subject);
	    }
	    update_subj_entry(old->subject);
	    if (global.mode == NewsModeThread) {
		ArtTreeNodeSetInner(main_widgets.arttree,
				    (ART_TREE_NODE *)old, True);
		if (old->hot > 0) {
		    ArtTreeNodeSetPixmap(main_widgets.arttree,
					 (ART_TREE_NODE *)old,
					 global.hot_pixmaps[old->hot]);
		}
	    }
	}

	if (old && old->subject->thread == art->subject->thread)
	    new_thread = False;
	else
	    new_thread = True;

	set_curr_art(art, True);
	if (read_article(art, False, NULL)) {
	    if (global.mode == NewsModeThread) {
		if (new_thread)
		    setNewsModeThread();
	    } else
		set_curr_art(art, False);
	}
	break;
    case NewsModeConnected: /* Rescan */
	remove_rescan_timeout();

	set_busy(True, True);
	set_message("Server contacted, waiting for response...", False);
	buffer = rescan();
	if (!buffer) {
	    reconnect_server(True);
	    unset_busy();
	    return;
	}
	unset_busy();

	if (atoi(buffer) == NNTP_OK_GROUPS)
	    setNewsModeConnected();
	else {
	    char	message[128];

	    if (strlen(buffer) > 80) buffer[80] = '\0';
	    sprintf(message, "Error!  Message from server is: %s",
		    buffer);
	    set_message(message, True);
	}
	break;
    case NewsModeDisconnected:
    case NewsModeAllgroups:
    case NewsModeNewgroups:
	break;
    }
}

/* Abort */
void knapp9_callback(Widget gw, XtPointer client_data, XtPointer call_data)
{
    if (global.busy && !global.aborted) {
	global.aborted = True;
	if (global.quit_nicely)
	    server_write_raw(main_server, "\r\nQUIT\r\n", 8);
	server_close(main_server);
    }
}

/* Save... */
void knapp10_callback(Widget gw, XtPointer client_data, XtPointer call_data)
{
    switch (global.mode) {
    case NewsModeGroup:
    case NewsModeThread:
	popup_save();
	break;
    case NewsModeDisconnected:
    case NewsModeConnected:
    case NewsModeAllgroups:
    case NewsModeNewgroups:
	if (global.bell)
	    XBell(XtDisplay(gw), 0);
	break;
    }
}

/* Search... */
void knapp11_callback(Widget gw, XtPointer client_data, XtPointer call_data)
{
    switch (global.mode) {
    case NewsModeGroup:
    case NewsModeThread:
    case NewsModeAllgroups:
	popup_search();
	break;
    case NewsModeDisconnected:
    case NewsModeConnected:
    case NewsModeNewgroups:
	if (global.bell)
	    XBell(XtDisplay(gw), 0);
	break;
    }
}

void top_vbar_callback(Widget gw, XtPointer client_data, XtPointer call_data)
{
    ScrBarReport	*report = (ScrBarReport *)call_data;
    long		shown, size;
    Arg			arg;

    switch (global.mode) {
    case NewsModeAllgroups:
    case NewsModeConnected:
    case NewsModeNewgroups:
	ScrListSetFirst(main_widgets.group_list, report->slider_position);
	ScrListGetFirstShownSize(main_widgets.group_list,
				 NULL, &shown, &size);
	if ((report->slider_length != shown ||
	     report->canvas_length != size) &&
	    report->slider_position + shown < size)
	    ScrBarSetLengthsAndPos(main_widgets.top_vbar,
				   size, shown, report->slider_position);
	break;
    case NewsModeGroup:
	ScrListSetFirst(main_widgets.thread_list, report->slider_position);
	ScrListGetFirstShownSize(main_widgets.thread_list,
				 NULL, &shown, &size);
	if ((report->slider_length != shown ||
	     report->canvas_length != size) &&
	    report->slider_position + shown < size)
	    ScrBarSetLengthsAndPos(main_widgets.top_vbar,
				   size, shown, report->slider_position);
	break;
    case NewsModeThread:
	XtSetArg(arg, XtNy, -report->slider_position);
	XtSetValues(main_widgets.arttree, &arg, 1);
	break;
    case NewsModeDisconnected:
	break;
    }
}

void top_hbar_callback(Widget gw, XtPointer client_data, XtPointer call_data)
{
    ScrBarReport	*report = (ScrBarReport *)call_data;

    if (global.mode == NewsModeThread) {
	Arg	arg;

	XtSetArg(arg, XtNx, -report->slider_position);
	XtSetValues(main_widgets.arttree, &arg, 1);
    }
}

void text_scrbar_callback(Widget gw, XtPointer client_data,
			  XtPointer call_data)
{
    ScrBarReport	*report = (ScrBarReport *)call_data;
    long		first, shown, size;

    ArtTextSetFirst(main_widgets.text, report->slider_position);
    ArtTextGetFirstShownSize(main_widgets.text, &first, &shown, &size);
    if (report->slider_position != first ||
	report->slider_length   != shown ||
	report->canvas_length   != size)
	ScrBarSetLengthsAndPos(main_widgets.text_scrbar, size, shown, first);
}

void top_manager_callback(Widget gw, XtPointer client_data,
			  XtPointer call_data)
{
    long	first, shown, size;
    Position	s_x, s_y;
    Dimension	s_w, s_h, c_w, c_h;
    Arg		args[4];

    switch (global.mode) {
    case NewsModeConnected:
    case NewsModeAllgroups:
    case NewsModeNewgroups:
	ScrListGetFirstShownSize(main_widgets.group_list,
				 &first, &shown, &size);
	ScrBarSetLengthsAndPos(main_widgets.top_vbar, size, shown, first);
	break;
    case NewsModeThread:
	XtSetArg(args[0], XtNx, &s_x);
	XtSetArg(args[1], XtNy, &s_y);
	XtSetArg(args[2], XtNwidth, &c_w);
	XtSetArg(args[3], XtNheight, &c_h);
	XtGetValues(main_widgets.arttree, args, 4);
	XtSetArg(args[0], XtNwidth, &s_w);
	XtSetArg(args[1], XtNheight, &s_h);
	XtGetValues(main_widgets.top_manager, args, 2);
	ScrBarSetLengthsAndPos(main_widgets.top_vbar, c_h, s_h, - s_y);
	ScrBarSetLengthsAndPos(main_widgets.top_hbar, c_w, s_w, - s_x);
	break;
    case NewsModeGroup:
	ScrListGetFirstShownSize(main_widgets.thread_list,
				 &first, &shown, &size);
	ScrBarSetLengthsAndPos(main_widgets.top_vbar, size, shown, first);
	break;
    case NewsModeDisconnected:
	break;
    }
}

void text_resize_callback(Widget gw, XtPointer client_data,
			  XtPointer call_data)
{
    if (global.mode == NewsModeGroup || global.mode == NewsModeThread) {
	ArtTextReport	*report = (ArtTextReport *)call_data;

	ScrBarSetLengthsAndPos(main_widgets.text_scrbar,
			       report->canvas_length,
			       report->slider_length,
			       report->slider_position);
    }
}

void arttree_scroll_callback(Widget gw,
			     XtPointer client_data,
			     XtPointer call_data)
{
    ArtTreeScrollReport	*report = (ArtTreeScrollReport *)call_data;

    if (global.mode == NewsModeThread) {
	ScrBarSetSliderPosition(main_widgets.top_hbar, report->slider_x);
	ScrBarSetSliderPosition(main_widgets.top_vbar, report->slider_y);
    }
}

void thread_list_sel_callback(Widget gw,
			      XtPointer client_data,
			      XtPointer call_data)
{
    long	row = (long)call_data;
    SUBJECT	*loop;

    if (global.busy || global.mode != NewsModeGroup)
	return;

    global.curr_art = NULL;
    if (global.curr_subj && global.curr_subj->disp == row)
	return;

    for (loop = get_subjects(main_thr) ; loop ;	loop = loop->next)
	if (loop->disp == row)
	    break;

    global.curr_subj = loop;
}

void thread_list_callback(Widget gw,
			  XtPointer client_data,
			  XtPointer call_data)
{
    long	row = (long)call_data;
    SUBJECT	*subj;
    ARTICLE	*art = NULL;

    if (global.busy || global.mode != NewsModeGroup)
	return;

    if (global.curr_subj && global.curr_subj->disp == row)
	subj = global.curr_subj;
    else {
	subj = get_subjects(main_thr);
	while (subj)
	    if (subj->disp == row)
		break;
	    else
		subj = subj->next;
	if (!subj)
	    return;
	global.curr_subj = subj;
    }

    for (art = subj->thread ; art ; art = next_in_thread_preorder(art))
	if (art->subject == subj && art->from && !art->read)
	    break;

    global.curr_art = art;
    read_article(art, False, NULL);

    if (!art) {
	if (global.bell)
	    XBell(XtDisplay(gw), 0);
	set_message("No unread article with that subject!", True);
    }
}

void arttree_sel_callback(Widget gw,
			  XtPointer client_data,
			  XtPointer call_data)
{
    ARTICLE	*art = (ARTICLE *)call_data;

    if (global.busy) return;

    if (global.mode == NewsModeThread) {
	set_curr_art(art, False);
	read_article(art, False, NULL);
    }	
}

void group_list_callback(Widget gw,
			 XtPointer client_data,
			 XtPointer call_data)
{
    long	row = (long)call_data;

    if (global.busy) return;

    global.curr_group = NULL;

    if (global.mode == NewsModeAllgroups) {
	popdown_search();
	global.curr_group = global.groups[row];
    } else if (global.mode == NewsModeConnected) {
	GROUP		**loop = global.groups;
	long		n = global.no_groups;

	while (n > 0 && (*loop)->subscribed) {
	    if ((*loop)->disp == row) {
		global.curr_group = *loop;
		break;
	    }
	    loop++;
	    n--;
	}

	if (!global.curr_group)
	    return;
    } else {
	return;
    }

    read_group(NULL, True, 0);
}

void group_list_sel_callback(Widget w,
			     XtPointer client_data,
			     XtPointer call_data)
{
    long	row;

    if (global.busy) return;

    switch (global.mode) {
    case NewsModeDisconnected:
    case NewsModeNewgroups:
    case NewsModeGroup:
    case NewsModeThread:
	break;
    case NewsModeConnected:
	row = (long)call_data;
	if (row >= 0) {
	    long	i;

	    for (i = 0 ; i < global.no_groups ; i++) {
		if (!global.groups[i]->subscribed)
		    break;
		if (global.groups[i]->disp == row) {
		    global.curr_group = global.groups[i];
		    return;
		}
	    }
	    global.curr_group = NULL;
	}
	break;
    case NewsModeAllgroups:
	row = (long)call_data;
	if (row >= 0 && row < global.no_groups)
	    global.curr_group = global.groups[row];
	else
	    global.curr_group = NULL;
	break;
    }
}

void group_list_dnd_callback(Widget w,
			     XtPointer client_data,
			     XtPointer call_data)
{
    long	*index = (long *)call_data;
    GROUP	*temp;
    long	i;

    index[2] = False;
    switch (global.mode) {
    case NewsModeAllgroups:
	if (global.groups[index[0]]->subscribed &&
	    global.groups[index[1]]->subscribed) {
	    index[2] = True;
	    temp = global.groups[index[0]];
	    if (index[0] < index[1])
		for (i = index[0] ; i < index[1] ; i++)
		    global.groups[i] = global.groups[i + 1];
	    else
		for (i = index[0] ; i > index[1] ; i--)
		    global.groups[i] = global.groups[i - 1];
	    global.groups[index[1]] = temp;
	    break;
	}
	/* fall through */
    case NewsModeDisconnected:
    case NewsModeGroup:
    case NewsModeThread:
    case NewsModeConnected:
    case NewsModeNewgroups:
	if (global.bell)
	    XBell(XtDisplay(w), 0);
	break;
    }
}

void delete_window_callback(Widget w,
			    XtPointer client_data,
			    XtPointer call_data)
{
    if (global.busy)
	return;

    quit(True);
}

void sash_callback(Widget w, XtPointer client_data, XtPointer call_data)
{
    int			y, i1, i2, i3;
    unsigned int	mods;
    Window		w1, w2;
    Arg			arg[2];
    Dimension		width, height;

    /*
     *  Avoid race condition.
     */

    XQueryPointer(XtDisplay(w), XtWindow(w),
		  &w1, &w2, &i1, &i2, &i3, &y, &mods);
    if (y == 0 || mods == 0)
	return;

    XtSetArg(arg[0], XtNheight, &height);
    XtSetArg(arg[1], XtNwidth, &width);
    XtGetValues(main_widgets.top_layout, arg, 2);
    y += height;
    if (y <= 0)
	y = 1;
    else if (y > 32767)
	y = 32767;
    (void)XtMakeResizeRequest(main_widgets.top_layout, width, y, NULL, NULL);
}
