/* This software is Copyright 1995 by Karl-Johan Johnsson
 *
 * Permission is hereby granted to copy, reproduce, redistribute or otherwise
 * use this software as long as: there is no monetary profit gained
 * specifically from the use or reproduction of this software, it is not
 * sold, rented, traded or otherwise marketed, and this copyright notice is
 * included prominently in any copy made. 
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. ANY USE OF THIS
 * SOFTWARE IS AT THE USER'S OWN RISK.
 */
#include "global.h"
#include <sys/stat.h>
#include <sys/wait.h>
#include "child.h"
#include "codes.h"
#include "connect.h"
#include "newsrc.h"
#include "post.h"
#include "psetup.h"
#include "resource.h"
#include "server.h"
#include "thread.h"
#include "util.h"
#include "widgets.h"
#include "xutil.h"

PostStatus post_article(PostContext *context, char *message)
{
    char	buffer[4096];
    char	*reply, *c;
    int		status;
    FILE	*fp;

    if (message)
	message[0] = '\0';

    fp = fopen(context->file_name, "r");
    if (!fp)
	return PostStatusOpen;

    set_busy(True, False); /* don't quit nicely */

    reply = server_comm(main_server, "POST\r\n", True);
    if (!reply) {
	reconnect_server(True);
	unset_busy();
	fclose(fp);
	return PostStatusBroken;
    }

    status = atoi(reply);
    if (status != NNTP_CONT_POST) {
	if (message) {
	    if (strlen(reply) > 256)
		reply[256] = '\0';
	    strcpy(message, reply);
	}
	fclose(fp);
	unset_busy();
	return PostStatusFailed;
    }

    buffer[0] = '\0';
    if (context->flags & NEEDS_SENDER) {
	char	*full_name = res_full_name();

	if (full_name)
	    sprintf(buffer, "Sender: %s@%s (%s)\r\n",
		    global.user_id, global.domain_name, full_name);
	else
	    sprintf(buffer, "Sender: %s@%s\r\n",
		    global.user_id, global.domain_name);
    }
    c = buffer + strlen(buffer);

    for (;;) {
	int	len = sizeof(buffer) - (c - buffer);

	if (len < 128) {
	    server_write(main_server, buffer);
	    c = buffer;
	    c[0] = '\0';
	    len = sizeof(buffer);
	}

	if (fgets(c, len - 8, fp)) {
	    char	*p;

	    if (c[0] == '.')     /* Duplicate initial dot */
		memmove(c + 1, c, strlen(c) + 1);

	    p = strchr(c, '\n');
	    if (!p)
		c += strlen(c);
	    else {
		p[0] = '\r';
		p[1] = '\n';
		p[2] = '\0';
		c = p + 2;
	    }
	} else {
	    strcpy(c, "\r\n.\r\n");
	    server_write(main_server, buffer);
	    break;
	}
    }

    fclose(fp);
    reply = server_read(main_server);
    if (!reply) {
	reconnect_server(True);
	unset_busy();
	return PostStatusBroken;
    }
    unset_busy();
    status = atoi(reply);
    if (status == NNTP_OK_POSTED) {
	context->flags |= POST_DONE;
	return PostStatusOk;
    } else {
	if (message) {
	    if (strlen(reply) > 256)
		reply[256] = '\0';
	    strcpy(message, reply);
	}
	return PostStatusFailed;
    }
}

MailStatus mail_article(PostContext *context, char *stderr_buf)
{
    int		fd, status;
    pid_t	pid, wpid;

    fd = open(context->file_name, O_RDONLY);
    if (fd < 0) {
	perror("knews: open");
	return MailStatusOpen;
    }

    pid = fork_nicely(NULL, NULL, True);

    if (pid < 0) {
	perror("knews: fork");
	close(fd);
	return MailStatusFork;
    } else if (pid == 0) { /* child */
	if (fd != STDIN_FILENO) {
	    fd = dup2(fd, STDIN_FILENO);
	    if (fd < 0) {
		perror("knews: dup2");
		_exit(127);
	    }
	}

	execl(BIN_SH, "sh", "-c", MAIL_COMMAND, (char *)0);
	perror("knews: execl " BIN_SH);
	_exit(127);
    }

    close(fd);
    set_busy(False, False);
    wpid = wait_for_pid(pid, &status, stderr_buf);
    unset_busy();

    if (wpid == pid && WIFEXITED(status))
	status = WEXITSTATUS(status);
    else
	status = 1;

    if (status == 0)
        return MailStatusOk;
    else if (status == 127)
	return MailStatusStart;
    else
	return MailStatusFailed;
}

PostStatus post_to_agent(PostContext *context, char *agent, char *stderr_buf)
{
    pid_t	pid, wpid;
    int		status;

    pid = fork_nicely(NULL, NULL, True);

    if (pid < 0)
	return PostStatusStart;

    if (pid == 0) {
	int	fd;

	fd = open(context->file_name, O_RDONLY);
	if (fd < 0) {
	    perror("knews: open");
	    _exit(126);
	}

	if (fd != STDIN_FILENO) {
	    if (dup2(fd, STDIN_FILENO) != STDIN_FILENO) {
		perror("knews: dup2");
		_exit(126);
	    }
	    close(fd);
	}

	execl(BIN_SH, "sh", "-c", agent, (char *)0);
	perror("knews: execl " BIN_SH);
	_exit(127);
    }

    set_busy(False, False);
    wpid = wait_for_pid(pid, &status, stderr_buf);
    unset_busy();

    if (wpid == pid && WIFEXITED(status))
	status = WEXITSTATUS(status);
    else
	status = 1;

    if (status == 0)
        return PostStatusOk;
    else if (status == 126)
	return PostStatusOpen;
    else if (status == 127)
	return PostStatusStart;
    else
	return PostStatusFailed;
}
