/* This software is Copyright 1995 by Karl-Johan Johnsson
 *
 * Permission is hereby granted to copy, reproduce, redistribute or otherwise
 * use this software as long as: there is no monetary profit gained
 * specifically from the use or reproduction of this software, it is not
 * sold, rented, traded or otherwise marketed, and this copyright notice is
 * included prominently in any copy made. 
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. ANY USE OF THIS
 * SOFTWARE IS AT THE USER'S OWN RISK.
 */
#include <X11/IntrinsicP.h>
#include <X11/StringDefs.h>

#include "Compat.h"
#include "PopdownSh.h"
#include "MenuP.h"

static XtResource resources[] = {
#define offset(field) XtOffsetOf(MenuRec, menu.field)
    {XtNcursor, XtCCursor, XtRCursor, sizeof(Cursor),
     offset(cursor), XtRImmediate, (XtPointer)None},
#undef offset
};

static void Initialize(Widget, Widget, ArgList, Cardinal*);
static void Destroy(Widget);
static void Resize(Widget);
static void Realize(Widget, XtValueMask*, XSetWindowAttributes*);
static void Redisplay(Widget, XEvent*, Region);
static Boolean SetValues(Widget, Widget, Widget, ArgList, Cardinal*);
static XtGeometryResult QueryGeometry(Widget, XtWidgetGeometry*,
				      XtWidgetGeometry*);

static void highlight(Widget, XEvent*, String*, Cardinal*);
static void unhighlight(Widget, XEvent*, String*, Cardinal*);

static XtActionsRec actions[] = {
    {"highlight",	highlight},
    {"unhighlight",	unhighlight},
};

static char translations[] =
"<Motion>:	highlight() \n"
"<LeaveWindow>:	unhighlight() \n";

MenuClassRec menuClassRec = {
    { /* core fields */
    	(WidgetClass) &shadowClassRec,		/* superclass		*/
    	"Menu",					/* class_name		*/
    	sizeof(MenuRec),			/* widget_size		*/
    	NULL,					/* class_initialize	*/
    	NULL,					/* class_part_initialize*/
    	FALSE,					/* class_inited		*/
    	Initialize,				/* initialize		*/
    	NULL,					/* initialize_hook	*/
    	Realize,				/* realize		*/
    	actions,				/* actions		*/
    	XtNumber(actions),			/* num_actions		*/
    	resources,				/* resources		*/
    	XtNumber(resources),			/* resource_count	*/
    	NULLQUARK,				/* xrm_class		*/
    	TRUE,					/* compress_motion	*/
#if (XtSpecificationRelease < 4)
        TRUE,					/* compress_exposure    */
#elif (XtSpecificationRelease < 6)
	XtExposeCompressMaximal,		/* compress_exposure	*/
#else
	XtExposeCompressMaximal | XtExposeNoRegion, /* compress_exposure*/
#endif
    	TRUE,					/* compress_enterleave	*/
    	TRUE,					/* visible_interest	*/
    	Destroy,				/* destroy		*/
    	Resize,					/* resize		*/
    	Redisplay,				/* expose		*/
    	SetValues,				/* set_values		*/
    	NULL,					/* set_values_hook	*/
    	XtInheritSetValuesAlmost,		/* set_values_almost	*/
    	NULL,					/* get_values_hook	*/
    	NULL,					/* accept_focus		*/
    	XtVersion,				/* version		*/
    	NULL,					/* callback_private	*/
    	translations,				/* tm_table		*/
    	QueryGeometry,				/* query_geometry       */
    	XtInheritDisplayAccelerator,		/* display_accelerator  */
    	NULL					/* extension            */
    },
    { /* shadow fields */
	XtInheritPixelOffset,		/* pixel offset			*/
	False,				/* use_arm_for_background	*/
	XtInheritAllocShadowColors,	/* alloc_shadow_colors		*/
	XtInheritAllocShadowPixmaps,	/* alloc_shadow_pixmaps		*/
	XtInheritAllocArmColor,		/* alloc_arm_color		*/
	XtInheritAllocArmPixmap,	/* alloc_arm_pixmap		*/
	XtInheritAllocGCs,		/* alloc_gcs			*/
	NULL,				/* extension			*/
    },
    { /* menu fields */
	NULL					/* extension            */
    }
};

WidgetClass menuWidgetClass = (WidgetClass) &menuClassRec;

/*************************************************************************/

static void highlight(Widget gw, XEvent *event,
		      String *params, Cardinal *no_params)
{
    MenuWidget		w = (MenuWidget)gw;
    MenuGadget		*loop = w->menu.children;
    int			n = w->menu.num_children;
    int			i;

    if (event->type != MotionNotify) {
	XtAppWarning(XtWidgetToApplicationContext((Widget)w),
		     "action highlight() only supported for Motion events.");
	return;
    }

    for (i = 0 ; i < n ; i++, loop++) {
	MenuGadget	g = *loop;
	int		dy = event->xmotion.y - g->rectangle.y;

	if (dy >= 0 && dy <= (int)g->rectangle.height)
	    break;
    }
    if (i == n)
	i = -1;

    if (i != w->menu.hl_index) {
	if (w->menu.hl_index >= 0) {
	    MenuGadget	g = w->menu.children[w->menu.hl_index];

	    g->menu_g.highlighted = False;
	    XClearArea(XtDisplay(w), XtWindow(w),
		       g->rectangle.x, g->rectangle.y,
		       g->rectangle.width, g->rectangle.height, False);
	    if (g->object.widget_class->core_class.expose)
		g->object.widget_class->core_class.expose((Widget)g, NULL, 0);
	    if (MenuGadgetChangeHighlighted(g))
		MenuGadgetChangeHighlighted(g)(g);
	}

	if (i >= 0 && XtIsSensitive((Widget)w->menu.children[i])) {
	    MenuGadget	g = w->menu.children[i];

	    g->menu_g.highlighted = True;
	    XClearArea(XtDisplay(w), XtWindow(w),
		       g->rectangle.x, g->rectangle.y,
		       g->rectangle.width, g->rectangle.height, False);
	    if (g->object.widget_class->core_class.expose)
		g->object.widget_class->core_class.expose((Widget)g, NULL, 0);
	    if (MenuGadgetChangeHighlighted(g))
		MenuGadgetChangeHighlighted(g)(g);

	    w->menu.hl_index = i;
	} else {
	    w->menu.hl_index = -1;
	}
    }
}

static void unhighlight(Widget gw, XEvent *event,
			String *params, Cardinal *no_params)
{
    MenuWidget	w = (MenuWidget)gw;

    if (w->menu.hl_index >= 0) {
	MenuGadget	g = w->menu.children[w->menu.hl_index];

	if (MenuGadgetUnhighlightOnLeave(g)) {
	    g->menu_g.highlighted = False;
	    XClearArea(XtDisplay(w), XtWindow(w),
		       g->rectangle.x, g->rectangle.y,
		       g->rectangle.width, g->rectangle.height, False);
	    if (g->object.widget_class->core_class.expose)
		g->object.widget_class->core_class.expose((Widget)g, NULL, 0);
	    if (MenuGadgetChangeHighlighted(g))
		MenuGadgetChangeHighlighted(g)(g);
	    w->menu.hl_index = -1;
	}
    }
}

static void callback(Widget gw, XtPointer client_data, XtPointer call_data)
{
    MenuWidget	w = (MenuWidget)client_data;

    if (w->menu.saved_hl_index >= 0) {
	MenuGadget	g = w->menu.children[w->menu.saved_hl_index];

	if (MenuGadgetCallCallbacks(g))
	    MenuGadgetCallCallbacks(g)(g);
	w->menu.saved_hl_index = -1;
    }
}

static void layout(MenuWidget w)
{
    MenuGadget		*loop;
    int			n = w->menu.num_children;
    int			sw = w->shadow.shadow_width;
    int			i, y, width = 0;

    for (y = sw, i = 0, loop = w->menu.children ; i < n ; i++, loop++) {
	if (!XtIsSubclass((Widget)*loop, menuGadgetClass))
	    XtAppError(XtWidgetToApplicationContext((Widget)w),
		       "Child of MenuWidget must be "
		       "subclass of MenuGadgetClass.");

	y += (*loop)->rectangle.height;
	if (width < (int)(*loop)->rectangle.width)
	    width = (int)(*loop)->rectangle.width;
    }
    y += sw;
    w->menu.pref_width = (Dimension)(width + 2 * sw);
    w->menu.pref_height = (Dimension)y;
    (void)XtMakeResizeRequest((Widget)w, w->menu.pref_width,
			      w->menu.pref_height, NULL, NULL);

    for (y = sw, i = 0, loop = w->menu.children ; i < n ; i++, loop++) {
	XtConfigureWidget((Widget)*loop, sw, y,
			  width, (*loop)->rectangle.height, 0);
	y += (*loop)->rectangle.height;
    }
}

static void pre_popdown_callback(Widget gw,
				 XtPointer client_data,
				 XtPointer call_data)
{
    MenuWidget	w = (MenuWidget)client_data;

    if (w->menu.hl_index >= 0) {
	MenuGadget	g = w->menu.children[w->menu.hl_index];

	if (g->menu_g.highlighted) {
	    g->menu_g.highlighted = False;
	    if (MenuGadgetChangeHighlighted(g))
		MenuGadgetChangeHighlighted(g)(g);
	}
	w->menu.saved_hl_index = w->menu.hl_index;
	w->menu.hl_index = -1;
    } else {
	w->menu.saved_hl_index = -1;
    }
}

/*************************************************************************/

static void Initialize (Widget grequest, Widget gnew,
			ArgList args, Cardinal *num_args)
{
    MenuWidget	new = (MenuWidget)gnew;

    new->menu.children = NULL;
    new->menu.num_children = 0;
    new->menu.num_slots = 0;
    new->menu.hl_index = -1;
    XtAddCallback(XtParent(new), XtNprePopdownCallback,
		  pre_popdown_callback, (XtPointer)new);
    XtAddCallback(XtParent(new), XtNcallback,
		  callback, (XtPointer)new);
}

static void Destroy(Widget gw)
{
    MenuWidget	w = (MenuWidget)gw;
    int		i;

    for (i = 0 ; i < w->menu.num_children ; i++)
	XtDestroyWidget((Widget)w->menu.children[i]);
}

static void Resize (Widget gw)
{
    /* we never expect to be resized */
}

static void Redisplay(Widget gw, XEvent *event, Region region)
{
    MenuWidget	w = (MenuWidget)gw;
    MenuGadget	*loop = w->menu.children;
    int		n = w->menu.num_children;
    int		y1, y2;

    if (event && event->type == Expose) {
	y1 = event->xexpose.y;
	y2 = y1 + event->xexpose.height;
    } else if (event && event->type == GraphicsExpose) { /* won't happen... */
	y1 = event->xgraphicsexpose.y;
	y2 = y1 + event->xgraphicsexpose.height;
    } else {
	y1 = 0;
	y2 = w->menu.pref_height;
    }

    while (n > 0) {
	MenuGadget	g = *loop;

	if (g->rectangle.y <= y2 &&
	    y1 <= g->rectangle.y + (int)g->rectangle.height) {
	    if (g->object.widget_class->core_class.expose) {
		g->object.widget_class->core_class.expose((Widget)g,
							  event, region);
	    }
	}

	loop++;
	n--;
    }

    ShadowDrawShadows((ShadowWidget)w, 0, 0,
		      w->menu.pref_width, w->menu.pref_height, False);
}

static void Realize(Widget gw, XtValueMask *mask,
		    XSetWindowAttributes *attributes)
{
    MenuWidget	w = (MenuWidget)gw;

    if (w->menu.cursor != None) {
	*mask |= CWCursor;
	attributes->cursor = w->menu.cursor;
    }

    shadowWidgetClass->core_class.realize((Widget)w, mask, attributes);
}

static Boolean SetValues(Widget gcurrent, Widget grequest, Widget gnew,
			 ArgList args, Cardinal *num_args)
{
    Boolean	redisplay = False;
    MenuWidget	new = (MenuWidget)gnew;
    MenuWidget	current = (MenuWidget)gcurrent;

    if (XtIsRealized((Widget)new) &&
	new->menu.cursor != current->menu.cursor)
	XDefineCursor(XtDisplay(new), XtWindow(new), new->menu.cursor);

    return redisplay;
}

static XtGeometryResult QueryGeometry(Widget gw,
				      XtWidgetGeometry *intended,
				      XtWidgetGeometry *preferred)
{
    MenuWidget	w = (MenuWidget)gw;
    Dimension	intended_width, intended_height;

    if (intended->request_mode & CWWidth)
	intended_width = intended->width;
    else
	intended_width = w->core.width;
    if (intended->request_mode & CWHeight)
	intended_height = intended->height;
    else
	intended_height = w->core.height;

    preferred->request_mode = CWWidth | CWHeight;
    preferred->width = w->menu.pref_width;
    preferred->height = w->menu.pref_height;

    if (preferred->width == w->core.width &&
	preferred->height == w->core.height)
	return XtGeometryNo;
    else if (preferred->width == intended_width &&
	     preferred->height == intended_height)
	return XtGeometryYes;
    else
	return XtGeometryAlmost;
}

/*************************************************************************/

extern Widget MenuCreateGadget(String name, WidgetClass widget_class,
			       Widget parent, ArgList args, Cardinal num_args)
{
    MenuWidget	w = (MenuWidget)parent;
    MenuGadget	child;
    WidgetClass	loop = widget_class;

    while (loop && loop != menuGadgetClass)
	loop = loop->core_class.superclass;
    if (!loop)
	XtAppError(XtWidgetToApplicationContext((Widget)w),
		   "MenuWidgetClass accepts children "
		   "that are subclass of MenuGadgetClass only.");

    if (w->menu.num_children >= w->menu.num_slots) {
	w->menu.num_slots = 2 * (w->menu.num_slots + 1);
	w->menu.children =
	    (MenuGadget *)XtRealloc((char *)w->menu.children,
				    w->menu.num_slots * sizeof(MenuGadget));
    }

    child =
	(MenuGadget)XtCreateWidget(name, widget_class,
				   (Widget)w, args, num_args);
    w->menu.children[w->menu.num_children] = child;
    w->menu.num_children++;

    layout(w);

    return (Widget)child;
}
