/************************************************************************
 * rpncalc.h								*
 *									*
 * A little RPN (Reverse Polish Notation) calculator,                   *
 * rudimentary emulating a HP 28S  					*
 * 									*
 * rpncalc is (c) David Frey, 1993, 1994, 1995				*
 * 									*
 * This program is free software; you can redistribute it and/or modify *
 * it under the terms of the GNU General Public License as published by *
 * the Free Software Foundation; either version 2 of the License, or    *
 * (at your option) any later version.                                  *
 *									* 
 * This program is distributed in the hope that it will be useful,      *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 * GNU General Public License for more details.                         *
 *									* 
 * You should have received a copy of the GNU General Public License    *
 * along with this program; if not, write to the Free Software          *
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.            *
 *									* 
 ************************************************************************/

/* $Id: rpncalc.h,v 1.0 1995/12/31 18:18:10 david Rel $
 * $Log: rpncalc.h,v $
 * Revision 1.0  1995/12/31 18:18:10  david
 * Initial revision
 *
 * Revision 1.0  1995/12/31 18:17:22  david
 * Initial revision
 *
 * Revision 1.2  1995/11/25 21:59:51  david
 * Cleaned up the cmd table. The stack operations have now the right
 * value in the cmd->argno field; even at the expense of superfluous
 * push/pops, unfortunately.
 * */

#define LINEFRAGMENT 32		/* allocate 32 bytes at a time */

extern double push(double elem);
extern double pop(void);
extern double pick(double op1);
extern double drop(void);
extern double dupel(double d);
extern double dupel2(double d);
extern double dupn(double n, double d);
extern double dropn(double n);
extern double clear(void);
extern double over(void);
extern double depth(void);
extern double swap(void);
extern double roll(void);
extern double showstack(void);
extern double sethex(void);
extern double setdec(void);
extern double setoct(void);
extern double pi(void);
extern double e(void);
extern double chs(double f);
extern double sqr(double f);
extern double inv(double f);
extern double log2(double f);
extern double fact(double n);
extern double prec(double p);
extern double not(double l);
extern double plus(double s1, double s2);
extern double minus(double s, double m);
extern double times(double f1, double f2);
extern double divide(double n, double d);
extern double idiv(double n, double d);
extern double mod(double n, double d);
extern double gcd(double n, double d);
extern double and(double l1, double l2);
extern double or(double l1, double l2);
extern double xor(double l1, double l2);
extern double sum(void);
extern double prod(void);
extern double help(void);
extern double warranty(void);

/* The idea of the following was taken out of the Bison Manual p. 31 */
/* Sorting and lsearch out of iX 11/1995 */
struct cmd 
{
  char *fname;
  short argno;			/* 0..2: no of arguments */
  short pushrestostack;
  double  (*fnct)();
};

static struct cmd cmdtab[] =
{
  { "show"    , 0, 0, showstack },
  { "push"    , 1, 0, push      },
  { "pop"     , 0, 0, pop       },
  { "pick"    , 1, 1, pick      },
  { "swap"    , 0, 0, swap      },
  { "over"    , 0, 0, over      },
  { "roll"    , 0, 0, roll      },
  { "dup"     , 1, 0, dupel     },
  { "dup2"    , 1, 0, dupel2    },
  { "dupn"    , 2, 0, dupn      },
  { "drop"    , 0, 0, drop      },
  { "dropn"   , 1, 0, dropn     },
  { "depth"   , 0, 1, depth     },
  { "prec"    , 1, 0, prec      },
  { "clear"   , 0, 0, clear     },
  { "chs"     , 1, 1, chs       },
  { "+"       , 2, 1, plus      },
  { "-"       , 2, 1, minus     },
  { "*"       , 2, 1, times     },
  { "/"       , 2, 1, divide    },
  { "^"       , 2, 1, pow       },
  { "inv"     , 1, 1, inv       },
  { "sqrt"    , 1, 1, sqrt      }, 
  { "sqr"     , 1, 1, sqr       },
  { "sin"     , 1, 1, sin       },
  { "cos"     , 1, 1, cos       },
  { "tan"     , 1, 1, tan       },
  { "asin"    , 1, 1, asin      },
  { "acos"    , 1, 1, acos      },
  { "atan"    , 1, 1, atan      },
  { "atan2"   , 2, 1, atan2     },
  { "sinh"    , 1, 1, sinh      },
  { "cosh"    , 1, 1, cosh      },
  { "tanh"    , 1, 1, tanh      }, 
  { "asinh"   , 1, 1, asinh     },
  { "acosh"   , 1, 1, acosh     },
  { "atanh"   , 1, 1, atanh     },
  { "ln"      , 1, 1, log       },
  { "log"     , 1, 1, log10     },
  { "ld"      , 1, 1, log2      },
  { "exp"     , 1, 1, exp       },
  { "alog"    , 1, 1, pow10     },
  { "shl"     , 1, 1, pow2      },
  { "j0"      , 1, 1, j0        },
  { "j1"      , 1, 1, j1        },
  { "jn"      , 2, 1, jn        },
  { "y0"      , 1, 1, y0        },
  { "y1"      , 1, 1, y1        },
  { "yn"      , 2, 1, yn        },
  { "erf"     , 1, 1, erf       },
  { "erfc"    , 1, 1, erfc      },
  { "lgamma"  , 1, 1, lgamma    },
  { "abs"     , 1, 1, fabs      },
  { "ceil"    , 1, 1, ceil      },
  { "fact"    , 1, 1, fact      },
  { "mod"     , 2, 1, mod       },
  { "div"     , 2, 1, idiv      },
  { "gcd"     , 2, 1, gcd       },
  { "sum"     , 0, 1, sum       },	/* special case, takes all elements */
  { "prod"    , 0, 1, prod      },	/* special case, takes all elements */
  { "hex"     , 0, 0, sethex    },
  { "dec"     , 0, 0, setdec    },
  { "oct"     , 0, 0, setoct    },
  { "and"     , 2, 1, and       },
  { "&"       , 2, 1, and       },
  { "or"      , 2, 1, or        },
  { "|"       , 2, 1, or        },
  { "xor"     , 2, 1, xor       },	/* ^ for XOR would be irritating */
  { "not"     , 1, 1, not       },
  { "!"       , 1, 1, not       },
  { "pi"      , 0, 1, pi        },
  { "e"       , 0, 1, e         },
  { "help"    , 0, 0, help      },
  { "?"       , 0, 0, help      },
  { "warranty", 0, 0, warranty  },
};

#define NCMDTAB (sizeof cmdtab / sizeof cmdtab[0])
